/* -*- mode: C++; tab-width: 4 -*- */
// ===========================================================================
//	PPStream.cpp			   1993-1998 Metrowerks Inc. All rights reserved.
// ===========================================================================
//
//	Abstract class for reading/writing an ordered sequence of bytes

#include "EmulatorCommon.h"
#include "PPStream.h"

// const	uae_s32	length_NilBlock = -1;

#pragma mark --- Construction & Destruction ---

// ---------------------------------------------------------------------------
//	 PPStream
// ---------------------------------------------------------------------------
//	Default Constructor

PPStream::PPStream()
{
	mMarker = 0;
	mLength = 0;
}


// ---------------------------------------------------------------------------
//	 ~PPStream
// ---------------------------------------------------------------------------
//	Destructor

PPStream::~PPStream()
{
}

#pragma mark --- Accessors ---

// ---------------------------------------------------------------------------
//	 SetMarker
// ---------------------------------------------------------------------------
//	Place the Read/Write Marker at an offset from a specified position
//
//	inFromWhere can be streamFrom_Start, streamFrom_End, or streamFrom_Marker

void
PPStream::SetMarker(
	uae_s32		inOffset,
	StreamFromType	inFromWhere)
{
	uae_s32	newMarker = mMarker;

	switch (inFromWhere) {

		case kStreamFromStart:
			newMarker = inOffset;
			break;

		case kStreamFromEnd:
			newMarker = GetLength() - inOffset;
			break;

		case kStreamFromMarker:
			newMarker += inOffset;
			break;
	}

	if (newMarker < 0) {				// marker must be between 0 and
		newMarker = 0;					// Length, inclusive
	} else if (newMarker > GetLength()) {
		newMarker = GetLength();
	}

	mMarker = newMarker;
}


// ---------------------------------------------------------------------------
//	 GetMarker
// ---------------------------------------------------------------------------
//	Return the Read/Write Marker position
//
//	Position is a byte offset from the start of the Stream

uae_s32
PPStream::GetMarker() const
{
	return mMarker;
}


// ---------------------------------------------------------------------------
//	 SetLength
// ---------------------------------------------------------------------------
//	Set the length, in bytes, of the Stream

void
PPStream::SetLength(
	uae_s32	inLength)
{
	uae_s32	oldLength = GetLength();
	mLength = inLength;
										// If making Stream shorter, call
										//   SetMarker to make sure that
	if (oldLength > inLength) {			//   marker is not past the end
		SetMarker(GetMarker(), kStreamFromStart);
	}
}


// ---------------------------------------------------------------------------
//	 GetLength
// ---------------------------------------------------------------------------
//	Return the length, in bytes, of the Stream

uae_s32
PPStream::GetLength() const
{
	return mLength;
}

#pragma mark --- Low-Level I/O ---

// ---------------------------------------------------------------------------
//	 PutBytes
// ---------------------------------------------------------------------------
//	Write bytes from a buffer to a Stream
//
//	Returns an error code and passes back the number of bytes actually
//	written, which may be less than the number requested if an error occurred.
//
//	Subclasses must override this function to support writing.
//
//	NOTE: You should not throw an Exception out of this function.

ErrCode
PPStream::PutBytes(
	const void*	inBuffer,
	uae_s32		ioByteCount)
{
	UNUSED_PARAM(inBuffer)
	UNUSED_PARAM(ioByteCount)

	return 1;
}


// ---------------------------------------------------------------------------
//	 GetBytes
// ---------------------------------------------------------------------------
//	Read bytes from a Stream to a buffer
//
//	Returns an error code and passes back the number of bytes actually
//	read, which may be less than the number requested if an error occurred.
//
//	Subclasses must override this function to support reading.
//
//	NOTE: You should not throw an Exception out of this function.

ErrCode
PPStream::GetBytes(
	void*	outBuffer,
	uae_s32	ioByteCount)
{
	UNUSED_PARAM(outBuffer)
	UNUSED_PARAM(ioByteCount)

	return 1;
}


// ---------------------------------------------------------------------------
//	 PeekData
// ---------------------------------------------------------------------------
//	Read data from a Stream to a buffer, without moving the Marker
//
//	Return the number of bytes actually read, which may be less than the
//	number requested if an error occurred

uae_s32
PPStream::PeekData(
	void	*outBuffer,
	uae_s32	inByteCount)
{
	uae_s32	currentMarker = GetMarker();

	uae_s32	bytesToPeek = inByteCount;
	GetBytes(outBuffer, bytesToPeek);

	SetMarker(currentMarker, kStreamFromStart);

	return bytesToPeek;
}

#pragma mark --- High-Level I/O ---

// ---------------------------------------------------------------------------
//	 WriteCString
// ---------------------------------------------------------------------------
//	Write a C string to a Stream
//
//	Returns the number of bytes written.

uae_s32
PPStream::WriteCString(
	const char *inString)
{
	assert (inString);

	uae_s32		strLen = 0;				// Find length of C string
	const char	*s = inString;
	while (*s++ != 0) {
		strLen++;
	}

		// Write C string as a 4-byte count followed by the characters.
		// Do not write the null terminator.

	*this << strLen;
	PutBytes(inString, strLen);

	return (strLen + (uae_s32) sizeof(strLen));
}


// ---------------------------------------------------------------------------
//	 ReadCString
// ---------------------------------------------------------------------------
//	Read a C string from a Stream
//
//	Returns the number of bytes read

uae_s32
PPStream::ReadCString(
	char	*outString)
{
	assert (outString);

		// C string is stored as a 4-byte count followed by the
		// characters. The null terminator is not stored and must
		// be added afterwards.

	uae_s32	strLen;
	*this >> strLen;

	GetBytes(outString, strLen);
	outString[strLen] = '\0';			// Null terminator

	return (strLen + (uae_s32) sizeof(uae_s32));
}

// ---------------------------------------------------------------------------
//	 WriteString
// ---------------------------------------------------------------------------
//	Write a C string to a Stream
//
//	Returns the number of bytes written.

uae_s32
PPStream::WriteString(
	const string&	inString)
{
	return this->WriteCString (inString.c_str ());
}


// ---------------------------------------------------------------------------
//	 ReadString
// ---------------------------------------------------------------------------
//	Read a C string from a Stream
//
//	Returns the number of bytes read

uae_s32
PPStream::ReadString(
	string&			outString)
{
		// C string is stored as a 4-byte count followed by the
		// characters. The null terminator is not stored and must
		// be added afterwards.

	uae_s32	strLen;
	*this >> strLen;

	outString.resize (strLen);
	if (strLen > 0)
	{
		GetBytes(&outString[0], strLen);
	}

	return (strLen + (uae_s32) sizeof(uae_s32));
}
