#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <string.h>
#include <limits.h>
#include <misc.h>
#include <fviews.h>
#include <subsys.h>
#include <sstream.h>
#include <netconf.h>
#include <hostinfo.h>
#include "../../paths.h"
#include "redhat.h"
#include "redhat.m"

/*
	Parse a ifcfg-dev path and return the file name and eliminate
	all the special devices, aliases and point to point stuff
	Return the name of the script only if
*/
static const char *devinfo_filtername (const char *path)
{
	const char *ret = NULL;
	const char *fname = strrchr (path,'/');
	if (fname != NULL){
		fname++;
		char last = fname[strlen(fname)-1];
		// Avoid file ending with ~
		if (last != '~'
			&& strcmp(fname,"ifcfg-lo")!=0
			&& strchr(fname,':')==NULL
			&& strncmp(fname+6,"ppp",3) != 0
			&& strncmp(fname+6,"slip",4) != 0
			&& strncmp(fname+6,"plip",4) != 0){
			// Avoid file ending with .bak
			char *pt = strstr(fname,".bak");
			if (pt == NULL || pt[4] != '\0'){
				ret = fname;
			}
		}
	}
	return ret;
}
const char scripts_path[]="/etc/sysconfig/network-scripts/ifcfg-";
/*
	Extract the list of ifcfg files related to LAN devices
*/
static int devinfo_getlist(SSTRINGS &tb)
{
	dir_getlist_p (scripts_path,tb);
	// Remove the ifcfg file which are not LAN devices
	for (int i=0; i<tb.getnb(); i++){
		SSTRING *s = tb.getitem(i);
		if (devinfo_filtername (s->get()) == NULL){
			tb.remove_del (s);
			i--;
		}
	}
	tb.sort();
	return tb.getnb();
}
static const char DEVICE[]="DEVICE";
static const char IPADDR[]="IPADDR";
static const char NETMASK[]="NETMASK";
static const char NETWORK[]="NETWORK";
static const char BROADCAST[]="BROADCAST";
static const char BOOTPROTO[]="BOOTPROTO";
static const char ONBOOT[]="ONBOOT";
static const char PCMCIA[]="PCMCIA";
static const char YES[]="yes";
static const char NO[]="no";


static const char *tbipxframe[]={
	"802_2","802_3","ETHERII","SNAP"
};
static const char K_IPXNETNUM[]="IPXNETNUM";
static const char K_IPXPRIMARY[]="IPXPRIMARY";
static const char K_IPXACTIVE[]="IPXACTIVE";

static void devinfo_setipxvars (
	int noframe,
	char varnetnum[20],
	char varprimary[20],
	char varactive[20])
{
	sprintf (varnetnum,"%s_%s",K_IPXNETNUM,tbipxframe[noframe]);
	sprintf (varprimary,"%s_%s",K_IPXPRIMARY,tbipxframe[noframe]);
	sprintf (varactive,"%s_%s",K_IPXACTIVE,tbipxframe[noframe]);
}


static void devinfo_ipx_loadinter (VIEWITEMS &items, IPX_INTER_INFO &itf)
{
	for (int f=0; f<NB_IPX_FRAME_TYPE; f++){
		IPX_FRAME_INFO *fra = &itf.frames[f];
		char varnetnum[20],varprimary[20],varactive[20];
		devinfo_setipxvars (f,varnetnum,varprimary,varactive);
		fra->netnum = items.locatehval (varnetnum);
		fra->primary = items.locatebval (varprimary);
		fra->active = items.locatebval (varactive);
	}
}

int devinfo_load(HOSTS &hosts, HOSTINFO &info)
{
	SSTRINGS tb;
	int n = devinfo_getlist (tb);
	int nbdev = 0;
	char hostname[PATH_MAX];
	filter_gethostname (hostname);
	info.hostname.setfrom (hostname);
	for (int i=0; i<n; i++){
		const char *path = tb.getitem(i)->get();
		VIEWITEMS items;
		CONFIG_FILE file(path,help_nil
			,CONFIGF_MANAGED|CONFIGF_OPTIONNAL
			,"root","root",0755);
		if (items.read (file) != -1){
			char tmp[1000];
			const char *ipnum = items.locateval(IPADDR,tmp);
			INTER_INFO *itf = &info.a[nbdev++];
			itf->ipaddr.setfrom (ipnum);
			itf->device.setfrom (items.locateval(DEVICE,tmp));
			itf->netmask.setfrom (items.locateval(NETMASK,tmp));
			/* #Specification: redhat ifcfg file / network and bcast
				The files /etc/sysconfig/network-scripts/ifcfg-DEV (eth0,...)
				had originally NETWORK and BROADCAST field. Linuxconf
				never update those fields and computes the value
				using the IP address and the netmask. So those
				value are ignored. Some redhat utilities still
				write them (the installation I think).
			*/
			//itf->network.setfrom (items.locateval(NETWORK,tmp));
			//itf->bcast.setfrom (items.locateval(BROADCAST,tmp));
			const char *proto = items.locateval(BOOTPROTO,tmp);
			if (proto == NULL || strcmp(proto,"none")==0){
				itf->confmode = INTER_MANUAL;
			}else if (strcmp(proto,"bootp")==0){
				itf->confmode = INTER_BOOTP;
			}else if (strcmp(proto,"dhcp")==0){
				itf->confmode = INTER_DHCP;
			}
			itf->enable = items.locatebval(ONBOOT);
			itf->pcmcia = items.locatebval(PCMCIA);
			devinfo_ipx_loadinter (items,itf->ipx);
		}
	}
	info.nbdev = nbdev;
	for (int j=0; j<info.nbdev; j++){
		INTER_INFO *pti = &info.a[j];
		HOST *hst = hosts.getbyip (pti->ipaddr.get());
		if (hst != NULL){
			pti->name.setfrom (hst->getname1());
			pti->others.setfrom (hst->getothers());
		}
	}
	return nbdev;
}

static void devinfo_ipx_saveinter (VIEWITEMS &items, IPX_INTER_INFO &itf)
{
	for (int f=0; f<NB_IPX_FRAME_TYPE; f++){
		IPX_FRAME_INFO *fra = &itf.frames[f];
		char varnetnum[20],varprimary[20],varactive[20];
		devinfo_setipxvars (f,varnetnum,varprimary,varactive);
		if (fra->netnum == 0){
			items.update (varnetnum,"");
		}else{
			items.updatehval (varnetnum,fra->netnum);
		}
		items.updatebval (varprimary,fra->primary);
		items.updatebval (varactive,fra->active);
	}
}

void devinfo_remove (
	VIEWITEMS &items,
	const char *var)
{
	VIEWITEM *it = items.locateassign(var);
	items.remove_del (it);
}


int devinfo_save(HOSTS &hosts, HOSTINFO &info)
{
	int ret = 0;
	SSTRINGS tb;
	int n = devinfo_getlist (tb);
	// First, remove all ifcfg-device which are not configured any more
	for (int i=0; i<n; i++){
		const char *path = tb.getitem(i)->get();
		VIEWITEMS items;
		CONFIG_FILE file(path,help_nil
			,CONFIGF_MANAGED|CONFIGF_OPTIONNAL
			,"root","root",0755);
		if (items.read (file) != -1){
			char tmp[1000];
			const char *dev = items.locateval(DEVICE,tmp);
			bool found = false;
			for (int d=0; d<NB_ETH; d++){
				INTER_INFO *itf = &info.a[d];
				if ((!itf->ipaddr.is_empty()	|| itf->confmode != INTER_MANUAL)
					&& itf->device.cmp(dev)==0){
					found = true;
					break;
				}
			}
			if (!found){
				file.unlink ();
			}
		}
	}
	// Now update or create the information
	for (int d=0; d<NB_ETH; d++){
		INTER_INFO *itf = &info.a[d];
		if (!itf->device.is_empty()){
			const char *ipaddr = itf->ipaddr.get();
			bool validip = ipaddr[0] != '\0';
			if (validip	|| itf->confmode != INTER_MANUAL){
				char path[PATH_MAX];
				sprintf (path,"%s%s",scripts_path,itf->device.get());
				VIEWITEMS items;
				CONFIG_FILE file(path,help_nil
					,CONFIGF_MANAGED|CONFIGF_OPTIONNAL
					,"root","root",0755);
				if (!file.exist() || items.read (file) != -1){
					char tmp[1000];
					const char *old_ipaddr = items.locateval(IPADDR,tmp);
					if (old_ipaddr == NULL
						|| strcmp(old_ipaddr, ipaddr) != 0) {
						// if IP address has changed, remove obsolete items
						devinfo_remove (items,BROADCAST);
						devinfo_remove (items,NETWORK);
					}
					items.update (DEVICE,itf->device);
					items.update (IPADDR,ipaddr);
					items.update (NETMASK,itf->netmask);
					items.update (ONBOOT,itf->enable ? YES : NO);
					static const char *tbproto[]={
						"none","dhcp","bootp"
					};
					items.update (BOOTPROTO,tbproto[itf->confmode]);
					devinfo_ipx_saveinter (items,itf->ipx);
					ret |= items.write (file,NULL);
				}
			}
			HOST *hst = NULL;
			const char *name = itf->name.get();
			if (name[0] != '\0'){
				hst = hosts.getitem (name);
			}
			const char *others = itf->others.get();
			if (hst == NULL && others[0] != '\0'){
				char first[PATH_MAX];
				str_copyword (first,others,PATH_MAX-1);
				hst = hosts.getitem (first);
			}
			if (validip){
				if (hst == NULL){
					hst = hosts.getbyip (ipaddr);
				}
				if (name[0] != '\0' || others[0] != '\0'){
					if (hst == NULL){
						hst = new HOST;
						hosts.add (hst);
					}
					hst->setipnum (ipaddr);
					hst->setname1 (name);
					hst->setothers (others);
				}else if (hst != NULL){
					hosts.remove_del (hst);
				}
			}else if (hst != NULL){
				// No IP number supplied
				hosts.remove_del (hst);
			}
		}
	}
	hosts.write ();
	filter_sethostname (info.hostname.get());
	return ret;
}

class CONFIG_FILE_DEVLIST: public CONFIG_FILE{
	/*~PROTOBEG~ CONFIG_FILE_DEVLIST */
public:
	CONFIG_FILE_DEVLIST (void);
	int archive (SSTREAM&ss)const;
	int extract (SSTREAM&ss);
	/*~PROTOEND~ CONFIG_FILE_DEVLIST */
};

PUBLIC CONFIG_FILE_DEVLIST::CONFIG_FILE_DEVLIST()
	: CONFIG_FILE ("/etc/sysconfig/landev.list",help_nil,CONFIGF_VIRTUAL
		,subsys_stationid)
{
}

PUBLIC int CONFIG_FILE_DEVLIST::archive(SSTREAM &ss) const
{
	configf_sendexist (ss,true);
	SSTRINGS tb;
	int n = devinfo_getlist (tb);
	for (int i=0; i<n; i++){
		const char *path = tb.getitem(i)->get();
		ss.printf ("%s\n",path);
	}
	return 0;
}

PUBLIC int CONFIG_FILE_DEVLIST::extract(SSTREAM &ss)
{
	SSTRINGS tbold;
	devinfo_getlist (tbold);
	SSTRINGS tb;
	char path[PATH_MAX];
	while (ss.gets(path,sizeof(path)-1) != NULL){
		strip_end (path);
		tb.add (new SSTRING (path));
	}
	for (int i=0; i<tb.getnb(); i++){
		const char *path = tb.getitem(i)->get();
		CONFIG_FILE conf (path,help_nil,CONFIGF_MANAGED|CONFIGF_OPTIONNAL
			,subsys_stationid);
		conf.extract();
	}
	// Erase the config files which were not part of the archive
	for (int j=0; j<tbold.getnb(); j++){
		const char *path = tbold.getitem(j)->get();
		if (tb.lookup(path)==-1){
			net_prtlog (NETLOG_CMD,MSG_U(I_REMOVING,"Removing device configuration: %s\n")
				,path);
			::unlink (path);
		}
	}
	return 0;
}

static CONFIG_FILE_DEVLIST devlist;

static void devinfo_listcfg()
{
	SSTRINGS lst;
	int n = devinfo_getlist (lst);
	for (int i=0; i<n; i++){
		new CONFIG_FILE (lst.getitem(i)->get(),help_nil
			,CONFIGF_MANAGED|CONFIGF_OPTIONNAL
			,subsys_stationid);
	}
}

static CONFIG_FILE_LISTER lister (devinfo_listcfg);

/*
	Extract the list of ifcfg alias files related to one LAN devices
*/
static int devinfo_ipalias_getlist(const char *devname, SSTRINGS &tb)
{
	char path[PATH_MAX];
	sprintf (path,"%s%s:",scripts_path,devname);
	dir_getlist_p (path,tb);
	tb.sort();
	return tb.getnb();
}


/*
	Read all the ifcfg-xxx:y file and store the IP address in tb
	Return the number of items collected.
*/
int devinfo_ipalias_load (
	const char *devname,
	SSTRINGS &tb,
	SSTRINGS &masks)
{
	SSTRINGS tbf;
	int n = devinfo_ipalias_getlist (devname,tbf);
	for (int i=0; i<n; i++){
		const char *path = tbf.getitem(i)->get();
		CONFIG_FILE conf (path,help_nil,CONFIGF_MANAGED|CONFIGF_OPTIONNAL
			,subsys_stationid);
		VIEWITEMS items;
		if (items.read (conf) != -1){
			char tmp[1000];
			const char *ip = items.locateval (IPADDR,tmp);
			tb.add (new SSTRING(ip));
			const char *mk = items.locateval (NETMASK,tmp);
			masks.add (new SSTRING(mk));
		}
	}
	return tb.getnb();
}

/*
	Update the ifcfg-xxx:y file
*/
int devinfo_ipalias_save (
	const char *devname,
	const SSTRINGS &tb,
	const SSTRINGS &masks)
{
	int ret = 0;
	SSTRINGS tbf;
	devinfo_ipalias_getlist (devname,tbf);
	int i;
	for (i=0; i<tb.getnb() && ret != -1; i++){
		char path[PATH_MAX];
		sprintf (path,"%s%s:%d",scripts_path,devname,i);
		CONFIG_FILE conf (path,help_nil,CONFIGF_MANAGED|CONFIGF_OPTIONNAL
			,subsys_stationid);
		VIEWITEMS items;
		items.read (conf);
		items.update (IPADDR,tb.getitem(i)->get());
		items.update (NETMASK,masks.getitem(i)->get());
		ret = items.write (conf,NULL);
		int lk = tbf.lookup (path);
		if (lk != -1) tbf.remove_del (lk);
	}
	if (ret != -1){
		// Remove all extra ifcfg-xxx:y file
		for (i=0; i<tbf.getnb(); i++) unlink (tbf.getitem(i)->get());
	}
	return ret;
}

