# copyright (C) 1997-1999 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: cpustats.tcl,v 1.23 1999/09/15 21:18:57 jfontain Exp $}


package provide cpustats 2.0
package require network 1.3

namespace eval cpustats {

    array set data {
        updates 0
        0,label CPU 0,type ascii 0,message {CPU number (0 for all CPUs)}
        1,label user 1,type real 1,message {percentage spent in user mode}
        2,label system 2,type real 2,message {percentage spent in system mode}
        3,label nice 3,type real 3,message {percentage spent in nice mode}
        4,label idle 4,type real 4,message {percentage spent in idle mode}
        sort {0 increasing}
        switches {-r 1 --remote 1}
    }
    set file [open [file join $::packageDirectory(cpustats) cpustats.htm]]
    set data(helpText) [read $file]                                                           ;# initialize HTML help data from file
    close $file

    proc initialize {optionsName} {
        upvar $optionsName options
        variable remote
        variable data
        variable statisticsFile

        set lookup [expr {![info exists options(-n)]&&![info exists options(--numeric)]}]            ;# host or network names lookup
        if {![catch {set locator $options(--remote)}]||![catch {set locator $options(-r)}]} {                   ;# remote monitoring
            set data(pollTimes) {20 10 30 60 120 300 600}                                ;# poll less often when remotely monitoring
            foreach {remote(user) remote(host)} [network::parseRemoteLocator $locator] {}
            network::checkRemoteOutputEmptiness $remote(host)
            set data(identifier) cpustats($remote(host))
            set file [open "| /usr/bin/rsh -nl $remote(user) $remote(host) cat /proc/stat"]
            fileevent $file readable {set ::cpustats::remote(busy) 0}
            vwait ::cpustats::remote(busy)
            if {[catch {close $file} message]} {
                puts stderr "on remote host $remote(host) as user $remote(user): $message"
                exit 1                                                                                        ;# detect errors early
            }
        } else {
            set data(pollTimes) {10 5 20 30 60 120 300 600}
            set statisticsFile [open /proc/stat]                                      ;# keep local file open for better performance
        }
    }

    set data(0,0) {}                                                                                                 ;# for all CPUs
    array set last {user 0 nice 0 system 0 idle 0}                                                    ;# stored values for last poll

    proc update {} {                                               ;# gather cpu statistics (based on the proc man page information)
        variable remote
        variable statisticsFile
        variable data

        if {[info exists remote]} {
            if {![info exists statisticsFile]} {                           ;# start data gathering process in a non blocking fashion
                if {$remote(busy)} return                                           ;# core invocation while waiting for remote data
                set remote(busy) 1
                set file [open "| /usr/bin/rsh -nl $remote(user) $remote(host) cat /proc/stat"]
                # do not hang GUI, allow other modules updates
                fileevent $file readable "set ::cpustats::statisticsFile $file; ::cpustats::update"
                return                                                                                       ;# wait for remote data
            }                                                                                 ;# else continue below to process data
        } else {
            seek $statisticsFile 0                                                                  ;# rewind before retrieving data
        }
        set index 0                                                     ;# first line is for all CPUs, data is in 100ths of a second
        scan [gets $statisticsFile] {cpu %d %d %d %d} user nice system idle
        updateRow $index $user $nice $system $idle
        while {[scan [gets $statisticsFile] {cpu%*u %d %d %d %d} user nice system idle]==4} {
            updateRow [incr index] $user $nice $system $idle
        }
        if {[info exists remote]} {
            close $statisticsFile                                                   ;# closing is necessary since seek does not work
            unset statisticsFile
            set remote(busy) 0
        }
        incr data(updates)
    }

    proc updateRow {index user nice system idle} {
        variable last
        variable data

        if {![info exists last($index,user)]} {        ;# calculate statistics during the last poll period except for the first pass
            set data($index,0) $index
            set last($index,user) 0
            set last($index,system) 0
            set last($index,nice) 0
            set last($index,idle) 0
        }
        set multiplier [expr {100.0/($user+$nice+$system+$idle)}]
        array set data [list\
            $index,1 [format %.1f [expr {$multiplier*($user-$last($index,user))}]]\
            $index,2 [format %.1f [expr {$multiplier*($system-$last($index,system))}]]\
            $index,3 [format %.1f [expr {$multiplier*($nice-$last($index,nice))}]]\
            $index,4 [format %.1f [expr {$multiplier*($idle-$last($index,idle))}]]\
        ]
        set last($index,user) $user
        set last($index,system) $system
        set last($index,nice) $nice
        set last($index,idle) $idle
    }

}
