#------------------------------------------------------------------------------
# Contains Wcb procedures for entry widgets.
#
# Copyright (c) 1999  Csaba Nemethi (E-mail: csaba.nemethi@t-online.de)
#------------------------------------------------------------------------------

#
# Simple before-insert callback routines for entry widgets
# ========================================================
#

#------------------------------------------------------------------------------
# wcb::checkStrForAlpha
#
# Checks whether the string str to be inserted into the entry widget w is
# alphabetic; if not, it cancels the insert operation.
#------------------------------------------------------------------------------
proc wcb::checkStrForAlpha {w idx str} {
    if {![regexp -nocase {^[a-z]*$} $str]} {
	cancel
    }
}

#------------------------------------------------------------------------------
# wcb::checkStrForNum
#
# Checks whether the string str to be inserted into the entry widget w is
# numeric; if not, it cancels the insert operation.
#------------------------------------------------------------------------------
proc wcb::checkStrForNum {w idx str} {
    if {![regexp {^[0-9]*$} $str]} {
	cancel
    }
}

#------------------------------------------------------------------------------
# wcb::checkStrForAlnum
#
# Checks whether the string str to be inserted into the entry widget w is
# alphanumeric; if not, it cancels the insert operation.
#------------------------------------------------------------------------------
proc wcb::checkStrForAlnum {w idx str} {
    if {![regexp -nocase {^[0-9a-z]*$} $str]} {
	cancel
    }
}

#------------------------------------------------------------------------------
# wcb::convStrToUpper
#
# Replaces the string str to be inserted into the entry widget w with its
# uppercase equivalent.
#------------------------------------------------------------------------------
proc wcb::convStrToUpper {w idx str} {
    replace 1 1 [string toupper $str]
    return ""
}

#------------------------------------------------------------------------------
# wcb::convStrToLower
#
# Replaces the string str to be inserted into the entry widget w with its
# lowercase equivalent.
#------------------------------------------------------------------------------
proc wcb::convStrToLower {w idx str} {
    replace 1 1 [string tolower $str]
    return ""
}

#
# Helper procs needed in some before-insert callback routines for entry widgets
# =============================================================================
#

#------------------------------------------------------------------------------
# wcb::postInsertEntryText
#
# Returns the text that would be contained in the entry widget w after
# inserting the string str before the character indicated by the index idx.
#------------------------------------------------------------------------------
proc wcb::postInsertEntryText {w idx str} {
    set oldText [$w get]
    set numIdx [$w index $idx]

    append newText [string range $oldText 0 [expr $numIdx - 1]] \
		   $str \
		   [string range $oldText $numIdx end]
    return $newText
}

#------------------------------------------------------------------------------
# wcb::postInsertEntryLen
#
# Returns the length of the text that would be contained in the entry widget w
# after inserting the string str.
#------------------------------------------------------------------------------
proc wcb::postInsertEntryLen {w str} {
    return [expr [$w index end] + [string length $str]]
}

#
# Further before-insert callback routines for entry widgets
# =========================================================
#

#------------------------------------------------------------------------------
# wcb::checkEntryForInt
#
# Checks whether the text contained in the entry widget w after inserting the
# string str before the character indicated by the index idx would represent
# (the starting part of) an integer number; if not, it cancels the insert
# operation.
#------------------------------------------------------------------------------
proc wcb::checkEntryForInt {w idx str} {
    set newText [postInsertEntryText $w $idx $str]
    if {![regexp {^[+-]?[0-9]*$} $newText]} {
	cancel
    }
}

#------------------------------------------------------------------------------
# wcb::checkEntryForReal
#
# Checks whether the text contained in the entry widget w after inserting the
# string str before the character indicated by the index idx would represent
# (the starting part of) a real number; if not, it cancels the insert
# operation.
#------------------------------------------------------------------------------
proc wcb::checkEntryForReal {w idx str} {
    set newText [postInsertEntryText $w $idx $str]
    if {![regexp {^[+-]?[0-9]*\.?[0-9]*([0-9]\.?[eE][+-]?[0-9]*)?$} $newText]} {
	cancel
    }
}

#------------------------------------------------------------------------------
# wcb::checkEntryLen
#
# Checks whether the length of the text contained in the entry widget w after
# inserting the string str would be greater than len; if yes, it cancels the
# insert operation.
#------------------------------------------------------------------------------
proc wcb::checkEntryLen {len w idx str} {
    if {[postInsertEntryLen $w $str] > $len} {
	cancel
    }
}

#
# Private procedure
# =================
#

#------------------------------------------------------------------------------
# wcb::buildNewEntryCmd
#
# Constructs a new command procedure for the entry widget w.  In the new
# procedure, the execution of the commands insert, delete, and icursor is
# preceded by calls to the corresponding before-callbacks and followed by
# calls to the corresponding after-callbacks, in the global scope.
#------------------------------------------------------------------------------
proc wcb::buildNewEntryCmd w {
    set W [list $w]
    set orig [list _$w]

    proc ::$w {option args} [format {
	set opLen [string length $option]

	if {[string match $option* insert] && $opLen >= 3} {
	    if {[llength $args] == 2} {
		return [eval {wcb::processCmd %s insert insert} $args]
	    } else {
		# Let Tk report the error
		return [eval {%s $option} $args]
	    }

	} elseif {[string match $option* delete] && $opLen >= 1} {
	    set len [llength $args]
	    if {$len == 1 || $len == 2} {
		return [eval {wcb::processCmd %s delete delete} $args]
	    } else {
		# Let Tk report the error
		return [eval {%s $option} $args]
	    }

	} elseif {[string match $option* icursor] && $opLen >= 2} {
	    if {[llength $args] == 1} {
		return [eval {wcb::processCmd %s motion icursor} $args]
	    } else {
		# Let Tk report the error
		return [eval {%s $option} $args]
	    }

	} else {
	    return [eval {%s $option} $args]
	}
    } $W $orig  $W $orig  $W $orig  $orig]
}
