/* $Header: /cvs/gnome/gIDE/src/gI_search.c,v 1.5 1999/12/05 19:10:48 jpr Exp $ */
/* gIDE
 * Copyright (C) 1998-2000 Steffen Kern
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>
#include <gnome.h>
#include <dirent.h>
#include <fnmatch.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "structs.h"

#ifdef HAVE_GNU_REGEXP
#include <regex.h>
#endif
#ifdef HAVE_GTKTEXT_PATCH
#include <gtksctext.h>
#endif

#include "gI_document.h"
#include "gI_compile.h"
#include "gI_window.h"
#include "gI_common.h"
#include "gI_search.h"


/* globals */
static GtkWidget *search_window = NULL;
static GtkWidget *search_entry;
static GtkWidget *replace_entry;
static glong kz_regexp = 0;
static glong kz_case = 0;
static glong kz_cursor = 0;
static glong kz_again = 0;
static gchar search_str[256];
static gchar replace_str[256];


/* externs */
extern gI_window *main_window;


static void search_window_destroy( GtkWidget *widget, gpointer data)
{
	gtk_widget_destroy( search_window );
	search_window = NULL;
}


static void _goto_line( gchar *line, gpointer data )
{
	gint _line;
	gI_document *current;

	if( !line )
	{
		return;
	}

	gI_window_clear_statusbar( main_window );

	current = gI_document_get_current( main_window );
	g_return_if_fail( current != NULL );

	_line = atoi( line );

	if( ( _line <= 0 ) || ( get_point_from_line( _line ) > gI_text_get_length( current->text ) ) )
	{
		/* error */
		gI_error_dialog( _("Invalid Line!") );
		return;
	}

	goto_line( _line );
}


/*
 ---------------------------------------------------------------------
     Function: search_goto_line()
     Desc: Callback-Function /Search/Goto Line
 ---------------------------------------------------------------------
*/
void search_goto_line( GtkWidget *widget, gpointer data )
{
	GtkWidget *dlg;
	gI_document *current = gI_document_get_current( main_window );

	if( !current )
	{
		return;
	}

	dlg = gnome_request_dialog( FALSE,
	                            _("Please enter the number of the line"),
	                            "1",
	                            10,
	                            _goto_line,
	                            NULL,
	                            NULL );

	gtk_window_set_title( GTK_WINDOW( dlg ), _("Goto Line...") );
	gtk_window_set_modal( GTK_WINDOW( dlg ), TRUE );

	gnome_dialog_run_and_close( GNOME_DIALOG( dlg ) );
}


static void _search_search( GtkWidget *widget, gpointer data )
{
	gchar *buf;
	gI_document *current;
	gint i;
	glong len,text_length;
	glong hit = 0;
	gchar search_result[STRLEN];
	glong found = 0;
#ifdef HAVE_GNU_REGEXP
	gchar errbuf[256];
	gchar errmsg[300];
	regex_t buffer;
	glong compile;
	gchar *text;
	regmatch_t regs[1];
	glong match;
#endif

	if(widget)  /* if not trying search again? */
		history_add_to_history(main_window->search_history,GTK_COMBO(search_entry));

	gI_window_clear_statusbar( main_window );

	current = gI_document_get_current( main_window );

	if( !kz_again )
	{
		gchar *search_p = gtk_entry_get_text( GTK_ENTRY( GTK_COMBO( search_entry )->entry ));

		g_return_if_fail( search_p != NULL );

		strcpy( search_str, gtk_entry_get_text(GTK_ENTRY( GTK_COMBO( search_entry )->entry )) );

		search_window_destroy( NULL, NULL );
	}

	buf = (gchar *) g_malloc0( sizeof( search_str ) );
	len = strlen( search_str );
	text_length = gI_text_get_length( current->text );

	if( !kz_cursor )
		i = 0;
	else
		i = gI_text_get_point( current->text );

	if( kz_regexp )
	{
#ifdef HAVE_GNU_REGEXP
		compile = regcomp( &buffer, search_str, REG_EXTENDED );
		if( compile != 0 )
		{
			regerror( compile, &buffer, errbuf, 256 );
			sprintf( errmsg, _("Regex Compile Error: %s"), errbuf );
			gI_error_dialog( errmsg );
			return;
		}

		/* the stuff below needs too much memory, find a better way */
		text = gtk_editable_get_chars( GTK_EDITABLE( current->text ), 0, text_length );
		match = regexec( &buffer, text, 1, regs, 0 );
		if( match != 0 )
		{
			g_snprintf( search_result, STRLEN, _("\n    [%s] not found!    \n"), search_str );
			gI_error_dialog( search_result);
			g_free( text );
		}
		else
		{
			/* scroll... */
			__goto_point( regs[0].rm_so );
			gtk_editable_select_region( GTK_EDITABLE( current->text ), regs[0].rm_so, regs[0].rm_eo );
			gI_text_set_point( current->text, regs[0].rm_eo+1 );
			regfree( &buffer );
			g_free( text );
		}
#else
		gI_error_dialog( _("Regular Expression Search only supported with the GNU regexp lib") );
#endif
		return;
	}

	gI_text_freeze( current->text );

	for(;i<=(text_length-len);i++)
	{
		buf = gtk_editable_get_chars( GTK_EDITABLE( current->text ), i, i+len );
		if( kz_case )
			hit = strcmp( buf, search_str );
		else
			hit = strcasecmp( buf, search_str );

		if( hit == 0 )
		{
			found = TRUE;

			gI_text_thaw( current->text );

			/* scroll... */
			__goto_point( i );
			gtk_editable_select_region( GTK_EDITABLE( current->text ), i, i+len );
			gI_text_set_point( current->text, i+len+1 );
			break;
		}
		g_free( buf );
	}

	if( !found )
	{
		g_snprintf( search_result, STRLEN, _("\n    [%s] not found!    \n"), search_str );
		gI_error_dialog( search_result);
	}
	else
	{
	}

	kz_again = FALSE;
}


static void cb_case_clicked( GtkWidget *widget, gpointer data )
{
	if( kz_case )
		kz_case = FALSE;
	else
		kz_case = TRUE;
}


static void cb_regexp_clicked( GtkWidget *widget, gpointer data )
{
	if( kz_regexp )
		kz_regexp = FALSE;
	else
		kz_regexp = TRUE;
}

/*
 ---------------------------------------------------------------------
     Function: search_search()
     Desc: Callback-Function /Search/Search
 ---------------------------------------------------------------------
*/

void search_search( GtkWidget *widget, gpointer data )
{
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *hsep;
	GtkWidget *button;
	GtkWidget *cb_case;
	GtkWidget *cb_regexp;

	if( search_window )
		return;

	kz_case = 0;
	kz_cursor = 0;
	kz_regexp = 0;

	search_window = gtk_window_new( GTK_WINDOW_TOPLEVEL );
	gtk_widget_set_usize( search_window, 300, 125 );
	gtk_window_set_title( GTK_WINDOW( search_window ), _("Find...") );
	gtk_signal_connect( GTK_OBJECT( search_window ), "destroy",
	                    GTK_SIGNAL_FUNC( search_window_destroy ), NULL );

	vbox = gtk_vbox_new( FALSE, 0 );
	gtk_container_add( GTK_CONTAINER( search_window ), vbox );
	gtk_widget_show( vbox );

	hbox = gtk_hbox_new( FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 10 );
	gtk_widget_show( hbox );

	label = gtk_label_new( _("Search for") );
	gtk_box_pack_start( GTK_BOX( hbox ), label, FALSE, TRUE, 5 );
	gtk_widget_show( label );

	search_entry = gtk_combo_new( );
	gtk_box_pack_start( GTK_BOX( hbox ), search_entry, TRUE, TRUE, 5 );
	gtk_signal_connect( GTK_OBJECT(GTK_COMBO( search_entry)->entry) , "activate",
	                    GTK_SIGNAL_FUNC( _search_search ), NULL );
	gtk_widget_show( search_entry );

	if(!main_window->search_history)
		main_window->search_history=g_list_alloc();
	else
		history_add_to_combo(main_window->search_history,GTK_COMBO(search_entry));

	hbox = gtk_hbox_new( FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 5 );
	gtk_widget_show( hbox );

	cb_case = gtk_check_button_new_with_label( _("case sensitive") );
	gtk_box_pack_start( GTK_BOX( hbox ), cb_case, TRUE, TRUE, 5 );
	gtk_signal_connect( GTK_OBJECT( cb_case ), "clicked",
	                    GTK_SIGNAL_FUNC( cb_case_clicked ), NULL );
	gtk_widget_show( cb_case );

	cb_regexp = gtk_check_button_new_with_label( _("regular expression") );
	gtk_box_pack_start( GTK_BOX( hbox ), cb_regexp, TRUE, TRUE, 5 );
	gtk_signal_connect( GTK_OBJECT( cb_regexp ), "clicked",
	                    GTK_SIGNAL_FUNC( cb_regexp_clicked ), NULL );
	gtk_widget_show( cb_regexp );

	hsep = gtk_hseparator_new();
	gtk_box_pack_start( GTK_BOX( vbox ), hsep, TRUE, TRUE, 5 );
	gtk_widget_show( hsep );

	hbox = gtk_hbox_new( FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 5 );
	gtk_widget_show( hbox );

	button = gnome_stock_button( GNOME_STOCK_BUTTON_OK );
	gtk_box_pack_start( GTK_BOX( hbox ), button, TRUE, TRUE, 5 );
	gtk_signal_connect( GTK_OBJECT( button ), "clicked",
	                    GTK_SIGNAL_FUNC( _search_search ), NULL );
	GTK_WIDGET_SET_FLAGS( button, GTK_CAN_DEFAULT );
	gtk_widget_grab_default( button );
	gtk_widget_show( button );

	button = gnome_stock_button( GNOME_STOCK_BUTTON_CANCEL );
	gtk_box_pack_end( GTK_BOX( hbox ), button, TRUE, TRUE, 5 );
	gtk_signal_connect( GTK_OBJECT( button ), "clicked",
	                    GTK_SIGNAL_FUNC( search_window_destroy ), NULL );
	gtk_widget_show( button );

	if( search_str )
		gtk_entry_set_text( GTK_ENTRY(GTK_COMBO( search_entry )->entry), search_str );

	gtk_widget_grab_focus( search_entry );

	gtk_widget_show( search_window );
}


static void _search_replace( GtkWidget *widget, gpointer data )
{
	gchar *buf;
	gI_document *current;
	gint i;
	glong len,text_length;
	glong hit = 0;
	glong replace = 0;
	gchar search_result[STRLEN];
	glong found = 0;

	if(widget)
		history_add_to_history(main_window->search_replace_history,GTK_COMBO(search_entry));

	gI_window_clear_statusbar( main_window );

	current = gI_document_get_current( main_window );

	strcpy( search_str, gtk_entry_get_text( GTK_ENTRY(GTK_COMBO( search_entry)->entry ) ) );
	strcpy( replace_str, gtk_entry_get_text( GTK_ENTRY( replace_entry ) ) );
	if( strlen( search_str ) == 0 )
	{
		/* do something */
		gI_error_dialog( _("Search String Length == 0") );
		return;
	}
	else
		search_window_destroy( NULL, NULL );

	buf = (gchar *) g_malloc0( sizeof( search_str ) );
	len = strlen( search_str );
	text_length = gI_text_get_length( current->text );

	if( !kz_cursor )
		i = 0;
	else
		i = gI_text_get_point( current->text );

	gI_text_freeze( current->text );

	for(;i<=(text_length-len-replace);i++)
	{
		buf = gtk_editable_get_chars( GTK_EDITABLE( current->text ), i, i+len );
		if( kz_case )
			hit = strcmp( buf, search_str );
		else
			hit = strcasecmp( buf, search_str );

		if( hit == 0 )
		{
			found = TRUE;

			gI_text_thaw( current->text );

			/* scroll... */
			__goto_point( i );
			gI_text_set_point( current->text, i+len );
			gtk_editable_select_region( GTK_EDITABLE( current->text ), i, i+len );

			gI_document_delete_selection( current );
			gtk_editable_insert_text( GTK_EDITABLE( current->text ), replace_str, strlen( replace_str ), &i );
			gI_text_set_point( current->text, i+strlen( replace_str ) );
			replace = replace + ( strlen( search_str ) - strlen( replace_str ) );
			gI_text_freeze( current->text );
		}
		g_free( buf );
	}

	if( !found )
	{
		g_snprintf( search_result, STRLEN, _("\n    [%s] not found!    \n"), search_str );
		/*        gtk_statusbar_push( GTK_STATUSBAR( main_window->statusbar ), 1, search_result );	*/
		gI_error_dialog( search_result);
	}


}


/*
 ---------------------------------------------------------------------
     Function: search_again()
     Desc: Callback-Function /Search/Search Again
 ---------------------------------------------------------------------
*/

void search_again( GtkWidget *widget, gpointer data )
{
	if( strlen( search_str ) == 0 )
		return;

	kz_cursor = TRUE;
	kz_again = TRUE;

	_search_search( NULL, NULL );
}


/*
 ---------------------------------------------------------------------
     Function: search_replace()
     Desc: Callback-Function /Search/Replace
 ---------------------------------------------------------------------
*/

void search_replace( GtkWidget *widget, gpointer data )
{
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *hsep;
	GtkWidget *button;
	GtkWidget *cb_case;
	GtkWidget *cb_regexp;

	if( search_window )
		return;

	kz_case = 0;
	kz_cursor = 0;
	kz_regexp = 0;

	search_window = gtk_window_new( GTK_WINDOW_TOPLEVEL );
	gtk_widget_set_usize( search_window, 300, 160 );
	gtk_window_set_title( GTK_WINDOW( search_window ), _("Find&Replace...") );
	gtk_signal_connect( GTK_OBJECT( search_window ), "destroy",
	                    GTK_SIGNAL_FUNC( search_window_destroy ), NULL );

	vbox = gtk_vbox_new( FALSE, 0 );
	gtk_container_add( GTK_CONTAINER( search_window ), vbox );
	gtk_widget_show( vbox );

	hbox = gtk_hbox_new( FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 10 );
	gtk_widget_show( hbox );

	label = gtk_label_new( _("Search for") );
	gtk_box_pack_start( GTK_BOX( hbox ), label, FALSE, TRUE, 5 );
	gtk_widget_show( label );

	search_entry=gtk_combo_new();
	gtk_box_pack_start( GTK_BOX( hbox ), search_entry, TRUE, TRUE, 5 );
	gtk_widget_show( search_entry );

	if(!main_window->search_replace_history)
		main_window->search_replace_history=g_list_alloc();
	else
		history_add_to_combo(main_window->search_replace_history,GTK_COMBO(search_entry));

	hbox = gtk_hbox_new( FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 5 );
	gtk_widget_show( hbox );

	label = gtk_label_new( _("Replace with") );
	gtk_box_pack_start( GTK_BOX( hbox ), label, FALSE, TRUE, 5 );
	gtk_widget_show( label );

	replace_entry = gtk_entry_new_with_max_length( 255 );
	gtk_box_pack_start( GTK_BOX( hbox ), replace_entry, TRUE, TRUE, 5 );
	gtk_signal_connect( GTK_OBJECT( replace_entry ), "activate",
	                    GTK_SIGNAL_FUNC( _search_replace ), NULL );
	gtk_widget_show( replace_entry );

	hbox = gtk_hbox_new( FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 5 );
	gtk_widget_show( hbox );

	cb_case = gtk_check_button_new_with_label( _("case sensitive") );
	gtk_box_pack_start( GTK_BOX( hbox ), cb_case, TRUE, TRUE, 5 );
	gtk_signal_connect( GTK_OBJECT( cb_case ), "clicked",
	                    GTK_SIGNAL_FUNC( cb_case_clicked ), NULL );
	gtk_widget_show( cb_case );

	cb_regexp = gtk_check_button_new_with_label( _("regular expression") );
	gtk_box_pack_start( GTK_BOX( hbox ), cb_regexp, TRUE, TRUE, 5 );
	gtk_signal_connect( GTK_OBJECT( cb_regexp ), "clicked",
	                    GTK_SIGNAL_FUNC( cb_regexp_clicked ), NULL );
	gtk_widget_show( cb_regexp );

	hsep = gtk_hseparator_new();
	gtk_box_pack_start( GTK_BOX( vbox ), hsep, TRUE, TRUE, 5 );
	gtk_widget_show( hsep );

	hbox = gtk_hbox_new( FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 5 );
	gtk_widget_show( hbox );

	button = gnome_stock_button( GNOME_STOCK_BUTTON_OK );
	gtk_box_pack_start( GTK_BOX( hbox ), button, TRUE, TRUE, 5 );
	gtk_signal_connect( GTK_OBJECT( button ), "clicked",
	                    GTK_SIGNAL_FUNC( _search_replace ), NULL );
	GTK_WIDGET_SET_FLAGS( button, GTK_CAN_DEFAULT );
	gtk_widget_grab_default( button );
	gtk_widget_show( button );

	button = gnome_stock_button( GNOME_STOCK_BUTTON_CANCEL );
	gtk_box_pack_end( GTK_BOX( hbox ), button, TRUE, TRUE, 5 );
	gtk_signal_connect( GTK_OBJECT( button ), "clicked",
	                    GTK_SIGNAL_FUNC( search_window_destroy ), NULL );
	gtk_widget_show( button );

	if( search_str )
		gtk_entry_set_text( GTK_ENTRY(GTK_COMBO( search_entry)->entry ), search_str );
	if( replace_str )
		gtk_entry_set_text( GTK_ENTRY( replace_entry ), replace_str );

	gtk_widget_grab_focus( search_entry );

	gtk_widget_show( search_window );
}

void history_add_to_history(GList *list,GtkCombo *widget)
{
	gchar *nstr;
	gchar *str;
	GList *cur;

	nstr= gtk_entry_get_text(GTK_ENTRY( GTK_COMBO( widget )->entry ));
	cur=g_list_first(list);
	while(cur)
	{
		str=(gchar *)cur->data;
		if(str)
		{
			if(!strcasecmp(str,nstr))
				return ; /* already one */
		}
		cur=g_list_next(cur);
	};
	g_list_append(list,g_strdup(nstr));
}

void history_add_to_combo(GList *list,GtkCombo *widget)
{
	GtkWidget *label;
	GtkWidget *listitem;
	gchar *str;
	GList *cur;

	cur=g_list_first(list);
	while(cur)
	{
		str=(gchar *)cur->data;
		if(str)
		{
			label=gtk_label_new(str);
			listitem=gtk_list_item_new();
			gtk_container_add(GTK_CONTAINER(listitem),label);
			gtk_widget_show(label);
			gtk_container_add(GTK_CONTAINER(GTK_COMBO(widget)->list),listitem);
			gtk_widget_show(listitem);
		}
		cur=g_list_next(cur);
	};
}


/*
 * this __goto_point() function really takes a point,
 * maybe we should re-name the functions one day
 * (there's a goto_point() function in gI_compile.[hc]
 * that takes a line as argument. the difference to
 * goto_line() is that it really goes to the line not
 * just makes them visible on the screen)
 */
void __goto_point( gint point )
{
	gI_document *current;
	double tl, ln, adjval;
	glong line;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	line = get_line_from_point( point );

	tl = (double) get_total_lines( current->text );
	ln = (double) line;

	if( tl < 10.0 )
	{
		/* is on the screen */
		return;
	}

	if( ln > tl )
	{
		/* should not happen */
		return;
	}

	gI_text_set_point( current->text, point );

	/* borrowed from gEdit */
	adjval = (ln * GTK_ADJUSTMENT( gI_text_vadj( current->text ) )->upper ) / tl - GTK_ADJUSTMENT( gI_text_vadj( current->text ) )->page_increment;

	gtk_adjustment_set_value( GTK_ADJUSTMENT( gI_text_vadj( current->text ) ), adjval );

	/* track movement and set statusbar */
	gI_document_track_movement( current );
	gI_window_set_statusbar( current );
}


static void find_in_file( gI_document *document, gchar *text, gint casein, gint regexp, gchar *filename )
{
	FILE *fp;
	gchar MsgStr[STRLEN];
	gint line;
	gint hit;
	gint i;
	gchar buf[STRLEN];
#ifdef HAVE_GNU_REGEXP
	gchar errbuf[256];
	gchar errmsg[300];
	regex_t buffer;
	gint compile;
	regmatch_t regs[1];
	gint match;
#endif

	fp = fopen( filename, "r" );
	if( !fp )
	{
		g_snprintf( MsgStr, sizeof(MsgStr), _("\nUnable to Open File: '%s'\n"), filename );
		gI_text_insert( document->text, NULL, NULL, NULL, MsgStr, strlen(MsgStr) );
		return;
	}

	if( regexp )
	{
#ifdef HAVE_GNU_REGEXP
		compile = regcomp( &buffer, text, REG_EXTENDED );
		if( compile != 0 )
		{
			regerror( compile, &buffer, errbuf, 256 );
			sprintf( errmsg, _("Regex Compile Error: %s"), errbuf );
			gI_error_dialog( errmsg);
			return;
		}
#else
		gI_error_dialog( _("Regular Expression Search only supported with the GNU regexp lib") );
		return;
#endif
	}

	line = 0;

	while( fgets( buf, sizeof(buf), fp ) )
	{
		line++;

		if( regexp )
		{
#ifdef HAVE_GNU_REGEXP
			match = regexec( &buffer, buf, 1, regs, 0 );
			if( match == 0 )
			{
				/* matched */
				g_snprintf( MsgStr, sizeof(MsgStr), "%s (%d): %s\n", filename, line, buf );
				gI_text_insert( document->text, NULL, NULL, NULL, MsgStr, strlen(MsgStr) );
			}
#endif

			continue;
		}

		if( strlen(buf) < strlen(text) )
			continue;

		for(i=0;i<=(strlen(buf)-strlen(text));i++)
		{
			if( casein )
			{
				hit = strncasecmp( &buf[i], text, strlen(text) );
			}
			else /* !casein && !regexp */
			{
				hit = strncmp( &buf[i], text, strlen(text) );
			}

			if( hit == 0 )
			{
				g_snprintf( MsgStr, sizeof(MsgStr), "%s (%d): %s\n", filename, line, buf );

				gI_text_insert( document->text, NULL, NULL, NULL, MsgStr, strlen(MsgStr) );
				break;
			}
		}
	}

	fclose( fp );

#ifdef HAVE_GNU_REGPEXP
	regfree( &buffer );
#endif
}


static void replace_in_file( gI_document *document, gchar *text, gint casein, gint regexp, gchar *repltext, gchar *filename )
{
	FILE *fp, *fp_out;
	gchar MsgStr[STRLEN];
	gint line;
	gint hit;
	gint i;
	gchar buf[STRLEN];
	gchar tmp[STRLEN];
	gchar tmpn[100];
	gchar *bakfile;
#ifdef HAVE_GNU_REGEXP
	gchar errbuf[256];
	gchar errmsg[300];
	regex_t buffer;
	gint compile;
	regmatch_t regs[1];
	gint match;
#endif

	fp = fopen( filename, "r" );
	if( !fp )
	{
		g_snprintf( MsgStr, sizeof(MsgStr), _("\nUnable to Open File: '%s'\n"), filename );
		gI_text_insert( document->text, NULL, NULL, NULL, MsgStr, strlen(MsgStr) );
		return;
	}

	strcpy( tmpn, "/tmp/replfile" );
	fp_out = fopen( tmpn, "w" );
	if( !fp_out )
	{
		g_snprintf( MsgStr, sizeof(MsgStr), _("\nUnable to Open File: '%s'\n"), tmpn );
		gI_text_insert( document->text, NULL, NULL, NULL, MsgStr, strlen(MsgStr) );
		return;
	}

	line = 0;

	while( fgets( buf, sizeof(buf), fp ) )
	{
		line++;

		if( regexp )
		{
#ifdef HAVE_GNU_REGEXP
			compile = regcomp( &buffer, text, REG_EXTENDED );
			if( compile != 0 )
			{
				regerror( compile, &buffer, errbuf, 256 );
				sprintf( errmsg, _("Regex Compile Error: %s"), errbuf );
				gI_error_dialog( errmsg);
				return;
			}

			match = regexec( &buffer, buf, 1, regs, 0 );
			if( match == 0 )
			{
				/* matched */
				g_print( "replace not implemented!\n" );
				return;
			}
#else
			/*error_dialog( "\n    Regular Expression Search only supported with the GNU regexp lib    \n", "Regexp Search Error!" );*/
			return;
#endif

			continue;
		}

		if( strlen(buf) < strlen(text) )
			continue;

		for(i=0;i<=(strlen(buf)-strlen(text));i++)
		{
			if( casein )
			{
				hit = strncasecmp( &buf[i], text, strlen(text) );
			}
			else /* !casein && !regexp */
			{
				hit = strncmp( &buf[i], text, strlen(text) );
			}

			if( hit == 0 )
			{
				strncpy( tmp, buf, i );
				tmp[i] = '\0';
				strcat( tmp, repltext );
				strcat( tmp, &buf[i+(strlen(text))] );
				strcpy( buf, tmp );
				i += strlen(repltext);
			}
		}

		fputs( buf, fp_out );
	}

	fclose( fp );
	fclose( fp_out );

	bakfile = g_strconcat( filename, ".bak", NULL );
	remove( bakfile );
	rename( filename, bakfile );
	g_free( bakfile );
	rename( tmpn, filename );
}


static gint search_dir( gI_document *document, gchar *text, gint casein, gint regexp, gint recursive, gchar *directory, gchar *pat, gint repl, gchar *repltext )
{
	struct stat sb;
	DIR *d;
	struct dirent *entry;
	gchar *fnwp;

	g_return_val_if_fail( document != NULL, -1 );
	g_return_val_if_fail( text != NULL, -1 );
	g_return_val_if_fail( directory != NULL, -1 );
	g_return_val_if_fail( pat != NULL, -1 );

	if( stat( directory, &sb ) < 0 )
	{
		return -1;
	}

	if( !S_ISDIR( sb.st_mode ) )
	{
		return -1;
	}

	d = opendir( directory );
	if( !d )
	{
		return -1;
	}

	while( (entry = readdir( d )) )
	{
		/* avoid endless recursion... */
		if( !strcmp( entry->d_name, "." )
		        || !strcmp( entry->d_name, ".." ) )
			continue;

		fnwp = g_strconcat( directory, "/", entry->d_name, NULL );

		if( stat( fnwp, &sb ) < 0 )
		{
			return -1;
		}

		if( S_ISDIR( sb.st_mode ) && !recursive )
		{
			continue;
		}

		if( S_ISDIR( sb.st_mode ) )
		{
			/* recursive.. */
			search_dir( document, text, casein, regexp, recursive, fnwp, pat, repl, repltext );
			continue;
		}

		/* check if its a regular file (or a symlink), here ... */
		if( !(S_ISREG( sb.st_mode ) || S_ISLNK( sb.st_mode )) )
		{
			continue;
		}

		if( fnmatch( pat, entry->d_name, FNM_NOESCAPE ) == 0 )
		{
			/* matched */
			if( repl )
			{
				replace_in_file( document, text, casein, regexp, repltext, fnwp );
			}
			else
			{
				find_in_file( document, text, casein, regexp, fnwp );
			}
		}

		g_free( fnwp );
	}

	closedir( d );

	return 0;
}


static void find_in_files_ok( GtkWidget *widget, gI_FiF_Window *win )
{
	gI_document *document;
	gchar *dir, *pat, *text;
	gchar MsgStr[STRLEN];
	gint casein, regexp, recursive;

	text = gtk_entry_get_text( GTK_ENTRY( win->text ) );
	dir = gtk_entry_get_text( GTK_ENTRY( win->dir ) );
	pat = gtk_entry_get_text( GTK_ENTRY( win->pat ) );
	casein = GTK_TOGGLE_BUTTON( win->casein )->active;
	regexp = GTK_TOGGLE_BUTTON( win->regexp )->active;
	recursive = GTK_TOGGLE_BUTTON( win->recursive )->active;

#ifndef HAVE_GNU_REGEXP
	if( regexp )
	{
		gI_error_dialog( _("Regular Expression Search only supported with the GNU regexp lib") );
		return;
	}
#endif

	if( isempty( text ) )
	{
		g_snprintf( MsgStr, sizeof(MsgStr), _("\n   Please enter a text to search for!   \n") );
		gI_error_dialog( MsgStr);
		return;
	}

	if( isempty( dir ) )
	{
		g_snprintf( MsgStr, sizeof(MsgStr), _("\n   Please enter a directory!   \n") );
		gI_error_dialog( MsgStr);
		return;
	}

	if( isempty( pat ) )
	{
		pat = "*.*";
	}

	document = gI_document_new( main_window );
	gtk_label_set( GTK_LABEL( document->label ), _("--- Find Results ---") );

	search_dir( document, text, casein, regexp, recursive, dir, pat, FALSE, NULL );

	gI_window_set_statusbar( document );

	gtk_widget_destroy( win->window );
}


static void find_in_files_cancel( GtkWidget *widget, gI_FiF_Window *win )
{
	gtk_widget_destroy( win->window );
}


void find_in_files( GtkWidget *widget, gpointer data )
{
	gI_FiF_Window *w;

	w = gI_FiF_Window_new();
	gtk_widget_show_all( w->window );
}


gI_FiF_Window *gI_FiF_Window_new()
{
	gI_FiF_Window *w;
	GtkWidget *label;
	GtkWidget *vbox, *hbox;
	GtkWidget *button;
	GtkWidget *packer;

	w = (gI_FiF_Window *) g_malloc0( sizeof( gI_FiF_Window ) );
	w->window = gtk_window_new( GTK_WINDOW_TOPLEVEL );
	gtk_window_set_title( GTK_WINDOW( w->window ), _("Find in Files") );
	gtk_signal_connect( GTK_OBJECT( w->window ), "destroy",
	                    GTK_SIGNAL_FUNC( gI_FiF_Window_destroy ), (gpointer) w );

	vbox = gtk_vbox_new( FALSE, 0 );
	gtk_container_add( GTK_CONTAINER( w->window ), vbox );
	gtk_container_set_border_width( GTK_CONTAINER( vbox ), 10 );

	/* ok, let us try the new packer widget */
	packer = gtk_packer_new();
	gtk_box_pack_start( GTK_BOX( vbox ), packer, TRUE, TRUE, 5 );

	label = gtk_label_new( _("Search for  ") );
	gtk_packer_add( GTK_PACKER( packer ),
	                label,
	                GTK_SIDE_LEFT,
	                GTK_ANCHOR_CENTER,
	                GTK_FILL_X,
	                0, 0, 0, 0, 0 );
	w->text = gtk_entry_new_with_max_length( 255 );
	gtk_packer_add( GTK_PACKER( packer ),
	                w->text,
	                GTK_SIDE_LEFT,
	                GTK_ANCHOR_CENTER,
	                GTK_FILL_X|GTK_EXPAND,
	                0, 10, 10, 0, 0 );
	gtk_widget_grab_focus( w->text );

	packer = gtk_packer_new();
	gtk_box_pack_start( GTK_BOX( vbox ), packer, TRUE, TRUE, 5 );

	label = gtk_label_new( _("Directory  ") );
	gtk_packer_add( GTK_PACKER( packer ),
	                label,
	                GTK_SIDE_LEFT,
	                GTK_ANCHOR_CENTER,
	                GTK_FILL_X,
	                0, 0, 0, 0, 0 );

	w->dir = gtk_entry_new_with_max_length( 255 );
	gtk_packer_add( GTK_PACKER( packer ),
	                w->dir,
	                GTK_SIDE_LEFT,
	                GTK_ANCHOR_CENTER,
	                GTK_FILL_X|GTK_EXPAND,
	                5, 10, 10, 0, 0 );

	packer = gtk_packer_new();
	gtk_box_pack_start( GTK_BOX( vbox ), packer, TRUE, TRUE, 5 );

	label = gtk_label_new( _("File Pattern") );
	gtk_packer_add( GTK_PACKER( packer ),
	                label,
	                GTK_SIDE_LEFT,
	                GTK_ANCHOR_CENTER,
	                GTK_FILL_X,
	                0, 0, 0, 0, 0 );

	w->pat = gtk_entry_new_with_max_length( 255 );
	gtk_packer_add( GTK_PACKER( packer ),
	                w->pat,
	                GTK_SIDE_LEFT,
	                GTK_ANCHOR_CENTER,
	                GTK_FILL_X|GTK_EXPAND,
	                0, 10, 10, 0, 0 );

	hbox = gtk_hbox_new( FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, FALSE, 10 );

	w->casein = gtk_check_button_new_with_label( _("case insensitive") );
	gtk_box_pack_start( GTK_BOX( hbox ), w->casein, TRUE, TRUE, 5 );

	w->regexp = gtk_check_button_new_with_label( _("regular expression") );
	gtk_box_pack_start( GTK_BOX( hbox ), w->regexp, TRUE, TRUE, 5 );

	w->recursive = gtk_check_button_new_with_label( _("search subdirectories") );
	gtk_box_pack_start( GTK_BOX( hbox ), w->recursive, TRUE, TRUE, 5 );

	label = gtk_hseparator_new();
	gtk_box_pack_start( GTK_BOX( vbox ), label, FALSE, TRUE, 5 );

	hbox = gtk_hbox_new( FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, FALSE, 7 );

	button = gnome_stock_button( GNOME_STOCK_BUTTON_OK );
	gtk_box_pack_start( GTK_BOX( hbox ), button, TRUE, TRUE, 5 );
	gtk_signal_connect( GTK_OBJECT( button ), "clicked",
	                    GTK_SIGNAL_FUNC( find_in_files_ok ), (gpointer) w );
	GTK_WIDGET_SET_FLAGS( button, GTK_CAN_DEFAULT );
	gtk_widget_grab_default( button );

	button = gnome_stock_button( GNOME_STOCK_BUTTON_CANCEL );
	gtk_box_pack_start( GTK_BOX( hbox ), button, TRUE, TRUE, 5 );
	gtk_signal_connect( GTK_OBJECT( button ), "clicked",
	                    GTK_SIGNAL_FUNC( find_in_files_cancel ), (gpointer) w );

	return w;
}


void gI_FiF_Window_destroy( GtkWidget *widget, gI_FiF_Window *win )
{
	g_free( win );
}


static void replace_in_files_ok( GtkWidget *widget, gI_RiF_Window *win )
{
	gI_document *document;
	gchar *dir, *pat, *text, *repltext;
	gchar MsgStr[STRLEN];
	gint casein, regexp, recursive;

	text = gtk_entry_get_text( GTK_ENTRY( win->text ) );
	repltext = gtk_entry_get_text( GTK_ENTRY( win->repltext ) );
	dir = gtk_entry_get_text( GTK_ENTRY( win->dir ) );
	pat = gtk_entry_get_text( GTK_ENTRY( win->pat ) );
	casein = GTK_TOGGLE_BUTTON( win->casein )->active;
	regexp = GTK_TOGGLE_BUTTON( win->regexp )->active;
	recursive = GTK_TOGGLE_BUTTON( win->recursive )->active;

#ifndef HAVE_GNU_REGEXP
	if( regexp )
	{
		gI_error_dialog( _("Regular Expression Search only supported with the GNU regexp lib") );
		return;
	}
#endif

	if( isempty( text ) )
	{
		g_snprintf( MsgStr, sizeof(MsgStr), _("\n   Please enter a text to search for!   \n") );
		gI_error_dialog( MsgStr);
		return;
	}

	if( isempty( dir ) )
	{
		g_snprintf( MsgStr, sizeof(MsgStr), _("\n   Please enter a directory!   \n") );
		gI_error_dialog( MsgStr);
		return;
	}

	if( isempty( pat ) )
	{
		pat = "*.*";
	}

	document = gI_document_new( main_window );
	gtk_label_set( GTK_LABEL( document->label ), _("-- Replace Results --") );

	search_dir( document, text, casein, regexp, recursive, dir, pat, TRUE, repltext );

	gI_window_set_statusbar( document );

	gtk_widget_destroy( win->window );
}


static void replace_in_files_cancel( GtkWidget *widget, gI_RiF_Window *win )
{
	gtk_widget_destroy( win->window );
}


void replace_in_files( GtkWidget *widget, gpointer data )
{
	gI_RiF_Window *w;

	w = gI_RiF_Window_new();
	gtk_widget_show_all( w->window );
}


gI_RiF_Window *gI_RiF_Window_new()
{
	gI_RiF_Window *w;
	GtkWidget *label;
	GtkWidget *vbox, *hbox;
	GtkWidget *button;
	GtkWidget *packer;

	w = (gI_RiF_Window *) g_malloc0( sizeof( gI_RiF_Window ) );
	w->window = gtk_window_new( GTK_WINDOW_TOPLEVEL );
	gtk_window_set_title( GTK_WINDOW( w->window ), _("Replace in Files") );
	gtk_signal_connect( GTK_OBJECT( w->window ), "destroy",
	                    GTK_SIGNAL_FUNC( gI_RiF_Window_destroy ), (gpointer) w );

	vbox = gtk_vbox_new( FALSE, 0 );
	gtk_container_add( GTK_CONTAINER( w->window ), vbox );
	gtk_container_set_border_width( GTK_CONTAINER( vbox ), 10 );

	packer = gtk_packer_new();
	gtk_box_pack_start( GTK_BOX( vbox ), packer, TRUE, TRUE, 5 );

	label = gtk_label_new( _("Search for   ") );
	gtk_packer_add( GTK_PACKER( packer ),
	                label,
	                GTK_SIDE_LEFT,
	                GTK_ANCHOR_CENTER,
	                GTK_FILL_X,
	                0, 0, 0, 0, 0 );
	w->text = gtk_entry_new_with_max_length( 255 );
	gtk_packer_add( GTK_PACKER( packer ),
	                w->text,
	                GTK_SIDE_LEFT,
	                GTK_ANCHOR_CENTER,
	                GTK_FILL_X|GTK_EXPAND,
	                0, 10, 10, 0, 0 );
	gtk_widget_grab_focus( w->text );

	packer = gtk_packer_new();
	gtk_box_pack_start( GTK_BOX( vbox ), packer, TRUE, TRUE, 5 );

	label = gtk_label_new( _("Replace with") );
	gtk_packer_add( GTK_PACKER( packer ),
	                label,
	                GTK_SIDE_LEFT,
	                GTK_ANCHOR_CENTER,
	                GTK_FILL_X,
	                0, 0, 0, 0, 0 );
	w->repltext = gtk_entry_new_with_max_length( 255 );
	gtk_packer_add( GTK_PACKER( packer ),
	                w->repltext,
	                GTK_SIDE_LEFT,
	                GTK_ANCHOR_CENTER,
	                GTK_FILL_X|GTK_EXPAND,
	                0, 10, 10, 0, 0 );

	packer = gtk_packer_new();
	gtk_box_pack_start( GTK_BOX( vbox ), packer, TRUE, TRUE, 5 );

	label = gtk_label_new( _("Directory   ") );
	gtk_packer_add( GTK_PACKER( packer ),
	                label,
	                GTK_SIDE_LEFT,
	                GTK_ANCHOR_CENTER,
	                GTK_FILL_X,
	                0, 0, 0, 0, 0 );

	w->dir = gtk_entry_new_with_max_length( 255 );
	gtk_packer_add( GTK_PACKER( packer ),
	                w->dir,
	                GTK_SIDE_LEFT,
	                GTK_ANCHOR_CENTER,
	                GTK_FILL_X|GTK_EXPAND,
	                5, 10, 10, 0, 0 );

	packer = gtk_packer_new();
	gtk_box_pack_start( GTK_BOX( vbox ), packer, TRUE, TRUE, 5 );

	label = gtk_label_new( _("File Pattern ") );
	gtk_packer_add( GTK_PACKER( packer ),
	                label,
	                GTK_SIDE_LEFT,
	                GTK_ANCHOR_CENTER,
	                GTK_FILL_X,
	                0, 0, 0, 0, 0 );

	w->pat = gtk_entry_new_with_max_length( 255 );
	gtk_packer_add( GTK_PACKER( packer ),
	                w->pat,
	                GTK_SIDE_LEFT,
	                GTK_ANCHOR_CENTER,
	                GTK_FILL_X|GTK_EXPAND,
	                0, 10, 10, 0, 0 );

	hbox = gtk_hbox_new( FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, FALSE, 10 );

	w->casein = gtk_check_button_new_with_label( _("case insensitive") );
	gtk_box_pack_start( GTK_BOX( hbox ), w->casein, TRUE, TRUE, 5 );

	w->regexp = gtk_check_button_new_with_label( _("regular expression") );
	gtk_box_pack_start( GTK_BOX( hbox ), w->regexp, TRUE, TRUE, 5 );

	w->recursive = gtk_check_button_new_with_label( _("search subdirectories") );
	gtk_box_pack_start( GTK_BOX( hbox ), w->recursive, TRUE, TRUE, 5 );

	label = gtk_hseparator_new();
	gtk_box_pack_start( GTK_BOX( vbox ), label, FALSE, TRUE, 5 );

	hbox = gtk_hbox_new( FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, FALSE, 7 );

	button = gnome_stock_button( GNOME_STOCK_BUTTON_OK );
	gtk_box_pack_start( GTK_BOX( hbox ), button, TRUE, TRUE, 5 );
	gtk_signal_connect( GTK_OBJECT( button ), "clicked",
	                    GTK_SIGNAL_FUNC( replace_in_files_ok ), (gpointer) w );
	GTK_WIDGET_SET_FLAGS( button, GTK_CAN_DEFAULT );
	gtk_widget_grab_default( button );

	button = gnome_stock_button( GNOME_STOCK_BUTTON_CANCEL );
	gtk_box_pack_start( GTK_BOX( hbox ), button, TRUE, TRUE, 5 );
	gtk_signal_connect( GTK_OBJECT( button ), "clicked",
	                    GTK_SIGNAL_FUNC( replace_in_files_cancel ), (gpointer) w );

	return w;
}


void gI_RiF_Window_destroy( GtkWidget *widget, gI_RiF_Window *win )
{
	g_free( win );
}
