/*
 * GSHIGH.C - Portable Graphics System
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "pgs.h"
#include "gsrast.h"

#define PAD_FACTOR            0.01

/* directions */
#define _UP 0
#define _DOWN 1
#define _LEFT 2
#define _RIGHT 3

/* sides */
#define LEFT_SIDE 0
#define TOP_SIDE 1
#define RIGHT_SIDE 2
#define BOTTOM_SIDE 3

#define ADD_CORNER(x, y, side1, side2, xmn, xmx, ymn, ymx)                 \
    {if (((side1 == TOP_SIDE) && (side2 == LEFT_SIDE)) ||                  \
         ((side1 == LEFT_SIDE) && (side2 == TOP_SIDE)))                    \
        {x = xmn;                                                          \
         y = ymx;}                                                         \
     else if (((side1 == TOP_SIDE) && (side2 == RIGHT_SIDE)) ||            \
              ((side1 == RIGHT_SIDE) && (side2 == TOP_SIDE)))              \
        {x = xmx;                                                          \
         y = ymx;}                                                         \
     else if (((side1 == RIGHT_SIDE) && (side2 == BOTTOM_SIDE)) ||         \
              ((side1 == BOTTOM_SIDE) && (side2 == RIGHT_SIDE)))           \
        {x = xmx;                                                          \
         y = ymn;}                                                         \
     else if (((side1 == LEFT_SIDE) && (side2 == BOTTOM_SIDE)) ||          \
              ((side1 == BOTTOM_SIDE) && (side2 == LEFT_SIDE)))            \
        {x = xmn;                                                          \
         y = ymn;};}

#define INCREMENT_SIDE(side, incr)                                         \
    {side += incr;                                                         \
     side = (side < 0) ? 3 : side;                                         \
     side = (side > 3) ? 0 : side;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_CENTER_LABEL - print a label centered on a line at a specified
 *                 - y value given in screen coordinates
 */

void PG_center_label(dev, sy, label)
   PG_device *dev;
   double sy;
   char *label;
   {REAL sx, dx, dy;

    if (PG_plot_labels == FALSE)
       return;

    PG_get_text_ext(dev, label, &dx, &dy);

    sx = dev->fxmin + 0.5*(dev->fxmax - dev->fxmin);
    sy = dev->fymin + sy*(dev->fymax - dev->fymin);

    StoW(dev, sx, sy);
    sx -= 0.5*dx;
    PG_write_WC(dev, sx, sy, "%s", label);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_SET_LIMITS - find limits of x and y arrays */
 
void PG_set_limits(dev, x, y, n, type)
   PG_device *dev;
   REAL *x, *y;
   int n, type;
   {REAL *px, *py;
    REAL xmin, xmax, ymin, ymax;
 
    switch(type)
       {case INSEL :
             xmax = 1.5;
             xmin = 0.0;
             ymax = y[0];
             ymin = y[0];
             for (px = x, py = y;px < x+n; px++, py++)
                 {if (*px > ymax)
                     ymax = *px;
                  if (*px < ymin)
                     ymin = *px;
                  if (*py > ymax)
                     ymax = *py;
                  if (*py < ymin)
                     ymin = *py;};
 
             if ((ymax - ymin) == 0)
                ymax = ymin + 1;
             break;

        case HISTOGRAM :
        case POLAR     :
        case CARTESIAN :
        default        :
             xmax = x[0];
             xmin = x[0];
             ymax = y[0];
             ymin = y[0];
             for (px = x, py = y; px < x+n; px++, py++)
                 {if (*px > xmax)
                     xmax = *px;
                  if (*px < xmin)
                     xmin = *px;
                  if (*py > ymax)
                     ymax = *py;
                  if (*py < ymin)
                     ymin = *py;};
 
             if ((xmax - xmin) == 0)
                xmax = xmin + 1;
             if ((ymax - ymin) == 0)
                ymax = ymin + 1;
             break;};
        
    PG_set_window(dev, xmin, xmax, ymin, ymax);  

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SET_PLOT_TYPE - set the plot and axis type */

pcons *PG_set_plot_type(info, plt, axs)
   pcons *info;
   int plt, axs;
   {if (plt == INSEL)
       axs = INSEL;

    else if (axs == INSEL)
       axs = plt;

    SC_CHANGE_VALUE_ALIST(info, int, SC_INTEGER_P_S,
			  "PLOT-TYPE", plt);

    SC_CHANGE_VALUE_ALIST(info, int, SC_INTEGER_P_S,
			  "AXIS-TYPE", axs);

    return(info);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_CLIP_DATA - clip the data points to the boundary region */

void PG_clip_data(dev, tx, ty, ptn, x, y, n)
   PG_device *dev;
   REAL *tx, *ty;
   int *ptn;
   REAL *x, *y;
   int n;
   {int i, tp, flag, xlf, ylf;
    REAL xmn, xmx, ymn, ymx, tmp;
    REAL x1, x2, y1, y2;

    PG_get_bound(dev, &xmn, &xmx, &ymn, &ymx);

    if (xmx < xmn)
       {tmp = xmx;
	xmx = xmn;
	xmn = tmp;}

    if (ymx < ymn)
       {tmp = ymx;
	ymx = ymn;
	ymn = tmp;}

    xlf = dev->ifxlog;
    ylf = dev->ifylog;
    tp  = 0;

    for (i = 1; i < n; i++)
        {x1 = x[i-1];
         x2 = x[i];
         y1 = y[i-1];
         y2 = y[i];

         flag = 0;
         PG_clip_line(&flag, &x1, &y1, &x2, &y2,
                      xmn, xmx, ymn, ymx,
		      xlf, ylf);

         if (((tp == 0) || (flag & 1)) && flag)
            {tx[tp] = x1;
             ty[tp] = y1;
             tp++;};

         if (flag & 2)
            {tx[tp] = x2;
             ty[tp] = y2;
             tp++;};};

    *ptn = tp;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_CLIP_LINE - clip the line X1 = (x1, y1) to X2 = (x2, y2) to the box
 *              - (xmn, ymn) to (xmx, ymx)
 *              - return FLAG value
 *              -    0 - take no end points
 *              -    1 - take the left end point
 *              -    2 - take the right end point
 *              -    3 - take both end points;
 */

void PG_clip_line(pfl, px1, py1, px2, py2, xmn, xmx, ymn, ymx,
			 xlf, ylf)
   int *pfl;
   REAL *px1, *py1, *px2, *py2;
   double  xmn, xmx, ymn, ymx;
   int xlf, ylf;
   {REAL x1, x2, y1, y2, tx, ty;
    REAL xta, xtb, xtc, yta, ytb, dx, dy;
    REAL lrgr, smllr;

    x1 = *px1;
    y1 = *py1;
    x2 = *px2;
    y2 = *py2;

    lrgr  = 1.0 + PAD_FACTOR;
    smllr = 1.0/lrgr;

    dx = PAD_FACTOR*ABS(xmx - xmn);
    dy = PAD_FACTOR*ABS(ymx - ymn);

/* if X1 and X2 are inside */
    if (((xmn <= x1) && (x1 <= xmx)) &&
        ((xmn <= x2) && (x2 <= xmx)) &&
        ((ymn <= y1) && (y1 <= ymx)) &&
        ((ymn <= y2) && (y2 <= ymx)))
       *pfl |= 2;

/* if both X1 -> X2 doesn't cross trivially */
    else if (((x1 < xmn) && (x2 < xmn)) ||
             ((x1 > xmx) && (x2 > xmx)) ||
             ((y1 < ymn) && (y2 < ymn)) ||
             ((y1 > ymx) && (y2 > ymx)))
       *pfl = 0;

/* if X1 is inside and X2 is outside we're leaving */
    else if (((xmn <= x1) && (x1 <= xmx)) &&
             ((ymn <= y1) && (y1 <= ymx)))
       {if (x2 < xmn)
           {if (xlf)
	       tx = smllr*xmn;
	    else
	       tx = xmn - dx;
            PM_interp(ty, tx, x1, y1, x2, y2);
	    y2 = ty;
	    x2 = tx;}

        else if (xmx < x2)
           {if (xlf)
	       tx = lrgr*xmx;
	    else
	       tx = xmx + dx;
            PM_interp(ty, tx, x1, y1, x2, y2);
            y2 = ty;
	    x2 = tx;};

        if (y2 < ymn)
           {if (ylf)
	       ty = smllr*ymn;
	    else
	       ty = ymn - dy;
            PM_interp(tx, ty, y1, x1, y2, x2);}
        else if (ymx < y2)
           {if (ylf)
	       ty = lrgr*ymx;
	    else
	       ty = ymx + dy;
            PM_interp(tx, ty, y1, x1, y2, x2);};

        *px2 = tx;
        *py2 = ty;

        *pfl |= 2;}

/* if X1 is outside and X2 is inside we're entering */
    else if (((xmn <= x2) && (x2 <= xmx)) &&
             ((ymn <= y2) && (y2 <= ymx)))
       {if (x1 < xmn)
           {if (xlf)
	       tx = smllr*xmn;
	    else
	       tx = xmn - dx;
            PM_interp(ty, tx, x1, y1, x2, y2);
	    y1 = ty;
	    x1 = tx;}

        else if (xmx < x1)
           {if (xlf)
	       tx = lrgr*xmx;
	    else
	       tx = xmx + dx;
            PM_interp(ty, tx, x1, y1, x2, y2);
            y1 = ty;
	    x1 = tx;};

        if (y1 < ymn)
           {if (ylf)
	       ty = smllr*ymn;
	    else
	       ty = ymn - dy;
            PM_interp(tx, ty, y1, x1, y2, x2);}
        else if (ymx < y1)
           {if (ylf)
	       ty = lrgr*ymx;
	    else
	       ty = ymx + dy;
            PM_interp(tx, ty, y1, x1, y2, x2);};

        *px1 = tx;
        *py1 = ty;

        *pfl |= 3;}

/* if X1 -> X2 crosses x = xmn going left to right interpolate and recurse */
    else if ((x1 < xmn) && (xmn <= x2))
       {tx = xmn;
        PM_interp(ty, tx, x1, y1, x2, y2);
       
        *px1 = tx;
        *py1 = ty;

        *pfl |= 1;

        PG_clip_line(pfl, px1, py1, px2, py2, xmn, xmx, ymn, ymx,
		     xlf, ylf);}

/* if X1 -> X2 crosses x = xmn going right to left interpolate and recurse */
    else if ((x2 < xmn) && (xmn <= x1))
       {tx = xmn;
        PM_interp(ty, tx, x1, y1, x2, y2);
       
        *px2 = tx;
        *py2 = ty;

        *pfl |= 2;

        PG_clip_line(pfl, px1, py1, px2, py2, xmn, xmx, ymn, ymx,
		     xlf, ylf);}

/* if X1 -> X2 crosses x = xmx going left to right interpolate and recurse */
    else if ((x1 <= xmx) && (xmx < x2))
       {tx = xmx;
        PM_interp(ty, tx, x1, y1, x2, y2);
       
        *px2 = tx;
        *py2 = ty;

        *pfl |= 2;

        PG_clip_line(pfl, px1, py1, px2, py2, xmn, xmx, ymn, ymx,
		     xlf, ylf);}

/* if X1 -> X2 crosses x = xmx going right to left interpolate and recurse */
    else if ((x2 <= xmx) && (xmx < x1))
       {tx = xmx;
        PM_interp(ty, tx, x1, y1, x2, y2);
       
        *px1 = tx;
        *py1 = ty;

        *pfl |= 1;

        PG_clip_line(pfl, px1, py1, px2, py2, xmn, xmx, ymn, ymx,
		     xlf, ylf);}

/* if X1 -> X2 crosses y = ymn going bottom to top interpolate and recurse */
    else if ((y1 < ymn) && (ymn <= y2))
       {ty = ymn;
        PM_interp(tx, ty, y1, x1, y2, x2);
       
        *px1 = tx;
        *py1 = ty;

        *pfl |= 1;

        PG_clip_line(pfl, px1, py1, px2, py2, xmn, xmx, ymn, ymx,
		     xlf, ylf);}

/* if X1 -> X2 crosses y = ymn going top to bottom interpolate and recurse */
    else if ((y2 < ymn) && (ymn <= y1))
       {ty = ymn;
        PM_interp(tx, ty, y1, x1, y2, x2);
       
        *px2 = tx;
        *py2 = ty;

        *pfl |= 2;

        PG_clip_line(pfl, px1, py1, px2, py2, xmn, xmx, ymn, ymx,
		     xlf, ylf);}

/* if X1 -> X2 crosses y = ymx going bottom to top interpolate and recurse */
    else if ((y1 <= ymx) && (ymx < y2))
       {ty = ymx;
        PM_interp(tx, ty, y1, x1, y2, x2);
       
        *px2 = tx;
        *py2 = ty;

        *pfl |= 2;

        PG_clip_line(pfl, px1, py1, px2, py2, xmn, xmx, ymn, ymx,
		     xlf, ylf);}

/* if X1 -> X2 crosses y = ymx going top to bottom interpolate and recurse */
    else if ((y2 <= ymx) && (ymx < y1))
       {ty = ymx;
        PM_interp(tx, ty, y1, x1, y2, x2);
       
        *px1 = tx;
        *py1 = ty;

        *pfl |= 1;

        PG_clip_line(pfl, px1, py1, px2, py2, xmn, xmx, ymn, ymx,
		     xlf, ylf);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_GAP_SEGS - calculate line segments around the edge of the clip rect
 *                between point 1 and point 2 going in direction dir.       
 */                     

static int _PG_gap_segs(x1, y1, p1_side, x2, y2, p2_side, dir,
                        xmn, xmx, ymn, ymx, xout, yout)
   REAL x1, y1;
   int p1_side;
   REAL x2, y2;
   int p2_side;
   int dir;
   REAL xmn, xmx, ymn, ymx;
   REAL *xout, *yout;
   {int count, incr, side1, side2;

    if (p1_side == p2_side)
       if (((dir == _DOWN)  && (y2 <= y1)) ||
           ((dir == _UP  )  && (y2 >= y1)) ||
           ((dir == _LEFT)  && (x2 <= x1)) ||
           ((dir == _RIGHT) && (x2 >= x1)))
           return (0);

    if (dir == _RIGHT)
       incr = (p1_side == TOP_SIDE) ? 1 : -1;
    else if (dir == _LEFT)
       incr = (p1_side == TOP_SIDE) ? -1 : 1;
    else if (dir == _UP)
       incr = (p1_side == LEFT_SIDE) ? 1 : -1;
    else
       incr = (p1_side == LEFT_SIDE) ? -1 : 1;
 
    side1 = p1_side;
    side2 = p1_side;
    INCREMENT_SIDE(side2, incr);

    count = 0;
    while (side1 != p2_side)
       {ADD_CORNER(xout[count], yout[count], side1, side2,
                   xmn, xmx, ymn, ymx);
        INCREMENT_SIDE(side1, incr);
        INCREMENT_SIDE(side2, incr);
        count++;}        

    return (count);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_POINT_EDGE - return which edge of a clip rectangle a point is on
 *
 */

static int _PG_point_edge(x1, y1, xmn, xmx, ymn, ymx)
   REAL x1, y1;
   REAL xmn, xmx, ymn, ymx;
   {int side;

    if ((x1 <= xmn) || (x1 >= xmx))
       side = (x1 <= xmn) ? LEFT_SIDE : RIGHT_SIDE;
    else if ((y1 <= ymn) || (y1 >= ymx))
       side = (y1 <= ymn) ? BOTTOM_SIDE : TOP_SIDE;
    else
       side = -1;

    return (side);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_INSIDE_POLYGON - test a point to see if it's inside a polygon
 *
 */

static int _PG_inside_polygon(x1, y1, x, y, n)
   REAL x1, y1;
   REAL *x, *y;
   int n;
   {int n_inter;
    REAL xmin, xmax, x2, y2, dx;

    PM_maxmin(x, &xmin, &xmax, n);

    dx = xmax - xmin + .001;

    x2 = x1 + dx;
    y2 = y1;

    n_inter = 0;

    if (PM_intersect_line_polygon(&x1, &y1, &x2, &y2, x, y, n, &n_inter))
       return(n_inter % 2);
    else
       return(FALSE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_CLOSE_GAPS - connect segments to close gaps left by clipping */

static void _PG_close_gaps(xin, yin, nin, gap, ngaps, xout, yout, nout,
                    x, y, n, xmn, xmx, ymn, ymx)
   REAL *xin, *yin;
   int nin;
   int *gap;
   int ngaps;
   REAL *xout, *yout;
   int *nout;
   REAL *x, *y;
   int n;
   REAL xmn, xmx, ymn, ymx;
   {int i, k, igap, incount, dir, ngap_segs;
    int p1_side, p2_side;
    REAL x1, y1, x2, y2;
    REAL xmin, xmax, ymin, ymax, xinc, yinc;
    REAL xgapsegs[4], ygapsegs[4];

    if (ngaps == 0)
       {for (i = 0; i < nin; i++)
            {xout[i] = xin[i];
             yout[i] = yin[i];}

        *nout = nin;
        return;}

    PM_maxmin(x, &xmin, &xmax, n);
    PM_maxmin(y, &ymin, &ymax, n);

    xinc = .001 * (xmax - xmin);
    yinc = .001 * (ymax - ymin);

    *nout = 0;
    incount = 0;    
    for (igap = 0; igap < ngaps; igap++)
        {for ( ; incount < gap[igap]; *nout += 1, incount++)
                   {xout[*nout] = xin[incount];
                    yout[*nout] = yin[incount];}

         if (gap[igap] == (nin - 1))
            {if (igap != (ngaps - 1))
/* if this is the last point in the input list, it has to be the last */
/* entry in the gap list.                                             */
                {PRINT(stdout, "internal error--_PG-close_gaps\n");
                 *nout = 0;
                 return;}
             else
/* gap is between the last point of the input and the first point. */
                {x1 = xin[gap[igap]];
                 y1 = yin[gap[igap]];
                 x2 = xin[0];
                 y2 = yin[0];};}

         else
            {x1 = xin[gap[igap]];
             y1 = yin[gap[igap]];
             x2 = xin[gap[igap]+1];
             y2 = yin[gap[igap]+1];}

/* determine which edges of the clip rectangle the points are on */
         p1_side = _PG_point_edge(x1, y1, xmn, xmx, ymn, ymx);
         p2_side = _PG_point_edge(x2, y2, xmn, xmx, ymn, ymx);
         if ((p1_side < 0) || (p2_side < 0))
            {PRINT(stdout, "Internal error--_PG_close_gaps\n");
             *nout = 0;
             return;}

/* decide which direction to draw the line to close the gap */
         if ((p1_side == LEFT_SIDE) || (p1_side == RIGHT_SIDE))
            {if (_PG_inside_polygon(x1, y1-yinc, x, y, n))
                    dir = _DOWN;
                 else
                    dir = _UP;}

         else
/* we're on the top or bottom of clip rectangle */
            {if (_PG_inside_polygon(x1-xinc, y1, x, y, n))
                    dir = _LEFT;
                 else
                    dir = _RIGHT;}

/* calculate the line segments between p1 and p2 */
         ngap_segs = _PG_gap_segs(x1, y1, p1_side, x2, y2, p2_side, dir,
                                  xmn, xmx, ymn, ymx, xgapsegs, ygapsegs);
         xout[*nout] = x1;
         yout[*nout] = y1;
         *nout += 1;

         for (k = 0; k < ngap_segs; k++, *nout += 1)
             {xout[*nout] = xgapsegs[k];
              yout[*nout] = ygapsegs[k];}

         xout[*nout] = x2;
         yout[*nout] = y2;
         *nout += 1;
         incount += 2;}

    for ( ; incount < nin; incount++, *nout += 1)
        {xout[*nout] = xin[incount];
         yout[*nout] = yin[incount];}


    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_CLIP_POLYGON - clip the data points to the boundary region */

void PG_clip_polygon(dev, tx, ty, ptn, x, y, n)
   PG_device *dev;
   REAL *tx, *ty;
   int *ptn;
   REAL *x, *y;
   int n;
   {int i, tp, flag, xlf, ylf, all_outside, ngaps;
    int last_point_flag, *gap;
    REAL xmn, xmx, ymn, ymx, tmp;
    REAL x1, x2, y1, y2;
    REAL *xtemp, *ytemp;

    PG_get_bound(dev, &xmn, &xmx, &ymn, &ymx);

    if (xmx < xmn)
       {tmp = xmx;
	xmx = xmn;
	xmn = tmp;}

    if (ymx < ymn)
       {tmp = ymx;
	ymx = ymn;
	ymn = tmp;}

    all_outside = TRUE;

/* check for all points outside the clip region */
    for (i = 0; i < n; i++)
        {if ((x[i] > xmn) && (x[i] < xmx) &&
             (y[i] > ymn) && (y[i] < ymx))
             {all_outside = FALSE;
              break;};}

    if (all_outside)
       {*ptn = 0;
        return;}

    gap   = FMAKE_N(int, 4*n, "PG_CLIP_POLYGON:gap");
    xtemp = FMAKE_N(REAL, 4*n, "PG_CLIP_POLYGON:xtemp");
    ytemp = FMAKE_N(REAL, 4*n, "PG_CLIP_POLYGON:ytemp");

    xlf             = dev->ifxlog;
    ylf             = dev->ifylog;
    tp              = 0;
    ngaps           = 0;
    last_point_flag = FALSE;

    for (i = 1; i < n; i++)
        {x1 = x[i-1];
         x2 = x[i];
         y1 = y[i-1];
         y2 = y[i];

         if ((xmn <= x1) && (x1 <= xmx) &&
             (xmn <= x2) && (x2 <= xmx) &&
             (ymn <= y1) && (y1 <= ymx) &&
             (ymn <= y2) && (y2 <= ymx))
/* entire line seg. inside clip window */
            {if (tp == 0)
                {xtemp[tp]   = x1;
                 ytemp[tp++] = y1;}
             
             xtemp[tp]   = x2;
             ytemp[tp++] = y2;}

         else if (((x1 < xmn) && (x2 < xmn)) ||
                  ((x1 > xmx) && (x2 > xmx)) ||
                  ((y1 < ymn) && (y2 < ymn)) ||
                  ((y1 > ymx) && (y2 > ymx)))
/* entire line seg. outside clip window */
             continue;

         else 
/* line needs to be clipped to window      */
/* clip the line, build the gap list       */
/* if the last entry in the gap list is    */
/* the last point in the polygon have to   */
/* close gap between last and first points */
            {flag = 0;
             PG_clip_line(&flag, &x1, &y1, &x2, &y2,
                          xmn, xmx, ymn, ymx,
		          xlf, ylf);
               
             if (flag == 0)
                continue;
             else
                {if ((tp == 0) || (flag & 1))
                    {xtemp[tp]   = x1;
                     ytemp[tp++] = y1;}
               
                 if (flag & 2)
                    {xtemp[tp]       = x2;
                     ytemp[tp++]     = y2;
                     if ((x2 != x[i]) ||
                         (y2 != y[i]))
                        gap[ngaps++] = tp - 1;};};};};

/* After clipping lines and building the gap list */
/* process the gaps, using the interior test to determine */
/* how to fill in the gaps. */

    _PG_close_gaps(xtemp, ytemp, tp, gap, ngaps, tx, ty, ptn, 
                   x, y, n, xmn, xmx, ymn, ymx);

    SFREE(gap);  
    SFREE(xtemp);
    SFREE(ytemp);

    return;}
                      
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

