/*
 * GSMM.C - memory management routines for PGS
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pgs.h"
#include "gsrast.h"

typedef DECLFPTR(void, PFRDev, (PG_device *dev));

static PROCESS
 *_PG_pp_me = NULL;

static HASHTAB
 *PG_device_table = NULL;

extern void
 SC_DECLARE(_PG_rl_markers, (byte));

static void
 SC_DECLARE(_PG_rl_device_table, (byte)),
 SC_DECLARE(_PG_rl_font_family, (PG_font_family *f));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_RL_ALL - release all memory allocated internally by pgs */

void PG_rl_all()
   {

    SFREE(_PG_text_format);
    SFREE(_PG_axis_label_x_format);
    SFREE(_PG_axis_label_y_format);
    SFREE(_PG_axis_type_face);
    _PG_rl_rst_fonts();
    _PG_rl_device_table();
    _PG_rl_markers();

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_MAKE_GRAPH_1D - assemble a bunch of stuff together in a graph struct
 *                  - which will describe a 1d graph and return a pointer
 *                  - to it
 */

PG_graph *PG_make_graph_1d(id, label, cp, n, x, y, xname, yname)
   int id;
   char *label;
   int cp, n;
   REAL *x, *y;
   char *xname, *yname;
   {PG_graph *g;
    pcons *info;
    PM_set *domain, *range;

/* build the domain set */
    domain = PM_make_set(xname, SC_REAL_S, cp, 1, n, 1, x);

/* build the range set */
    range = PM_make_set(yname, SC_REAL_S, cp, 1, n, 1, y);

/* set default info attributes */
    info = PG_set_line_info(NULL, CARTESIAN, CARTESIAN, SOLID, FALSE,
			    0, 1, 0, 0.0);
    g    = PG_make_graph_from_sets(label, domain, range, N_CENT,
				   SC_PCONS_P_S, info, id, NULL);

    return(g);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_MAKE_GRAPH_R2_R1 - assemble a bunch of stuff together in a graph struct
 *                     - which will describe a mapping from R2 to R1 
 *                     - and return a pointer to it
 */

PG_graph *PG_make_graph_r2_r1(id, label, cp, kmax, lmax, centering,
                              x, y, r, dname, rname)
   int id;
   char *label;
   int cp, kmax, lmax, centering;
   REAL *x, *y, *r;
   char *dname, *rname;
   {PG_graph *g;
    PM_set *domain, *range;

/* build the domain set */
    domain = PM_make_set(dname, SC_REAL_S, cp, 2, kmax, lmax, 2, x, y);

/* build the range set */
    range = PM_make_set(rname, SC_REAL_S, cp, 2, kmax, lmax, 1, r);

    g = PG_make_graph_from_sets(label, domain, range, centering,
                                SC_PCONS_P_S, NULL, id, NULL);

    return(g);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SET_LINE_INFO - set information describing how to
 *                  - render line plots
 */

pcons *PG_set_line_info(info, type, axis_type,
			style, scatter, marker, color, start, width)
   pcons *info;
   int type, axis_type, style, scatter, marker, color, start;
   double width;
   {

    info = PG_set_plot_type(info, type, axis_type);

    SC_CHANGE_VALUE_ALIST(info, int, SC_INTEGER_P_S,
			  "HIST-START", start);

    SC_CHANGE_VALUE_ALIST(info, int, SC_INTEGER_P_S,
			  "SCATTER", scatter);

    SC_CHANGE_VALUE_ALIST(info, int, SC_INTEGER_P_S,
			  "MARKER-INDEX", marker);

    SC_CHANGE_VALUE_ALIST(info, int, SC_INTEGER_P_S,
			  "LINE-COLOR", color);

    SC_CHANGE_VALUE_ALIST(info, int, SC_INTEGER_P_S,
			  "LINE-STYLE", style);

    SC_CHANGE_VALUE_ALIST(info, double, SC_DOUBLE_P_S,
			  "LINE-WIDTH", width);

    return(info);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SET_TDS_INFO - set information describing how to
 *                 - render 2D scalar plots
 */

pcons *PG_set_tds_info(info, type, axis_type, style, color, nlev, ratio,
		       width, theta, phi, chi, d)
   pcons *info;
   int type, axis_type, style, color, nlev;
   double ratio, width, theta, phi, chi, d;
   {

    info = PG_set_plot_type(info, type, axis_type);

    SC_CHANGE_VALUE_ALIST(info, int, SC_INTEGER_P_S,
			  "N-LEVELS", nlev);

    SC_CHANGE_VALUE_ALIST(info, int, SC_INTEGER_P_S,
			  "LINE-COLOR", color);

    SC_CHANGE_VALUE_ALIST(info, int, SC_INTEGER_P_S,
			  "LINE-STYLE", style);

    SC_CHANGE_VALUE_ALIST(info, double, SC_DOUBLE_P_S,
			  "LINE-WIDTH", width);

    SC_CHANGE_VALUE_ALIST(info, double, SC_DOUBLE_P_S,
			  "RATIO", ratio);

    SC_CHANGE_VALUE_ALIST(info, double, SC_DOUBLE_P_S,
			  "THETA", theta);

    SC_CHANGE_VALUE_ALIST(info, double, SC_DOUBLE_P_S,
			  "PHI", phi);

    SC_CHANGE_VALUE_ALIST(info, double, SC_DOUBLE_P_S,
			  "CHI", chi);

    SC_CHANGE_VALUE_ALIST(info, double, SC_DOUBLE_P_S,
			  "DISTANCE", d);

    return(info);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SET_TDV_INFO - set information describing how to
 *                 - render 2D vector plots
 */

pcons *PG_set_tdv_info(info, type, axis_type, style, color, width)
   pcons *info;
   int type, axis_type, style, color;
   double width;
   {

    info = PG_set_plot_type(info, type, axis_type);

    SC_CHANGE_VALUE_ALIST(info, int, SC_INTEGER_P_S,
			  "LINE-COLOR", color);

    SC_CHANGE_VALUE_ALIST(info, int, SC_INTEGER_P_S,
			  "LINE-STYLE", style);

    SC_CHANGE_VALUE_ALIST(info, double, SC_DOUBLE_P_S,
			  "LINE-WIDTH", width);

    return(info);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_MAKE_GRAPH_FROM_SETS - given a domain, range, and attributes bind them
 *                         - all together in a freshly allocated PG_graph
 *                         - and return a pointer to it
 */

PG_graph *PG_make_graph_from_sets(name, domain, range, centering,
                                  info_type, info, id, next)
   char *name;
   PM_set *domain, *range;
   int centering;
   char *info_type;
   byte *info;
   int id;
   PG_graph *next;
   {PM_mapping *f;
    PG_graph *g;

/* build the mapping */
    if (domain->topology == NULL)
       f = PM_make_mapping(name, PM_LR_S, domain, range, centering, NULL);
    else
       f = PM_make_mapping(name, PM_AC_S, domain, range, centering, NULL);

    g = PG_make_graph_from_mapping(f, info_type, info, id, NULL);
    g->next = next;

    return(g);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_MAKE_GRAPH_FROM_MAPPING - given a mapping and attributes bind them
 *                            - all together in a freshly allocated PG_graph
 *                            - and return a pointer to it
 */

PG_graph *PG_make_graph_from_mapping(f, info_type, info, id, next)
   PM_mapping *f;
   char *info_type;
   byte *info;
   int id;
   PG_graph *next;
   {PG_graph *g;

/* build the graph */
    g             = FMAKE(PG_graph, "PG_MAKE_GRAPH_FROM_MAPPING:g");
    g->f          = f;
    g->info_type  = info_type;
    g->info       = info;
    g->identifier = id;
    g->use        = NULL;
    g->render     = NULL;
    g->mesh       = FALSE;
    g->next       = next;

    return(g);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_RL_GRAPH - free a graph */

void PG_rl_graph(g, rld, rlr)
   PG_graph *g;
   int rld, rlr;
   {pcons *inf;

    if (g->info_type != NULL)
       {if (strcmp(g->info_type, SC_PCONS_P_S) == 0)
	   {inf = (pcons *) g->info;
	    if (inf != NULL)
	       SC_free_alist(inf, 3);};};

    if (g->f != NULL)
       PM_rel_mapping(g->f, rld, rlr);

    SFREE(g);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_REGISTER_DEVICE - let users add their own devices
 *                    - to do so they must register a setup function
 *                    - under the device name
 */

void PG_register_device(name, fnc)
   char *name;
   PFVoid fnc;
   {SC_address ad;
    static char *type = NULL;

    if (PG_device_table == NULL)
       {type = SC_strsavef("PFVoid", "char*:PG_REGISTER_DEVICE:type");
	PG_device_table = SC_make_hash_table(HSZSMALL, NODOC);
/* register the default set of devices */

#ifdef HAVE_WINDOW_DEVICE
	PG_register_device("SCREEN", PG_setup_window_device);
#endif

#ifdef HAVE_JPEGLIB
	PG_register_device("JPEG", PG_setup_jpeg_device);
#endif

#ifndef MSC
#ifndef MAC
	PG_register_device("MPEG", PG_setup_mpeg_device);
#endif
#endif

	PG_register_device("PS", PG_setup_ps_device);
	PG_register_device("CGM", PG_setup_cgm_device);};

    if ((name != NULL) && (fnc != NULL))
       {ad = SC_SET_FUNC_ADDR(fnc);

	_SC_install(name, SC_GET_MEM_ADDR(byte *, ad),
		    type, PG_device_table, FALSE, TRUE);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RL_DEVICE_TABLE - release memory associated with the device table */

static void _PG_rl_device_table()
   {hashel *np;

    if (PG_device_table != NULL)
       {np = SC_lookup("PS", PG_device_table);
	SFREE(np->type);
	SC_rl_hash_table(PG_device_table);
        PG_device_table = NULL;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_INIT_DEV_PARALLEL - initialize parallel rendering info for a device */

static PG_par_rend_info *_PG_init_dev_parallel(dd, dp)
   PG_device *dd;
   int dp;
   {int ip, np;
    int *map;
    PG_par_rend_info *pri;
    
    pri = NULL;
    if (_PG_pp_me != NULL)
       {np = PG_get_number_processors();
	ip = PG_get_processor_number();

	map = (ip == dp) ?
	      FMAKE_N(int, np, "_PG_INIT_DEV_PARALLEL:map") :
	      NULL;

	pri = FMAKE(PG_par_rend_info, "_PG_INIT_DEV_PARALLEL:pri");
	pri->dd        = dd;
	pri->pp        = _PG_pp_me;
	pri->ip        = dp;
	pri->have_data = TRUE;
	pri->map       = map;
	pri->alist     = NULL;
	pri->label     = NULL;
	pri->render    = 0;
	pri->polar     = FALSE;};

    return(pri);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_MAKE_RAW_DEVICE - initialize a graphical device to lowest order */

static PG_device *PG_make_raw_device(name, type, title, dd, dp)
   char *name, *type, *title;
   PG_device *dd;
   int dp;
   {PG_device *d;
    char bf[MAXLINE], *pttl, *pname;
    PFRDev fnc;

#ifndef HAVE_JPEGLIB
    if (!strcmp(name, "JPEG"))
       return(NULL);
#endif

    d = FMAKE(PG_device, "PG_MAKE_RAW_DEVICE:d");

    if (d == NULL)
       return(NULL);

#if 0
    sprintf(bf, "%s [PROC:%d]", title, dp);
    pttl = bf;
#else
    pttl = title;
#endif

/* parallel graphics info */
    d->pri = _PG_init_dev_parallel(dd, dp);

/* generic device info */

    d->absolute_n_color        = 256;
    d->autodomain              = TRUE;
    d->autorange               = TRUE;

    d->background_color_white  = TRUE;
    d->border_width            = 1;
    d->botspace                = 0.01;

/* character controls */
    d->char_frac               = 0.015;
    d->char_height_s           = 0.0;
    d->char_path_x             = 1.0;
    d->char_path_y             = 0.0;
    d->char_precision          = 1;
    d->char_space              = 0.0;
    d->char_space_s            = 0.0;
    d->char_up_x               = 0.0;
    d->char_up_y               = 1.0;
    d->char_width_s            = 0.0;

    d->clipping                = FALSE;
    d->color_table             = NULL;
    d->current_palette         = NULL;
    d->draw_fill_bound         = FALSE;
    d->file                    = NULL;
    d->fill_color              = 0;
    d->finished                = TRUE;

    d->font_family             = NULL;

/* frame setup */
    d->fxmin                   = 0.0;
    d->fxmax                   = 1.0;
    d->fymin                   = 0.0;
    d->fymax                   = 1.0;

/* WC limits of viewport */
    d->gxmin                   = 0.0;
    d->gxmax                   = 1.0;
    d->gymin                   = 0.0;
    d->gymax                   = 1.0;

/* graphics cursor location */
    d->gcurx                   = 0.0;
    d->gcury                   = 0.0;

    d->gprint_flag             = TRUE;
    d->grid                    = OFF;

/* TRUE for hard copy devices */
    d->hard_copy_device        = FALSE;

    d->ifxlog                  = FALSE;
    d->ifylog                  = FALSE;
    d->interface_objects       = NULL;
    d->is_visible              = TRUE;
    d->leftspace               = 0.01;
    d->line_style              = LINE_SOLID;
    d->line_color              = 1;
    d->line_width              = 0.1;
    d->logical_op              = GS_COPY;

    d->marker_scale            = 0.01;
    d->marker_orientation      = 0.0;
    d->match_rgb_colors        = NULL;

    d->max_intensity           = 1.0;
    d->max_interface_objects   = 0;
    d->max_red_intensity       = 1.0;
    d->max_green_intensity     = 1.0;
    d->max_blue_intensity      = 1.0;

/* color mapping flags */
    d->map_text_color          = TRUE;
    d->map_line_color          = TRUE;
    d->map_fill_color          = TRUE;

/* device limits in pixel coordinates */
    d->min_pc_x                = 0;
    d->min_pc_y                = 0;
    d->max_pc_x                = 0;
    d->max_pc_y                = 0;

    d->mode                    = -1;
    d->name                    = SC_strsavef(name, "char*:PG_MAKE_RAW_DEVICE:name");
    d->ncolor                  = 16;
    d->n_interface_objects     = 0;
    d->palettes                = NULL;
    d->print_labels            = FALSE;
    d->quadrant                = 0;
    d->range_n_extrema         = 0;
    d->range_extrema           = NULL;
    d->re_open                 = FALSE;
    d->resolution_scale_factor = 1;
    d->rightspace              = 0.01;
    d->scatter                 = 0;
    d->supress_setup           = FALSE;
    d->text_color              = 1;
    d->title                   = SC_strsavef(pttl,
                                 "char*:PG_MAKE_RAW_DEVICE:title");

/* text cursor location */
    d->tcurx                   = 0.0;
    d->tcury                   = 0.0;

    d->topspace                = 0.01;

/* text window */
    d->txmax                   = 1.0;
    d->txmin                   = 0.0;
    d->tymax                   = 1.0;
    d->tymin                   = 0.0;

    d->txt_ratio               = 1.0;
    d->type                    = SC_str_upper(SC_strsavef(type,
                                 "char*:PG_MAKE_RAW_DEVICE:type"));

    d->type_face               = SC_strsavef("helvetica",
                                 "char*:PG_MAKE_RAW_DEVICE:face");
    d->type_style              = SC_strsavef("medium",
                                 "char*:PG_MAKE_RAW_DEVICE:style");
    d->type_size               = 12;

/* viewport location/size in window in NDC */
#ifdef HAVE_WINDOW_DEVICE
    d->window                  = 0;
#endif
    d->view_aspect             = 1.0;
    d->view_height             = 0.0;
    d->view_width              = 0.0;
    d->view_x                  = 0.0;
    d->view_y                  = 0.0;

    d->xmin                    = 0.0;
    d->xmax                    = 1.0;
    d->ymin                    = 0.0;
    d->ymax                    = 1.0;

    d->sxmin                   = 0.0;
    d->sxmax                   = 1.0;
    d->sxmin                   = 0.0;
    d->sxmax                   = 1.0;

/* window location/size in NDC */
    d->window_height           = 0.0;
    d->window_width            = 0.0;
    d->window_x                = 0.0; 
    d->window_y                = 0.0;

    d->window_x_off            = 0;
    d->window_y_off            = 0;

    d->xor_parity              = 0;

    d->expose_event_handler.fnc      = NULL;
    d->expose_event_handler.lang     = _C_LANG;

    d->update_event_handler.fnc      = NULL;
    d->update_event_handler.lang     = _C_LANG;

    d->mouse_down_event_handler.fnc  = NULL;
    d->mouse_down_event_handler.lang = _C_LANG;

    d->mouse_up_event_handler.fnc    = NULL;
    d->mouse_up_event_handler.lang   = _C_LANG;

    d->motion_event_handler.fnc      = NULL;
    d->motion_event_handler.lang     = _C_LANG;

    d->key_down_event_handler.fnc    = NULL;
    d->key_down_event_handler.lang   = _C_LANG;

    d->key_up_event_handler.fnc      = NULL;
    d->key_up_event_handler.lang     = _C_LANG;

    d->default_event_handler.fnc     = NULL;
    d->default_event_handler.lang    = _C_LANG;

    if (PG_device_table == NULL)
       PG_register_device(NULL, NULL);

    strcpy(bf, name);
    pname = SC_firsttok(bf, " \t\r\n");
    fnc = SC_LOOKUP_FUNCTION(PFRDev, pname, PG_device_table);
    if (fnc != NULL)
       (*fnc)(d);

    else
       {fnc = SC_LOOKUP_FUNCTION(PFRDev, "SCREEN", PG_device_table);
	if (fnc != NULL)
	   (*fnc)(d);
	else
	   {_PG_rl_device(d);
	    return(NULL);};};

/* by default make world coordinates be the same as NDC */
    PG_set_viewport(d, 0.0, 1.0, 0.0, 1.0);
    PG_set_window(d, 0.0, 1.0, 0.0, 1.0);

    return(d);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_MAKE_DEVICE - initialize a graphical device */

PG_device *PG_make_device(name, type, title)
   char *name, *type, *title;
   {int ip, np;
    char *fname;
    PG_device *d, *dd;
    static int dp = 0;

    dd    = NULL;
    fname = name;

    np = PG_get_number_processors();
    if (np > 1)
       {ip = PG_get_processor_number();
	if (ip == dp)
	   dd = PG_make_raw_device(name, type, title, NULL, dp);

	fname = "IMAGE";}
	
    d  = PG_make_raw_device(fname, type, title, dd, dp);
    dp = (dp + 1) % np;

    return(d);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RL_DEVICE - release the memory associated with a graphical device */

void _PG_rl_device(dev)
   PG_device *dev;
   {int i, n;
    PG_font_family *f, *nxt_f;
    PG_interface_object **iobs;
    PG_palette *pal, *nxt_p;
    PG_device *dd;
    PG_par_rend_info *pri;

    SFREE(dev->name);
    SFREE(dev->range_extrema);
    SFREE(dev->title);
    SFREE(dev->type);
    SFREE(dev->type_face);
    SFREE(dev->type_style);

/* free any parallel rendering info */
    pri = dev->pri;
    if (pri != NULL)
       {dd = pri->dd;

/* NOTE: leave pri->pp it is really a global guy at this time */
	if (dd != NULL)
	   PG_close_device(dd);

	SFREE(pri->map);
	SFREE(pri->label);
        SFREE(pri);};

/* free any font info */
    for (f = dev->font_family; f != NULL; f = nxt_f)
	{nxt_f = f->next;
	 _PG_rl_font_family(f);};

/* free interface objects */
    iobs = dev->interface_objects;
    if (iobs != NULL)
       {n = dev->n_interface_objects;
        for (i = 0; i < n; i++)
            {if (iobs[i] != NULL)
                _PG_rl_interface_object(iobs[i], TRUE);};
        SFREE(iobs);};

/* free palettes and color tables */
    if (dev->palettes != NULL)
       {for (pal = dev->palettes; pal->next != NULL; pal = nxt_p)
	    {nxt_p = pal->next;
	     PG_rl_palette(pal);
	     if (nxt_p == dev->palettes)
	        break;};};

    if (dev->color_table != NULL)
       PG_rl_palette(dev->color_table);

/* free device dependent info */

#ifdef USE_SUNCORE

    if (dev->vwsurf != NULL)
       {if (dev->vwsurf->ptr != NULL)
           {SFREE(*(dev->vwsurf->ptr));
            SFREE(dev->vwsurf->ptr);};
        SFREE(dev->vwsurf);};
 
#endif

    SFREE(dev);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_INIT_PARALLEL - initialize parallel graphics */

int PG_init_parallel(argv, pp)
   char **argv;
   PROCESS *pp;
   {int ret;
    PDBfile *vif;

    if (pp == NULL)
       {_PG_pp_me = PC_open_member(argv, NULL);

	vif = _PG_pp_me->vif;

	PD_def_mapping(vif);

        ret = TRUE;}

    else
       {_PG_pp_me = pp;

	ret = FALSE;};

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_FIN_PARALLEL - terminate parallel graphics */

int PG_fin_parallel(trm)
   int trm;
   {

    if (trm)
       PC_close_member(_PG_pp_me);

    return(trm);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_MAKE_FONT_FAMILY - encapsulate the information for a family of 
 *                     - text fonts
 */

#ifdef PCC

PG_font_family *PG_make_font_family(dev, name, next, n, va_alist)
   PG_device *dev;
   char *name;
   PG_font_family *next;
   int n;
   va_dcl

#endif

#ifdef ANSI

PG_font_family *PG_make_font_family(PG_device *dev, char *name,
                                    PG_font_family *next, int n, ...)

#endif

   {PG_font_family *f;
    int i;
    char **fs;

    f  = FMAKE(PG_font_family, "PG_MAKE_FONT_FAMILY:f");
    fs = FMAKE_N(char *, n, "PG_MAKE_FONT_FAMILY:fs");

    SC_VA_START(n);

    for (i = 0; i < n; i++)
        fs[i] = SC_strsavef(SC_VA_ARG(char *),
                "char*:PG_MAKE_FONT_FAMILY:fs");

    SC_VA_END;

    f->type_face   = SC_strsavef(name,
                     "char*:PG_MAKE_FONT_FAMILY:name");
    f->n_styles    = n;
    f->type_styles = fs;
    f->next        = next;

    return(f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RL_FONT_FAMILY - release the memory associated with a font family */

static void _PG_rl_font_family(f)
   PG_font_family *f;
   {int i, n;
    char **fs;

    SFREE(f->type_face);

    n  = f->n_styles;
    fs = f->type_styles;
    for (i = 0; i < n; i++)
        SFREE(fs[i]);
    SFREE(fs);

    f->next = NULL;

    SFREE(f);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_MAKE_VIEW_ATTRIBUTES - initialize and set a view attributes struct */

PG_view_attributes *PG_make_view_attributes(dev)
   PG_device *dev;
   {PG_view_attributes *d;

    d = FMAKE(PG_view_attributes, "PG_MAKE_VIEW_ATTRIBUTES:d");
    PG_save_view_attributes(d, dev);

    return(d);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SAVE_VIEW_ATTRIBUTES - save the view attributes */

void PG_save_view_attributes(d, dev)
   PG_view_attributes *d;
   PG_device *dev;
   {

    d->botspace    = dev->botspace;
    d->leftspace   = dev->leftspace;
    d->rightspace  = dev->rightspace;
    d->topspace    = dev->topspace;

/* transformations between WC and NDC */
    d->axs_w       = dev->axs_w;
    d->axw_s       = dev->axw_s;
    d->ayw_s       = dev->ayw_s;
    d->ays_w       = dev->ays_w;
    d->bxs_w       = dev->bxs_w;
    d->bxw_s       = dev->bxw_s;
    d->bys_w       = dev->bys_w;
    d->byw_s       = dev->byw_s;

    d->finished    = dev->finished;

/* graphics cursor location in WC */
    d->gcurx       = dev->gcurx;
    d->gcury       = dev->gcury;

/* clipping limits in WC */
    d->gxmax       = dev->gxmax;
    d->gxmin       = dev->gxmin;
    d->gymax       = dev->gymax;
    d->gymin       = dev->gymin;

    d->ifxlog      = dev->ifxlog;
    d->ifylog      = dev->ifylog;

/* viewport limits in NDC */
    d->sxmax       = dev->sxmax;
    d->sxmin       = dev->sxmin;
    d->symax       = dev->symax;
    d->symin       = dev->symin;
    
/* text cursor location in WC */
    d->tcurx       = dev->tcurx;
    d->tcury       = dev->tcury;

/* text window in WC */
    d->txmax       = dev->txmax;
    d->txmin       = dev->txmin;
    d->tymax       = dev->tymax;
    d->tymin       = dev->tymin;

/* viewport location/size in window in NDC */
    d->view_aspect = dev->view_aspect;
    d->view_height = dev->view_height;
    d->view_width  = dev->view_width;
    d->view_x      = dev->view_x;
    d->view_y      = dev->view_y;

/* window limits in WC */
    d->xmax        = dev->xmax;
    d->xmin        = dev->xmin;
    d->ymax        = dev->ymax;
    d->ymin        = dev->ymin;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_RESTORE_VIEW_ATTRIBUTES - restore the PG_view_attributes to the
 *                            - graphical device
 */

void PG_restore_view_attributes(dev, d)
   PG_device *dev;
   PG_view_attributes *d;
   {dev->botspace    = d->botspace;
    dev->leftspace   = d->leftspace;
    dev->rightspace  = d->rightspace;
    dev->topspace    = d->topspace;

/* transformations between WC and NDC */
    dev->axs_w       = d->axs_w;
    dev->axw_s       = d->axw_s;
    dev->ayw_s       = d->ayw_s;
    dev->ays_w       = d->ays_w;
    dev->bxs_w       = d->bxs_w;
    dev->bxw_s       = d->bxw_s;
    dev->bys_w       = d->bys_w;
    dev->byw_s       = d->byw_s;

    dev->finished    = d->finished;

/* graphics cursor location in WC */
    dev->gcurx       = d->gcurx;
    dev->gcury       = d->gcury;

/* clipping limits in WC */
    dev->gxmax       = d->gxmax;
    dev->gxmin       = d->gxmin;
    dev->gymax       = d->gymax;
    dev->gymin       = d->gymin;

    dev->ifxlog      = d->ifxlog;
    dev->ifylog      = d->ifylog;

/* viewport limits in NDC */
    dev->sxmax       = d->sxmax;
    dev->sxmin       = d->sxmin;
    dev->symax       = d->symax;
    dev->symin       = d->symin;
    
/* text cursor location in WC */
    dev->tcurx       = d->tcurx;
    dev->tcury       = d->tcury;

/* text window in WC */
    dev->txmax       = d->txmax;
    dev->txmin       = d->txmin;
    dev->tymax       = d->tymax;
    dev->tymin       = d->tymin;

/* viewport location/size in window in NDC */
    dev->view_aspect = d->view_aspect;
    dev->view_height = d->view_height;
    dev->view_width  = d->view_width;
    dev->view_x      = d->view_x;
    dev->view_y      = d->view_y;

/* window limits in WC */
    dev->xmax        = d->xmax;
    dev->xmin        = d->xmin;
    dev->ymax        = d->ymax;
    dev->ymin        = d->ymin;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_MAKE_CURVE - allocate and initialize a curve
 *               - MODE: WORLDC, NORMC, PIXELC
 */

PG_curve *PG_make_curve(dev, mode, closed, n, xo, yo, xd, yd)
   PG_device *dev;
   int mode, closed, n;
   double xo, yo;
   REAL *xd, *yd;
   {int i, ix, iy, ixo, iyo;
    int *xi, *yi;
    REAL *px, *py;
    double x1, y1;
    PG_curve *crv;

    px = xd;
    py = yd;

    crv = FMAKE(PG_curve, "PG_MAKE_CURVE:crv");
    xi  = FMAKE_N(int, n, "PG_MAKE_CURVE:xi");
    yi  = FMAKE_N(int, n, "PG_MAKE_CURVE:yi");
    switch (mode)
       {case WORLDC :
             if (dev->ifxlog)
                xo = log10(ABS(xo) + SMALL);
             if (dev->ifylog)
                yo = log10(ABS(yo) + SMALL);
          
	     WtoS(dev, xo, yo);
	     StoP(dev, xo, yo, ixo, iyo);

             for (i = 0; i < n; i++)
                 {x1 = *px++;
                  y1 = *py++;
          
                  if (dev->ifxlog)
                     x1 = log10(ABS(x1) + SMALL);
                  if (dev->ifylog)
                     y1 = log10(ABS(y1) + SMALL);
          
                  WtoS(dev, x1, y1);
                  StoP(dev, x1, y1, ix, iy);

                  xi[i] = ix - ixo;
                  yi[i] = iy - iyo;};

             break;

        case NORMC  :
	     StoP(dev, xo, yo, ixo, iyo);

             for (i = 0; i < n; i++)
                 {x1 = *px++;
                  y1 = *py++;
          
                  StoP(dev, x1, y1, ix, iy);
                  xi[i] = ix - ixo;
                  yi[i] = iy - iyo;};

             break;

        case PIXELC :
             ixo = xo;
             iyo = yo;

             for (i = 0; i < n; i++)
                 {ix = *px++;
                  ix = *py++;
          
                  xi[i] = ix - ixo;
                  yi[i] = iy - iyo;};

             break;};

    crv->closed   = closed;
    crv->n        = n;
    crv->x        = xi;
    crv->y        = yi;
    crv->x_origin = ixo;
    crv->y_origin = iyo;

    return(crv);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_RELEASE_CURVE - free a curve */

void PG_release_curve(crv)
   PG_curve *crv;
   {SFREE(crv->x);
    SFREE(crv->y);
    SFREE(crv);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_MAKE_BOX_CURVE - make a curve from a rectangle description */

PG_curve *PG_make_box_curve(dev, mode, xo, yo, xmn, xmx, ymn, ymx)
   PG_device *dev;
   int mode;
   double xo, yo;
   double xmn, xmx, ymn, ymx;
   {REAL px[5], py[5];

    px[0] = xmn;
    py[0] = ymn;

    px[1] = xmx;
    py[1] = ymn;

    px[2] = xmx;
    py[2] = ymx;

    px[3] = xmn;
    py[3] = ymx;

    px[4] = xmn;
    py[4] = ymn;

    return(PG_make_curve(dev, mode, TRUE, 5, xo, yo, px, py));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
