/*
 * PCPOSIX.C - POSIX operating systems routines
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "ppc.h"

static int
 SC_DECLARE(_PC_posix_setup_tty, (PROCESS *pp, int child)),
 SC_DECLARE(_PC_posix_exec, 
            (PROCESS *cp, char **argv, char **env, char *mode)),
 SC_DECLARE(_PC_posix_in_ready, (PROCESS *pp)),
 SC_DECLARE(_PC_posix_release, (PROCESS *pp)),
 SC_DECLARE(_PC_posix_close, (PROCESS *pp)),
 SC_DECLARE(_PC_posix_status, (PROCESS *pp)),
 SC_DECLARE(_PC_posix_flush, (PROCESS *pp)),
 SC_DECLARE(_PC_posix_gets, (char *bf, int len, PROCESS *pp)),
 SC_DECLARE(_PC_posix_printf, (PROCESS *pp, char *buffer));

static void
 SC_DECLARE(_PC_latch_parent, (int in, int out, int medium));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_MK_PROCESS - initialize and return a PROCESS */

PROCESS *PC_mk_process(argv, mode, type)
   char **argv;
   char *mode;
   int type;
   {PROCESS *pp;

    pp = FMAKE(PROCESS, "PC_MK_PROCESS:pp");

    pp->in_ring  = FMAKE_N(unsigned char, SIZE_BUF,
                   "PC_MK_PROCESS:in_ring");
    pp->ib_in    = 0;
    pp->ob_in    = 0;

    pp->type     = PC_LOCAL;
    pp->rcpu     = -1;
    pp->acpu     = -1;

    pp->line_mode = FALSE;
    pp->line_sep  = 0x0a;
    pp->blocking  = FALSE;

    pp->n_pending = 0;
    pp->pending   = NULL;

    if (PC_BINARY_MODEP(mode))

/* contrive a very special looking PDBfile around the PROCESS descriptors */
       {pp->vif = PD_open_vif(argv[0]);
        pp->vif->virtual_internal = FALSE;
        pp->vif->stream = (FILE *) pp;}

    else
       pp->vif = NULL;

/* assign procedures to service this process */
    pp->release    = _PC_posix_release;
    pp->exec       = _PC_posix_exec;
    pp->close      = _PC_posix_close;
    pp->statusp    = _PC_posix_status;
    pp->flush      = _PC_posix_flush;
    pp->read       = _PC_bin_read;
    pp->write      = _PC_bin_write;
    pp->printf     = _PC_posix_printf;
    pp->gets       = _PC_posix_gets;
    pp->in_ready   = _PC_posix_in_ready;
    pp->setup      = _PC_posix_setup_tty;

    if (type == PC_PARENT)
       _PC_manage_process(pp);

    return(pp);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_POSIX_RELEASE - close the file descriptors and release the PROCESS */

static int _PC_posix_release(pp)
   PROCESS *pp;
   {

#ifdef HAVE_PROCESS_CONTROL

    if (pp == NULL)
       return(FALSE);

#ifdef UNIX

    {int in, out;

     in  = pp->in;
     out = pp->out;

     SIGNAL(SIGALRM, SIG_IGN);
     ALARM(1);
     if (in >= 0)
        close(in);
     ALARM(0);
     if ((in != out) && (out >= 0))
        close(out);};

#endif

    _PC_rl_process(pp);

#endif

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_POSIX_SETUP_TTY - set up the communication channel to the child
 *                     - as if it were a terminal
 */

static int _PC_posix_setup_tty(pp, child)
   PROCESS *pp;
   int child;
   {int in, medium;

#ifndef MSC

    if (pp == NULL)
       return(FALSE);

    medium = pp->medium;

    if (child)

/* when using PTY's now is the time to open the slave side of the PTY */
       {if (medium == USE_PTYS)
#ifndef OPENNT                              
           {in = open(pp->spty, O_RDWR & ~O_NDELAY);          
#else                                                      
           {in = open(pp->spty, O_RDWR & ~O_NONBLOCK);
#endif 
            if (in < 0)
               PC_error("COULDN'T OPEN SLAVE %s - _PC_POSIX_SETUP_TTY",
                        pp->spty);

/* now close the master */
            if (in != pp->in)
               {if (close(pp->in) < 0)
                   PC_error("COULDN'T CLOSE MASTER - _PC_POSIX_SETUP_TTY");};

            pp->out = pp->in = in;};};

    in = pp->in;

#ifndef LATCH_AFTER_RECONNECT

/* latch up with the parent */
    _PC_latch_parent(in, pp->out, medium);

#endif

    if ((medium == USE_PTYS) && (PC_original_terminal_state != NULL))
       PC_set_term_state(PC_original_terminal_state, in);

    PC_block_fd(in);

    return(TRUE);}
#else
    return(FALSE);}
#endif
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_POSIX_EXEC - exec the named program together with args and env
 *                - in the new pro
 *                - if successful this routine should not return
 *                - return FALSE indicating failure
 */

static int _PC_posix_exec(cp, argv, env, mode)
   PROCESS *cp;
   char **argv;
   char **env;
   char *mode;
   {int child_in, child_out;

#ifdef HAVE_PROCESS_CONTROL

    if (cp == NULL)
       return(FALSE);

/* break the association with the control terminal */

#ifdef SYSV
    if (setsid() < 0)
        PC_error("COULDN'T DITCH CONTROLLING TERMINAL - _PC_POSIX_EXEC");
#endif

#ifdef BSD
    {int fd;

     fd = open("/dev/tty", O_RDWR);
     if (fd >= 0)
        {if (ioctl(fd, TIOCNOTTY, NULL) < 0)
	    PC_error("COULDN'T DITCH CONTROLLING TERMINAL - _PC_POSIX_EXEC");
	 close(fd);};};
#endif

/* setup the tty first */
    cp->setup(cp, TRUE);

    child_in  = cp->in;
    child_out = cp->out;

/* set the process stdin, stdout, and stderr to those from the pipe */
    if (dup2(child_in, 0) < 0)
       PC_error("COULDN'T DUP %d TO STDIN (%d) - _PC_POSIX_EXEC",
                child_in, errno);

    if (dup2(child_out, 1) < 0)
       PC_error("COULDN'T DUP %d TO STDOUT (%d) - _PC_POSIX_EXEC",
                child_out, errno);

    if (dup2(child_out, 2) < 0)
       PC_error("COULDN'T DUP %d TO STDERR (%d) - _PC_POSIX_EXEC",
                child_out, errno);

/* now that they are copied release the old values */
    if (child_out != child_in)
       close(child_out);

    if (close(child_in) < 0)
       PC_error("COULDN'T CLOSE DESCRIPTOR - _PC_POSIX_EXEC");

#ifdef LATCH_AFTER_RECONNECT

/* latch up with the parent */
    _PC_latch_parent(0, 1, cp->medium);

#endif

/* if this is a binary connection inform the parent of the binary formats */
    if (cp->data_type == BINARY)

/* keep the info about the parent for use in binary I/O */
       {cp->in = 0;
	cp->out = 1;
	PC_terminal = cp;
        if (!PC_send_formats())
           PC_error("COULDN'T SEND FORMATS - _PC_POSIX_EXEC");}
    else
       PC_terminal = NULL;

/* exec the process with args and environment - this won't return */
    execvp(argv[0], argv);

/* if we get here the exec failed */
    PC_error("COULDN'T EXEC %s - _PC_POSIX_EXEC", argv[0]);

#endif

    return(FALSE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_LATCH_PARENT - latch up with the parent
 *                  - this is done to prevent the child from running
 *                  - to completion before the parent even registers
 *                  - the child's existence so as to be able to catch
 *                  - the SIGCHLD that will be sent when the child exits
 */

static void _PC_latch_parent(in, out, medium)
   int in, out, medium;
   {char exch[10];

#ifdef NO_LATCH_PTYS
    if (medium == USE_PTYS)
       return;
#endif

    PC_block_fd(in);

/* set the alarm */
    PC_alarm(5, NULL);

    exch[0] = '\0';
    exch[1] = '\0';
    while (exch[0] != 'A')
       {if (read(in, exch, 1) != 1)
           PC_error("CHILD READ FAILED (%d) - _PC_LATCH_PARENT", errno);};

    PC_unblock_fd(in);

#ifdef AIX
    read(in, exch, 1);
#else
    while (read(in, exch, 1) > 0);
#endif

/* reset the alarm */
    PC_alarm(0, NULL);

    exch[0] = 'B';
    exch[1] = '\n';
    exch[2] = '\0';
    if (write(out, exch, 2) != 2)
       PC_error("CAN'T WRITE TO PARENT - _PC_LATCH_PARENT");

    PC_block_fd(in);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_POSIX_STATUS - return the status of the process */

static int _PC_posix_status(pp)
   PROCESS *pp;
   {

/* flush socket first to generate SIGPIPE if other end is gone */
    if (pp->medium == USE_SOCKETS)
       PC_flush(pp);
/*
       {char c;
        c = '\021';
        write(pp->out, &c, 1);};
*/
    if (pp == NULL)
       return(EXITED);
    else
       return(pp->status & ~CHANGED);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_POSIX_GETS - get bytes from the input stream
 *                - and return the number of bytes placed in
 *                - the buffer
 */

static int _PC_posix_gets(bf, len, pp)
   char *bf;
   int len;
   PROCESS *pp;
   {int n;

    PC_flush(pp);

    n = read(pp->in, bf, len);
    switch (n)
       {case 0 :
             return(0);
        case -1 :
             return(0);
        default :
             return(n);};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_POSIX_IN_READY - return input ready status of process */

static int _PC_posix_in_ready(pp)
   PROCESS *pp;
   {int status;

#ifdef MSC
    return(0);}
#else
    status = fcntl(pp->in, F_GETFL);

#ifdef OPENNT
    return(status & O_NONBLOCK);}                     
#else
    return(status & O_NDELAY);}                               
#endif
#endif

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_POSIX_CLOSE - kill the process in effect closing the file
 *                 - release the PROCESS (be careful about using pp
 *                 - after this!!) clean up the details too
 */

static int _PC_posix_close(pp)
   PROCESS *pp;
   {int id;

    PC_flush(pp);

    if (pp == NULL)
       return(FALSE);

    id = pp->id;

    _PC_delete_process(pp);

    PC_catch_io_interrupts(FALSE);

/* kill the process itself
 * NOTE: if the process is remote, then we don't know the pid
 *       and id will be 0
 */
    if (id > 0)
       kill(id , 9);

/* close the communications pipe */
    pp->release(pp);

    PC_catch_io_interrupts(TRUE);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_POSIX_FLUSH - flush the input and output streams
 *                 - for the given process
 */

static int _PC_posix_flush(pp)
   PROCESS *pp;
   {int iv, ov;

    iv = -2;
    ov = -2;

    if (pp == NULL)
       return(FALSE);

    _PC_current_flushed_process = pp->id;

#ifdef BSD

    iv = ioctl(pp->in, TIOCFLUSH, &One_I);
    if (pp->out != pp->in)
       ov = ioctl(pp->out, TIOCFLUSH, &One_I);
    else
       ov = iv;

#endif

#ifdef SYSV

/* System V appears to interpret flush as "throw it away" */

    iv = ov = 0;

#endif

    return((iv < 0) | ((ov < 0) << 1));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_POSIX_PRINTF - do an fprintf style output to a process */

static int _PC_posix_printf(pp, buffer)
   PROCESS *pp;
   char *buffer;
   {int len, nb;

    len = strlen(buffer);

    PC_ERR_TRAP(FALSE);

    nb = write(pp->out, buffer, len);
    if (nb != len)
       PC_error("FAILED WRITE %d OF %d BYTES - _PC_POSIX_PRINTF",
                nb, len);

    PC_flush(pp);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
