/*
 * SHREAD.C - C and Scheme Primitive Input Routines
 *
 * Source Version: 4.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "scheme.h"

#define NEXT_CHAR (*SS_pr_ch_in)
#define PUSH_CHAR (*SS_pr_ch_un)

PFInt
 SS_pr_ch_in;

PFVoid
 SS_pr_ch_un;

object
 *SS_indev;

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_PR_READ - the C level reader */

object *_SS_pr_read(str)
   object *str;
   {int c;

    c = NEXT_CHAR(str, TRUE);
    switch (c)
       {case EOF : return(SS_eof);
        case '(' : return(SS_rd_lst(str));
        case '\"': return(SS_rd_str(str));
        case '\'': return(SS_mk_cons(SS_quoteproc,
                                     SS_mk_cons(_SS_pr_read(str),
                                                SS_null)));
        case '`' : return(SS_mk_cons(SS_quasiproc,
                                     SS_mk_cons(_SS_pr_read(str),
                                                SS_null)));
        case ',' : if ((c = NEXT_CHAR(str, TRUE)) == '@')
                      return(SS_mk_cons(SS_unqspproc,
                                        SS_mk_cons(_SS_pr_read(str),
                                                   SS_null)));
                   else
                      {PUSH_CHAR(c, str);
                       return(SS_mk_cons(SS_unqproc,
                                         SS_mk_cons(_SS_pr_read(str),
                                                    SS_null)));};
#ifdef LARGE
        case '#' : if ((c = NEXT_CHAR(str, FALSE)) == '(')
                      return(SS_rd_vct(str));
                   else
                      {PUSH_CHAR(c, str);
                       PUSH_CHAR('#', str);
                       return(SS_rd_atm(str));};
#endif
        default  : PUSH_CHAR(c, str);
                   return(SS_rd_atm(str));};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_READ - the C level reader
 *          - this level is here to implement the transcript mechanism
 */

object *_SS_read(str)
   object *str;
   {object *obj;

    obj = _SS_pr_read(str);

    switch (SS_hist_flag)
       {case STDIN_ONLY : if (str != SS_indev)
                             break;
        case ALL        : _SS_print(obj, "", "\r\n", SS_histdev);
        default         : break;};

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_READ - the Scheme level reader which invokes the C level reader */

object *SS_read(obj)
   object *obj;
   {object *op;

    if (SS_nullobjp(obj))
       return(_SS_read(SS_indev));

    else if (SS_inportp(op = SS_car(obj)))
       return(_SS_read(op));

    else
       SS_error("ARGUMENT TO READ NOT INPUT-PORT", obj);

    return(SS_null);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_OPN_IN - open-input-file in Scheme */

object *SS_opn_in(obj)
   object *obj;
   {FILE *str;
    char *s, *t;

    if (SS_stringp(obj))
       s = SS_STRING_TEXT(obj);
    else if (SS_variablep(obj))
       s = SS_VARIABLE_NAME(obj);
    else
       SS_error("BAD STRING TO OPEN-INPUT-FILE", obj);

    str = io_open(s, "r");
    if (str == NULL)
       {t = SC_search_file(SC_path, s);
        if (t == NULL)
           SS_error("CAN'T FIND FILE - OPEN-INPUT-FILE", obj);

        str = io_open(t, "r");
        if (str == NULL)
           SS_error("CAN'T OPEN FILE - OPEN-INPUT-FILE", obj);};

    return(SS_mk_inport(str));}

/*--------------------------------------------------------------------------*/

#ifdef LARGE

/*--------------------------------------------------------------------------*/

/* SS_CALL_IF - call-with-input-file in Scheme */

object *SS_call_if(argl)
   object *argl;
   {FILE *str;
    char *s;
    object *obj, *old_indev, *ret;

    obj = SS_car(argl);

    if (SS_stringp(obj))
       s = SS_STRING_TEXT(obj);
    else if (SS_variablep(obj))
       s = SS_VARIABLE_NAME(obj);
    else
       SS_error("BAD STRING TO OPEN-INPUT-FILE", obj);

    str = io_open(s, BINARY_MODE_R);
    if (str == NULL)
       SS_error("CAN'T OPEN FILE - OPEN-INPUT-FILE", obj);

    old_indev = SS_indev;
    SS_indev  = SS_mk_inport(str);
    ret       = SS_exp_eval(SS_cdr(argl));

    SS_cls_in(SS_indev);

    SS_indev = old_indev;

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_CURR_IP - current-input-port in Scheme */

object *SS_curr_ip()
    {return(SS_indev);}

/*--------------------------------------------------------------------------*/

#endif

/*--------------------------------------------------------------------------*/

/* SS_CLS_IN - close-input-port in Scheme */

object *SS_cls_in(obj)
   object *obj;
   {if (!SS_inportp(obj))
       SS_error("BAD INPUT-PORT TO CLOSE-INPUT-PORT", obj);

    io_close(SS_INSTREAM(obj));

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_IPORTP - input-port? at Scheme level */

object *SS_iportp(obj)
   object *obj;
   {return(SS_inportp(obj) ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_STRPRT - string->port at the Scheme level */

object *SS_strprt(arg)
   object *arg;
   {object *port;

    if (!SS_stringp(arg))
       SS_error("BAD STRING TO STRING->PORT", arg);

    port = SS_mk_inport(NULL);
    strcpy(SS_BUFFER(port), SS_STRING_TEXT(arg));
    SS_PTR(port) = SS_BUFFER(port);

    return(port);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_RD_LST - read a list or cons */

object *SS_rd_lst(str)
   object *str;
   {object *First, *This, *Next;
    int c;

    First = SS_null;
    This  = SS_null;
    while (TRUE)
       switch (c = NEXT_CHAR(str, TRUE))
          {case ')' :
	        return(First);

           case '.' :
	        c = NEXT_CHAR(str, FALSE);
	        if (strchr(" \t\r\n", c) != NULL)
                   {_SS_setcdr(This, _SS_pr_read(str));
                    while ((c = NEXT_CHAR(str, TRUE)) != ')');
		    return(First);};

	        PUSH_CHAR(c, str);
                PUSH_CHAR('.', str);
                c = ' ';

           default :
               PUSH_CHAR(c, str);
	       Next = _SS_pr_read(str);
	       if (SS_eofobjp(Next))
                  {if ((c = NEXT_CHAR(str, TRUE)) == EOF)
                      {SS_clr_strm(str);
                       SS_error("UNEXPECTED END OF FILE - READ-LIST",
                                SS_null);}
                   else
                      PUSH_CHAR(c, str);};
                SS_end_cons(First, This, Next);
	        break;};}

/*--------------------------------------------------------------------------*/

#ifdef LARGE

/*--------------------------------------------------------------------------*/

/* SS_RD_LINE - read a line of text */

object *SS_rd_line(str)
   object *str;
   {FILE *s;
    char *t, *t1;
    object *ret;

    if (SS_nullobjp(str))
       str = SS_indev;
    else if (!SS_inportp(str = SS_car(str)))
       SS_error("ARGUMENT NOT INPUT-PORT - READ-LINE", str);

    s = SS_INSTREAM(str);
    t = SS_BUFFER(str);
    SS_PTR(str) = t;
    if (GETLN(t, MAXLINE, s) == NULL)
       return(SS_eof);

    for (t1 = t; *t1 != '\n'; t1++);
    *t1 = '\0';

    ret = SS_mk_string(t);
    *t = '\0';

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_RD_VCT - read a vector */

object *SS_rd_vct(str)
   object *str;
   {object *lst, *vct;

    lst = SS_rd_lst(str);
    SS_MARK(lst);
    vct = SS_lstvct(lst);
    SS_GC(lst);

    return(vct);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_RD_CHR - read-char for Scheme */

object *SS_rd_chr(arg)
   object *arg;
   {object *str;

    if (SS_nullobjp(arg))
       {*SS_PTR(SS_indev) = '\0';
        return(SS_mk_char((int) NEXT_CHAR(SS_indev, FALSE)));}

    else if (SS_inportp(str = SS_car(arg)))
       {*SS_PTR(str) = '\0';
        return(SS_mk_char((int) NEXT_CHAR(str, FALSE)));}

    else
       SS_error("ARGUMENT TO READ NOT INPUT-PORT", arg);

    return(SS_null);}

/*--------------------------------------------------------------------------*/

#endif

/*--------------------------------------------------------------------------*/

/* SS_CLR_STRM - clear the buffer associated with the stream object */

void SS_clr_strm(str)
   object *str;
   {SS_PTR(str) = SS_BUFFER(str);
    *SS_PTR(str) = '\0';

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_RD_ATM - read and make an atomic object */

object *SS_rd_atm(str)
   object *str;
   {int c;
    char token[MAXLINE], *pt;
    object *op;

    pt = token;
    while ((c = NEXT_CHAR(str, FALSE)) != EOF)
       {if (strchr(" \t\n\r();", c) == NULL)
           *pt++ = c;
        else
           {if ((c == '(') || (c == ')') || (c == ';'))
               PUSH_CHAR(c, str);
            break;};};
    *pt = '\0';

#ifdef LARGE
    if (strncmp(token, "#\\", 2) == 0)
       return(SS_mk_char(token[2]));
#endif

    if (SC_intstrp(token, Radix))
       return(SS_mk_integer((BIGINT)STRTOL(token, &pt, Radix)));

    if (SC_fltstrp(token))
       return(SS_mk_float(ATOF(token)));

    op = (object *) SC_def_lookup(token, SS_symtab);
    if (op != NULL)
       return(op);

    op = SS_mk_variable(token, SS_null);
    SS_UNCOLLECT(op);
    if (SC_install(token, op, SS_POBJECT_S, SS_symtab) == NULL)
       longjmp(SC_top_lev, ABORT);

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_RD_STR - read and make an string object */

object *SS_rd_str(str)
   object *str;
   {int i, c, nc, bfsz;
    int delta;
    char *pt, *bf;
    object *ret;

    bfsz = MAXLINE;
    bf   = FMAKE_N(char, bfsz + 2, "SS_RD_STR:bf");
    while (TRUE)
       {pt = bf + bfsz - MAXLINE;
        for (i = 0; i < MAXLINE;)
            {c = NEXT_CHAR(str, FALSE);
             if (c == '\\')
                {nc = NEXT_CHAR(str, FALSE);
                 if (nc == '\"')
                    pt[i++] = nc;
                 else if (nc == '\n')
                    continue;
                 else
                    {pt[i++] = c;
                     pt[i++] = nc;};}
             else if (c != '\"')
                pt[i++] = c;
             else
                break;};

        if (i < MAXLINE)
           break;

        else
           {bfsz += MAXLINE;
            REMAKE_N(bf, char, bfsz + 2);};};

    pt[i] = '\0';
    delta = SC_arrlen(bf);
    ret   = SS_mk_string(bf);

    SFREE(bf);

/* help out with the byte count */
    SC_mem_stats_acc(-delta, -delta);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_LOAD - Scheme load
 *         - an optional final argument causes the load to happen in the
 *         - global environment if SS_t
 */

object *SS_load(argl)
   object *argl;
   {object *obj, *strm, *flag;
    int c;

    flag = SS_f;
    obj  = SS_car(argl);
    argl = SS_cdr(argl);
    if (SS_consp(argl))
       {flag = SS_car(argl);
        if (SS_true(flag))
           {SS_Save(SS_Env);
            SS_Env = SS_Global_Env;};};

    strm  = SS_opn_in(obj);

/* check for the first line starting with #! */
    c = NEXT_CHAR(strm, TRUE);
    if (c == '#')
       {c = NEXT_CHAR(strm, FALSE);
        if (c == '!')
           *SS_PTR(strm) = '\0';
        else
           {PUSH_CHAR(c, strm);
            PUSH_CHAR('#', strm);};}
    else
       PUSH_CHAR(c, strm);

    SS_Save(SS_rdobj);
    SS_Save(SS_evobj);

    while (TRUE)
       {SS_Assign(SS_rdobj, _SS_read(strm));
        if (SS_post_read_hook != NULL)
           (*SS_post_read_hook)(strm);

        if (SS_eofobjp(SS_rdobj))
           {SS_cls_in(strm);
            break;};
        SS_Save(SS_Env);
        SS_Assign(SS_evobj, SS_exp_eval(SS_rdobj));
        SS_Restore(SS_Env);

        if (SS_post_eval_hook != NULL)
           (*SS_post_eval_hook)(strm);};

    SS_Restore(SS_evobj);
    SS_Restore(SS_rdobj);

    if (SS_true(flag))
       {SS_Restore(SS_Env);};

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
