/*
 * ULAUXF.C - auxillary functions used by ultra
 *
 * Source Version: 4.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "ultra.h"

#define MAX_ARY(a, ap, n, amax)                                              \
    for (ap = a, amax = *a; ap-a < n; amax = max(amax, *a++))

static object
 *SC_DECLARE(_UL_fft, 
             (object *argl, char *type, int no, int flag, int ordr)),
 *SC_DECLARE(UL_crv_label, (object *obj)),
 *SC_DECLARE(UL_crv_domain, (object *obj)),
 *SC_DECLARE(UL_crv_range, (object *obj)),
 *SC_DECLARE(UL_crv_npts, (object *obj)),
 *SC_DECLARE(UL_crv_attr, (object *obj)),
 *SC_DECLARE(UL_write_abs, (PG_device *dev, object *argl)),
 *SC_DECLARE(UL_edit, (int j));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_FFT - the main controlling routine for the FFT command */

object *UL_fft(j)
   int j;
   {REAL *y, *x;
    int n;
    complex *cy;
    REAL xmn, xmx;
    int i;
    char label[MAXLINE];
    curve *crv;
    object *cre, *cim;

    crv = &SX_dataset[j];
    xmn = crv->xmin;
    xmx = crv->xmax;

    n = PM_fft_sc_real_data(&cy, &x, crv->xp, crv->yp, crv->n,
                            xmn, xmx, TRUE);
    if (n == 0)
       SS_error("FFT FAILED - UL_FFT", SS_null);

    y = FMAKE_N(REAL, n, "UL_FFT:y");
    if (y == NULL)
       SS_error("INSUFFICIENT MEMORY - UL_FFT", SS_null);

/* extract the real part */
    for (i = 0; i < n; i++)
        y[i] = PM_REAL_C(cy[i]);
    sprintf(label, "Real part FFT %c", crv->id);
    cre = _SX_mk_curve(n, x, y, label, NULL,
		       (PFVoid) UL_plot);

/* extract the imaginary part */
    for (i = 0; i < n; i++)
        y[i] = PM_IMAGINARY_C(cy[i]);
    sprintf(label, "Imaginary part FFT %c", crv->id);
    cim = _SX_mk_curve(n, x, y, label, NULL,
		       (PFVoid) UL_plot);

    SFREE(x);
    SFREE(y);
    SFREE(cy);

    return(SS_make_list(SS_OBJECT_I, cre,
                        SS_OBJECT_I, cim,
                        0));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_IFFT - main controlling routine for IFFT */

object *UL_ifft(argl)
   object *argl;
   {return(_UL_fft(argl, "IFFT", 0, -1, TRUE));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_CFFT - main controlling routine for complex FFT */

object *UL_cfft(argl)
   object *argl;
   {return(_UL_fft(argl, "CFFT", 0, 1, TRUE));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _UL_FFT - work horse for complex FFT's in ULTRA */

static object *_UL_fft(argl, type, no, flag, ordr)
   object *argl;
   char *type;
   int no, flag, ordr;
   {REAL *y, *x, *ypr, *ypi;
    int n, i, jr, ji, n0;
    complex *cy, *icy;
    REAL xmn, xmx;
    char label[MAXLINE];
    curve *crvr, *crvi;
    object *ch1, *ch2;

    SX_prep_arg(argl);

    jr = -1;
    ji = -1;
    SS_args(argl,
            UL_CURVE_INDEX_I, &jr,
            UL_CURVE_INDEX_I, &ji,
            0);

    if ((jr < 0) || (ji < 0))
       SS_error("BAD CURVES - _UL_FFT", argl);

    crvr = &SX_dataset[jr];
    crvi = &SX_dataset[ji];

    xmn = crvr->xmin;
    xmx = crvr->xmax;
    ypr = crvr->yp;
    ypi = crvi->yp;

    n0 = max(crvr->n, crvi->n);
    n  = PM_near_power(n0, 2);

/* make an array of complex values out of the real and imaginary values */
    icy = FMAKE_N(complex, n0, "_UL_FFT:icy");
    for (i = 0; i < n0; i++)
        icy[i] = PM_COMPLEX(ypr[i], ypi[i]);

/* GOTTCHA: this assumes a common set of domain values for the curves */
    n = PM_fft_sc_complex_data(&cy, &x, crvr->xp, icy, n0,
                               xmn, xmx, flag, ordr);

    no += n;

    y = FMAKE_N(REAL, no, "_UL_FFT:y");

/* extract the real part */
    for (i = 0; i < no; i++)
        y[i] = PM_REAL_C(cy[i]);
    sprintf(label, "Real part %s %c %c", type, crvr->id, crvi->id);
    ch1 = _SX_mk_curve(no, x, y, label, NULL,
		       (PFVoid) UL_plot);

/* extract the imaginary part */
    for (i = 0; i < no; i++)
        y[i] = PM_IMAGINARY_C(cy[i]);
    sprintf(label, "Imaginary part %s %c %c", type, crvr->id, crvi->id);
    ch2 = _SX_mk_curve(no, x, y, label, NULL,
		       (PFVoid) UL_plot);

    SFREE(x);
    SFREE(y);
    SFREE(icy);
    SFREE(cy);

    return(SS_make_list(SS_OBJECT_I, ch1,
                        SS_OBJECT_I, ch2,
                        0));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _UL_EFFECTIVE_DX - estimate a sampling rate for an FFT
 *                  - dx(eff) is the summed part of the domain
 *                  - where f > 0.5*fmax
 *                  - (for a gaussian this would be the half width)
 *                  - return dx(eff)/16
 *                  - the 16 allows for the Nyquist factor of 2
 *                  - and 8 effective widths
 */

static double _UL_effective_dx(x, y, n)
   REAL *x, *y;
   int n;
   {int i;
    double yx, dxe, v;

    yx = 0.0;
    for (i = 0; i < n; i++)
        {v  = ABS(y[i]);
	 yx = max(yx, v);};
    yx *= 0.5;

    dxe = 0.0;
    for (i = 1; i < n; i++)
        {v = y[i];
         if (v > yx)
            dxe += (x[i] - x[i-1]);};

    return(0.0625*dxe);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_CONVLV - compute and return the convolution of two curves */

static object *UL_convlv(argl)
   object *argl;
   {REAL *y, *x, *gx, *gy, *hx, *hy, dt, dxeg, dxeh;
    int jg, jh, n, gn, hn;
    char label[MAXLINE];
    curve *crvg, *crvh;
    object *ch;

    SX_prep_arg(argl);

    jg = -1;
    jh = -1;
    dt = HUGE;
    SS_args(argl,
            UL_CURVE_INDEX_I, &jg,
            UL_CURVE_INDEX_I, &jh,
            SC_REAL_I, &dt,
            0);

    if ((jg < 0) || (jh < 0))
       SS_error("BAD CURVES - UL_CONVLV", argl);

    crvg = &SX_dataset[jg];
    crvh = &SX_dataset[jh];

    gx = crvg->xp;
    gy = crvg->yp;
    gn = crvg->n;
    hx = crvh->xp;
    hy = crvh->yp;
    hn = crvh->n;

/* estimate dt */
    dxeg = _UL_effective_dx(gx, gy, gn);
    dxeh = _UL_effective_dx(hx, hy, hn);
    if (dt == HUGE)
       dt = min(dxeg, dxeh);

    PM_convolve(gx, gy, gn, hx, hy, hn, dt, &x, &y, &n);

    sprintf(label, "Convolution %c %c", crvg->id, crvh->id);
    ch = _SX_mk_curve(n, x, y, label, NULL,
		      (PFVoid) UL_plot);

    return(ch);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_GET_VALUE - get a list of values from the given curve and return it */

object *UL_get_value(sp, vp, val, n, id)
   REAL *sp, *vp;
   double val;
   int n, id;
   {int i;
    char flag = 'f';
    REAL *z, *z1;
    REAL test, y;
    REAL xta, xtb, xtc, yta, ytb;
    object *ret;

    if (SS_interactive == ON)
       PRINT(stdout, "\nCurve %c\n", id);

    ret = SS_null;
    for (i = 0, z = sp, z1 = sp+1; i < n-1; i++, z++, z1++)
        {flag = 'f';
         test = (*z-val)*(*z1-val);
         if (test < 0)
            {PM_interp(y, val, *z, vp[i], *z1, vp[i+1]);
             flag = 't';}
         else if (PM_CLOSETO_REL(test, 0.0))
            {if ((*z == val) || ((i == 0) && PM_CLOSETO_REL(*z - val, 0.0)))
                {y = vp[i];
                 flag = 't';}
             else if (i == (n-2) && PM_CLOSETO_REL(*z1 - val, 0.0))
                {y = vp[i+1];
                 flag = 't';};};
         if (flag == 't')
            {if (SS_interactive == ON)
                {PRINT(stdout, "    ");
                 PRINT(stdout, SX_ascii_output_format, val);
                 PRINT(stdout, "    ");
                 PRINT(stdout, SX_ascii_output_format, y);
                 PRINT(stdout, "\n");};
                 
             SS_Assign(ret, SS_mk_cons(SS_mk_float(y), ret));};};

    if (!SS_nullobjp(ret))
       {SS_Assign(ret, SS_reverse(ret));
        SC_mark(ret, -1);
        return(ret);}
    else
       return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_CURVE_EVAL - find the apropriate y value for the given x value */

object *UL_curve_eval(arg)
   object *arg;
   {int i;
    REAL value;
    object *ret;

    i = SX_data_index[_SX_curve_id(*SS_GET(procedure, SS_Fun)->name)];
    if (i < 0)
       SS_error("CURVE DELETED, NO PROCEDURE - CURVE-EVAL", SS_Fun);

    if (SS_integerp(arg))
       value = (REAL) *SS_GET(BIGINT, arg);
    else
       value = (REAL) *SS_GET(double, arg);

    ret = UL_get_value(SX_dataset[i].xp, SX_dataset[i].yp, value,
                       SX_dataset[i].n, SX_dataset[i].id);
    SS_MARK(ret);

    UL_pause(FALSE);

    if (SS_true(ret))
       {SX_prep_ret(ret);
        return(ret);}
    else
       SS_error("ARGUMENT OUT OF DOMAIN - CURVE-EVAL", arg);

    return(SS_null);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_GETX - return the x values associated with the given y value */

object *UL_getx(obj, tok)
   object *obj, *tok;
   {int j;
    REAL value;
    object *tmp;

    j = SX_get_curve(obj);
    if (SS_integerp(tok))
       value = (REAL) *SS_GET(BIGINT, tok);
    else
       value = (REAL) *SS_GET(double, tok);

    tmp = UL_get_value(SX_dataset[j].yp, SX_dataset[j].xp, value,
                       SX_dataset[j].n, SX_dataset[j].id);

    return(SS_make_list(SS_OBJECT_I, obj,
                        SS_OBJECT_I, tok,
                        SS_OBJECT_I, tmp,
                        0));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_GETY - return the y values associated with the given x value */

object *UL_gety(obj, tok)
   object *obj, *tok;
   {int j;
    REAL value;
    object *tmp;

    j = SX_get_curve(obj);
    if (SS_integerp(tok))
       value = (REAL) *SS_GET(BIGINT, tok);
    else
       value = (REAL) *SS_GET(double, tok);

    tmp = UL_get_value(SX_dataset[j].xp, SX_dataset[j].yp, value,
                       SX_dataset[j].n, SX_dataset[j].id);

    return(SS_make_list(SS_OBJECT_I, obj,
                        SS_OBJECT_I, tok,
                        SS_OBJECT_I, tmp,
                        0));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_LSQ - find least squares fit */

PM_matrix *UL_lsq(a, ay)
   PM_matrix *a, *ay;
   {PM_matrix *at, *b, *by;

    at = PM_transpose(a);
    b  = PM_times(at, a);
    by = PM_times(at, ay);
        
/* solution is returned in by */
    PM_solve(b, by);

/* clean up the mess */
    PM_destroy(at);
    PM_destroy(b);

    return(by);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_FIT - curve fitting routine */

object *UL_fit(obj, tok)
   object *obj, *tok;
   {int j, n, k, sgn;
    PM_matrix *a, *ay, *solution;
    int i, j1, order, aord;
    REAL x, *xp, *yp, xpi, xmin, xmax, delx;
    REAL accum, temp;
    object *ret;

    if (SS_integerp(tok))
       order = SS_INTEGER_VALUE(tok);

    else
       SS_error("FIT POWER MUST BE INTEGER - UL_FIT", tok);

    aord = abs(order) + 1;
    j    = SX_get_curve(obj);

    n    = SX_dataset[j].n;
    xp   = SX_dataset[j].xp;
    yp   = SX_dataset[j].yp;
    xmin = SX_dataset[j].xmin;
    xmax = SX_dataset[j].xmax;
    ay   = PM_create(n, 1);
    a    = PM_create(n, aord);

    if ((order < 0) && (xmin*xmax <= 0.0))
       SS_error("BAD DOMAIN FOR NEGATIVE POWER FIT - UL_FIT", SS_null);

    for (i = 1; i <= n; i++)
        {temp  = (order < 0) ? 1.0/xp[i-1] : xp[i-1];
         accum = 1;
         PM_element(ay, i, 1) = yp[i-1];
         for (k = 1; k <= aord; k++)
             {PM_element(a, i, k) = accum;
              accum *= temp;};};
        
    solution = UL_lsq(a, ay);
    PM_destroy(a);
    PM_destroy(ay);

/* display coefficients and cons up the return list */
    PRINT(stdout, "\nCurve %c\n", SX_dataset[j].id);

    ret = SS_null;
    sgn = (order < 0) ? -1 : 1;
    for (i = 1 ; i <= aord; i++)
        {PRINT(stdout, "    ");
         PRINT(stdout, SX_ascii_output_format, PM_element(solution, i, 1));
         PRINT(stdout, " *x^%d\n", sgn*(i-1));
         ret = SS_mk_cons(SS_mk_float(PM_element(solution, i, 1)), ret);};
        
/* create curve of fit */
    UL_buf1x = FMAKE_N(REAL, SX_default_npts, "UL_FIT:buf1x");
    UL_buf1y = FMAKE_N(REAL, SX_default_npts, "UL_FIT:buf1y");
    delx     = (xmax - xmin)/(SX_default_npts - 1);
    for (i = 0; i < SX_default_npts; i++)
        {xpi         =  xmin + i*delx;
	 x           = (order < 0) ? 1.0/xpi : xpi;
         UL_buf1x[i] = xpi;
         UL_buf1y[i] = 0;
         accum       = 1;
         for (j1 = 1; j1 <= aord; j1++)
             {UL_buf1y[i] += (PM_element(solution, j1, 1))*accum;
              accum       *= x;};};

    sprintf(pbuffer, "Fit %c %d", SX_dataset[j].id, order);
    ret = SS_mk_cons(SS_reverse(ret),
                     _SX_mk_curve(SX_default_npts, UL_buf1x, UL_buf1y,
                                  pbuffer, NULL,
				  (PFVoid) UL_plot));

/* clean up */
    SFREE(UL_buf1x);
    SFREE(UL_buf1y);      
    PM_destroy(solution);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_FIT_CURVE - generate least squares fits to the given curves */

object *UL_fit_curve(argl)
   object *argl;
   {int j, i, k, n, order, aord, id;
    object *ch;
    int *curid;
    PM_matrix *ay, *a, *solution;
    REAL *xp, *yp, *xp0, *yp0, yv;
    REAL xta, xtb, xtc, yta, ytb;
    char frag[MAXLINE];

    SS_args(argl,
            UL_CURVE_INDEX_I, &j,
            0);

    if (j < 0)
       SS_error("BAD CURVE - UL_FIT_CURVE", argl);
    n = SX_dataset[j].n;

    argl  = SS_cdr(argl);
    order = _SS_length(argl);
    aord  = abs(order) + 1;
    curid = FMAKE_N(int, order, "UL_FIT_CURVE:curid");
    for (i = 0; i < order; i++, argl = SS_cdr(argl))
        curid[i] = SX_get_curve(SS_car(argl));
        
    ay = PM_create(n, 1);
    a  = PM_create(n, order);

    xp0 = SX_dataset[j].xp - 1;
    yp0 = SX_dataset[j].yp - 1;
    for (i = 1; i <= n; i++)
        PM_element(ay, i, 1) = yp0[i];

    for (k = 1; k < aord; k++)
        {xp = SX_dataset[curid[k-1]].xp;
         yp = SX_dataset[curid[k-1]].yp;
         for (i = 1; i <= n; i++)
             {while (xp0[i] > xp[1])
                 {xp++;
                  yp++;};
              PM_interp(yv, xp0[i], *xp, *yp, xp[1], yp[1]);
              PM_element(a, i, k) = yv;};};

    PRINT(stdout, "\n    Fit curves\n\n");

    solution = UL_lsq(a, ay);

/* display coefficients */
    id = SX_dataset[j].id;
    PRINT(stdout, "Fit to curve %c\n\n", id);
    sprintf(pbuffer, "Fit(%c ;", id);
    for (i = 1; i <= order; i++)
        {id = SX_dataset[curid[i-1]].id;
         PRINT(stdout, "    ");
         PRINT(stdout, SX_ascii_output_format, PM_element(solution, i, 1));
         PRINT(stdout, " * curve %c\n", id);
         sprintf(frag, " %c", id);
         strcat(pbuffer, frag);};
    strcat(pbuffer, ")");

/* find fit */
    PM_destroy(ay);
    ay = PM_times(a, solution);

    UL_buf1y = FMAKE_N(REAL, n, "UL_FIT_CURVE:buf1y");
    for (i = 1; i <= n; i++)
        UL_buf1y[i-1] = PM_element(ay, i, 1);

    PRINT(stdout, "\n");
    ch = _SX_mk_curve(SX_dataset[j].n, SX_dataset[j].xp, UL_buf1y,
                      pbuffer, NULL,
		      (PFVoid) UL_plot);

/* clean up */
    PM_destroy(a);
    PM_destroy(ay);
    PM_destroy(solution);
    SFREE(UL_buf1y);
    SFREE(curid);
        
    return(ch);}
        
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_ERROR_PLOT - plot error bars on the specified curve */

static object *UL_error_plot(argl)
   object *argl;
   {pcons *info;
    object *c, *yp, *ym, *xp, *xm;
    curve *crv;
    byte *ypc, *ymc, *xpc, *xmc;

    yp = NULL;
    ym = NULL;
    xp = NULL;
    xm = NULL;
    SS_args(argl,
            SS_OBJECT_I, &c,
            SS_OBJECT_I, &yp,
            SS_OBJECT_I, &ym,
            SS_OBJECT_I, &xp,
            SS_OBJECT_I, &xm,
	    0);

    crv  = SX_dataset + SX_get_curve(c);
    info = crv->info;

    info = PG_set_plot_type(info, ERROR_BAR, CARTESIAN);

    ypc  = ((yp == NULL) || SS_nullobjp(yp)) ?
           NULL : (byte *) (SX_dataset + SX_get_curve(yp))->yp;
    info = SC_change_alist(info, "DY-PLUS", SC_REAL_P_S, ypc);

    ymc  = ((ym == NULL) || SS_nullobjp(ym)) ?
           NULL : (byte *) (SX_dataset + SX_get_curve(ym))->yp;
    info = SC_change_alist(info, "DY-MINUS", SC_REAL_P_S, ymc);

    xpc  = ((xp == NULL) || SS_nullobjp(xp)) ?
           NULL : (byte *) (SX_dataset + SX_get_curve(xp))->yp;
    info = SC_change_alist(info, "DX-PLUS", SC_REAL_P_S, xpc);

    xmc  = ((xm == NULL) || SS_nullobjp(xm)) ?
           NULL : (byte *) (SX_dataset + SX_get_curve(xm))->yp;
    info = SC_change_alist(info, "DX-MINUS", SC_REAL_P_S, xmc);

    crv->info = info;

    return((object *) crv->obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_MK_PALETTE - make a new palette */

static object *UL_mk_palette(argl)
   object *argl;
   {int nc, wbck;
    char *name, fname[MAXLINE];
    PG_device *dev;
    PG_palette *pal;

    name = NULL;
    nc   = 8;
    wbck = TRUE;
    SS_args(argl,
            SC_STRING_I, &name,
            SC_INTEGER_I, &nc,
            SC_INTEGER_I, &wbck,
            0);

    dev = SX_graphics_device;
    pal = PG_make_palette(dev, name, nc, wbck);

/* write the palette for future reference */
    if (pal != NULL)
       {dev->current_palette = pal;
	sprintf(fname, "%s.pal", name);
	PG_wr_palette(dev, pal, fname);};

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_RD_PALETTE - read a new palette */

static object *UL_rd_palette(argl)
   object *argl;
   {char *name;
    PG_device *dev;
    PG_palette *pal;

    name = NULL;
    SS_args(argl,
            SC_STRING_I, &name,
            0);

    dev = SX_graphics_device;

/* write the palette for future reference */

    pal = PG_rd_palette(dev, name);
    if (pal != NULL)
       dev->current_palette = pal;

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_MARK_CURVE_POINTS - graphically mark curve points for some later
 *                      - operation
 */

static void UL_mark_curve_points(xp, yp, n, indx)
   REAL *xp, *yp;
   int n;
   char *indx;
   {int i, ix, iy, btn, mod, color, off;
    REAL xm, ym, xc, yc;
    PG_device *dev;

    dev = SX_graphics_device;

    i  = 0;
    xc = xp[0];
    yc = yp[0];

    while (TRUE)
       {PG_query_pointer(dev, &ix, &iy, &btn, &mod);
	PG_print_pointer_location(dev,
				  SX_show_mouse_location_x,
				  SX_show_mouse_location_y,
				  TRUE);
        if (btn)
	   {PtoS(dev, ix, iy, xm, ym);
	    StoW(dev, xm, ym);

	    PM_nearest_point(xp, yp, n, xm, ym, &xc, &yc, &i);

	    off = (mod & KEY_SHIFT) ? FALSE : TRUE;
	    switch (btn)
	       {case MOUSE_LEFT :
		     indx[i] = off;
                     color   = off ? dev->RED : dev->GREEN;

/* toggle the color of the point between green and red */
                     PG_set_line_color(dev, color);
                     PG_draw_markers(dev, 1, &xc, &yc, 0);

		     break;

		case MOUSE_MIDDLE :
                     break;

		case MOUSE_RIGHT :
                     return;};};};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_EDIT - graphically edit a curve */

static object *UL_edit(j)
   int j;
   {int n, i, k, oldc;
    int *pc, *scn, *cln, *clo;
    REAL *xp, *yp;
    char *indx;
    PG_device *dev;
    pcons *info, *infn;
    curve *crv, *ocv;

    dev = SX_graphics_device;

    ocv  = SX_dataset + j;
    info = ocv->info;

    crv  = SX_dataset + SX_get_curve(UL_copy_curve(j));
    n    = crv->n;
    xp   = crv->xp;
    yp   = crv->yp;
    infn = crv->info;

    indx = FMAKE_N(char, n, "UL_EDIT:indx");

/* remember the original curve's color */
    SC_assoc_info(info,
                  "LINE-COLOR", &pc,
		  NULL);
    oldc = (pc != NULL) ? *pc : dev->WHITE;

/* redraw original curve in gray */
    SC_CHANGE_VALUE_ALIST(info, int, SC_INTEGER_P_S,
			  "LINE-COLOR", dev->GRAY);
    ocv->info = info;

/* draw scatter version of curve in green */
    SC_CHANGE_VALUE_ALIST(infn, int, SC_INTEGER_P_S,
			  "SCATTER", TRUE);

    SC_CHANGE_VALUE_ALIST(infn, int, SC_INTEGER_P_S,
			  "LINE-COLOR", dev->GREEN);
    crv->info = infn;

    UL_plot();

/* mark the points to remove */
    UL_mark_curve_points(xp, yp, n, indx);

/* remove the marked points */
    for (i = 0; i < n; i++)
        {if (indx[i])
            {n--;
             for (k = i--; k < n; k++)
	         {xp[k]   = xp[k+1];
		  yp[k]   = yp[k+1];
                  indx[k] = indx[k+1];};};};

    SFREE(indx);

/* resize the xp and yp arrays */
    REMAKE_N(crv->xp, REAL, n);
    REMAKE_N(crv->yp, REAL, n);

    crv->n = n;

/* change rendering of new curve */
    *scn  = FALSE;
    *cln  = _SX_next_color(dev); 

/* restore original curve color */
    *clo = oldc;

    return((object *) crv->obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_DUPX - make a new curve consisting of x vs. x for the given curve */

object *UL_dupx(j)
   int j;
   {

    sprintf(pbuffer, "Dupx %c", SX_dataset[j].id);

    return(_SX_mk_curve(SX_dataset[j].n, SX_dataset[j].xp,
			SX_dataset[j].xp, pbuffer, NULL,
			(PFVoid) UL_plot));}
        
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_STAT - return a list of relevant statistics for the given curve */

object *UL_stat(j)
   int j;
   {REAL sumy = 0, sumy2 = 0, sumx = 0, sumx2 = 0;
    REAL xmean, xstd, ymean, ystd;
    REAL x, y;
    REAL *yp, *xp, *ypmax, fn;
    object *ret;

    PRINT(stdout, "\nCurve %c\n", SX_dataset[j].id);

    fn = SX_dataset[j].n;
    ypmax = SX_dataset[j].yp+SX_dataset[j].n;
    for (yp = SX_dataset[j].yp, xp = SX_dataset[j].xp; yp < ypmax; yp++, xp++)
        {y = *yp;
         x = *xp;
         sumy  += y;
         sumy2 += y*y;
         sumx  += x;
         sumx2 += x*x;};
        
    xmean = sumx / fn;
    xstd  = sqrt((fn*sumx2-sumx*sumx)/(fn*(fn-1)));
    ymean = sumy / fn;
    ystd  = sqrt((fn*sumy2-sumy*sumy)/(fn*(fn-1)));

    ret = SS_make_list(SC_REAL_I, &xmean,
                       SC_REAL_I, &xstd,
                       SC_REAL_I, &ymean,
                       SC_REAL_I, &ystd,
                       0);

    PRINT(stdout, "\nX Mean =               ");
    PRINT(stdout, SX_ascii_output_format, xmean);
    PRINT(stdout, "\nX Standard deviation = ");
    PRINT(stdout, SX_ascii_output_format, xstd);
    PRINT(stdout, "\nY Mean =               ");
    PRINT(stdout, SX_ascii_output_format, ymean);
    PRINT(stdout, "\nY Standard deviation = ");
    PRINT(stdout, SX_ascii_output_format, ystd);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_STATS - control the computation of means and standard deviations for
 *          - a specified set of curves
 */

object *UL_stats(argl)
   object *argl;
   {int j;
    object *ret;

    SX_prep_arg(argl);
    ret = SS_null;
    for ( ; SS_consp(argl); argl = SS_cdr(argl))
       {j = SX_get_curve(SS_car(argl));
        if (j != -1)
           ret = SS_mk_cons(UL_stat(j), ret);};
         
    UL_pause(FALSE);
   
    if (_SS_length(ret) == 1)
       return(SS_car(ret));
    else
       return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_DISP - display the curve points in the given domain */

object *UL_disp(j, xmin, xmax)
   int j;
   double xmin, xmax;
   {REAL *xp, *yp;
    int n, i;

    xp   = SX_dataset[j].xp;
    yp   = SX_dataset[j].yp;
    n    = SX_dataset[j].n;

    PRINT(stdout, "\n Curve %c (%s) from ", SX_dataset[j].id, SX_dataset[j].text);
    PRINT(stdout, SX_ascii_output_format, xmin);
    PRINT(stdout, " to ");
    PRINT(stdout, SX_ascii_output_format, xmax);
    PRINT(stdout, "\n\n");

    for (i = 0; i < n; i++)
        {if ((xp[i] >= xmin) && (xp[i] <= xmax))
            {PRINT(stdout, "    ");
             PRINT(stdout, SX_ascii_output_format, xp[i]);
             PRINT(stdout, " ");
             PRINT(stdout, SX_ascii_output_format, yp[i]);
             PRINT(stdout, "\n");};};

    PRINT(stdout, "\n");
        
    return((object *) SX_dataset[j].obj);}
                
/*--------------------------------------------------------------------------*/

/*                       SCHEME CURVE ACCESS ROUTINES                       */

/*--------------------------------------------------------------------------*/

/* UL_CRV_LABEL - return the label for the given curve */

static object *UL_crv_label(obj)
   object *obj;
   {int j;

    if (!SX_curvep_a(obj))
       SS_error("BAD CURVE - UL_CRV_LABEL", obj);

    j = SX_get_curve(obj);
    if (j != -1)
       {if (SS_interactive == ON)
           PRINT(stdout, "\n Label: %s\n\n", SX_dataset[j].text);
        return(SS_mk_string(SX_dataset[j].text));}
    else
       return(SS_null);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_CRV_DOMAIN - return a cons whose car and cdr are the min and max of
 *               - the given curve's domain
 */

static object *UL_crv_domain(obj)
   object *obj;
   {int j;

    j = -1;
    SS_args(obj,
            UL_CURVE_INDEX_I, &j,
            0);

    if (j != -1)
       {if (SS_interactive == ON)
           {PRINT(stdout, "\n Domain: (");
            PRINT(stdout, SX_ascii_output_format, SX_dataset[j].xmin);
            PRINT(stdout, " . ");
            PRINT(stdout, SX_ascii_output_format, SX_dataset[j].xmax);
            PRINT(stdout, ")\n\n");};
        return(SS_mk_cons(SS_mk_float(SX_dataset[j].xmin),
                          SS_mk_float(SX_dataset[j].xmax)));}
    else
       return(SS_null);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_CRV_RANGE - return a cons whose car and cdr are the min and max of
 *              - the given curve's range
 */

static object *UL_crv_range(obj)
   object *obj;
   {int j;

    j = -1;
    SS_args(obj,
            UL_CURVE_INDEX_I, &j,
            0);

    if (j != -1)
       {if (SS_interactive == ON)
           {PRINT(stdout, "\n Range: (");
            PRINT(stdout, SX_ascii_output_format, SX_dataset[j].ymin);
            PRINT(stdout, " . ");
            PRINT(stdout, SX_ascii_output_format, SX_dataset[j].ymax);
            PRINT(stdout, ")\n\n");};
        return(SS_mk_cons(SS_mk_float(SX_dataset[j].ymin),
                          SS_mk_float(SX_dataset[j].ymax)));}
    else
       return(SS_null);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_CRV_NPTS - return the number of points in the given curve */

static object *UL_crv_npts(obj)
   object *obj;
   {int j;

    j = -1;
    SS_args(obj,
            UL_CURVE_INDEX_I, &j,
            0);

    if (j != -1)
       {if (SS_interactive == ON)
           PRINT(stdout, "\n Number of points: %ld\n\n", SX_dataset[j].n);
        return(SS_mk_integer((BIGINT) SX_dataset[j].n));}
    else
       return(SS_null);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_CRV_ATTR - return a list of the given curve's attributes:
 *             -      (color width style)
 */

static object *UL_crv_attr(obj)
   object *obj;
   {int j;

    j = -1;
    SS_args(obj,
            UL_CURVE_INDEX_I, &j,
            0);

    if (j != -1)
       {int *plc, *pls, lncol, lnsty;
        double lnwid, *plw;
        pcons *info;

        info = SX_dataset[j].info;
	SC_assoc_info(info,
		      "LINE-COLOR", &plc,
		      "LINE-STYLE", &pls,
		      "LINE-WIDTH", &plw,
		      NULL);
	lncol   = (plc != NULL) ? *plc : SX_graphics_device->BLUE;
	lnwid   = (plw != NULL) ? *plw : 0.0;
	lnsty   = (pls != NULL) ? *pls : SOLID;

        if (SS_interactive == ON)
           {PRINT(stdout, "\n Color, width, style: (%ld ", lncol);
            PRINT(stdout, SX_ascii_output_format, lnwid);
            PRINT(stdout, " %ld)\n\n", lnsty);};

        return(SS_make_list(SC_INTEGER_I, &lncol,
                            SC_DOUBLE_I, &lnwid,
                            SC_INTEGER_I, &lnsty,
                            0));}
    else
       return(SS_null);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_WRITE_ABS - let ULTRA make print stuff on the screen
 *              - convert from NDC/Screen coordinates to WC
 */

static object *UL_write_abs(dev, argl)
   PG_device *dev;
   object *argl;
   {double x, y;
    char *text;

    x    = 0.0;
    y    = 0.0;
    text = NULL;
    SS_args(argl,
            SC_DOUBLE_I, &x,
            SC_DOUBLE_I, &y,
            SC_STRING_I, &text,
            0);

    StoW(dev, x, y);
    PG_write_WC(SX_graphics_device, x, y, text);
   
    return(SS_f);}
    
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_INSTALL_AUX_FUNCS - install the functions defined in this file */

void UL_install_aux_funcs()
   {SS_install("convolve",
               "Procedure: Convolve the two functions\n     Usage: convolve <curve> <curve> [<dt>]",
               SS_nargs,
               UL_convlv, SS_PR_PROC);

    SS_install("edit",
               "Procedure: Graphically edit a curve\n     Usage: edit <curve>",
               UL_uc,
               UL_edit, SS_PR_PROC);

    SS_install("error-bar",
               "Procedure: Do a scatter plot with error bars on the given curve\n     Usage: error-bar <curve> <y+curve> <y-curve> [<x+curve> [<x-curve>]]",
               SS_nargs,
               UL_error_plot, SS_PR_PROC);

    SS_install("get-label",
               "Procedure: Return the given curve's label\n     Usage: get-label <curve>",
               SS_sargs,
               UL_crv_label, SS_PR_PROC);

    SS_install("get-domain",
               "Procedure: Return the given curve's domain\n     Usage: get-domain <curve>",
               SS_sargs,
               UL_crv_domain, SS_PR_PROC);

    SS_install("get-range",
               "Procedure: Return the given curve's range\n     Usage: get-range <curve>",
               SS_sargs,
               UL_crv_range, SS_PR_PROC);

    SS_install("get-number-points",
               "Procedure: Return the given curve's number of points\n     Usage: get-number-points <curve>",
               SS_sargs,
               UL_crv_npts, SS_PR_PROC);

    SS_install("get-attributes",
               "Procedure: Return the given curve's attributes, (color width style)\n     Usage: get-attributes <curve>",
               SS_sargs,
               UL_crv_attr, SS_PR_PROC);

    SS_install("mk-pal",
               "Macro: Make a new palette and write it to a file\n     Usage: mk-pal <file-name> <integer> [on | off]",
               SS_nargs,
               UL_mk_palette, SS_UR_MACRO);

    SS_install("rd-pal",
               "Macro: Read palette file\n     Usage: rd-pal <file-name>",
               SS_sargs,
               UL_rd_palette, SS_UR_MACRO);

    SS_install("write-text",
               "Procedure: Write text to screen\n     Usage: write_text <x> <y> <text>",
               SS_nargs,
               UL_write_abs, SS_PR_PROC);


    SS_install_cf("fft",
                  "Procedure: Compute Fast Fourier Transform of real curve. Return real and imaginary parts.\n     Usage: fft <curve>",
                  UL_uc, 
                  UL_fft);
    SS_install_cf("getx",
                  "Procedure: Return x values for a given y\n     Usage: getx <curve-list> <value>",
                  UL_bltocnp, 
                  UL_getx);
    SS_install_cf("gety",
                  "Procedure: Return y values for a given x\n     Usage: gety <curve-list> <value>",
                  UL_bltocnp, 
                  UL_gety);
    SS_install_cf("fit",
                  "Procedure: Find least-squares fit to the specified curves for a polynomial of order n\n     Usage: fit <curve-list> n",
                  UL_bltocnp, 
                  UL_fit);
    SS_install_cf("dupx",
                  "Procedure: Duplicate x values so that y = x for each of the specified curves\n     Usage: dupx <curve-list>",
                  UL_uc, 
                  UL_dupx);
    SS_install_cf("disp",
                  "Procedure: Display actual values in specified curves between min and max points\n     Usage: disp <curve-list> <xmin> <xmax>",
                  UL_ul2tocnp, 
                  UL_disp);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
