/* Implementation of TLLLambda class.
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLLLambda.m,v 1.3 1998/02/23 14:17:31 tiggr Exp $  */

#import "tl/support.h"
#import "tl/TLLLambda.h"
#import "tl/TLCons.h"
#import "tl/subr.h"
#import "tl/TLGC.h"
#import "tl/TLPatchedRoots.h"

/* Lambda keyword symbols and its value.  */
TLSymbol *Slambda_list_keywords;
TLCons *Qlambda_list_keywords;

/* Important list keywords.  */
static TLSymbol *Slambda_optional, *Slambda_uneval, *Slambda_rest;

@implementation TLLLambda

+initialize
{
  if (!Slambda_list_keywords)
    {
      Slambda_list_keywords = [CO_TLSymbol symbolWithName:
			       @"lambda-list-keywords"];
      [Slambda_list_keywords setCValue: &Qlambda_list_keywords
       encoding: @encode (__typeof__ (Qlambda_list_keywords))];

      Slambda_optional = [CO_TLSymbol symbolWithName: @"&optional"];
      Slambda_rest = [CO_TLSymbol symbolWithName: @"&rest"];
      Slambda_uneval = [CO_TLSymbol symbolWithName: @"&uneval"];

      /* Add implemented keywords to this list.  */
      [Slambda_list_keywords
       setVarValue: CONS (Slambda_optional, CONS (Slambda_rest,
						  CONS (Slambda_uneval, nil)))];
    }
  return (self);
} /* +initialize */

+(TLLLambda *) lambdaWithArguments: (TLCons *) arglist body: (TLCons *) b
{
  return ([[self gcAlloc] initWithArguments: arglist body: b]);
} /* +lambdaWithArguments:body: */

-(TLCons *) body
{
  return (body);
} /* -body */

-(id <TLString>) documentation
{
  return (documentation);
} /* -documentation */

-evalWithArguments: (TLCons *) in_args
{
  id *arg_values, arg, next_arg, proto, next_proto, retval;
  int i, entry_binding_stack, optional, uneval, rest;
  int nargs = in_args ? [in_args length] : 0;
  GCDECL2;

#if SUPPORT_DEBUG
  int invocation_level = tll_invocation_num - 1;
  tll_invocation_info *ii;
#endif

  arg_values = alloca (nargs * sizeof (*arg_values));
  GCPRO2 (in_args, in_args);
  _gcpro2.n = 0;
  _gcpro2.v = arg_values;
  
  for (arg = in_args; arg; _gcpro2.n++, arg = next_arg)
    {
      DECONS (arg, arg, next_arg);
      arg_values[_gcpro2.n] = arg;
      ASGN_SPROT (arg);
    }

#if SUPPORT_DEBUG
  ii = &tll_invocation_stack[invocation_level];
  if (!tll_invocation_num || ii->argc != -2)
    {
      invocation_level = tll_invocation_new ();
      ii = &tll_invocation_stack[invocation_level];
      ii->name = nil;
    }
  ii->receiver = nil;
  ii->argc = -1;
#endif

  /* XXX Some of these checks should not be here but in -init...  */
  for (proto = args_proto, i = optional = uneval = 0, rest = -1;
       proto || (rest != -1 && i != _gcpro2.n); proto = next_proto)
    {
      if (proto)
	DECONS (proto, proto, next_proto);

      if (proto == Slambda_optional)
	if (optional)
	  [self error: "too many &optional"];
	else
	  optional = 1;
      else if (proto == Slambda_uneval)
	if (uneval)
	  [self error: "too many &uneval"];
	else
	  uneval = 1;
      else if (proto == Slambda_rest)
	if (rest != -1)
	  [self error: "too many &rest"];
	else
	  {
	    if ([next_proto cdr])
	      [self error: "too many arguments after &rest"];
	    rest = i;
	  }
      else if (i < _gcpro2.n)
	{
	  if (!uneval)
	    {
	      arg_values[i] = EVAL (arg_values[i]);
	      ASGN_SPROT (arg_values[i]);
	    }
	  i++;
	}
      else if (!(optional || rest != -1))
	[self error: "not enough arguments to lambda"];
    }
  if (i != _gcpro2.n)
    [self error: "too many arguments to lambda"];

#if SUPPORT_DEBUG
  ii->argv = arg_values;
  ii->argc = i;
#endif

  /* Assign the variable values.  */
  entry_binding_stack = [CO_TLSymbol bindingLevel];
  for (proto = args_proto, i = optional = 0; proto; proto = next_proto)
    {
      DECONS (proto, proto, next_proto);

      if (proto == Slambda_optional)
	optional = 1;
      else if (proto == Slambda_uneval);
      else if (proto == Slambda_rest)
	break;
      else if (i < _gcpro2.n)
	[proto pushVarValue: arg_values[i++]];
      else
	[proto pushVarValue: nil];
    }

  /* Handle &rest.  */
  if (proto)
    {
      /* Stupid compiler.  */
      TLCons *list, *last, *a;

      for (list = last = nil; i < _gcpro2.n; i++)
	if (list)
	  {
	    a = CONS (arg_values[i], nil);
	    [last setCdr: a];
	    last = a;
	  }
	else
	  list = last = CONS (arg_values[i], nil);
      [[next_proto car] pushVarValue: list];
    }

  retval = Fprogn (body);

  [CO_TLSymbol popVarValues: entry_binding_stack];
#if SUPPORT_DEBUG
  tll_invocation_pop (invocation_level);
#endif
  GCUNPRO;
  return (retval);
} /* -evalWithArguments: */

-initWithArguments: (TLCons *) arglist body: (TLCons *) b
{
  id d, e;

  ASGN_IVAR (args_proto, arglist);

  DECONS (b, d, e);
  if (STRINGP (d))
    {
      ASGN_IVAR (documentation, d);
      b = e;
    }

  ASGN_IVAR (body, b);
  return (self);
} /* -initWithArguments:body: */

-lambdap
{
  return (Qt);
} /* -lambdap */

-(void) print: (id <TLMutableStream>) stream quoted: (BOOL) qp
{
  [stream writeBytes: 8 fromBuffer: "(lambda "];
  print (args_proto, stream, 1);
  if (documentation)
    formac (stream, @" %#", documentation);
  if (body)
    print_list_element (body, stream, 1);
  else
    [stream writeByte: ')'];
} /* -print:quoted: */

-(TLCons *) prototype
{
  return (args_proto);
} /* -prototype */

/******************** garbage collection ********************/

-(void) gcReference
{
  MARK (args_proto);
  MARK (body);
  MARK (documentation);
} /* -gcReference */

@end
