/* C routines for streams and date.
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>
          and Michael L.H. Brouwer <michael@thi.nl>

   Copyright (C) 1996-1998 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: timespace.c,v 1.42 1999/01/05 11:56:43 tiggr Exp $  */

#include "trt.h"
#include <tom/tom-r.h>
#include <config/config.h>

#if HAVE_LIBC_H
#include <libc.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_FCNTL_H
#include <fcntl.h>
#endif

#include <stdio.h>
#include <sys/file.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifndef S_ISDIR
#define S_ISDIR(X)  (((X) & _S_IFMT) == _S_IFDIR)
#endif
#ifndef S_ISCHR
#define S_ISCHR(X)  (((X) & _S_IFMT) == _S_IFCHR)
#endif
#ifndef S_ISBLK
#define S_ISBLK(X)  (((X) & _S_IFMT) == _S_IFBLK)
#endif
#ifndef S_ISREG
#define S_ISREG(X)  (((X) & _S_IFMT) == _S_IFREG)
#endif
#ifndef S_ISLNK
#define S_ISLNK(X)  (((X) & _S_IFMT) == _S_IFLNK)
#endif
#ifndef S_ISSOCK
#define S_ISSOCK(X)  (((X) & _S_IFMT) == _S_IFSOCK)
#endif
#ifndef S_ISFIFO
#define S_ISFIFO(X)  (((X) & _S_IFMT) == _S_IFIFO)
#endif

#define UNIX_OFFSET	-978307200.0

#if HAVE_DIRENT_H
# include <dirent.h>
# define NAMLEN(dirent) strlen((dirent)->d_name)
#else
# define dirent direct
# define NAMLEN(dirent) (dirent)->d_namlen
# if HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
# if HAVE_SYS_DIR_H
#  include <sys/dir.h>
# endif
# if HAVE_NDIR_H
#  include <ndir.h>
# endif
#endif

void
c_tom_Date_v_load_r (tom_object self, selector cmd, tom_object args)
{
  struct timeval tv;

  gettimeofday (&tv, NULL);
  c_tom_Date_relative_offset = UNIX_OFFSET + tv.tv_sec;
  c_tom_Date_distant_future = TRT_SEND (_PI_, self, SEL (r_alloc));
  TRT_SEND (_PI_, c_tom_Date_distant_future,
	    SEL (r_initWithTimeIntervalSinceReferenceDate_d),
	    (tom_double) 1e+100);
  c_tom_Date_distant_past = TRT_SEND (_PI_, self, SEL (r_alloc));
  TRT_SEND (_PI_, c_tom_Date_distant_past,
	    SEL (r_initWithTimeIntervalSinceReferenceDate_d),
	    (tom_double) -1e+100);
}

double
c_tom_Date_d_timeIntervalSinceReferenceDate (tom_object self, selector cmd)
{
  struct timeval tv;

  gettimeofday (&tv, NULL);
  return UNIX_OFFSET + tv.tv_sec + tv.tv_usec * 1e-6;
}

double
c_tom_Date_d_relativeTimeIntervalSinceNow (tom_object self, selector cmd)
{
  struct timeval tv;

  gettimeofday (&tv, NULL);
  return ((UNIX_OFFSET - c_tom_Date_relative_offset)
	  + tv.tv_sec + tv.tv_usec * 1e-6);
}

tom_object
c_tom_File_r_current_directory (tom_object self, selector cmd)
{
  char *buf;
  int len;

#if HAVE_GETCWD
  buf = alloca (10000);

  if (!getcwd (buf, 9999))
    return 0;
#elif HAVE_GETWD
  if (!getwd (buf))
    return 0;
#else
#error need getwd or getcwd (or this must be fixed)
#endif

  len = strlen (buf);
  buf[len] = '/';
  buf[len + 1] = 0;

  return byte_string_with_c_string (buf);
}

void
c_tom_File_v_set_current_directory_r (tom_object self, selector cmd,
				      tom_object directory)
{
  tom_int len;
  char *s;

  C_STRING_WITH_TOM_STRING (s, len, directory);
  if (chdir (s))
    trt_raise (0, self, 0, c_tom_Conditions_file_error,
	       "%s: %s", s, OS_ERROR_MSG);
}

tom_object
c_tom_File_r_filenames_in_directory_r (tom_object self, selector cmd,
				       tom_object dir_name)
{
  struct dirent *de;
  tom_object array;
  tom_int len;
  char *s;
  DIR *d;

  C_STRING_WITH_TOM_STRING (s, len, dir_name);

  d = opendir (s);
  if (!d)
    array = trt_raise (1, self, cmd, c_tom_Conditions_file_error,
		       "%s: %s", s, OS_ERROR_MSG);
  else
    {
      array = TRT_SEND (_PI_, CREF (tom_MutableObjectArray), SEL (r_new));
      while ((de = readdir (d)) != NULL)
	{
	  tom_object e = byte_string_with_string (de->d_name, NAMLEN (de));
	  TRT_SEND (, array, SEL (v_add_r), e);
	}
      closedir (d);
    }

  return array;
}

tom_byte
c_tom_File_o_file_exists_r (tom_object self, selector cmd, tom_object name)
{
  struct stat st;
  tom_int len;
  char *s;

  if (!name)
    return 0;
  C_STRING_WITH_TOM_STRING (s, len, name);
  return stat (s, &st) ? 0 : 1;
}

/* Return the file type of the file of which ST contains the vital
   information.  */
static tom_int
tom_file_type (struct stat *st)
{
  if (S_ISSOCK (st->st_mode))
    return c_tom_Constants_FILE_TYPE_SOCKET;
  if (S_ISLNK (st->st_mode))
    return c_tom_Constants_FILE_TYPE_LINK;
  if (S_ISREG (st->st_mode))
    return c_tom_Constants_FILE_TYPE_REGULAR;
  if (S_ISBLK (st->st_mode))
    return c_tom_Constants_FILE_TYPE_BLOCK;
  if (S_ISDIR (st->st_mode))
    return c_tom_Constants_FILE_TYPE_DIRECTORY;
  if (S_ISCHR (st->st_mode))
    return c_tom_Constants_FILE_TYPE_CHARACTER;
  if (S_ISFIFO (st->st_mode))
    return c_tom_Constants_FILE_TYPE_FIFO;
  return c_tom_Constants_FILE_TYPE_OTHER;
}

tom_int
c_tom_File_i_type_of_file_r_follow_link__o (tom_object self, selector cmd,
					    tom_object name,
					    tom_byte follow_link)
{
  struct stat st;
  tom_int len;
  char *s;

  C_STRING_WITH_TOM_STRING (s, len, name);
  if (follow_link ? stat (s, &st) : lstat (s, &st))
    {
      trt_raise (1, self, cmd, c_tom_Conditions_file_error,
		 "%s", OS_ERROR_MSG);
      return c_tom_Constants_FILE_TYPE_OTHER;
    }
  return tom_file_type (&st);
}

tom_int
i_tom_Descriptor_i_type_of_file (tom_object self, selector cmd)
{
  struct _es_i_tom_Descriptor *bs;
  struct stat st;

  bs = trt_ext_address (self, _ei_i_tom_Descriptor);
  if (bs->descriptor == -1)
    {
      trt_raise (1, self, cmd, c_tom_Conditions_file_error,
		 "not open");
      return c_tom_Constants_FILE_TYPE_NONEXISTENT;
    }
  if (fstat (bs->descriptor, &st))
    {
      trt_raise (1, self, cmd, c_tom_Conditions_file_error,
		 "%s", OS_ERROR_MSG);
      return c_tom_Constants_FILE_TYPE_OTHER;
    }
  return tom_file_type (&st);
}

tom_object
i_tom_File_r_reopen (tom_object self, selector cmd)
{
  struct _es_i_tom_File *fs = trt_ext_address (self, _ei_i_tom_File);
  struct _es_i_tom_Descriptor *bs;
  char input_p, output_p, exists;
  int flags = 0;
  tom_int l;
  char *s;

  bs = trt_ext_address (self, _ei_i_tom_Descriptor);

  if (bs->descriptor != -1)
    TRT_SEND (, self, SEL (v_close));

  if (bs->descriptor != -1)
    {
      trt_raise (1, self, cmd, c_tom_Conditions_file_error,
		 "unable to close for reopen");
      bs->descriptor = -1;
    }

  input_p = !!(fs->flags & c_tom_File_OPEN_INPUT);
  output_p = !!(fs->flags & c_tom_File_OPEN_OUTPUT);

  /* Get the filename in a way C can understand.  */
  C_STRING_WITH_TOM_STRING (s, l, fs->name);

  if (!access (s, F_OK))
    {
      /* It exists.  */
      if (fs->flags & c_tom_Constants_FILE_EXIST_NOTHING)
	return 0;
      else if (fs->flags & c_tom_Constants_FILE_EXIST_RAISE)
	trt_raise (0, self, cmd, c_tom_Conditions_file_error, "file exists");
      exists = 1;
    }
  else
    {
      /* It does not exist (as far as we're concerned).  */
      if (fs->flags & c_tom_Constants_FILE_NOT_EXIST_NOTHING)
	return 0;
      else if (fs->flags & c_tom_Constants_FILE_NOT_EXIST_RAISE)
	trt_raise (0, self, cmd, c_tom_Conditions_file_error,
		   "%s", OS_ERROR_MSG);
      exists = 0;
    }

  if (output_p)
    {
      if (fs->flags & c_tom_Constants_FILE_NOT_EXIST_CREATE)
	flags |= O_CREAT;
      if (fs->flags & c_tom_Constants_FILE_EXIST_TRUNCATE)
	flags |= O_TRUNC;
      if (fs->flags & c_tom_Constants_FILE_APPEND)
	flags |= O_APPEND;
    }

  if (input_p)
    {
      if (output_p)
	flags |= O_RDWR;
      else if (!exists)
	/* Read only and it doesn't exist  */
	return 0;
      else
	flags |= O_RDONLY;
    }
  else if (output_p)
    {
      /* Output-only.  */
      flags |= O_WRONLY | O_TRUNC;
    }
  else
    {
      /* Probe only, which has been done already.  */
      return exists ? self : 0;
    }

  /* XXX The 0666 should be an argument to this method.  */
  bs->descriptor = open (s, flags, 0666);
  if (bs->descriptor == -1)
    trt_raise (0, self, cmd, c_tom_Conditions_file_error, "%s", OS_ERROR_MSG);

  return self;
}

tom_long
i_tom_File_l_position (tom_object self, selector cmd)
{
  struct _es_i_tom_Descriptor *this
    = trt_ext_address (self, _ei_i_tom_Descriptor);
  tom_long i = lseek (this->descriptor, 0, SEEK_CUR);

  if (i == -1)
    trt_raise (0, self, cmd, c_tom_Conditions_stream_error, "%s", OS_ERROR_MSG);

  return i;
}

void
i_tom_File_v_seek_l_relative__i (tom_object self, selector cmd,
				 tom_long offset, tom_int how)
{
  struct _es_i_tom_Descriptor *this
    = trt_ext_address (self, _ei_i_tom_Descriptor);

  /* This should be a no-op.  */
  how = (how == c_tom_Constants_STREAM_SEEK_SET ? SEEK_SET
	 : how == c_tom_Constants_STREAM_SEEK_CUR ? SEEK_CUR
	 : how == c_tom_Constants_STREAM_SEEK_END ? SEEK_END
	 : -1);

  if (how == -1 || offset > 0x7fffffff)
    /* XXX */
    ABORT ();

  if (lseek (this->descriptor, offset, how) == -1)
    trt_raise (0, self, cmd, c_tom_Conditions_stream_error, "%s", OS_ERROR_MSG);
}

tom_int
i_tom_ByteStream_i_writeBytes_i_from_p
  (tom_object self, selector cmd, tom_int length, void *address)
{
  struct _es_i_tom_Descriptor *this
    = trt_ext_address (self, _ei_i_tom_Descriptor);

  return length <= 0 ? 0 : write (this->descriptor, address, length);
}

tom_byte
i_tom_ByteStream_b_read (tom_object self, selector cmd)
{
  struct _es_i_tom_Descriptor *this;
  tom_byte c;
  int i;

  this = trt_ext_address (self, _ei_i_tom_Descriptor);
  i = read (this->descriptor, &c, 1);
  if (i != 1)
    if (i == 0)
      trt_raise (0, self, cmd, c_tom_Conditions_stream_eos, 0);
    else if (i < 0)
      trt_raise (0, self, cmd, c_tom_Conditions_stream_error,
		 "%s", OS_ERROR_MSG);

  return c;
}

tom_int
i_tom_ByteStream_i_read (tom_object self, selector cmd)
{
  struct _es_i_tom_Descriptor *this;
  tom_byte c;
  int i;

  this = trt_ext_address (self, _ei_i_tom_Descriptor);
  i = read (this->descriptor, &c, 1);
  if (i < 0)
    trt_raise (0, self, cmd, c_tom_Conditions_stream_error, "%s", OS_ERROR_MSG);

  return i ? c : -1;
}

void
i_tom_ByteStream_v_write_b (tom_object self, selector cmd, tom_byte b)
{
  struct _es_i_tom_Descriptor *this;
  int i;

  this = trt_ext_address (self, _ei_i_tom_Descriptor);
  i = write (this->descriptor, &b, 1);
  if (i != 1)
    if (i == 0)
      trt_raise (0, self, cmd, c_tom_Conditions_stream_eos, 0);
    else
      trt_raise (0, self, cmd, c_tom_Conditions_stream_error,
		 "%s", OS_ERROR_MSG);
}

tom_int
i_tom_ByteStream_i_write_b (tom_object self, selector cmd, tom_byte b)
{
  struct _es_i_tom_Descriptor *this;
  int i;

  this = trt_ext_address (self, _ei_i_tom_Descriptor);
  i = write (this->descriptor, &b, 1);
  if (i < 0)
    trt_raise (0, self, cmd, c_tom_Conditions_stream_error, "%s", OS_ERROR_MSG);

  return i;
}

tom_int
i_tom_MutableByteArray_i_readRange__ii__fromByteStream_r
  (tom_object self, selector cmd, tom_int start, tom_int num, tom_object file)
{
  struct _es_i_tom_MutableArray *m;
  struct _es_i_tom_Array *a;
  int i, fd = TRT_SEND ((int_imp), file, SEL (i_descriptor));

  m = trt_ext_address (self, _ei_i_tom_MutableArray);
  a = trt_ext_address (self, _ei_i_tom_Array);

  if (start < 0 || num < 0 || start > a->length)
    trt_raise (0, self, cmd, c_tom_Conditions_program_condition,
	       "bad range (%d, %d)", (int) start, (int) num);

  if (m->capacity < start + num)
    trt_array_resize (a, m, 1, start + num - a->length);

  i = read (fd, (tom_byte *) a->contents + start, num);
  if (i < 0)
    trt_raise (1, self, cmd, c_tom_Conditions_stream_error, "%s", OS_ERROR_MSG);

  if (a->length < start + i)
    a->length = start + i;

  return i;
}

void
init_streams (void)
{
  struct _es_i_tom_Descriptor *this;

  c_tom_stdio_in = TRT_SEND (_PI_, _mr_c_tom_ByteStream, SEL (r_alloc));
  this = trt_ext_address (c_tom_stdio_in, _ei_i_tom_Descriptor);
  this->descriptor = 0;

  c_tom_stdio_out = TRT_SEND (_PI_, _mr_c_tom_ByteStream, SEL (r_alloc));
  this = trt_ext_address (c_tom_stdio_out, _ei_i_tom_Descriptor);
  this->descriptor = 1;

  c_tom_stdio_err = TRT_SEND (_PI_, _mr_c_tom_ByteStream, SEL (r_alloc));
  this = trt_ext_address (c_tom_stdio_err, _ei_i_tom_Descriptor);
  this->descriptor = 2;
}

void
tini_streams (void)
{
  if (c_tom_stdio_in)
    TRT_SEND (, c_tom_stdio_in, SEL (v_close));
  if (c_tom_stdio_out)
    TRT_SEND (, c_tom_stdio_out, SEL (v_close));
  if (c_tom_stdio_err)
    TRT_SEND (, c_tom_stdio_err, SEL (v_close));
}

void
c_tom_stdio_v_close_i (tom_object self, selector cmd, tom_int d)
{
  if (close (d))
    TRT_SIGNAL (self, c_tom_Conditions_stream_error,
		byte_string_with_c_string (OS_ERROR_MSG));
}

i_tom_OutputStream
i_tom_OutputStream_r_print_p (tom_object self, selector cmd, void *p)
{
  POINTER_INT_TYPE np = (POINTER_INT_TYPE) p;
  tom_object buf;
  tom_int len;
  char *s;
  int i;

  buf = TRT_SEND ((reference_imp), self, SEL (r_print_buffer));
  TRT_SEND (, buf, SEL (v_resize_i), (tom_int) 2 * sizeof (void *));
  s = TRT_SEND ((pointer_imp), buf, SEL (_pi__pointerToElements__ii_),
		(tom_int) 0, (tom_int) -1, &len);

  for (i = 2 * sizeof (void *) - 1; i >= 0; i--, np >>= 4)
    s[i] = "0123456789abcdef"[np & 0xf];

  return TRT_SEND (_PI_, buf, SEL (r_write_r), self);
}

#define PRINT(T, FT)  \
i_tom_OutputStream    							\
C2 (i_tom_OutputStream_r_print_, FT)					\
  (tom_object self, selector cmd, T arg)				\
{		      							\
  tom_object buf;							\
  tom_int len;								\
  char *s;								\
  int i, neg = 0;							\
									\
  buf = TRT_SEND ((reference_imp), self, SEL (r_print_buffer));		\
  TRT_SEND (, buf, SEL (v_resize_i), (tom_int) 32);			\
  s = TRT_SEND ((pointer_imp), buf, SEL (_pi__pointerToElements__ii_),	\
		(tom_int) 0, (tom_int) -1, &len);			\
  i = len - 1;								\
		      							\
  if (arg < 0)	      							\
    neg = 1;								\
		      							\
  do		      							\
    {		      							\
      int digit = arg % 10;						\
      s[i] = '0' + (neg ? -digit : digit);				\
      arg /= 10;      							\
      i--;	      							\
    } while (arg);    							\
		      							\
  if (neg)	      							\
    s[i--] = '-';							\
		      							\
  memmove (s, s + i + 1, len - i - 1);					\
  TRT_SEND (, buf, SEL (v_truncate_i), len - i - 1);			\
									\
  return TRT_SEND (_PI_, buf, SEL (r_write_r), self);			\
}

PRINT (tom_byte, o)
PRINT (tom_char, c)
PRINT (tom_int, i)
PRINT (tom_long, l)

static i_tom_OutputStream
print_fp (tom_object self, int ndig, double arg)
{
  tom_object buf;
  tom_int len;
  char *s;

  buf = TRT_SEND ((reference_imp), self, SEL (r_print_buffer));
  /* This 512 should suffice, even for printing all digits in a
     DBL_MAX (at whatever accuracy).  */
  TRT_SEND (, buf, SEL (v_resize_i), (tom_int) 512);
  s = TRT_SEND ((pointer_imp), buf, SEL (_pi__pointerToElements__ii_),
		(tom_int) 0, (tom_int) -1, &len);

  len = sprintf (s, "%.*g", ndig, arg);
  if (!strpbrk (s, ".e"))
    {
      strcpy (s + len, ".0");
      len += 2;
    }

  TRT_SEND (, buf, SEL (v_truncate_i), len);

  return TRT_SEND (_PI_, buf, SEL (r_write_r), self);
}

i_tom_OutputStream
i_tom_OutputStream_r_print_d (tom_object self, selector cmd, double arg)
{
  return print_fp (self, 15, arg);
}

i_tom_OutputStream
i_tom_OutputStream_r_print_f (tom_object self, selector cmd, float arg)
{
  return print_fp (self, 6, arg);
}

tom_object
i_tom_Sink_r_print_x (tom_object self, selector cmd, ...)
{
  /* Nothing much to do for a sink...  */
  return self;
}

i_tom_OutputStream
i_tom_OutputStream_r_print_x (tom_object self, selector cmd, ...)
{
  va_list ap;
  int i;

  va_start (ap, cmd);

  for (i = 0; i < cmd->in->num; i++)
    switch (cmd->in->args[i])
      {
      case TRT_TE_BOOLEAN:
	self = TRT_SEND (_PI_, self, SEL (r_print_o), VA_ARG_BYTE (ap));
	break;
      case TRT_TE_BYTE:
	self = TRT_SEND (_PI_, self, SEL (r_print_b), VA_ARG_BYTE (ap));
	break;
      case TRT_TE_CHAR:
	self = TRT_SEND (_PI_, self, SEL (r_print_c), VA_ARG_CHAR (ap));
	break;
      case TRT_TE_INT:
	self = TRT_SEND (_PI_, self, SEL (r_print_i), va_arg (ap, tom_int));
	break;
      case TRT_TE_LONG:
	self = TRT_SEND (_PI_, self, SEL (r_print_l), va_arg (ap, tom_long));
	break;
      case TRT_TE_FLOAT:
	self = TRT_SEND ((tom_object (*) (tom_object, selector, tom_float)),
			 self, SEL (r_print_f), VA_ARG_FLOAT (ap));
	break;
      case TRT_TE_DOUBLE:
	self = TRT_SEND (_PI_, self, SEL (r_print_d), va_arg (ap, tom_double));
	break;
      case TRT_TE_POINTER:
	self = TRT_SEND (_PI_, self, SEL (r_print_p), va_arg (ap, void *));
	break;
      case TRT_TE_REFERENCE:
	self = TRT_SEND (_PI_, self, SEL (r_print_r), va_arg (ap, tom_object));
	break;
      case TRT_TE_SELECTOR:
	{
	  selector sel = va_arg (ap, selector);
	  TRT_SEND (_PI_, self, SEL (i_writeBytes_i_from_p),
		    (tom_int) sel->name.len, sel->name.s);
	}
	break;

      default:
	fatal ("print_x: unhandled type %d", cmd->in->args[i]);
	break;
      }

  va_end (ap);

  return (void *) self;
}
