(**
   A cycle gadget as nown from the amiga.
**)

MODULE VOCycle;

(*
    Implements a cycle gadget.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D  := VODisplay,
       E  := VOEvent,
       F  := VOFrame,
       G  := VOGUIObject,
       O  := VOObject,
       V  := VOVecImage,
       VM := VOValue;

CONST
  selectedMsg* = 0;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the cycle is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 frame*,
                 image*      : LONGINT;
                 imageRight* : BOOLEAN;
                 highlight*  : BOOLEAN;
               END;


  Cycle*     = POINTER TO CycleDesc;
  CycleDesc* = RECORD (G.GroupDesc)
                 prefs    : Prefs;
                 frame    : F.Frame;
                 image    : V.VecImage;

                 current  : G.Object;
                 model    : VM.ValueModel;

                 selected : BOOLEAN;
               END;

  SelectedMsg*      = POINTER TO SelectedMsgDesc;
  SelectedMsgDesc*  = RECORD (O.MessageDesc)
                      END;


VAR
  prefs* : Prefs;


  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.double3DOut;
    p.image:=V.simpleCycle;
    p.imageRight:=TRUE;
    p.highlight:=TRUE;
  END Init;

  PROCEDURE (p : Prefs) SetPrefs(c : Cycle);

  BEGIN
    c.prefs:=p;   (* We set the prefs *)

    IF p.background#NIL THEN
      c.SetBackgroundObject(p.background.Copy());
      c.backgroundObject.source:=c;
    END;
  END SetPrefs;

  PROCEDURE (c : Cycle) Init*;

  BEGIN
    c.Init^;

    c.SetBackground(D.buttonBackgroundColor);

    INCL(c.flags,G.canFocus);
    EXCL(c.flags,G.stdFocus);

    prefs.SetPrefs(c);

    NEW(c.frame);
    c.frame.Init;
    c.frame.SetFlags({G.horizontalFlex,G.verticalFlex});

    NEW(c.image);
    c.image.Init;
    c.image.SetFlags({G.horizontalFlex,G.verticalFlex});

    c.model:=NIL;
    c.current:=NIL;
    c.selected:=FALSE;
  END Init;

  (**
    Assign the model for the cycle object. The cycle object will show the nth entry
    given by the value of the integer model.
  **)

  PROCEDURE (c : Cycle) SetModel*(model : VM.ValueModel);

  BEGIN
    IF c.model#NIL THEN
      c.UnattachModel(c.model);
    END;
    c.model:=model;
    model.SetLongint(1);
    c.AttachModel(model);
  END SetModel;

  PROCEDURE (c : Cycle) CalcSize*(display : D.Display);

  VAR
    w,mw,h,mh : LONGINT;
    object    : G.Object;

  BEGIN
    c.frame.SetInternalFrame(c.prefs.frame);
    c.frame.CalcSize(display);

    c.image.Set(c.prefs.image);
    c.CopyBackground(c.image);
    c.image.CalcSize(display);


    c.height:=c.frame.topBorder+c.frame.bottomBorder;
    c.minHeight:=c.height;

    w :=0; h :=0;
    mw:=0; mh:=0;

    object:=c.list;
    WHILE object#NIL DO
      IF object.StdFocus() THEN
        INCL(c.flags,G.stdFocus);
      END;

      object.SetFlags({G.horizontalFlex,G.verticalFlex});
      IF ~c.prefs.highlight THEN
        object.SetFlags({G.noHighlight});
      ELSE
        object.RemoveFlags({G.noHighlight});
      END;

      object.CalcSize(display);
      w:=G.MaxLong(w,object.width);
      mw:=G.MaxLong(mw,object.minWidth);
      h:=G.MaxLong(h,object.height);
      mh:=G.MaxLong(mh,object.minHeight);

      object:=object.next;
    END;

    IF ~c.StdFocus() & c.MayFocus() THEN
      object:=c.list;
      WHILE object#NIL DO
        (* tell the object to reserve space for focus displaying *)
        INCL(object.flags,G.mayFocus);
        object:=object.next;
      END;
    END;

    h:=G.MaxLong(h+display.spaceHeight,c.image.height);
    mh:=G.MaxLong(mh+display.spaceHeight,c.image.height);
    INC(c.height,h);
    INC(c.minHeight,mh);

    (* A little trick to make cycle-Button a little bit nicer *)
    c.image.Resize(c.height-c.frame.topBorder-c.frame.bottomBorder,
                   c.height-c.frame.topBorder-c.frame.bottomBorder);
    c.width:=c.frame.leftBorder+c.image.width+display.spaceWidth+c.frame.rightBorder;
    c.minWidth:=c.width;

    INC(c.width,w+display.spaceWidth);
    INC(c.minWidth,mw+display.spaceWidth);

    c.CalcSize^(display);
  END CalcSize;

  PROCEDURE (c : Cycle) GetFocus*(event : E.Event):G.Object;

  BEGIN
    IF ~c.visible OR c.disabled OR (c.model=NIL) OR c.model.IsNull() THEN
      RETURN NIL;
    END;

    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseDown) & c.PointIsIn(event.x,event.y)
      & (event.qualifier={}) & (event.button=E.button1) THEN
        c.selected:=TRUE;
        c.Redraw;
        RETURN c;
      END;
    ELSE
    END;
    RETURN NIL;
  END GetFocus;

  PROCEDURE (c : Cycle) HandleEvent*(event : E.Event):BOOLEAN;

  VAR
    selected : SelectedMsg;

  BEGIN
    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseUp) & (event.button=E.button1) THEN
        IF c.PointIsIn(event.x,event.y) THEN
          IF c.model.GetLongint()>=c.count THEN
            c.model.SetLongint(1);
          ELSE
            c.model.Inc;
          END;
          NEW(selected);
          c.Send(selected,selectedMsg);
          (* Action: Next entry selected *)
        END;
        c.selected:=FALSE;
        c.Redraw;
        RETURN TRUE;
      END;
    | event : E.MotionEvent DO
      IF c.PointIsIn(event.x,event.y) THEN
        IF ~c.selected THEN
          c.selected:=TRUE;
          c.Redraw;
        END;
      ELSE
        IF c.selected THEN
          c.selected:=FALSE;
          c.Redraw;
        END;
      END;
    ELSE
    END;
    RETURN FALSE;
  END HandleEvent;

  PROCEDURE (c : Cycle) HandleFocusEvent*(event : E.KeyEvent):BOOLEAN;

  VAR
    keysym   : LONGINT;
    selected : SelectedMsg;

  BEGIN
    IF event.type=E.keyDown THEN
      keysym:=event.GetKey();
      IF (keysym=E.down) OR (keysym=E.space) THEN
        IF c.model.GetLongint()>=c.count THEN
          c.model.SetLongint(1);
        ELSE
          c.model.Inc;
        END;
        NEW(selected);
        c.Send(selected,selectedMsg);
        RETURN TRUE;
      ELSIF keysym=E.up THEN
        IF c.model.GetLongint()<=1 THEN
          c.model.SetLongint(c.count);
        ELSE
          c.model.Dec;
        END;
        NEW(selected);
        c.Send(selected,selectedMsg);
        RETURN TRUE;
      END;
    END;
    RETURN FALSE;
  END HandleFocusEvent;

  PROCEDURE (c : Cycle) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  VAR
    i : LONGINT;
    w : LONGINT;

  BEGIN
    c.Draw^(x,y,draw);

    IF c.selected THEN
      draw.mode:={D.selected};
    END;

    c.frame.Resize(c.width,c.height);
    c.frame.Draw(c.x,c.y,draw);

    i:=1;
    c.current:=c.list;
    IF (c.model#NIL) & ~c.model.IsNull() THEN
      WHILE (c.current#NIL) & (i<c.model.GetLongint()) DO
        INC(i);
        c.current:=c.current.next;
      END;
    END;

    IF c.current#NIL THEN
      w:=c.width-c.image.width-c.frame.leftBorder-c.frame.rightBorder;
      c.current.Resize(w,c.height-c.frame.topBorder-c.frame.bottomBorder);
      c.CopyBackground(c.current);

      IF c.HasFocus() THEN
        (* tell the object to display focus *)
        INCL(c.current.flags,G.showFocus);
      ELSE
        (* tell the object to not display focus *)
        EXCL(c.current.flags,G.showFocus);
      END;

      IF c.prefs.imageRight THEN
        c.current.Draw(c.x+c.frame.leftBorder+(w-c.current.width) DIV 2,
                       c.y+(c.height-c.current.height) DIV 2,draw);
      ELSE
        c.current.Draw(c.x+c.frame.leftBorder+c.image.width+(w-c.current.width) DIV 2,
                       c.y+(c.height-c.current.height) DIV 2,draw);
      END;

    ELSE
      (* TODO: Clear display *)
    END;

    c.image.Resize(-1,c.height-c.frame.topBorder-c.frame.bottomBorder);
    IF c.prefs.imageRight THEN
      c.image.Draw(c.x+c.width-c.image.width-c.frame.rightBorder,c.y+c.frame.topBorder,draw);
    ELSE
      c.image.Draw(c.x+c.frame.leftBorder,c.y+c.frame.topBorder,draw);
    END;

    draw.mode:={};

    IF c.disabled THEN
      c.DrawDisabled;
    END;
  END Draw;

  (**
    Refreshes the group object by refreshing all objects in the object list.
    The refresh retangle is preserved and will be handled to all subobjects.
  **)

  PROCEDURE (c : Cycle) Refresh*(x,y,w,h : LONGINT);

  BEGIN
    IF c.visible & c.Intersect(x,y,w,h) THEN
      c.frame.Refresh(x,y,w,h);
      c.image.Refresh(x,y,w,h);
      IF c.current#NIL THEN
        c.current.Refresh(x,y,w,h);
      END;

      IF c.StdFocus() & (c.focus#NIL) THEN
        c.focus.Refresh(x,y,w,h);
      END;
    END;
  END Refresh;


  PROCEDURE (c : Cycle) DrawFocus*;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    IF (c.current#NIL) & ~c.current.StdFocus() THEN
      IF c.selected THEN
        c.draw.mode:={D.selected};
      ELSE
        c.draw.mode:={};
      END;
      c.current.DrawFocus;
      c.draw.mode:={};
    ELSE
      (* Delegate drawing to the baseclass *)
      c.DrawFocus^;
    END;
  END DrawFocus;

  (**
    Hide the keyboard focus.
  **)

  PROCEDURE (c : Cycle) HideFocus*;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    IF (c.current#NIL) & ~c.current.StdFocus() THEN
      IF c.selected THEN
        c.draw.mode:={D.selected};
      ELSE
        c.draw.mode:={};
      END;
      c.current.HideFocus;
      c.draw.mode:={};
    ELSE
      (* Delegate drawing to the baseclass *)
     c.HideFocus^;
    END;
  END HideFocus;

  PROCEDURE (c : Cycle) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    c.Redraw;
  END Resync;

  PROCEDURE (c : Cycle) Hide*;

  BEGIN
    IF c.visible THEN
      c.frame.Hide;
      c.image.Hide;
      IF c.current#NIL THEN
        c.current.Hide;
      END;
      c.Hide^;
    END;
  END Hide;

BEGIN
  NEW(prefs);
  prefs.Init;

END VOCycle.