/************************************************************************/
/*  Management of X11 fonts: Translation of understandable fonts to	*/
/*  X11 structures.							*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<debugon.h>

#   include	<X11/Xatom.h>

#   include	"appFrame.h"
#   include	"appUtil.h"

/************************************************************************/
/*  Represents a font chooser.						*/
/************************************************************************/
typedef struct AppFontToolResources
    {
    char *	aftrFamily;
    char *	aftrFace;
    char *	aftrSize;
    char *	aftrRevert;
    char *	aftrPreview;
    char *	aftrSet;
    } AppFontToolResources;

typedef struct AppFontChooser
    {
    Widget		afcTopWidget;
    Widget		afcMainWidget;

    Widget		afcFamilyList;
    Widget		afcFaceList;
    Widget		afcSizeList;

    Widget		afcPreviewButton;

    Widget		afcSampleLabel;
    XmFontList		afcSampleFontList;

    int			afcFamilyChosen;
    int			afcFaceChosen;
    int			afcBoldChosen;
    int			afcSlantedChosen;
    int			afcSizeChosen;

    int			afcFamilySet;
    int			afcFaceSet;
    int			afcBoldSet;
    int			afcSlantedSet;
    int			afcSizeSet;

    double		afcMagnifiedPixelsPerTwip;

    void *		afcTarget;

    AppToolDestroy	afcDestroy;
    FontChooserSetFont	afcSetFont;
    
    AppFontFamily *	afcFamilies;
    int			afcFamilyCount;
    } AppFontChooser;

/************************************************************************/
/*  Display the fonts panel.						*/
/************************************************************************/
#   include	<Xm/MainW.h>
#   include	<Xm/Form.h>
#   include	<Xm/Label.h>
#   include	<Xm/List.h>
#   include	<Xm/ScrolledW.h>
#   include	<Xm/RowColumn.h>
#   include	<Xm/Label.h>

#   include	<Xm/Protocols.h>

/************************************************************************/
/*  Value lists that can be used when no particular font is selected.	*/
/************************************************************************/
static int EditFontToolDummySizes[]=
			    { 8, 9, 10, 11, 12, 14, 16, 18, 24, 36, 48, 64 };

#   define DEFSIZES \
    EditFontToolDummySizes, sizeof(EditFontToolDummySizes)/sizeof(int)

static AppFontTypeface EditFontToolDummyFace=
	{ "Any", DEFSIZES, 0, 0, 0, 0 };

static AppFontTypeface EditFontToolDefaultFaces[]=
    {
	{ "Medium", DEFSIZES, 0, 0, 1, 0 },
	{ "Bold", DEFSIZES, 1, 0, 1, 0 },
	{ "Italic", DEFSIZES, 0, 1, 1, 0 },
	{ "Bold Italic", DEFSIZES, 1, 1, 1, 0 },
    };

static AppFontFamily EditFontToolDummyFamily=
    {
    "Any", (char *)0,
	EditFontToolDefaultFaces,
	sizeof(EditFontToolDefaultFaces)/sizeof(AppFontTypeface),
	0, 1,
    };

/************************************************************************/
/*  Get current.							*/
/************************************************************************/
static void appFontGetCurrent(		AppFontFamily **	pEff,
					AppFontTypeface **	pEft,
					int *			pFontSize,
					AppFontChooser *	afc	)
    {
    AppFontFamily *	aff= (AppFontFamily *)0;
    AppFontTypeface *	aft= (AppFontTypeface *)0;
    int			size= -1;

    if  ( afc->afcFamilyChosen >= 0 )
	{
	aff= afc->afcFamilies+ afc->afcFamilyChosen;

	if  ( afc->afcFaceChosen >= 0 )
	    { aft= aff->affFaces+ afc->afcFaceChosen; }
	}
    else{
	aff= &EditFontToolDummyFamily;

	if  ( afc->afcFaceChosen >= 0 )
	    { aft= aff->affFaces+ afc->afcFaceChosen; }
	}

    size= afc->afcSizeChosen;

    if  ( pEff )
	{ *pEff= aff;	}
    if  ( pEft )
	{ *pEft= aft;	}
    if  ( pFontSize )
	{ *pFontSize= size;	}
    }

/************************************************************************/
/*  Reflect the currently selected font.				*/
/************************************************************************/
static void appFontShowCurrent(	AppFontChooser *	afc	)
    {
    XmString		labelString;
    char		familyFormat[120];

    AppFontFamily *	aff= (AppFontFamily *)0;
    AppFontTypeface *	aft= (AppFontTypeface *)0;
    int			size= -1;

    XtVaSetValues( afc->afcSampleLabel,
		XmNfontList,		afc->afcSampleFontList,
		NULL );

    appFontGetCurrent( &aff, &aft, &size, afc );

    appFontFormatCurrent( familyFormat, aff, aft, size );

    labelString= XmStringCreateLocalized( familyFormat );

    XtVaSetValues( afc->afcSampleLabel,
			XmNlabelString,		labelString,
			NULL	);

    XmStringFree( labelString );

    XtSetSensitive( afc->afcPreviewButton,
				afc->afcFamilyChosen >= 0	&&
				afc->afcFaceChosen >= 0		&&
				afc->afcSizeChosen >= 0		);

    return;
    }

/************************************************************************/
/*  refill size list and reselect selected size.			*/
/************************************************************************/
static void appFontFillSizes(		AppFontChooser *	afc,
					AppFontTypeface *	aft,
					int			fontSize )
    {
    int		pos;
    int		sizeChosen= -1;

    XmListDeleteAllItems( afc->afcSizeList );

    for ( pos= 0; pos < aft->aftSizeCount; pos++ )
	{
	XmString	labelString;
	char		scratch[80];

	sprintf( scratch, "%d", aft->aftSizes[pos] );
	labelString= XmStringCreateLocalized( scratch );

	XmListAddItemUnselected( afc->afcSizeList, labelString, 0 );

	XmStringFree( labelString );

	if  ( fontSize > 0 && aft->aftSizes[pos] == fontSize )
	    { sizeChosen= pos;	}
	}

    if  ( sizeChosen >= 0 )
	{ XmListSelectPos( afc->afcSizeList, sizeChosen+ 1, False ); }

    afc->afcSizeChosen= fontSize;

    return;
    }

/************************************************************************/
/*									*/
/*  A font family has been chosen, or set.				*/
/*  Try to maintain the properties in the other listboxes.		*/
/*									*/
/************************************************************************/

static void appFontReflectFamily(	AppFontChooser *	afc,
					int			familyNumber,
					AppFontFamily *		aff,
					int			fontBold,
					int			fontSlanted,
					int			fontSize )
    {
    AppFontTypeface *		aft;
    int				pos;
    int				faceChosen= -1;

    if  ( afc->afcFamilyChosen != familyNumber )
	{
	XmListDeleteAllItems( afc->afcFaceList );

	aft= aff->affFaces;
	for ( pos= 0; pos < aff->affFaceCount; aft++, pos++ )
	    {
	    XmString	labelString;

	    labelString= XmStringCreateLocalized( aft->aftFaceName );

	    XmListAddItemUnselected( afc->afcFaceList, labelString, 0 );

	    XmStringFree( labelString );

	    if  ( fontBold == (int)aft->aftIsBold	&&
		  fontSlanted == (int)aft->aftIsSlanted	)
		{ faceChosen= pos;	}
	    }

	afc->afcFamilyChosen= familyNumber;
	afc->afcFaceChosen= -1;
	}
    else{
	aft= aff->affFaces;
	for ( pos= 0; pos < aff->affFaceCount; aft++, pos++ )
	    {
	    if  ( fontBold == (int)aft->aftIsBold	&&
		  fontSlanted == (int)aft->aftIsSlanted	)
		{ faceChosen= pos;	}
	    }
	}

    if  ( faceChosen >= 0 )
	{
	XmListSelectPos( afc->afcFaceList, faceChosen+ 1, False );

	if  ( afc->afcFaceChosen != faceChosen )
	    {
	    afc->afcFaceChosen= faceChosen;

	    appFontFillSizes( afc, aff->affFaces+ faceChosen, fontSize );
	    }
	else{
	    int		sizeChosen= -1;

	    aft= aff->affFaces+ faceChosen;
	    for ( pos= 0; pos < aft->aftSizeCount; pos++ )
		{
		if  ( fontSize > 0 && aft->aftSizes[pos] == fontSize )
		    { sizeChosen= pos;	}
		}

	    XmListDeselectAllItems( afc->afcSizeList );

	    if  ( sizeChosen >= 0 )
		{ XmListSelectPos( afc->afcSizeList, sizeChosen+ 1, False ); }

	    afc->afcSizeChosen= fontSize;
	    }
	}
    else{
	afc->afcFaceChosen= faceChosen;

	XmListDeselectAllItems( afc->afcFaceList );

	appFontFillSizes( afc, &EditFontToolDummyFace, fontSize );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  A font family has been chosen					*/
/*									*/
/************************************************************************/
static void appFontFamilyChosen(	Widget		w,
					XtPointer	voidafc,
					XtPointer	voidlcs	 )
    {
    XmListCallbackStruct *	lcs= (XmListCallbackStruct *)voidlcs;
    AppFontChooser *		afc= (AppFontChooser *)voidafc;

    AppFontFamily *		aff;
    int				affCount;

    int				fontSize;

    int				familyNumber;

    if  ( lcs->item_position- 1 == afc->afcFamilyChosen )
	{ return;	}

    appFontGetCurrent( NULL, NULL, &fontSize, afc );

    aff= afc->afcFamilies;
    affCount= afc->afcFamilyCount;

    afc->afcFaceChosen= -1;

    familyNumber= lcs->item_position- 1;
    if  ( familyNumber < 0 || familyNumber >= affCount )
	{
	LLDEB(familyNumber,affCount);
	afc->afcFamilyChosen= -1;
	return;
	}

    aff= afc->afcFamilies+ familyNumber;
    appFontReflectFamily( afc, familyNumber, aff,
			afc->afcBoldChosen, afc->afcSlantedChosen, fontSize );

    appFontShowCurrent( afc );

    return;
    }

/************************************************************************/
/*  A font style has been chosen					*/
/************************************************************************/
static void appFontFaceChosen(	Widget		w,
				XtPointer	voidafc,
				XtPointer	voidlcs	 )
    {
    XmListCallbackStruct *	lcs= (XmListCallbackStruct *)voidlcs;
    AppFontChooser *		afc= (AppFontChooser *)voidafc;

    AppFontFamily *		aff;
    AppFontTypeface *		aft;
    int				affCount;

    int				fontSize;

    AppFontTypeface *		faceList= EditFontToolDefaultFaces;
    int				faceCount= sizeof(EditFontToolDefaultFaces)/
						    sizeof(AppFontTypeface);

    if  ( lcs->item_position- 1 == afc->afcFaceChosen )
	{ return;	}

    appFontGetCurrent( NULL, NULL, &fontSize, afc );

    aff= afc->afcFamilies;
    affCount= afc->afcFamilyCount;

    afc->afcFaceChosen= -1;

    if  ( afc->afcFamilyChosen >= 0 && afc->afcFamilyChosen < affCount )
	{
	aff= afc->afcFamilies+ afc->afcFamilyChosen;

	faceList= aff->affFaces;
	faceCount= aff->affFaceCount;
	}

    afc->afcFaceChosen= lcs->item_position- 1;
    if  ( afc->afcFaceChosen < 0 || afc->afcFaceChosen >= faceCount )
	{
	LLDEB(afc->afcFaceChosen,aff->affFaceCount);
	afc->afcFaceChosen= -1;
	return;
	}

    aft= aff->affFaces+ afc->afcFaceChosen;

    afc->afcBoldChosen= aft->aftIsBold;
    afc->afcSlantedChosen= aft->aftIsSlanted;

    appFontFillSizes( afc, aft, fontSize );

    appFontShowCurrent( afc );

    return;
    }

/************************************************************************/
/*  A font size has been chosen						*/
/************************************************************************/
static void appFontSizeChosen(		Widget		w,
					XtPointer	voidafc,
					XtPointer	voidlcs	 )
    {
    XmListCallbackStruct *	lcs= (XmListCallbackStruct *)voidlcs;
    AppFontChooser *		afc= (AppFontChooser *)voidafc;

    int				sizeChosen;

    int *			sizeList= EditFontToolDummySizes;
    int				sizeCount= sizeof(EditFontToolDummySizes)/
								sizeof(int);
    if  ( afc->afcFamilyChosen >= 0			&&
	  afc->afcFamilyChosen <  afc->afcFamilyCount	)
	{
	AppFontFamily *	aff= afc->afcFamilies+ afc->afcFamilyChosen;

	if  ( afc->afcFaceChosen >= 0			&&
	      afc->afcFaceChosen <  aff->affFaceCount	)
	    {
	    AppFontTypeface *	aft= aff->affFaces+ afc->afcFaceChosen;

	    sizeList= aft->aftSizes;
	    sizeCount= aft->aftSizeCount;
	    }
	}

    sizeChosen= lcs->item_position- 1;
    if  ( sizeChosen < 0 || sizeChosen >= sizeCount )
	{ LLDEB(sizeChosen,sizeCount); afc->afcSizeChosen= -1; return; }

    afc->afcSizeChosen= sizeList[sizeChosen];

    appFontShowCurrent( afc );

    return;
    }

/************************************************************************/
/*  Fill the list with families.					*/
/************************************************************************/
static void appFontFillFamilies(	Display *		display,
					AppFontChooser *	afc	)
    {
    int			i;
    AppFontFamily *	aff;

    aff=  afc->afcFamilies;
    for ( i= 0; i < afc->afcFamilyCount; aff++, i++ )
	{
	XmString		labelString;

	if  ( aff->affFontFamilyText )
	    { labelString= XmStringCreateLocalized( aff->affFontFamilyText ); }
	else{ labelString= XmStringCreateLocalized( aff->affFontFamilyName ); }

	XmListAddItemUnselected( afc->afcFamilyList, labelString, 0 );

	XmStringFree( labelString );
	}

    return;
    }

/************************************************************************/
/*  A fontchooser must be destroyed.					*/
/************************************************************************/
static void appCloseFontChooser(		Widget		w,
						XtPointer	voidafc,
						XtPointer	voidlcs	 )
    {
    AppFontChooser *		afc= (AppFontChooser *)voidafc;

    if  ( afc->afcSampleFontList )
	{ XmFontListFree( afc->afcSampleFontList );	}

    if  ( ! afc->afcDestroy )
	{ XDEB(afc->afcDestroy);		}
    else{ (*afc->afcDestroy)( afc->afcTarget );	}

    free( voidafc );

    XtDestroyWidget( w );

    return;
    }

/************************************************************************/
/*  Create a listbox with a title.					*/
/************************************************************************/
static void appFontToolTitledList(	Widget		chooseForm,
					Dimension	left,
					Dimension	right,
					Widget *	pForm,
					Widget *	pList,
					int		visibleItems,
					char *		title		)
    {
    XmString	titleString;
    Widget	label;

    Arg			al[20];
    int			ac= 0;

    ac= 0;
    XtSetArg( al[ac], XmNtopAttachment,		XmATTACH_POSITION ); ac++;
    XtSetArg( al[ac], XmNtopPosition,		0 ); ac++;
    XtSetArg( al[ac], XmNleftAttachment,	XmATTACH_POSITION ); ac++;
    XtSetArg( al[ac], XmNleftPosition,		left ); ac++;
    XtSetArg( al[ac], XmNrightAttachment,	XmATTACH_POSITION ); ac++;
    XtSetArg( al[ac], XmNrightPosition,		right ); ac++;
    XtSetArg( al[ac], XmNbottomAttachment,	XmATTACH_POSITION ); ac++;
    XtSetArg( al[ac], XmNbottomPosition,	9 ); ac++;

    *pForm= XmCreateForm( chooseForm, WIDGET_NAME, al, ac );

    titleString= XmStringCreateLocalized( title );

    ac= 0;
    XtSetArg( al[ac], XmNtopAttachment,		XmATTACH_FORM ); ac++;
    XtSetArg( al[ac], XmNleftAttachment,	XmATTACH_FORM ); ac++;
    XtSetArg( al[ac], XmNrightAttachment,	XmATTACH_FORM ); ac++;
    XtSetArg( al[ac], XmNbottomAttachment,	XmATTACH_NONE ); ac++;
    XtSetArg( al[ac], XmNtopOffset,		5 ); ac++;
    XtSetArg( al[ac], XmNleftOffset,		5 ); ac++;
    XtSetArg( al[ac], XmNrightOffset,		5 ); ac++;
    XtSetArg( al[ac], XmNbottomOffset,		5 ); ac++;
    XtSetArg( al[ac], XmNlabelString,		titleString ); ac++;

    label= XmCreateLabel( *pForm, WIDGET_NAME, al, ac );

    XmStringFree( titleString );

    ac= 0;
    XtSetArg( al[ac], XmNtopAttachment,		XmATTACH_WIDGET ); ac++;
    XtSetArg( al[ac], XmNtopWidget,		label ); ac++;
    XtSetArg( al[ac], XmNleftAttachment,	XmATTACH_FORM ); ac++;
    XtSetArg( al[ac], XmNrightAttachment,	XmATTACH_FORM ); ac++;
    XtSetArg( al[ac], XmNbottomAttachment,	XmATTACH_FORM ); ac++;
    XtSetArg( al[ac], XmNtopOffset,		5 ); ac++;
    XtSetArg( al[ac], XmNleftOffset,		5 ); ac++;
    XtSetArg( al[ac], XmNrightOffset,		5 ); ac++;
    XtSetArg( al[ac], XmNbottomOffset,		5 ); ac++;
    XtSetArg( al[ac], XmNvisibleItemCount,	visibleItems ); ac++;

    *pList= XmCreateScrolledList( *pForm, WIDGET_NAME, al, ac );

    XtManageChild( label );
    XtManageChild( *pList );
    XtManageChild( *pForm );

    return;
    }

/************************************************************************/
/*  make the font chooser worm with three listboxes.			*/
/************************************************************************/
static void appFontMakeChooseForm(	Widget			fontsForm,
					AppFontToolResources *	aftr,
					AppFontChooser *	afc,
					Widget			above,
					Widget			below	)
    {
    Widget		chooseForm;

    Widget		familyForm;
    Widget		faceForm;
    Widget		sizeForm;

    Arg			al[20];
    int			ac= 0;

    ac= 0;
    XtSetArg( al[ac],	XmNfractionBase,	9 ); ac++;
    XtSetArg( al[ac],	XmNtopAttachment,	XmATTACH_WIDGET ); ac++;
    XtSetArg( al[ac],	XmNtopWidget,		above ); ac++;
    XtSetArg( al[ac],	XmNtopOffset,		5 ); ac++;
    XtSetArg( al[ac],	XmNleftAttachment,	XmATTACH_FORM ); ac++;
    XtSetArg( al[ac],	XmNrightAttachment,	XmATTACH_FORM ); ac++;
    XtSetArg( al[ac],	XmNbottomAttachment,	XmATTACH_WIDGET ); ac++;
    XtSetArg( al[ac],	XmNbottomWidget,	below ); ac++;
    XtSetArg( al[ac],	XmNbottomOffset,	5 ); ac++;

    chooseForm= XmCreateForm( fontsForm, WIDGET_NAME, al, ac );

    /********************************************************************/
    /*  Parents with listbox and label:					*/
    /********************************************************************/
    appFontToolTitledList( chooseForm, 0, 3, &familyForm,
				    &afc->afcFamilyList, 9, aftr->aftrFamily );

    XtAddCallback( afc->afcFamilyList,
	    XmNbrowseSelectionCallback, appFontFamilyChosen, (void *)afc );

    appFontToolTitledList( chooseForm, 3, 6, &faceForm,
				    &afc->afcFaceList, 9, aftr->aftrFace );

    XtAddCallback( afc->afcFaceList,
	    XmNbrowseSelectionCallback, appFontFaceChosen, (void *)afc );

    appFontToolTitledList( chooseForm, 6, 9, &sizeForm,
				    &afc->afcSizeList, 12, aftr->aftrSize );

    XtAddCallback( afc->afcSizeList,
	    XmNbrowseSelectionCallback, appFontSizeChosen, (void *)afc );

    XtManageChild( chooseForm );

    return;
    }

/************************************************************************/
/*  make the label to preview the font.					*/
/************************************************************************/
static Widget appFontMakePreviewText(	Display *		display,
					int			screen,
					Widget			fontsForm,
					AppFontChooser *	afc,
					int			height )
    {
    XmFontList		fl;

    Arg			al[20];
    int			ac= 0;

    XtSetArg( al[ac], XmNtopAttachment, XmATTACH_FORM ); ac++;
    XtSetArg( al[ac], XmNtopOffset, 5 ); ac++;
    XtSetArg( al[ac], XmNleftAttachment, XmATTACH_FORM ); ac++;
    XtSetArg( al[ac], XmNleftOffset, 5 ); ac++;
    XtSetArg( al[ac], XmNrightAttachment, XmATTACH_FORM ); ac++;
    XtSetArg( al[ac], XmNrightOffset, 5 ); ac++;
    XtSetArg( al[ac], XmNbottomAttachment, XmATTACH_OPPOSITE_FORM ); ac++;
    XtSetArg( al[ac], XmNbottomOffset, -height+ 5 ); ac++;

    XtSetArg( al[ac], XmNbackground, WhitePixel( display, screen ) ); ac++;
    XtSetArg( al[ac], XmNforeground, BlackPixel( display, screen ) ); ac++;

    afc->afcSampleLabel= XmCreateLabel( fontsForm, WIDGET_NAME, al, ac );

    XtVaGetValues( afc->afcSampleLabel,
		    XmNfontList,	&fl,
		    NULL );

    afc->afcSampleFontList= XmFontListCopy( fl );

    XtManageChild( afc->afcSampleLabel );

    return afc->afcSampleLabel;
    }

/************************************************************************/
/*  The preview button of a font tool has been pushed.			*/
/************************************************************************/
static void appFontPreviewPushed(	Widget		w,
					XtPointer	voidafc,
					XtPointer	voidpbcs	 )
    {
    AppFontChooser *		afc= (AppFontChooser *)voidafc;

    char		scratch[120];
    XFontStruct *	xfs;
    XmFontList		fontList;

    AppFontFamily *	aff;
    AppFontTypeface *	aft;
    int			pointSize;

    appFontGetCurrent( &aff, &aft, &pointSize, afc );

    if  ( appFontXFont( scratch, XtDisplay( afc->afcSampleLabel ),
				    afc->afcMagnifiedPixelsPerTwip,
				    aff, aft, 20* pointSize, DOCfontREGULAR ) )
	{ SDEB(aff->affFontFamilyName); return;	}

    xfs= XLoadQueryFont( XtDisplay( afc->afcSampleLabel ), scratch );
    if  ( ! xfs )
	{ SXDEB(scratch,xfs); return;	}

    fontList= XmFontListCreate( xfs, XmSTRING_DEFAULT_CHARSET );

    XtVaSetValues( afc->afcSampleLabel, XmNfontList, fontList, NULL );

    XmFontListFree( fontList );

    return;
    }

/************************************************************************/
/*  The set button of a font tool has been pushed.			*/
/************************************************************************/
static void appFontSetPushed(		Widget		w,
					XtPointer	voidafc,
					XtPointer	voidpbcs	 )
    {
    AppFontChooser *		afc= (AppFontChooser *)voidafc;

    AppFontFamily *		aff;
    AppFontTypeface *		aft;
    int				size;

    char *			familyText= (char *)0;

    if  ( ! afc->afcSetFont )
	{ XDEB(afc->afcSetFont); return;	}

    appFontGetCurrent( &aff, &aft, &size, afc );

    if  ( aff )
	{
	if  ( aff->affFontFamilyText )
	    { familyText= aff->affFontFamilyText;	}
	else{ familyText= aff->affFontFamilyName;	}
	}

    if  ( aft )
	{
	(*afc->afcSetFont)( afc->afcTarget, familyText, afc->afcFamilyChosen,
				    aft->aftIsBold, aft->aftIsSlanted, size );
	}
    else{
	(*afc->afcSetFont)( afc->afcTarget, familyText, afc->afcFamilyChosen,
								-1, -1, size );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  The 'Revert' button of a font tool has been pushed.			*/
/*									*/
/************************************************************************/

static void appFontRevertPushed(	Widget		w,
					XtPointer	voidafc,
					XtPointer	voidpbcs	 )
    {
    AppFontChooser *		afc= (AppFontChooser *)voidafc;

    if  ( afc->afcFamilySet < 0 )
	{ XmListDeselectAllItems( afc->afcFamilyList ); }
    else{
	AppFontFamily *	aff= afc->afcFamilies+ afc->afcFamilySet;

	XmListSelectPos( afc->afcFamilyList, afc->afcFamilySet+ 1, False );

	appFontReflectFamily( afc, afc->afcFamilySet, aff,
				afc->afcBoldSet, afc->afcSlantedSet,
				afc->afcSizeSet );
	}

    afc->afcFamilyChosen= afc->afcFamilySet;
    afc->afcFaceChosen= afc->afcFaceSet;
    afc->afcSizeChosen= afc->afcSizeSet;
    afc->afcBoldChosen= afc->afcBoldSet;
    afc->afcSlantedChosen= afc->afcSlantedSet;

    appFontShowCurrent( afc );

    return;
    }

/************************************************************************/
/*									*/
/*  Reflect/Remember the current font of the application.		*/
/*									*/
/*  1)  Adapt the old fashioned internals to new style interface.	*/
/*									*/
/************************************************************************/

int appFontSetCurrentFont(	void *		voidafc,
				int		fontFamilyNumber,
				unsigned int	updMask,
				TextAttribute	taNew )
    {
    AppFontChooser *	afc= (AppFontChooser *)voidafc;

    int			fontBold= -1;
    int			fontSlanted= -1;
    int			fontSize= -1;

    /*  1  */
    if  ( updMask & TAmaskFONTBOLD )
	{ fontBold= taNew.taFontIsBold;	}
    if  ( updMask & TAmaskFONTSLANTED )
	{ fontSlanted= taNew.taFontIsSlanted;	}
    if  ( updMask & TAmaskFONTSIZE )
	{ fontSize= taNew.taFontSizeHalfPoints/ 2;	}

    afc->afcFamilySet= fontFamilyNumber;
    afc->afcBoldChosen= afc->afcBoldSet= fontBold;
    afc->afcSlantedChosen= afc->afcSlantedSet= fontSlanted;
    afc->afcSizeSet= fontSize;

    if  ( fontFamilyNumber < 0 )
	{
	XmListDeselectAllItems( afc->afcFamilyList );

	appFontReflectFamily( afc, fontFamilyNumber,
		&EditFontToolDummyFamily, fontBold, fontSlanted, fontSize );

	afc->afcFaceSet= afc->afcFaceChosen;
	}
    else{
	AppFontFamily *	aff= afc->afcFamilies+ fontFamilyNumber;

	XmListSelectPos( afc->afcFamilyList, fontFamilyNumber+ 1, False );

	appFontReflectFamily( afc, fontFamilyNumber, aff,
			    afc->afcBoldSet, afc->afcSlantedSet, fontSize );

	afc->afcFaceSet= afc->afcFaceChosen;
	}

    appFontShowCurrent( afc );

    return 0;
    }

/************************************************************************/
/*  make the form with the three buttons.				*/
/************************************************************************/
static Widget appFontMakeButtonForm(	Widget			fontsForm,
					AppFontToolResources *	aftr,
					AppFontChooser *	afc	)
    {
    Widget	buttonForm;

    Widget	revertButton;
    Widget	setButton;

    Arg		al[20];
    int		ac= 0;

    ac= 0;
    XtSetArg( al[ac],	XmNfractionBase,	3 ); ac++;
    XtSetArg( al[ac],	XmNleftAttachment,	XmATTACH_FORM ); ac++;
    XtSetArg( al[ac],	XmNrightAttachment,	XmATTACH_FORM ); ac++;
    XtSetArg( al[ac],	XmNbottomAttachment,	XmATTACH_FORM ); ac++;
    XtSetArg( al[ac],	XmNleftOffset,		5 ); ac++;
    XtSetArg( al[ac],	XmNrightOffset,		5 ); ac++;
    XtSetArg( al[ac],	XmNbottomOffset,	5 ); ac++;

    buttonForm= XmCreateForm( fontsForm, WIDGET_NAME, al, ac );

    revertButton= appMakeRowButton( buttonForm, aftr->aftrRevert,
				appFontRevertPushed, (void *)afc, 0, 0 );
    afc->afcPreviewButton= appMakeRowButton( buttonForm, aftr->aftrPreview,
				appFontPreviewPushed, (void *)afc, 1, 0 );
    setButton= appMakeRowButton( buttonForm, aftr->aftrSet,
				appFontSetPushed, (void *)afc, 2, 0 );

    XtManageChild( revertButton );
    XtManageChild( afc->afcPreviewButton );
    XtManageChild( setButton );

    XtManageChild( buttonForm );

    return buttonForm;
    }

/************************************************************************/
/*  make a fonts tool.							*/
/************************************************************************/
# define xx(x)	x,x

static XtResource APP_FontToolResourceTable[]=
    {
	{ xx("fontToolFamily"), XtRString, sizeof(char *),
		    offsetof(AppFontToolResources,aftrFamily),
		    XtRString, "Family" },
	{ xx("fontToolFace"), XtRString, sizeof(char *),
		    offsetof(AppFontToolResources,aftrFace),
		    XtRString, "Typeface" },
	{ xx("fontToolSize"), XtRString, sizeof(char *),
		    offsetof(AppFontToolResources,aftrSize),
		    XtRString, "Size" },
	{ xx("fontToolRevert"), XtRString, sizeof(char *),
		    offsetof(AppFontToolResources,aftrRevert),
		    XtRString, "Revert" },
	{ xx("fontToolPreview"), XtRString, sizeof(char *),
		    offsetof(AppFontToolResources,aftrPreview),
		    XtRString, "Preview" },
	{ xx("fontToolSet"), XtRString, sizeof(char *),
		    offsetof(AppFontToolResources,aftrSet),
		    XtRString, "Set" },
    };

void * appMakeFontsTool(	Widget			fontsOption,
				Widget			topWidget,
				const char *		applicationName,
				const char *		widgetName,
				Pixmap			iconPixmap,
				Atom			closeAtom,
				AppToolDestroy		destroy,
				FontChooserSetFont	setFont,
				double			magnifiedPixelsPerTwip,
				EditApplication *	ea )
    {
    Display *		display= XtDisplay( fontsOption );
    int			screen= DefaultScreen( display );
    AppFontChooser *	afc;
    
    Arg			al[20];
    int			ac= 0;

    static AppFontToolResources	aftr;
    static int			gotResources;

    if  ( ! gotResources )
	{
	XtGetApplicationResources( topWidget, (void *)&aftr,
		APP_FontToolResourceTable, XtNumber(APP_FontToolResourceTable),
		NULL, 0 );

	gotResources= 1;
	}

    afc= (AppFontChooser *)malloc( sizeof(AppFontChooser) );
    if  ( ! afc )
	{ XDEB(afc); return (void *)0;	}

    afc->afcTarget= (void *)ea;
    afc->afcDestroy= destroy;
    afc->afcSetFont= setFont;
    afc->afcMagnifiedPixelsPerTwip=  magnifiedPixelsPerTwip;
    afc->afcSampleFontList= (XmFontList)0;

    XtSetArg( al[ac], XmNdeleteResponse, XmDO_NOTHING ); ac++;
    XtSetArg( al[ac], XmNinput,		True ); ac++;

    if  ( iconPixmap )
	{ XtSetArg( al[ac], XmNiconPixmap,	iconPixmap ); ac++; }

    afc->afcTopWidget= XtAppCreateShell( applicationName,
					    widgetName,
					    applicationShellWidgetClass,
					    display, al, ac );

    XtAddEventHandler( afc->afcTopWidget, StructureNotifyMask, False,
					appSetSizeAsMinimum, (void *)afc );

    appSetShellTitle( afc->afcTopWidget, fontsOption, applicationName );

    if  ( closeAtom > 0 )
	{
	XmAddWMProtocolCallback( afc->afcTopWidget, closeAtom,
				    appCloseFontChooser, (XtPointer)afc );
	}

    ac= 0;
    XtSetArg( al[ac], XmNuserData, (void *)afc ); ac++;
    afc->afcMainWidget= XmCreateForm( afc->afcTopWidget, WIDGET_NAME, al, ac );

    {
    Widget	previewText;
    Widget	buttonForm;
    int		previewHeight= 60;

    previewText= appFontMakePreviewText( display, screen,
				    afc->afcMainWidget, afc, previewHeight );
    buttonForm= appFontMakeButtonForm( afc->afcMainWidget, &aftr, afc );

    appFontMakeChooseForm( afc->afcMainWidget, &aftr,
						afc, previewText, buttonForm );
    }

    XtManageChild( afc->afcMainWidget );

    if  ( psFontCatalog( ea->eaAfmDirectory,
				&afc->afcFamilies, &afc->afcFamilyCount ) )
	{
	SDEB(ea->eaAfmDirectory);
	afc->afcFamilies= (AppFontFamily *)0;
	afc->afcFamilyCount= 0;
	}

    appFontFillFamilies( display, afc );

    afc->afcFamilySet= afc->afcFamilyChosen= -2; /* force fill of styles */
    afc->afcFaceSet= afc->afcFaceChosen= -1;
    afc->afcBoldSet= afc->afcBoldChosen= -1;
    afc->afcSlantedSet= afc->afcSlantedChosen= -1;
    afc->afcSizeSet= afc->afcSizeChosen= -1;

    XtRealizeWidget( afc->afcTopWidget );
    XtMapWidget( afc->afcTopWidget );

    return (void *)afc;
    }

/************************************************************************/
/*  Draw a fonts tool to front.						*/
/************************************************************************/
void appShowFontsTool(		void *	voidafc	)
    {
    AppFontChooser *	afc= (AppFontChooser *)voidafc;

    XtVaSetValues( afc->afcTopWidget, XmNinitialState, NormalState, NULL );
    XtMapWidget( afc->afcTopWidget );
    XRaiseWindow( XtDisplay( afc->afcTopWidget ),
					    XtWindow( afc->afcTopWidget ) );
    }

void appEnableFontsTool(	void *	voidafc,
				int	enabled )
    {
    AppFontChooser *	afc= (AppFontChooser *)voidafc;

    XtSetSensitive( afc->afcMainWidget, enabled != 0 );

    return;
    }

void appFontToolTextReadonly(	void *	voidafc,
				int	readonly )
    {
    AppFontChooser *	afc= (AppFontChooser *)voidafc;

    XtSetSensitive( afc->afcMainWidget, readonly == 0 );

    return;
    }
