/************************************************************************/
/*  A spelling tool.							*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<debugon.h>

#   include	<sys/stat.h>
#   include	<unistd.h>
#   include	<errno.h>

#   include	<X11/Xatom.h>
#   include	<X11/IntrinsicP.h>
#   include	<Xm/Text.h>
#   include	<Xm/Form.h>
#   include	<Xm/Label.h>
#   include	<Xm/Frame.h>
#   include	<Xm/List.h>
#   include	<Xm/PanedW.h>
#   include	<Xm/Label.h>
#   include	<Xm/RowColumn.h>

#   include	<Xm/Protocols.h>

#   include	"appSpellTool.h"
#   include	<appSystem.h>
#   include	<appFrame.h>
#   include	"appUtil.h"

#   include	<charnames.h>
#   include	<ind.h>

/************************************************************************/
/*									*/
/*  The list of dictionaries.						*/
/*									*/
/************************************************************************/

static SpellCheckContext *	EditSpellDictionaries;
static int			EditSpellDictionaryCount;

/************************************************************************/
/*									*/
/*  Resources for the spell tool.					*/
/*									*/
/************************************************************************/

typedef struct AppSpellToolResources
    {
    char *	astrDictionary;
    char *	astrLearn;
    char *	astrForget;
    char *	astrGuesses;
    char *	astrIgnore;
    char *	astrFindNext;
    char *	astrGuess;
    char *	astrCorrect;
    char *	astrPrivateDictionaries;
    char *	astrSystemDictionaries;
    char *	astrDirNoAccess;
    char *	astrNoSuchDir;
    char *	astrDirNoDicts;
    char *	astrNoDicts;
    char *	astrSysDictNoAccess;
    char *	astrPrivDictDirNotMade;
    char *	astrPrivDictNoAccess;
    char *	astrPrivDictWrongFormat;
    } AppSpellToolResources;

# define xx(x)	x,x

static XtResource APP_SpellToolResourceTable[]=
    {
    /************************/
    /*  Spell Tool.		*/
    /************************/
    { xx("spellToolDictTitle"), XtRString, sizeof(char *),
		offsetof(AppSpellToolResources,astrDictionary),
		XtRString, "Dictionary" },
    { xx("spellToolLearn"), XtRString, sizeof(char *),
		offsetof(AppSpellToolResources,astrLearn),
		XtRString, "Learn" },
    { xx("spellToolForget"), XtRString, sizeof(char *),
		offsetof(AppSpellToolResources,astrForget),
		XtRString, "Forget" },
    { xx("spellToolGuesses"), XtRString, sizeof(char *),
		offsetof(AppSpellToolResources,astrGuesses),
		XtRString, "Guesses" },
    { xx("spellToolIgnore"), XtRString, sizeof(char *),
		offsetof(AppSpellToolResources,astrIgnore),
		XtRString, "Ignore" },
    { xx("spellToolFindNext"), XtRString, sizeof(char *),
		offsetof(AppSpellToolResources,astrFindNext),
		XtRString, "Find Next" },
    { xx("spellToolGuess"), XtRString, sizeof(char *),
		offsetof(AppSpellToolResources,astrGuess),
		XtRString, "Guess" },
    { xx("spellToolCorrect"), XtRString, sizeof(char *),
		offsetof(AppSpellToolResources,astrCorrect),
		XtRString, "Correct" },
    { xx("spellToolNoDicts"), XtRString, sizeof(char *),
		offsetof(AppSpellToolResources,astrNoDicts),
		XtRString, "None" },

    { xx("spellToolPrivateDicts"), XtRString, sizeof(char *),
		offsetof(AppSpellToolResources,astrPrivateDictionaries),
		XtRString, ".Dictionaries" },
    { xx("spellToolSystemDicts"), XtRString, sizeof(char *),
		offsetof(AppSpellToolResources,astrSystemDictionaries),
		XtRString, INDDIR },

    { xx("spellToolDirNoAccess"), XtRString, sizeof(char *),
		offsetof(AppSpellToolResources,astrDirNoAccess),
		XtRString, "Could not access directory for dictionaries." },
    { xx("spellToolDirNoSuchDir"), XtRString, sizeof(char *),
		offsetof(AppSpellToolResources,astrNoSuchDir),
		XtRString, "This directory could not be found." },
    { xx("spellToolDirNoDicts"), XtRString, sizeof(char *),
		offsetof(AppSpellToolResources,astrDirNoDicts), XtRString,
		"No dictionaries were found in this directory." },
    { xx("spellToolSysDictNoAccess"), XtRString, sizeof(char *),
		offsetof(AppSpellToolResources,astrSysDictNoAccess),
		XtRString, "Could not read system dictionary." },
    { xx("spellToolPrivDictDirNotMade"), XtRString, sizeof(char *),
		offsetof(AppSpellToolResources,astrPrivDictDirNotMade),
		XtRString, "Could not make private directory." },
    { xx("spellToolPrivDictNoAccess"), XtRString, sizeof(char *),
		offsetof(AppSpellToolResources,astrPrivDictNoAccess),
		XtRString, "Could not read private dictionary." },
    { xx("spellToolPrivDictWrongFormat"), XtRString, sizeof(char *),
		offsetof(AppSpellToolResources,astrPrivDictWrongFormat),
		XtRString, "Private dictionary has an illegal format." },
    };

/************************************************************************/
/*									*/
/*  Represents a spell tool.						*/
/*									*/
/************************************************************************/
#   define	FILEL	400

typedef struct AppSpellTool
    {
    EditApplication *		astApplication;

    AppSpellToolResources *	astResources;

    Widget			astTopWidget;
    Widget			astMainWidget;

    Widget			astDictionaryMenu;
    Widget			astDictionaryPulldown;
    Widget			astGuessList;
    Widget			astWordText;

    Widget			astLearnButton;
    Widget			astForgetButton;
    Widget			astCorrectButton;
    Widget			astGuessButton;
    Widget			astIgnoreButton;

    AppToolDestroy		astDestroy;
    SpellToolFindNext		astFindNext;
    SpellToolCorrect		astCorrect;
    void *			astTarget;

    SpellCheckContext *		astDictionaryContexts;
    int				astDictionaryCount;

    int				astCurrentDictionary;

    char *			astPrivateDictionaries;
    char *			astSystemDictionaries;
    } AppSpellTool;

static FILE *	appSpellOpenDictionary(	const char *	prefix,
					AppSpellTool *	ast,
					int		readonly );

/************************************************************************/
/*  Open index files for a certain dictionary.				*/
/************************************************************************/

static int appSpellToolOpenIndices(	SpellCheckContext *	scc,
					AppSpellTool *		ast )
    {
    char *		systemdicts= ast->astSystemDictionaries;

    char		scratch[FILEL];
    char *		format= "%s/%s.ind";

    FILE *		privateDict;

    sprintf( scratch, format, systemdicts, scc->sccDictionaryPrefix );

    scc->sccStaticInd= indRead( scratch, 1 );

    if  ( ! scc->sccStaticInd )
	{
	AppSpellToolResources *	astr= ast->astResources;

	appQuestionRunSubjectErrorDialog( ast->astApplication,
			ast->astTopWidget, (Widget)0,
			scratch, astr->astrSysDictNoAccess );
	return -1;
	}

    privateDict= appSpellOpenDictionary( scc->sccDictionaryPrefix, ast, 1 );

    if  ( privateDict )
	{
	if  ( indReadPrivateDictionary( privateDict,
				&scc->sccLearntInd, &(scc->sccForgotInd) ) )
	    {
	    AppSpellToolResources *	astr= ast->astResources;

	    appQuestionRunSubjectErrorDialog( ast->astApplication,
			ast->astTopWidget, (Widget)0,
			scc->sccDictionaryPrefix,
			astr->astrPrivDictWrongFormat );
	    }

	fclose( privateDict );
	}

    return 0;
    }

/************************************************************************/
/*  'Learn' button has been pushed.					*/
/************************************************************************/
static void appSpellToolLearnPushed(	Widget		w,
					XtPointer	voidast,
					XtPointer	voidpbcs	 )
    {
    AppSpellTool *	ast= (AppSpellTool *)voidast;
    unsigned char *	word;
    SpellCheckContext *	scc;
    FILE *		privateDict;

    if  ( ast->astCurrentDictionary < 0 )
	{ LDEB(ast->astCurrentDictionary); return;	}

    scc= ast->astDictionaryContexts+ ast->astCurrentDictionary;

    if  ( ! scc->sccStaticInd			&&
	  appSpellToolOpenIndices( scc, ast )	)
	{ return;	}

    if  ( ! scc->sccLearntInd )
	{
	scc->sccLearntInd= indMake();
	if  ( ! scc->sccLearntInd )
	    { XDEB(scc->sccLearntInd); return;	}
	}

    privateDict= appSpellOpenDictionary( scc->sccDictionaryPrefix, ast, 0 );
    if  ( ! privateDict )
	{ return;	}

    word= (unsigned char *)XmTextGetString( ast->astWordText );

    if  ( indLearnWord( privateDict,
			    scc->sccLearntInd, scc->sccForgotInd, word ) )
	{ SDEB((char *)word);	}

    XtFree( (char *)word );

    fclose( privateDict );

    XtVaSetValues( ast->astLearnButton,
			XmNsensitive,		False,
			NULL );
    XtVaSetValues( ast->astForgetButton,
			XmNsensitive,		True,
			NULL );
    return;
    }

/************************************************************************/
/*  'Forget' button has been pushed.					*/
/************************************************************************/
static void appSpellToolForgetPushed(	Widget		w,
					XtPointer	voidast,
					XtPointer	voidpbcs	 )
    {
    AppSpellTool *	ast= (AppSpellTool *)voidast;
    unsigned char *	word;
    SpellCheckContext *	scc;
    FILE *		privateDict;

    if  ( ast->astCurrentDictionary < 0 )
	{ LDEB(ast->astCurrentDictionary); return;	}

    scc= ast->astDictionaryContexts+ ast->astCurrentDictionary;

    if  ( ! scc->sccStaticInd			&&
	  appSpellToolOpenIndices( scc, ast )	)
	{ return;	}

    if  ( ! scc->sccForgotInd )
	{
	scc->sccForgotInd= indMake();
	if  ( ! scc->sccForgotInd )
	    { XDEB(scc->sccForgotInd); return;	}
	}

    privateDict= appSpellOpenDictionary( scc->sccDictionaryPrefix, ast, 0 );
    if  ( ! privateDict )
	{ SXDEB(scc->sccDictionaryPrefix,privateDict); return;	}

    word= (unsigned char *)XmTextGetString( ast->astWordText );

    if  ( indForgetWord( privateDict,
			    scc->sccLearntInd, scc->sccForgotInd, word ) )
	{ SDEB((char *)word);	}

    XtFree( (char *)word );

    fclose( privateDict );

    XtVaSetValues( ast->astForgetButton,
			XmNsensitive,		False,
			NULL );
    XtVaSetValues( ast->astLearnButton,
			XmNsensitive,		True,
			NULL );

    return;
    }

/************************************************************************/
/*  Adapt buttons to different situations..				*/
/************************************************************************/

static void appSpellToolSomethingFound(	AppSpellTool *	ast,
					int		yes_no )
    {
    XtVaSetValues( ast->astLearnButton,
			XmNsensitive,		yes_no,
			NULL );
    XtVaSetValues( ast->astIgnoreButton,
			XmNsensitive,		yes_no,
			NULL );

    return;
    }

static void appSpellToolGotAlternative(	AppSpellTool *	ast,
					int		yes_no )
    {
    XtVaSetValues( ast->astForgetButton,
			XmNsensitive,		yes_no,
			NULL );
    XtVaSetValues( ast->astCorrectButton,
			XmNsensitive,		yes_no,
			NULL );
    XtVaSetValues( ast->astGuessButton,
			XmNsensitive,		yes_no,
			NULL );

    return;
    }

/************************************************************************/
/*  'Find Next' button has been pushed.					*/
/************************************************************************/
static void editSpellToolFindNext(	Widget		w,
					XtPointer	voidast,
					XtPointer	voidpbcs	 )
    {
    AppSpellTool *	ast= (AppSpellTool *)voidast;
    SpellCheckContext *	scc;

    if  ( ast->astCurrentDictionary < 0 )
	{ LDEB(ast->astCurrentDictionary); return;	}

    if  ( ! ast->astFindNext < 0 )
	{ XDEB(ast->astFindNext); return;	}

    scc= ast->astDictionaryContexts+ ast->astCurrentDictionary;

    XmTextSetString( ast->astWordText, "" );
    XmListDeleteAllItems( ast->astGuessList );

    if  ( ! scc->sccStaticInd			&&
	  appSpellToolOpenIndices( scc, ast )	)
	{ return;	}

    if  ( ! (*ast->astFindNext)( ast->astTarget, scc ) )
	{ appSpellToolSomethingFound( ast, 1 );	}
    else{ appSpellToolSomethingFound( ast, 0 );	}

    appSpellToolGotAlternative( ast, 0 );
    }

/************************************************************************/
/*									*/
/*  'Ignore' button has been pushed.					*/
/*									*/
/************************************************************************/

static void editSpellToolIgnorePushed(	Widget		w,
					XtPointer	voidast,
					XtPointer	voidpbcs	 )
    {
    AppSpellTool *	ast= (AppSpellTool *)voidast;
    unsigned char *	word;
    SpellCheckContext *	scc;

    if  ( ast->astCurrentDictionary < 0 )
	{ LDEB(ast->astCurrentDictionary); return;	}

    scc= ast->astDictionaryContexts+ ast->astCurrentDictionary;

    if  ( ! scc->sccLearntInd )
	{
	scc->sccLearntInd= indMake();
	if  ( ! scc->sccLearntInd )
	    { XDEB(scc->sccLearntInd); return;	}
	}

    word= (unsigned char *)XmTextGetString( ast->astWordText );

    if  ( indMoveWord( scc->sccForgotInd, scc->sccLearntInd, word ) )
	{ SDEB((char *)word); return;	}

    editSpellToolFindNext( w, voidast, voidpbcs );

    XtFree( (char *)word );

    return;
    }

/************************************************************************/
/*  The user typed something in the Coreection, Turn on the the		*/
/*  'Correct' button.							*/
/************************************************************************/
static void editSpellCorrectionTyped(	Widget		w,
					XtPointer	voidast,
					XtPointer	ignored	 )
    {
    AppSpellTool *	ast= (AppSpellTool *)voidast;

    appSpellToolGotAlternative( ast, 1 );

    return;
    }

/************************************************************************/
/*  'Correct' button has been pushed.					*/
/*  Or a double click on the listbox with guesses.			*/
/************************************************************************/
static void editSpellToolCorrect(	Widget		w,
					XtPointer	voidast,
					XtPointer	ignored	 )
    {
    AppSpellTool *	ast= (AppSpellTool *)voidast;
    char *	guess;

    if  ( ! ast->astCorrect < 0 )
	{ XDEB(ast->astCorrect); return;	}

    guess= XmTextGetString( ast->astWordText );

    (*ast->astCorrect)( ast->astTarget, (unsigned char *)guess );

    XtFree( guess );

    editSpellToolFindNext( w, voidast, ignored );

    return;
    }

/************************************************************************/
/*  The 'Guess' button has been pushed.					*/
/************************************************************************/
static void editSpellGuessButtonPushed(	Widget		w,
					XtPointer	voidast,
					XtPointer	voidpbcs	 )
    {
    AppSpellTool *	ast= (AppSpellTool *)voidast;
    char *		word;

    word= XmTextGetString( ast->astWordText );

    appSpellMakeGuesses( voidast, (unsigned char *)word );

    XtFree( word );

    return;
    }

/************************************************************************/
/*  A guess in the list has been selected.				*/
/************************************************************************/
static void editSpellGuessChosen(	Widget		w,
					XtPointer	voidast,
					XtPointer	voidlcs	 )
    {
    XmListCallbackStruct *	lcs= (XmListCallbackStruct *)voidlcs;
    AppSpellTool *		ast= (AppSpellTool *)voidast;

    char *			text;

    if  ( XmStringGetLtoR( lcs->item, XmFONTLIST_DEFAULT_TAG, &text ) )
	{
	XmTextSetString( ast->astWordText, text );
	XtFree( text );
	}

    appSpellToolGotAlternative( ast, 1 );

    return;
    }

/************************************************************************/
/*									*/
/*  Make a row of buttons for the Spell Tool.				*/
/*									*/
/************************************************************************/

static void appSpellToolMakeButtonRow(	Widget *	pRow,
					Widget *	pLeftButton,
					Widget *	pRightButton,
					char *		leftLabel,
					char *		rightLabel,
					XtCallbackProc	leftCallback,
					XtCallbackProc	rightCallback,
					AppSpellTool *	ast,
					Widget		above,
					Widget		parent )

    {
    Widget		row;

    Widget		leftButton;
    Widget		rightButton;

    row= appMakeButtonRow( parent, 2 );

    leftButton= appMakeRowButton( row, leftLabel,
					leftCallback, (void *)ast, 0, False );
    rightButton= appMakeRowButton( row, rightLabel,
					rightCallback, (void *)ast, 1, False );

    XtManageChild( row );

    *pRow= row; *pLeftButton= leftButton, *pRightButton= rightButton;

    return;
    }

/************************************************************************/
/*									*/
/*  Create the dictionary part of the spelling tool.			*/
/*									*/
/************************************************************************/

static Widget appSpellMakeDictionaryFrame( Widget		parent,
					AppSpellToolResources * astr,
					AppSpellTool *		ast	)
    {
    Widget		frame;
    Widget		bboard;
    Widget		paned;

    Widget		buttonRow;

    appMakeVerticalFrame( &frame, &bboard, &paned,
					    parent, astr->astrDictionary );

    appMakePulldownList( &ast->astDictionaryPulldown,
					    &ast->astDictionaryMenu, paned );

    appSpellToolMakeButtonRow( &buttonRow,
		&(ast->astLearnButton), &(ast->astForgetButton),
		astr->astrLearn, astr->astrForget,
		appSpellToolLearnPushed, appSpellToolForgetPushed,
		ast, ast->astDictionaryMenu, paned );

    XtManageChild( ast->astDictionaryMenu );
    XtManageChild( buttonRow );

    XtManageChild( paned );
    XtManageChild( bboard );
    XtManageChild( frame );

    return frame;
    }

/************************************************************************/
/*									*/
/*  Make the button part of the spelling Tool.				*/
/*									*/
/************************************************************************/

static void appSpellMakeButtonRows(	Widget			spellForm,
					Widget			above,
					AppSpellToolResources * astr,
					AppSpellTool *		ast )
    {
    Widget		buttonRow;

    Widget		findNextButton;

    appSpellToolMakeButtonRow( &buttonRow,
		&(ast->astIgnoreButton), &(findNextButton),
		astr->astrIgnore, astr->astrFindNext,
		editSpellToolIgnorePushed, editSpellToolFindNext,
		ast, above, spellForm );

    above= buttonRow;

    appSpellToolMakeButtonRow( &buttonRow,
		&(ast->astGuessButton), &(ast->astCorrectButton),
		astr->astrGuess, astr->astrCorrect,
		editSpellGuessButtonPushed, editSpellToolCorrect,
		ast, above, spellForm );

    return;
    }

/************************************************************************/
/*  A spell tool must be destroyed.					*/
/************************************************************************/
static void appCloseSpellTool(		Widget		w,
					XtPointer	voidast,
					XtPointer	voidlcs	 )
    {
    AppSpellTool *	ast= (AppSpellTool *)voidast;

    if  ( ast->astDestroy )
	{ (*ast->astDestroy)( ast->astTarget );	}

    XtDestroyWidget( w );

    free( ast );

    return;
    }

/************************************************************************/
/*									*/
/*  Initialise character classification.				*/
/*									*/
/*  This should be based on some kind of knowledge, NOT on the name of	*/
/*  the language.							*/
/*									*/
/*  The font of the XmList and the XmText should be adapted to the	*/
/*  character set of the language. Now things look ridiculous.		*/
/*									*/
/************************************************************************/

static void appSpellIso1CharacterKinds(		SpellCheckContext *	scc )
    {
    int		i;

    memset( scc->sccCharKinds, 0, 256 );
    for ( i= 0; i < 256; i++ )
	{ scc->sccCharShifts[i]= i; }

    for ( i= 0; i < 256; i++ )
	{
	if  ( ISO1_islower( i ) )
	    {
	    scc->sccCharKinds[i] |= CHARisLOWER;
	    if  ( ! ISO1_isupper( i ) )
		{ scc->sccCharShifts[i]= ISO1_toupper( i );	}
	    }

	if  ( ISO1_isupper( i ) )
	    {
	    scc->sccCharKinds[i] |= CHARisUPPER;
	    if  ( ! ISO1_islower( i ) )
		{ scc->sccCharShifts[i]= ISO1_tolower( i );	}
	    }

	if  ( ISO1_isdigit( i ) )
	    { scc->sccCharKinds[i] |= CHARisDIGIT;	}
	}

    return;
    }

static void appSpellIso2CharacterKinds(		SpellCheckContext *	scc )
    {
    int		i;

    memset( scc->sccCharKinds, 0, 256 );
    for ( i= 0; i < 256; i++ )
	{ scc->sccCharShifts[i]= i; }

    for ( i= 0; i < 256; i++ )
	{
	if  ( ISO2_islower( i ) )
	    {
	    scc->sccCharKinds[i] |= CHARisLOWER;
	    if  ( ! ISO2_isupper( i ) )
		{ scc->sccCharShifts[i]= ISO2_toupper( i );	}
	    }

	if  ( ISO2_isupper( i ) )
	    {
	    scc->sccCharKinds[i] |= CHARisUPPER;
	    if  ( ! ISO2_islower( i ) )
		{ scc->sccCharShifts[i]= ISO2_tolower( i );	}
	    }

	if  ( ISO2_isdigit( i ) )
	    { scc->sccCharKinds[i] |= CHARisDIGIT;	}
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Obtain the list of dictionaries.					*/
/*									*/
/************************************************************************/

static int appSpellAddDictionary(	const char *		filename,
					void *			voisast )
    {
    SpellCheckContext *	fresh;
    char *		name;

    const char *	s;
    int			l;

    s= strrchr( filename, '/' );
    if  ( s )
	{ s++;		}
    else{ s= filename;	}

    l= strlen( s );
    if  ( l == 0 )
	{ LDEB(l); return 0;	}

    if  ( strcmp( s+ l- 4, ".ind" ) )
	{ SSDEB(filename,s+ l- 4); return 0;	}

    fresh= (SpellCheckContext *)realloc( EditSpellDictionaries,
		    (EditSpellDictionaryCount+ 1)*sizeof(SpellCheckContext) );
    if  ( ! fresh )
	{ XDEB(fresh); return -1;	}
    EditSpellDictionaries= fresh;

    name= (char *)malloc( l- 3 );
    if  ( ! name )
	{ LXDEB(l,name); return -1;       }
    strncpy( name, s, l- 4 )[l- 4]= '\0';

    fresh += EditSpellDictionaryCount++;

    fresh->sccDictionaryPrefix= name;
    fresh->sccStaticInd= (void *)0;
    fresh->sccForgotInd= (void *)0;
    fresh->sccLearntInd= (void *)0;

    /********************************************/
    /*  Wrong.. But concentrated in one place.	*/
    /********************************************/
    if  ( ! strcmp( name, "Czech" )	||
	  ! strcmp( name, "Polish" )	||
	  ! strcmp( name, "Slovak" )	)
	{ appSpellIso2CharacterKinds( fresh );	}
    else{ appSpellIso1CharacterKinds( fresh );	}

    return 0;
    }

static int appSpellGetDictionaries(	AppSpellTool *		ast,
					AppSpellToolResources *	astr )
    {
    char *		systemdicts= ast->astSystemDictionaries;

    if  ( EditSpellDictionaryCount > 0 )
	{
	ast->astDictionaryContexts= EditSpellDictionaries;
	ast->astDictionaryCount= EditSpellDictionaryCount;
	return 0;
	}

    if  ( appForAllFiles( systemdicts, "ind",
					(void *)ast, appSpellAddDictionary ) )
	{ SDEB(systemdicts); return -1;	}

    if  ( EditSpellDictionaryCount == 0 )
	{
	if  ( appTestDirectory( systemdicts ) )
	    {
	    appQuestionRunSubjectErrorDialog( ast->astApplication,
			(Widget)0, (Widget)0,
			systemdicts, astr->astrNoSuchDir );
	    }
	else{
	    if  ( appTestFileReadable( systemdicts ) )
		{
		appQuestionRunSubjectErrorDialog( ast->astApplication,
			    (Widget)0, (Widget)0,
			    systemdicts, astr->astrDirNoAccess );
		}
	    else{
		appQuestionRunSubjectErrorDialog( ast->astApplication,
			    (Widget)0, (Widget)0,
			    systemdicts, astr->astrDirNoDicts );
		}
	    }
	}

    ast->astDictionaryContexts= EditSpellDictionaries;
    ast->astDictionaryCount= EditSpellDictionaryCount;

    return 0;
    }

/************************************************************************/
/*  A dictionary has been selected.					*/
/************************************************************************/
static void editSpellDictionaryChosen(	Widget		w,
					XtPointer	voidast,
					XtPointer	voidpbcs	 )
    {
    short		dictionaryChosen= -1;
    AppSpellTool *	ast= (AppSpellTool *)voidast;

    XtVaGetValues( w,
			XmNpositionIndex,	&dictionaryChosen,
			NULL );

    if  ( dictionaryChosen < 0 )
	{ LDEB(dictionaryChosen); return;	}

    ast->astCurrentDictionary= dictionaryChosen;

    return;
    }

/************************************************************************/
/*  Fill the list of dictionaries.					*/
/************************************************************************/
static void appSpellFillDictionaryMenu(	AppSpellTool *		ast,
						AppSpellToolResources *	astr )
    {
    Dimension		width;
    int			i;

    Widget		child0= (Widget)0;

    XtVaGetValues( ast->astDictionaryMenu,
			XmNwidth,		&width,
			NULL );

    appEmptyPulldownList( ast->astDictionaryPulldown );

    for ( i= 0; i < ast->astDictionaryCount; i++ )
	{
	Widget		fresh;

	fresh= appPulldownMakeOption( ast->astDictionaryPulldown,
			    ast->astDictionaryContexts[i].sccDictionaryPrefix,
			    width, editSpellDictionaryChosen, (void *)ast );

	if  ( i == 0 )
	    { child0= fresh;	}
	}

    if  ( ast->astDictionaryCount == 0 )
	{
	child0= appPulldownMakeOption( ast->astDictionaryPulldown,
			    astr->astrNoDicts,
			    width, editSpellDictionaryChosen, (void *)ast );

	XtSetSensitive( ast->astTopWidget, 0 );
	}

    if  ( child0 )
	{
	XtVaSetValues( ast->astDictionaryMenu,
			    XmNmenuHistory,		child0,
			    NULL );

	ast->astCurrentDictionary= 0;
	}

    appPulldownSetWidth( ast->astDictionaryMenu, width );
    }

/************************************************************************/
/*  Make the listbox with guesses.					*/
/************************************************************************/
static Widget appSpellGuessList(	Widget			parent,
					AppSpellToolResources * astr,
					AppSpellTool *		ast )
    {
    XmString	titleString;

    Widget	label;
    Widget	list;

    Arg		al[20];
    int		ac= 0;

    titleString= XmStringCreateLocalized( astr->astrGuesses );

    ac= 0;
    XtSetArg( al[ac],	XmNskipAdjust,	True ); ac++;
    XtSetArg( al[ac],	XmNlabelString,	titleString ); ac++;
    label= XmCreateLabel( parent, WIDGET_NAME, al, ac );

    XmStringFree( titleString );

    ac= 0;
    XtSetArg( al[ac],	XmNskipAdjust,		False ); ac++;
    XtSetArg( al[ac],	XmNvisibleItemCount,	6 ); ac++;
    list= XmCreateScrolledList( parent, WIDGET_NAME, al, ac );

    XtManageChild( label );
    XtManageChild( list );

    return list;
    }

/************************************************************************/
/*  make a spell tool.							*/
/************************************************************************/
void * appMakeSpellTool(	Widget			spellOption,
				EditApplication *	ea,
				const char *		widgetName,
				Pixmap			iconPixmap,
				Atom			closeAtom,
				AppToolDestroy		destroy,
				SpellToolFindNext	findNext,
				SpellToolCorrect	correct,
				void *			target		)
    {
    AppSpellTool *	ast;
    
    Arg			al[20];
    int			ac= 0;

    Widget		dictionaryFrame;

    Dimension		width;

    static AppSpellToolResources	astr;
    static int			gotResources;

    if  ( ! gotResources )
	{
	XtGetApplicationResources( ea->eaTopWidget, (void *)&astr,
	    APP_SpellToolResourceTable, XtNumber(APP_SpellToolResourceTable),
	    NULL, 0 );

	gotResources= 1;
	}

    ast= (AppSpellTool *)malloc( sizeof(AppSpellTool) );
    if  ( ! ast )
	{ XDEB(ast); return (void *)0;	}

    ast->astApplication= ea;

    ast->astResources= &astr;

    ast->astDestroy= destroy;
    ast->astFindNext= findNext;
    ast->astCorrect= correct;
    ast->astTarget= target;
    ast->astPrivateDictionaries= astr.astrPrivateDictionaries;
    ast->astSystemDictionaries= astr.astrSystemDictionaries;

    ast->astDictionaryContexts= (SpellCheckContext *)0;
    ast->astDictionaryCount= 0;
    ast->astCurrentDictionary= -1;

    appSpellGetDictionaries( ast, &astr );

    XtSetArg( al[ac], XmNuserData,	(void *)ast ); ac++;
    XtSetArg( al[ac], XmNdeleteResponse, XmDO_NOTHING ); ac++;
    XtSetArg( al[ac], XmNinput,		True ); ac++;

    if  ( iconPixmap )
	{ XtSetArg( al[ac], XmNiconPixmap,	iconPixmap ); ac++; }

    ast->astTopWidget= XtAppCreateShell( ea->eaApplicationName,
					    widgetName,
					    applicationShellWidgetClass,
					    ea->eaDisplay, al, ac );

    XtAddEventHandler( ast->astTopWidget, StructureNotifyMask, False,
					appSetSizeAsMinimum, (void *)ast );

    appSetShellTitle( ast->astTopWidget, spellOption, ea->eaApplicationName );

    if  ( closeAtom > 0 )
	{
	XmAddWMProtocolCallback( ast->astTopWidget, closeAtom,
				    appCloseSpellTool, (XtPointer)ast );
	}

    ac= 0;
    XtSetArg( al[ac],	XmNuserData,		(void *)ast ); ac++;
    XtSetArg( al[ac],	XmNsashWidth,		1 ); ac++;
    XtSetArg( al[ac],	XmNsashHeight,		1 ); ac++;
    XtSetArg( al[ac],	XmNseparatorOn,		False ); ac++;
    XtSetArg( al[ac],	XmNmarginWidth,		5 ); ac++;
    XtSetArg( al[ac],	XmNmarginHeight,	5 ); ac++;
    XtSetArg( al[ac],	XmNspacing,		5 ); ac++;
    XtSetArg( al[ac],	XmNskipAdjust,		False ); ac++;
    ast->astMainWidget= XmCreatePanedWindow( ast->astTopWidget, WIDGET_NAME, al, ac );

    dictionaryFrame= appSpellMakeDictionaryFrame( ast->astMainWidget,
								&astr, ast );
    ast->astGuessList= appSpellGuessList( ast->astMainWidget, &astr, ast );

    XtAddCallback( ast->astGuessList, XmNbrowseSelectionCallback,
					editSpellGuessChosen, (void *)ast );

    XtAddCallback( ast->astGuessList, XmNdefaultActionCallback,
					editSpellToolCorrect, (void *)ast );

    appMakeColumnText( &(ast->astWordText), ast->astMainWidget, 0, True );

    XtAddCallback( ast->astWordText, XmNvalueChangedCallback,
				    editSpellCorrectionTyped, (void *)ast );

    appSpellMakeButtonRows( ast->astMainWidget,
					    ast->astWordText, &astr, ast );


    XtManageChild( ast->astMainWidget );

    appSpellFillDictionaryMenu( ast, &astr );

    XtRealizeWidget( ast->astTopWidget );
    XtMapWidget( ast->astTopWidget );

    XtVaGetValues( ast->astDictionaryMenu,
			    XmNwidth,		&width,
			    NULL );

    appPulldownSetWidth( ast->astDictionaryMenu, width );

    appSpellToolSomethingFound( ast, 0 );
    appSpellToolGotAlternative( ast, 0 );

    return (void *)ast;
    }

/************************************************************************/
/*  Draw a spell tool to front.						*/
/************************************************************************/
void appShowSpellTool(		void *	voidast	)
    {
    AppSpellTool *		ast= (AppSpellTool *)voidast;

    XmTextSetString( ast->astWordText, "" );
    XmListDeleteAllItems( ast->astGuessList );

    XtVaSetValues( ast->astTopWidget, XmNinitialState, NormalState, NULL );
    XtMapWidget( ast->astTopWidget );
    XRaiseWindow( XtDisplay( ast->astTopWidget ),
					    XtWindow( ast->astTopWidget ) );

    appSpellToolSomethingFound( ast, 0 );
    appSpellToolGotAlternative( ast, 0 );

    }

void appSpellMakeGuesses(	void *			voidast,
				const unsigned char *	word	)
    {
    AppSpellTool *		ast= (AppSpellTool *)voidast;
    SpellGuessContext		sgc;
    SpellCheckContext *		scc;

    int				limit;
    int				i;

    static IndGuessList		igl;

    if  ( ast->astCurrentDictionary < 0 )
	{ LDEB(ast->astCurrentDictionary); return;	}

    scc= ast->astDictionaryContexts+ ast->astCurrentDictionary;

    if  ( ! scc->sccStaticInd			&&
	  appSpellToolOpenIndices( scc, ast )	)
	{ return;	}

    sgc.sgcGuessList= &igl;
    sgc.sgcCheckContext= scc;

    indCleanGuesses( &igl );

    XmListDeleteAllItems( ast->astGuessList );
    XmTextSetString( ast->astWordText, (char *)word );

    if  ( scc->sccStaticInd )
	{
	indGuessWord( scc->sccStaticInd, word, &sgc,
		    (const GuessSubstitution *)0, 0,
		    scc->sccCharKinds, scc->sccCharShifts );
	}

    if  ( scc->sccLearntInd )
	{
	indGuess( scc->sccLearntInd, word, &sgc, INDhASIS,
		    (const GuessSubstitution *)0, 0,
		    scc->sccCharKinds, scc->sccCharShifts );
	}

    indSortGuesses( &igl );

    if  ( igl.iglGuessCount > 6 )
	{
	limit= ( igl.iglGuesses[0].igsScore+
		    2* igl.iglGuesses[igl.iglGuessCount-1].igsScore )/ 3;
	}
    else{ limit= 0;	}

    for ( i= 0; i < igl.iglGuessCount; i++ )
	{
	XmString	str;

	if  ( igl.iglGuesses[i].igsScore < limit )
	    { break;	}

	str= XmStringCreateLocalized( (char *)igl.iglGuesses[i].igsWord );
	XmListAddItemUnselected( ast->astGuessList, str, 0 );
	XmStringFree( str );
	}

    appSpellToolGotAlternative( ast, 0 );
    }

/************************************************************************/
/*									*/
/*  Open a private dictionary.						*/
/*									*/
/************************************************************************/

static FILE *	appSpellOpenDictionary(	const char *	prefix,
					AppSpellTool *	ast,
					int		readonly )
    {
    char *		dir= ast->astPrivateDictionaries;
    char		scratch[FILEL+ 1];
    char		home[FILEL+ 1];
    int			len;

    FILE *		f;

    char *		mode;

    len= appHomeDirectory( home, FILEL );
    if  ( len < 0 )
	{ LDEB(len); return (FILE *)0;	}

    if  ( dir[0] == '/' )
	{ strcpy( scratch, dir );			}
    else{ sprintf( scratch, "%s/%s", home, dir );	}

    if  ( appTestDirectory( scratch ) )
	{
	if  ( readonly )
	    { return (FILE *)0;	}

	if  ( appMakeDirectory( scratch ) )
	    {
	    AppSpellToolResources *	astr= ast->astResources;

	    appQuestionRunSubjectErrorDialog( ast->astApplication,
			ast->astTopWidget, (Widget)0,
			scratch, astr->astrPrivDictDirNotMade );

	    return (FILE *)0;
	    }
	}

    if  ( dir[0] == '/' )
	{ sprintf( scratch, "%s/%s.changes", dir, prefix );		}
    else{ sprintf( scratch, "%s/%s/%s.changes", home, dir, prefix );	}

    if  ( readonly )
	{ mode= "r";	}
    else{ mode= "a";	}

    f= fopen( scratch, mode );
    if  ( ! f && ! readonly )
	{ SLDEB(scratch,errno); return (FILE *)0; }

    return f;
    }

void appEnableSpellTool(	void *	voidast,
				int	enabled )
    {
    AppSpellTool *	ast= (AppSpellTool *)voidast;

    XtSetSensitive( ast->astMainWidget, enabled != 0 );

    return;
    }
