//  UFont.cpp version 1.5
//  yudit package - Unicode Editor for the X Window System (and Linux) 
//
//  Author: gsinai@iname.com (Gaspar Sinai)
//  GNU Copyright (C) 1997,1998,1999  Gaspar Sinai
// 
//  yudit version 1.5  Copyright(C) 30 November, 1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.4  Copyright(C) 25 November, 1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.3  Copyright(C)  5 April,    1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.2  Copyright(C) 10 December, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.1  Copyright(C) 23 August,   1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.0  Copyright(C) 17 May,      1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.99 Copyright(C)  4 April,    1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.97 Copyright(C)  4 February, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.95 Copyright(C) 10 January,  1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.94 Copyright(C) 17 December, 1997, Tokyo Japan  Gaspar Sinai
//  yudit version 0.9 Copyright (C)  8 December, 1997, Tokyo Japan  Gaspar Sinai
//  yutex version 0.8 Copyright (C)  5 November, 1997, Tokyo Japan  Gaspar Sinai
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
//
//
#include 	"UFont.h"
#include 	"UCache.h"
#include 	"UCommon.h"
#include 	<string.h>
#include 	<strings.h>

const char*	weightMap[] = {"*", "regular", "medium", "demibold", "bold"};
const char*	slantMap[] = {"*", "r", "i", "o"};
const char*	spacingMap[] = {"*", "m", "c", "p"};

static          UCache<UFontX11Cache*>* cache=0;
static          UCache<UFontIndexCache*>* fontIndexCache=0;

UFontCache::UFontCache(void)
{
}
UFontCache::~UFontCache()
{
}
int
UFontCache::getDirection()
{
	return 0;
}

int
UFontCache::getWidth (const XChar2b& char2B)
{
	return 0;
}

void 
UFontCache::drawImageString (Display* dsp, Window window, GC gc,
	int x, int y, int ascent, const XChar2b* char2B, int count, UPixmapCache* pmc)
{
}

void 
UFontCache::drawString (Display* dsp, Window window, GC gc,
	int x, int y, int ascent, const XChar2b* char2B, int count, UPixmapCache* pmc)
{
}

//
// Load an X11 font
//
UFontX11Cache::UFontX11Cache (const char *name, Display* displayIn, 
	const UFontMapStruct* fontMapIn, const int pixelIn, 
	const UFont::UWeight weightIn, const UFont::USlant slantIn,
	const UFont::USpacing spacingIn,
	const char* addStyle, const char* avgWidth)
{
	font = 0;
	cacheName = new char [strlen (name) +1];
	CHECKNULL (cacheName);
	strcpy (cacheName, name);

	// try with this font 
	fontName = getFontName (fontMapIn, pixelIn, weightIn, slantIn, spacingIn, addStyle, avgWidth);
	display = displayIn;
	
	// You have all the info to load the font here
	font = XLoadQueryFont (display, fontName);
	if (font==0)
	{
		cerr << "warn: could not load font '" << fontName << "'.\n";
	}

	return;
}

int
UFontX11Cache::getDirection ()
{
	if (font == 0) return 1;
	return font->direction;
}

int
UFontX11Cache::getWidth (const XChar2b& char2B)
{
	XCharStruct		overall;
	int			cellIndex;
	int			overallWidth;
	int			descent;
	int			ddescent;
	int			ddirection;

	if (font == 0) return 0;
	if (font->direction!=0) return 0;
	if (	   char2B.byte1 < font->min_byte1
		|| char2B.byte1 > font->max_byte1
		|| char2B.byte2 < font->min_char_or_byte2
		|| char2B.byte2 > font->max_char_or_byte2)
	{
		return 0;
	}

	if (font->per_char==0)
	{
		if (!XQueryTextExtents16 (display,
			font->fid, &char2B, 1, &ddirection,
			&descent, &ddescent, &overall))
		{
			return 0;
		}
		return overall.width;
	}
	cellIndex = (char2B.byte1-font->min_byte1)
		* (font->max_char_or_byte2 -font->min_char_or_byte2+1)
		+ char2B.byte2-font->min_char_or_byte2;
	overallWidth = font->per_char[cellIndex].width;
	//if (overallWidth==0) overallWidth = font->max_bounds.width;
	return overallWidth;
	
}

void
UFontX11Cache::drawImageString (Display* dsp, Window window, GC gc,
	int x, int y, int ascent, const XChar2b* char2B, int count, UPixmapCache* pmc)
{
	XSetFont (display, gc, font->fid);
	XDrawImageString16(display, window, gc,
			x, y+ascent, char2B, count);
}

void
UFontX11Cache::drawString (Display* dsp, Window window, GC gc,
	int x, int y, int ascent, const XChar2b* char2B, int count, UPixmapCache* pmc)
{
	XSetFont (display, gc, font->fid);
	XDrawString16(display, window, gc,
			x, y+ascent, char2B, count);
}

UFontX11Cache::~UFontX11Cache ()
{
	if (font) XFreeFont (display, font);
	delete cacheName;
	delete fontName;
}

char *
UFontX11Cache::getFontName(
	const UFontMapStruct* fontMapStruct, const int pixelIn, 
	const UFont::UWeight weightIn, const UFont::USlant slantIn,
	const UFont::USpacing spacingIn,
	const char* addStyle, const char* avgWidth)
{
	char*		name;
	/* The fontname is this */
	name = new char[strlen (fontMapStruct->foundry) 
		+ strlen (fontMapStruct->family) 
		+ strlen (fontMapStruct->registry)
		+ strlen (fontMapStruct->encoding) + 256];
	CHECKNULL (name);
	sprintf (name, "-%s-%s-%s-%s-*-%s-%d-*-*-*-%s-%s-%s-%s",
		fontMapStruct->foundry, 
		fontMapStruct->family,
		weightMap[weightIn],
		slantMap[slantIn],
		addStyle,
		pixelIn,
		spacingMap[spacingIn],
		avgWidth,
		fontMapStruct->registry,
		fontMapStruct->encoding);
	return name;
}

UFont::UFont (UFontMap* fontMapIn)
{
	fontAscent = fontDescent = fontWidth = fontHeight = 0;
	minFontWidth=0;
	fontMap = fontMapIn;
	pixel = 12;
	weight = ANYWEIGHT;
	slant = ANYSLANT;
	spacing = ANYSPACING;
	freeType = 0;
	addStyle = "*";
	avgWidth = "*";
}

UFont::UFont (UFreeType* freeTypeIn)
{
	fontAscent = fontDescent = fontWidth = fontHeight = 0;
	minFontWidth=0;
	fontMap = 0;
	pixel = 12;
	weight = ANYWEIGHT;
	slant = ANYSLANT;
	spacing = ANYSPACING;
	freeType = freeTypeIn;
	addStyle = "*";
	avgWidth = "*";
}

UFontCache *
UFont::getFont (const UCS2* text, const int from, int* to)
{
	return 0;
}

UFont::~UFont()
{
}

void
UFont::setPixel (const int pixelIn)
{
	pixel = pixelIn;
}

void
UFont::setWeight (const UWeight weightIn)
{
	weight = weightIn;
}

void
UFont::setSlant (const USlant slantIn)
{
	slant = slantIn;
}


void
UFont::setSpacing (const USpacing spacingIn)
{
	spacing = spacingIn;
}

void
UFont::setAddStyle (const char* addStyleIn)
{
	if (addStyleIn==0 || strcasecmp (addStyleIn, "Any") == 0)
	{
		addStyle = "*";
	}
	else
	{
		addStyle = addStyleIn;
	}
}

void
UFont::setAvgWidth (const char* avgWidthIn)
{
	if (avgWidthIn==0 || strcasecmp (avgWidthIn, "Any") == 0) 
	{
		avgWidth = "*";
	}
	else
	{
		avgWidth = avgWidthIn;
	}
}

void
UFont::setWeight (const char* weightIn)
{
	if (weightIn==0 
		|| strcasecmp (weightIn, "Any") == 0 
		|| strcasecmp (weightIn, "Any Weight") == 0 
		|| strcmp (weightIn, "*") == 0 )
	{
		weight = ANYWEIGHT;
		return;
	}
	if (strcasecmp (weightIn, "Regular") ==0)
	{
		weight = REGULAR;
		return;
	}
	if (strcasecmp (weightIn, "Medium") ==0)
	{
		weight = MEDIUM;
		return;
	}
	if (strcasecmp (weightIn, "Demibold") ==0
		|| strcasecmp (weightIn, "Demi bold")==0)
	{
		weight = DEMIBOLD;
		return;
	}
	if (strcasecmp (weightIn, "Bold") ==0)
	{
		weight = BOLD;
		return;
	}
	cerr << "warn: weight is unknown '" << weightIn << "'\n";
	weight = ANYWEIGHT;
}

void
UFont::setSlant (const char* slantIn)
{
	if (slantIn==0 
		|| strcasecmp (slantIn, "Any") == 0 
		|| strcasecmp (slantIn, "Any Slant") == 0 
		|| strcmp (slantIn, "*") == 0 )
	{
		slant = ANYSLANT;
		return;
	}
	if (strcasecmp (slantIn, "Roman") ==0)
	{
		slant = ROMAN;
		return;
	}
	if (strcasecmp (slantIn, "Italic") ==0)
	{
		slant = ITALIC;
		return;
	}
	if (strcasecmp (slantIn, "Oblique") ==0)
	{
		slant = OBLIQUE;
		return;
	}
	cerr << "warn: slant is unknown '" << slantIn << "'\n";
	slant = ANYSLANT;
}

void
UFont::setSpacing (const char* spacingIn)
{
	if (spacingIn==0 
		|| strcasecmp (spacingIn, "Any") == 0 
		|| strcasecmp (spacingIn, "Any Spacing") == 0 
		|| strcmp (spacingIn, "*") == 0 )
	{
		spacing = ANYSPACING;
		return;
	}
	if (strcasecmp (spacingIn, "Monospace") ==0)
	{
		spacing = MONOSPACE;
		return;
	}
	if (strcasecmp (spacingIn, "Condensed") ==0)
	{
		spacing = CONDENSED;
		return;
	}
	if (strcasecmp (spacingIn, "CharCell") ==0)
	{
		spacing = CONDENSED;
		return;
	}
	if (strcasecmp (spacingIn, "Proportional") ==0)
	{
		spacing = PROPORTIONAL;
		return;
	}
	cerr << "warn: spacing is unknown '" << spacingIn << "'\n";
	spacing = ANYSPACING;
}

int
UFont::isA (UFontType ftype)
{
	if (ftype==UFONT) return 1;
	return 0;
}

UFontIndexCache::UFontIndexCache (const char* name_)
{
	name = new char [strlen (name_) + 1];
	CHECKNULL (name);
	strcpy (name, name_);
	memset (index, 0, 256 * sizeof (unsigned char*));
}

UFontIndexCache::~UFontIndexCache ()
{
	if (name) delete name;
}

int
UFontX11::isA (UFontType ftype)
{
	if (ftype==UX11) return 1;
	return (UFont::isA (ftype));
}


UFontX11::UFontX11 (UFontX11 &fontCpy) : UFont (fontCpy.fontMap)
{
	fontChanged = 1;
	display = fontCpy.display;
	screen = fontCpy.screen;
	spacing = fontCpy.spacing;
	slant = fontCpy.slant;
	pixel = fontCpy.pixel;
	weight = fontCpy.weight;
	addStyle = (const char*) fontCpy.addStyle;
	avgWidth = (const char*) fontCpy.avgWidth;
	size = 0;
	fonts = 0;
	fontIndex = 0;
}

UFontX11::UFontX11(UFontMap* fontMapIn, Display *displayIn, int screenIn) : UFont (fontMapIn)
{
	fontChanged = 1;
	display = displayIn;
	screen = screenIn;
	spacing = ANYSPACING;
	slant = ANYSLANT;
	weight = ANYWEIGHT;
	size = 0;
	fonts = 0;
	fontIndex = 0;
}

UFontX11::~UFontX11()
{
	int	i;

	for (i=0; i<size; i++)
	{
		if (fonts[i] == 0) continue;
		cache->unuseItem (fonts[i]->cacheName);
	}
	if (fonts!=0) delete  fonts;
	if (fontIndex != 0)
	{
		fontIndexCache->unuseItem (fontIndex->name);
	}
	
	fonts = 0;
}

void
UFontX11::setPixel (const int pixelIn)
{
	if (pixel != pixelIn) clear ();
	pixel = pixelIn;
}
void
UFontX11::setWeight (const UWeight weightIn)
{
	if (weight != weightIn) clear ();
	weight = weightIn;
}
void
UFontX11::setSlant (const USlant slantIn)
{
	if (slant != slantIn) clear();
	slant = slantIn;
}
void
UFontX11::setSpacing (const USpacing spacingIn)
{
	if (spacing != spacingIn) clear();
	spacing = spacingIn;
}

void
UFontX11::setAddStyle (const char* addStyleIn)
{
	clear ();
	UFont::setAddStyle (addStyleIn);
}
void
UFontX11::setAvgWidth (const char* avgWidthIn)
{
	clear();
	UFont::setAvgWidth (avgWidthIn);
}

void
UFontX11::setWeight (const char* weightIn)
{
	clear ();
	UFont::setWeight (weightIn);
}
void
UFontX11::setSlant (const char* slantIn)
{
	clear();
	UFont::setSlant (slantIn);
}
void
UFontX11::setSpacing (const char* spacingIn)
{
	clear();
	UFont::setSpacing (spacingIn);
}

void
UFontX11::clear ()
{
	int	i;
	if (fontChanged == 1) return;
	// clear font buffers, decrement cache count
	for (i=0; i<size; i++)
	{
		if (fonts[i] == 0) continue;
		cache->unuseItem (fonts[i]->cacheName);
	}
	if (fonts!=0) delete  fonts;
	fonts = 0;
	size = 0;
	fontChanged = 1;
	if (fontIndex != 0)
	{
		fontIndexCache->unuseItem (fontIndex->name);
		fontIndex = 0;
	}
}

const XFontStruct*
UFontX11::getFontAt (const int sizeIn)
{
	if (fontChanged) getFont();
	if (sizeIn < 0 && sizeIn >= size) return 0;
	return fonts[sizeIn]->font;
}

UFontCache *
UFontX11::getFont (const UCS2* text, const int from, int* to)
{
	int 				i;
	int				index;
	int				tryAscent[2];
	int				tryDescent[2];
	char*				cacheName;

	if (text != 0 && text[from]<0x20)
	{
		if (to!=0) *to=from+1;
		return 0;
	}

	if (cache == 0)
	{
		cache = new UCache<UFontX11Cache*> (FONTCACHE_SIZE);
		CHECKNULL (cache);
	}
	if (fontIndexCache==0)
	{
		// one charges 1k so it is 8 k
		fontIndexCache = new UCache<UFontIndexCache*> (8);
		CHECKNULL (cache);
	}
	if (fontChanged)
	{
		if (fontMap==0)
		{
			if (to!=0) *to=from+1;
			return 0;
		}
		fontMap->init();
		size = fontMap->size;
		fonts = new UFontX11Cache* [size];
		CHECKNULL (fonts);

		fontAscent = fontDescent = fontWidth = fontHeight = 0;
		minFontWidth = 0;
		// Get the font Index cache
		if (fontIndex !=0)
		{
			fontIndexCache->unuseItem (fontIndex->name);
		} 
		cacheName = getFontIndexCacheName ();
		fontIndex = fontIndexCache->getItem (cacheName);
		if (fontIndex ==0)
		{
			fontIndex = new UFontIndexCache (cacheName);
			CHECKNULL (fontIndex);
			(void) fontIndexCache->addItem (cacheName, fontIndex);
		}
		else
		{
			fontIndexCache->useItem (cacheName);
		}
		delete cacheName;
		// fill in the fonts
		for (i=0; i<size; i++)
		{
			cacheName = getCacheName (fontMap->fonts[i],
				(const char*) addStyle, 
				(const char*) avgWidth);

			fonts[i] = cache->getItem (cacheName);
			if (fonts[i] != 0) 
			{
				cache->useItem (cacheName);
			}
			else
			{
				fonts[i] = new UFontX11Cache (cacheName, 
					display, 
					fontMap->fonts[i], pixel, 
					weight, slant, spacing, 
					(const char*) addStyle, 
					(const char*) avgWidth);

				(void) cache->addItem (cacheName, fonts[i]);
				CHECKNULL (fonts[i]);
			}
			if (fonts[i]->font != 0)
			{
				//
				// Deal with broken fonts.
				//
				tryAscent[0] = fonts[i]->font->ascent;
				tryAscent[1] = fonts[i]->font->max_bounds.ascent;
				if (tryAscent[1] < tryAscent[0] 
				 || (tryAscent[0] +2) * 2 < tryAscent[1])
				{
					cerr << "Fixing max_bounds.ascent for '" << cacheName << "' from " << tryAscent[1] << " to " << tryAscent[0] << ".\n";
					tryAscent[1] = tryAscent[0];
				}

				tryDescent[0] = fonts[i]->font->descent;
				tryDescent[1] = fonts[i]->font->max_bounds.descent;
				if (tryDescent[1] < tryDescent[0] 
				   || (tryDescent[0] +2) * 2 < tryDescent[1])
				{
					cerr << "Fixing max_bounds.descent for '" << cacheName << "' from " << tryDescent[1] << " to " << tryDescent[0] << ".\n";
					tryDescent[1] = tryDescent[0];
				}
				
				if (tryDescent[1] > fontDescent)
				{
					fontDescent = tryDescent[1];
				}
				if (tryAscent[1] > fontAscent)
				{
					fontAscent = tryAscent[1];
				}
				if (fonts[i]->font->max_bounds.width > fontWidth)
				{
					fontWidth = fonts[i]->font->max_bounds.width;
				}
				if (minFontWidth == 0 || fonts[i]->font->max_bounds.width < minFontWidth)
				{
					minFontWidth =  fonts[i]->font->max_bounds.width;
				}
			}
			delete cacheName;
			continue;
		}
		fontHeight = fontAscent + fontDescent;
		fontChanged = 0;
	}
	// the guy just wanted to set the font bounding box.
	if (text == 0) { return 0; }
	i=from;

	// get things with the same font.
	index = getFontIndex (text[i]);
	
	i++;
	if (to!=0) *to=i;
	if (index == -1) return 0;
	if (fonts[index]->font != 0) while (text[i] != 0)
	{
		if (getFontIndex (text[i])!= index) break;
		i++;
	}
	if (to!=0) *to=i;
	return fonts[index];
}

//
// return the index in the font structure or -1
// The whole thing should be cached. (Later)
int
UFontX11::getFontIndex (const UCS2 in)
{
	unsigned int upper;
	unsigned int lower;
	int		i;
	UCS2		encoded;
	XChar2b		char2B;

	upper = in >> 8;
	lower = in & 0xff;
	// See if the array is initialized
	if (fontIndex->index[upper] == 0)
	{
		fontIndex->index[upper] = new unsigned char[256];
		CHECKNULL (fontIndex->index[upper]);
		memset (fontIndex->index[upper], 0, 256 * sizeof (char ));
	}
	// map available font
	if (fontIndex->index[upper][lower] ==0)
	{
		fontIndex->index[upper][lower] = 1;
		for (i=0; i<size; i++)
		{
			if (fonts[i] != 0  
				&& fontMap->fonts[i]->encoder != 0)
			{
				encoded=fontMap->fonts[i]->encoder->encode (in);
				char2B.byte1 = encoded >> 8;
				char2B.byte2 = encoded & 0xff;
				// second level of check.
				if (encoded != 0 && fonts[i]->font != 0 
				 && fonts[i]->getWidth (char2B) > 0)
				{
					fontIndex->index[upper][lower] = i + 2;
				}
			}
		}
	}
	return (fontIndex->index[upper][lower] - 2);
}

const UFontMapStruct*
UFont::getFontMapStruct (const UCS2 input)
{
	return 0;
}

const UFontMapStruct*
UFontX11::getFontMapStruct (const UCS2 input)
{
	int	index;

	if (fontChanged) getFont();
	return (((index=getFontIndex (input)) == -1) 
		? 0 : fontMap->fonts[index]);
}

//
// Get a unique name for a given fontmap and fonts
//
char *
UFontX11::getFontIndexCacheName ()
{
	char*	name;
	/* The fontname is this */
	name = new char[strlen (fontMap->getName()) + 256];
	CHECKNULL (name);
	sprintf (name, "%s %d %d %d %d ", fontMap->getName(), 
		(int) pixel, (int) slant, (int) weight, (int) spacing); 
	return name;
}

char *
UFontX11::getCacheName (const UFontMapStruct*  fontMapStruct,
	const char* _addStyle, const char* _avgWidth)
{
	char*	name;
	/* The fontname is this */
	name = new char[
		strlen (fontMapStruct->mapName)
		+ strlen (fontMapStruct->foundry) 
		+ strlen (fontMapStruct->family) 
		+ strlen (fontMapStruct->registry)
		+ strlen (fontMapStruct->encoding)
		+ strlen (_addStyle)
		+ strlen (_avgWidth)
		+ 256];
	CHECKNULL (name);
	sprintf (name, "%d %d %d %d ", weight, slant, pixel, spacing); 
	strcat (name, fontMapStruct->mapName);
	strcat (name, " ");
	strcat (name, fontMapStruct->foundry);
	strcat (name, " ");
	strcat (name, fontMapStruct->family);
	strcat (name, " ");
	strcat (name, fontMapStruct->registry);
	strcat (name, " ");
	strcat (name, fontMapStruct->encoding);
	strcat (name, " ");
	strcat (name, _addStyle);
	strcat (name, " ");
	strcat (name, _avgWidth);
	return name;
}

void
UDeinitFontX11 ()
{
	if (cache != 0)
	{
		delete cache;
		cache = 0;
	}
}

void
UTuneFontX11 (int cacheSize, int batchSize)
{
	if (cache)
	{
		cache->tune (cacheSize, batchSize);
	}
}

#include "UFontFreeType.h"

UFont*
UNewFont(Display* disp, int screen, const char* name)
{
	UFontMap*	fMap;
	UFreeType*	fType;
	UFont*		font;

	if (disp==0) return 0;

	fType = 0;
	fMap = UGetFontMap (name);

	if (fMap == 0)
	{
		fType = UGetFreeType (name);
		if (fType==0) return 0;
	}

	if (fMap!=0)
	{
		font = new UFontX11 (fMap, disp, screen);
	}
	else
	{
		font = new UFontFreeType (fType, disp, screen);
	}
	return font;
}

void
UDeleteFont (UFont* font)
{
	delete font;
}
