;;; gEDA - GNU Electronic Design Automation
;;; gnetlist - GNU Netlist
;;; Copyright (C) 1998 Ales V. Hvezda
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;
;; Verilog netlist backend written by Mike Jarabek starts here
;;
;; --------------------------------------------------------------------------

;; return a list of nets whose pins have the desired attribute name/value
;; pair
(define verilog:get-matching-nets
  (lambda (attribute value)
    (map car (verilog:filter attribute value packages))))

;; This function takes an attribute name, desired value, and a list of
;; packages.  For each of the packages, it looks up that attribute, and
;; if it matches, that package name is added to the list, and the function
;; recurses on the remaining packages.  If the attribute does not match, 
;; the function just recuses on the remaing packages. Thanks to Mohina Lal
;; for this trick.
;;

(define verilog:filter 
  (lambda (attribute value package-list)
    (cond ((null? package-list) '())
	  ((string=? (gnetlist:get-package-attribute (car package-list) 
						      attribute) value)
	   (cons 
	    (map (lambda (pin)
		   (car (gnetlist:get-nets (car package-list) pin)))
		 (pins (car package-list)))
	    (verilog:filter attribute value (cdr package-list))))
	  (else (verilog:filter attribute value (cdr package-list)))))
)


;;
;; Output the guts of the module ports here
;;
;; Scan through the list of components, and pins from each one, finding the
;; pins that have PINTYPE == CHIPIN, CHIPOUT, CHIPTRI (for inout)
;; build three lists one each for the inputs, outputs and inouts
;; return the a list of three lists that contain the pins in the order 
;; we want.
(define verilog:get-port-list
  (lambda ()
    ;; construct list
    (list (verilog:get-matching-nets "device" "IPAD")
	  (verilog:get-matching-nets "device" "OPAD")
	  (verilog:get-matching-nets "device" "IOPAD"))))

;;
;; output the meat of the module port section
;;
;; each line in the declaration is formatted like this:
;;
;;       PORTNAME , <newline>
;;
(define verilog:write-module-declaration
  (lambda (module-name port-list p)
    (begin
      (display "module " p)
      (display module-name p)
      (display " (" p)
      (newline p)
      (let ((the-pins ( append (car port-list)     ; build up list of pins
			       (cadr  port-list)
			       (caddr port-list))))
	(begin
	  ;(display the-pins)
	  ; do pins, but take care of last comma
	  (if (not (null? the-pins))
	      (begin
		(display "       " p) 
		(display (car the-pins) p)
		(if (not (null? (cdr the-pins)))
		    (for-each (lambda (pin)   ; loop over outputs
				(begin
				  (display " ," p)
				  (newline p)
				  (display "       " p)
				  (display pin p)))
			      (cdr the-pins) ))))
	(newline p)
	(display "      );" p)
	(newline p))))))
;;
;; output the module direction section
;;
(define verilog:write-port-directions
  (lambda (port-list p)
    (let ((in    (car   port-list))    ; extract list of pins 
	  (out   (cadr  port-list))
	  (inout (caddr port-list)))
      (begin
	(display "/* Port directions begin here */" p)
	(newline p)
	(for-each (lambda (pin)
		    (begin
		      (display "input " p)
		      (display pin p)
		      (display " ;" p)
		      (newline p))) in)       ; do each input

	(for-each (lambda (pin)
		    (begin
		      (display "output " p)
		      (display pin p)
		      (display " ;" p)
		      (newline p))) out)      ; do each output

	(for-each (lambda (pin)
		    (begin
		      (display "inout " p)
		      (display pin p)
		      (display " ;" p)
		      (newline p))) inout)    ; do each inout
		      
	(newline p)))))
;;
;; Top level header
;;

(define verilog:write-top-header
	(lambda (p)
	  (let ((port-list (verilog:get-port-list)))
	    (begin
	      (display "/* structural Verilog generated by gnetlist */" p)
	      (newline p)
	      (verilog:write-module-declaration "top" port-list p)
	      (newline p)
	      (verilog:write-port-directions port-list p)
	      (newline p)))))

;;
;; Footer for file
;;
(define verilog:write-bottom-footer
  (lambda (p)
    (display "endmodule" p)
    (newline p)
    )
)

;;
;;  Display wires from the design
;;
(define verilog:write-wires
  (lambda (p)
    (display "/* Wires from the design */" p)
    (newline p)
    (for-each (lambda (wire)          ; print a wire statement for each
		(display "wire " p)   ; net in the design
		(display wire p)
		(display " ;" p)
		(newline p)) all-unique-nets)
    (newline p)))

;;  Output any continuous assignment statements generated
;; by placing `high' and `low' components on the board 
(define verilog:write-continuous-assigns
  (lambda (p)
    (display "/* continuous assignments */" p) (newline p)
    (for-each (lambda (wire)             ; do high values
		(begin
		  (display "assign " p) 
		  (display wire p) (display " = 1'b1;" p)
		  (newline p)))
	      (verilog:get-matching-nets "device" "HIGH"))

    (for-each (lambda (wire)
		(begin
		  (display "assign " p) 
		  (display wire p) (display " = 1'b0;" p)
		  (newline p)))
	      (verilog:get-matching-nets "device" "LOW"))
    (newline p))
)



;;
;; Top level component writing 
;;
;; Output a compoment instatantiation for each of the
;; components on the board
;; 
;; use the format:
;;
;;  device-attribute refdes (
;;        .pinname ( net_name ),
;;        ...
;;    );
;;
(define verilog:components
  (lambda (packages port)
    (begin
      (display "/* Package instantiations */" port)
      (newline port)
      (for-each (lambda (package)         ; loop on packages
		  (begin
		    (let ((device (get-device package)))
 		      (if (not (memv (string->symbol device) ; ignore specials
 				     (map string->symbol
 					  (list "IOPAD" "IPAD" "OPAD"
 						"HIGH" "LOW"))))
 			  (begin
 			    (display (get-device package) port)
 			    (display " " port)
 			    (display package port)
 			    (display " ( " port)
			    ; if this module wants positional pins, 
			    ; then output that format, otherwise
			    ; output normal named declaration
			    (verilog:display-connections 
			     package 
			     (string=? (gnetlist:get-package-attribute
					package "VERILOG_PORTS" )
				       "POSITIONAL")
			     port)
 			    (display "    );" port)
	 		    (newline port)
 			    (newline port))))))
 		packages)))
)

;; output a module connection for the package given to us with named ports
;;
(define verilog:display-connections
   (lambda (package positional port)
     (begin
       (let ((pin-list (gnetlist:get-pins-nets package)))
 	(if (not (null? pin-list))
 	    (begin
	      (newline port)
 	      (verilog:display-pin (car pin-list) positional port)
 	      (for-each (lambda (pin)
 			  (display "," port)
 			  (newline port)
 			  (verilog:display-pin pin positional port))
 			(cdr pin-list))
 	      (newline port))))))
)


;;
;; Display the individual net connections
;;  in this format if positional is true:
;;
;;    /* PINNAME */ NETNAME
;;
;;  otherwise emit:
;; 
;;      .PINNAME ( NETNAME )
;;
(define verilog:display-pin
    (lambda (pin positional port)
      (begin
	(if positional
	    (begin    ; output a positional port instanace
	      (display "  /* " port)
	      (display (car pin) port)  ; add in name for debugging
	      (display " */ " port )
	      (verilog:display-conditional-pin pin port))
	    (begin    ; else output a named port instance 
	      (display "    ." port)
	      (display (car pin) port) ; name of pin 
	      (display " ( " port)
	      (verilog:display-conditional-pin pin port)
	      (display " )" port)))))
)
	 
;; display a pin only if it is not "unconnected_pin"
(define verilog:display-conditional-pin
  (lambda (pin port)
    (begin
      (if (not (string=? "unconnected_pin" (cdr pin)))
	  (display (cdr pin) port))))
)



;;; Highest level function
;;; Write Structural verilog representation of the schematic
;;;
(define verilog 
  (lambda (output-filename)
    (let ((port (open-output-file output-filename)))
      (begin

;;      Following is no longer needed
;;	(gnetlist:set-netlist-mode "SPICE")  ;; don't want 'duplicate' nets
	(verilog:write-top-header port)
	(verilog:write-wires port)
	(verilog:write-continuous-assigns port)
	(verilog:components packages port)
	(verilog:write-bottom-footer port)
	)
      (close-output-port port)
      )
    )
  ) 

;;
;; Verilog netlist backend written by Mike Jarabek ends here
;;
;; --------------------------------------------------------------------------

