/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994,1995,1996 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@rz.uni-ulm.de
 *
 */

static	char	*rcsid = "$Id: draw.c,v 1.6.1.1 1998/03/01 22:58:33 cad Exp $";

/* drawing routines
 */

/* ---------------------------------------------------------------------------
 * define TO_SCREEN before macro.h is included from global.h
 */
#define	SWAP_IDENT		SwapOutput
#define TO_SCREEN(a)	((ZoomValue < 0) ? (a) << -ZoomValue : (a) >> ZoomValue)
#define XORIG dxo
#define YORIG dyo

#include "global.h"

#include "crosshair.h"
#include "data.h"
#include "draw.h"
#include "error.h"
#include "mymem.h"
#include "misc.h"
#include "rotate.h"
#include "search.h"
#include "select.h"

/* ---------------------------------------------------------------------------
 * some local types
 */
typedef struct
{
	float	X,
		Y;
} FloatPolyType, *FloatPolyTypePtr;

/* ---------------------------------------------------------------------------
 * some local identifiers
 */
static	int		ZoomValue;	/* zoom, drawable and mirror */
static	Window		DrawingWindow;	/* flag common to all */
static	Boolean		SwapOutput;	/* all drawing routines */
static	XPoint		Outline[MAX_SIZE+1][8];
static  XRectangle	UpdateRect;
static  BoxType		Block;
static	Boolean		Gathering = True;
static  int		Erasing = False;
static	Position	dxo, dyo;


/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	void	Redraw(Boolean);
static	void	DrawEverything(void);
static	void	DrawTop(void);
static	void	DrawLayer(LayerTypePtr, int);
static	void	InitSpecialPolygon(void);
static	void	DrawSpecialPolygon(Drawable, GC, Position, Position, XPoint *);
static	void	DrawPinOrViaLowLevel(PinTypePtr, Boolean);
static	void	ClearOnlyPin(PinTypePtr);
static	void	ThermPin(LayerTypePtr, PinTypePtr);
static	void	DrawPlainPin(PinTypePtr, int);
static	void	DrawPlainVia(PinTypePtr, int);
static	void	DrawPlainElementPinsAndPads(ElementTypePtr, int);
static	void	DrawPinOrViaNameLowLevel(PinTypePtr);
static	void	DrawPadLowLevel(PadTypePtr);
static	void	DrawPadNameLowLevel(PadTypePtr);
static	void	DrawLineLowLevel(LineTypePtr, Boolean);
static	void	DrawTextLowLevel(TextTypePtr);
static	void	DrawRegularText(LayerTypePtr, TextTypePtr, int);
static	void	DrawPolygonLowLevel(PolygonTypePtr, Boolean);
static	void	DrawArcLowLevel(ArcTypePtr);
static	void	DrawElementPackageLowLevel(ElementTypePtr Element, int);
static	void	DrawPlainPolygon(LayerTypePtr Layer, PolygonTypePtr Polygon);
static	void	AddPart(void);
static	void	SetPVColor(PinTypePtr, int);
static	void	DrawGrid(void);
static	void	DrawEMark(Position, Position, Boolean); 
static	void	ClearLine(LineTypePtr);
static	void	ClearArc(ArcTypePtr);
static	void	ClearPad(PadTypePtr);
static	void	DrawHole(PinTypePtr);

/*--------------------------------------------------------------------------------------
 * setup color for pin or via
 */
static	void	SetPVColor(PinTypePtr Pin, int Type)
{
	if (Type == VIA_TYPE)
	{
		if (TEST_FLAG(WARNFLAG | SELECTEDFLAG | FOUNDFLAG, Pin))
			{
				if (TEST_FLAG(WARNFLAG, Pin))
					XSetForeground(Dpy, Output.fgGC, PCB->WarnColor);
				else if (TEST_FLAG(SELECTEDFLAG, Pin))
					XSetForeground(Dpy, Output.fgGC, PCB->ViaSelectedColor);
				else
					XSetForeground(Dpy, Output.fgGC, PCB->ConnectedColor);
			}
			else
				XSetForeground(Dpy, Output.fgGC, PCB->ViaColor);
	}
	else
	{
		if (TEST_FLAG(WARNFLAG | SELECTEDFLAG | FOUNDFLAG, Pin))
		{
			if (TEST_FLAG(WARNFLAG, Pin))
				XSetForeground(Dpy, Output.fgGC, PCB->WarnColor);
			else if (TEST_FLAG(SELECTEDFLAG, Pin))
				XSetForeground(Dpy, Output.fgGC, PCB->PinSelectedColor);
			else
				XSetForeground(Dpy, Output.fgGC, PCB->ConnectedColor);
		}
		else
			XSetForeground(Dpy, Output.fgGC, PCB->PinColor);
	}
}

/*---------------------------------------------------------------------------
 *  Adds the update rect to the update region
 */
static void AddPart()
{
	Block.X1 = MIN(Block.X1, UpdateRect.x);
	Block.X2 = MAX(Block.X2, UpdateRect.x + UpdateRect.width);
	Block.Y1 = MIN(Block.Y1, UpdateRect.y);
	Block.Y2 = MAX(Block.Y2, UpdateRect.y + UpdateRect.height);
}

/*
 * force the whole output to be updated
 */
void UpdateAll(void)
{
	Block.X1 = 1;
	Block.Y1 = 1;
	Block.Y2 = MAX_COORD -1;
	Block.X2 = MAX_COORD -1;
	Draw();
}

/*
 * initiate the actual drawing to the pixmap/screen
 * make the update block slightly larger to handle round-off
 * caused by the TO_SCREEN operation
 */
void Draw(void)
{
	static	XExposeEvent	erased;

	render = True;
	if (VALID_PIXMAP(Offscreen))
	{
		XFillRectangle(Dpy, Offscreen, Output.bgGC, Block.X1 -1,
			Block.Y1 -1, Block.X2 - Block.X1 + 2,
			Block.Y2 - Block.Y1 + 2);
			/* create an update event */
		erased.type = Expose;
		erased.display = Dpy;
		erased.window = Output.OutputWindow;
		erased.x = Block.X1 -1;
		erased.y = Block.Y1 -1;
		erased.width = Block.X2 - Block.X1 + 2;
		erased.height = Block.Y2 - Block.Y1 + 2;
		erased.count = 0;
		XSendEvent(Dpy, Output.OutputWindow, False, ExposureMask, (XEvent *) &erased);
	}
	else
	{
		HideCrosshair(True);
			/* clear and create event if not drawing to a pixmap */
		XClearArea(Dpy, Output.OutputWindow, Block.X1 -1, Block.Y1 -1,
			Block.X2 - Block.X1 + 2, Block.Y2 - Block.Y1 + 2, True);
		RestoreCrosshair(True);
	}
}

/* ---------------------------------------------------------------------------
 * redraws the output area without clearing it
 */
void RedrawOutput(void)
{
	Redraw(False);
}

/* ---------------------------------------------------------------------------
 * redraws the output area after clearing it
 */
void ClearAndRedrawOutput(void)
{
	render = True;
	Gathering = False;
/*
	Redraw(True);
*/
	UpdateAll();
}

/* ---------------------------------------------------------------------- 
 * redraws all the data
 * all necessary sizes are already set by the porthole widget and
 * by the event handlers
 */
static void Redraw(Boolean ClearWindow)
{
		/* make sure window exists */
	if (Output.OutputWindow && (render || ClearWindow || !VALID_PIXMAP(Offscreen)))
	{
			/* shrink the update block */
		Block.X1 = TO_SCREEN(PCB->MaxWidth);
		Block.Y1 = TO_SCREEN(PCB->MaxHeight);
		Block.X2 = Block.Y2 = 0;

			/* switch off crosshair, set up drawing window and redraw
			 * everything with Gather = False
			 */
		HideCrosshair(True);
		SwitchDrawingWindow(PCB->Zoom,
			VALID_PIXMAP(Offscreen) ? Offscreen : Output.OutputWindow,
			Settings.ShowSolderSide, False);

			/* clear the background
			 * of the drawing area
			 */
		XFillRectangle(Dpy, DrawingWindow, Output.bgGC, 0, 0,
			TO_DRAWABS_X(PCB->MaxWidth),
			TO_DRAWABS_Y(PCB->MaxHeight));
		XSetForeground(Dpy, Output.fgGC, Settings.OffLimitColor);
		XFillRectangle(Dpy, DrawingWindow, Output.fgGC,
			TO_DRAWABS_X(PCB->MaxWidth), 0, MAX_COORD, MAX_COORD);
		XFillRectangle(Dpy, DrawingWindow, Output.fgGC,
			0, TO_DRAWABS_Y(PCB->MaxHeight), MAX_COORD, MAX_COORD);
		if (ClearWindow)
			Crosshair.On = False;

		DrawEverything();

		RestoreCrosshair(True);
	}
	Gathering = True;
	render = False;
}

/* ----------------------------------------------------------------------
 * setup of zoom and output window for the next drawing operations
 */
Boolean	SwitchDrawingWindow(int Zoom, Window OutputWindow, Boolean Swap, Boolean Gather)
{
	Boolean	oldGather = Gathering;

	Gathering = Gather;
	ZoomValue = Zoom;
	DrawingWindow = OutputWindow;
	if (OutputWindow == Offscreen || OutputWindow == Output.OutputWindow)
	{
		dxo = Xorig;
		dyo = Yorig;
	}
	else
	{
		dxo = 0;
		dyo = 0;
	}
	SwapOutput = Swap;
	if (Zoom < 0)
		Zoom = 0;
	if (Zoom > 4)
		Zoom = 4;
	XSetFont(Dpy, Output.fgGC, Settings.PinoutFont[Zoom]->fid);
	XSetFont(Dpy, Output.bgGC, Settings.PinoutFont[Zoom]->fid);
	InitSpecialPolygon();
	return(oldGather);
}

/* ---------------------------------------------------------------------------
 * initializes some identifiers for a new zoom factor and redraws whole screen
 */
static void DrawEverything(void)
{
	int i;

		/*
		 * first draw all 'invisible' stuff
		 */
	if (PCB->InvisibleObjectsOn)
	{
		ELEMENT_LOOP(PCB->Data,
			if (!FRONT(element))
			{
				if (PCB->ElementOn)
				{
					if (VELEMENT(element))
						DrawElementPackage(element, 0);
					if (VELTEXT(element))
						DrawElementName(element, 0);
				}
				if (PCB->PinOn && VELEMENT(element))
					PAD_LOOP(element, DrawPad(pad, 0));
			}
		);
		if (PCB->ElementOn)
			DrawLayer(&PCB->Data->Layer[MAX_LAYER +
				(SWAP_IDENT ? COMPONENT_LAYER : SOLDER_LAYER)], 0);
	}
		/* draw all layers in layerstack order */
	for (i = MAX_LAYER-1; i >= 0; i--)
		if ((LAYER_ON_STACK(i))->On)
			DrawLayer(LAYER_ON_STACK(i), 0);
		/* Draw pins, pads, vias */
	DrawTop();
		/* Draw top silkscreen */
	if (PCB->ElementOn)
	{
		DrawLayer(&PCB->Data->Layer[MAX_LAYER +
			(SWAP_IDENT ? SOLDER_LAYER : COMPONENT_LAYER)], 0);
		ELEMENT_LOOP(PCB->Data,
			if (FRONT(element))
			{
				if (VELEMENT(element))
					DrawElementPackage(element, 0);
				if (VELTEXT(element))
					DrawElementName(element, 0);
			}
				/* Draw pin holes */
			if (PCB->PinOn && VELEMENT(element))
			{
				PIN_LOOP(element, DrawHole(pin));
				DrawEMark(element->MarkX, element->MarkY,
					!FRONT(element));
			}
		);
	}
	else if (PCB->PinOn)
			/* Draw pin holes */
		ELEMENT_LOOP(PCB->Data,
			if (VELEMENT(element))
				PIN_LOOP(element, DrawHole(pin));
		);
		/* Draw via holes */
	if (PCB->ViaOn)
		VIA_LOOP(PCB->Data, if (VVIA(via)) DrawHole(via));
		/* Draw rat lines on top */
	if (PCB->RatOn)
		RAT_LOOP(PCB->Data, if (VLINE(line)) DrawRat(line, 0));
	if (Settings.DrawGrid)
		DrawGrid();
}

static void DrawEMark(Position X, Position Y, Boolean invisible)
{
	if (!PCB->InvisibleObjectsOn && invisible)
		return;
	XSetForeground(Dpy, Output.fgGC, invisible ? PCB->InvisibleMarkColor : PCB->ElementColor);
	XSetLineAttributes(Dpy, Output.fgGC, 1, LineSolid, CapRound, JoinRound);
	XDrawLine(Dpy, DrawingWindow, Output.fgGC, TO_DRAW_X(X - EMARK_SIZE),
		TO_DRAW_Y(Y), TO_DRAW_X(X), TO_DRAW_Y(Y - EMARK_SIZE));
	XDrawLine(Dpy, DrawingWindow, Output.fgGC, TO_DRAW_X(X + EMARK_SIZE),
		TO_DRAW_Y(Y), TO_DRAW_X(X), TO_DRAW_Y(Y - EMARK_SIZE));
	XDrawLine(Dpy, DrawingWindow, Output.fgGC, TO_DRAW_X(X - EMARK_SIZE),
		TO_DRAW_Y(Y), TO_DRAW_X(X), TO_DRAW_Y(Y + EMARK_SIZE));
	XDrawLine(Dpy, DrawingWindow, Output.fgGC, TO_DRAW_X(X + EMARK_SIZE),
		TO_DRAW_Y(Y), TO_DRAW_X(X), TO_DRAW_Y(Y + EMARK_SIZE));
}

/* ---------------------------------------------------------------------------
 * draws pins pads and vias
 */
static void DrawTop(void)
{
		/* draw element pins */
	if (PCB->PinOn)
		ELEMENT_LOOP(PCB->Data,
			if (VELEMENT(element))
				 DrawPlainElementPinsAndPads(element, 0);
	);

		/* draw vias */
	if (PCB->ViaOn)
		VIA_LOOP(PCB->Data, if (VVIA(via)) DrawPlainVia(via, 0););
}

/* ---------------------------------------------------------------------------
 * draws one layer
 */
static void DrawLayer(LayerTypePtr Layer, int unused)
{
        int		layernum = GetLayerNumber(PCB->Data, Layer);
	Cardinal	group = GetLayerGroupNumberByNumber(layernum);
	Cardinal	entry;
        
        int PIPFlag = L0PIPFLAG << layernum;
/* in order to render polygons with line cut-outs:
 * draw a solid (or stippled) 1-bit pixmap, then erase
 * the clearance areas.  Use that as the mask when
 * drawing the actual polygons
 */
	if (layernum < MAX_LAYER && Layer->PolygonN)
	{
		XSetFunction(Dpy, Output.pmGC, GXcopy);
        	if (Settings.StipplePolygons)
        	{
			XSetBackground(Dpy, Output.pmGC, 0);
                	XSetStipple(Dpy, Output.pmGC, Stipples[layernum]);
                	XSetFillStyle(Dpy, Output.pmGC, FillOpaqueStippled);
        	}
			/* fill whole map first */
		XSetForeground(Dpy, Output.pmGC, AllPlanes);
		XFillRectangle(Dpy, Offmask, Output.pmGC, 0, 0 , Output.Width, Output.Height);
        	if (Settings.StipplePolygons)
                	XSetFillStyle(Dpy, Output.pmGC, FillSolid);
			/* make clearances around lines, arcs, pins and vias */
		XSetFunction(Dpy, Output.pmGC, GXclear);
		for (entry = 0; entry < PCB->LayerGroups.Number[group]; entry++)
		{
			Cardinal guest = PCB->LayerGroups.Entries[group][entry];

			if (guest < MAX_LAYER)
			{
				LayerTypePtr	guestLayer = &PCB->Data->Layer[guest];

				LINE_LOOP(guestLayer,
					if (TEST_FLAG(CLEARLINEFLAG, line) && VLINE(line))
					 	ClearLine(line);
				);
				ARC_LOOP(guestLayer,
					if (TEST_FLAG(CLEARLINEFLAG, arc) && VARC(arc))
						ClearArc(arc);
				);
			}
		}
        	ALLPIN_LOOP(PCB->Data,
                	if (TEST_FLAG(PIPFlag, pin))
                        	ClearOnlyPin(pin);
        	);
        	VIA_LOOP(PCB->Data,
                	if (TEST_FLAG(PIPFlag, via))
                        	ClearOnlyPin(via);
        	);
		if (group == GetLayerGroupNumberByNumber(MAX_LAYER + SOLDER_LAYER))
			ALLPAD_LOOP(PCB->Data,
				if (TEST_FLAG(ONSOLDERFLAG, pad))
					ClearPad(pad);
			);
		if (group == GetLayerGroupNumberByNumber(MAX_LAYER + COMPONENT_LAYER))
			ALLPAD_LOOP(PCB->Data,
				if (!TEST_FLAG(ONSOLDERFLAG, pad))
					ClearPad(pad);
			);
		XSetClipMask(Dpy, Output.fgGC, Offmask);
	}
	if (Layer->PolygonN)
	{
        	POLYGON_LOOP(Layer, if (VPOLY(polygon)) DrawPlainPolygon(Layer, polygon));
			/* restore the clip region */
		XCopyGC(Dpy, Output.bgGC, GCClipMask, Output.fgGC);
		if (layernum < MAX_LAYER)
		{
        		PIPFlag = L0THERMFLAG << layernum;
        		ALLPIN_LOOP(PCB->Data,
               			if (TEST_FLAG(PIPFlag, pin))
                       			ThermPin(Layer, pin);
        		);
        		VIA_LOOP(PCB->Data,
               			if (TEST_FLAG(PIPFlag, via))
                       			ThermPin(Layer, via);
        		);
		}
	} 
        LINE_LOOP(Layer, if (VLINE(line)) DrawLine(Layer, line, unused));
	ARC_LOOP(Layer, if (VARC(arc)) DrawArc(Layer, arc, unused));
        TEXT_LOOP(Layer, if (VTEXT(text)) DrawRegularText(Layer, text, unused));
} 

/* ---------------------------------------------------------------------------
 * initializes some zoom dependend information for pins and lines
 * just to speed up drawing a bit
 */
static void InitSpecialPolygon(void)
{
	int	i, j;
	static	FloatPolyType	p[8] = {{       0.5, -TAN_22_5_DEGREE_2},
					{ TAN_22_5_DEGREE_2,       -0.5},
					{-TAN_22_5_DEGREE_2,       -0.5},
					{      -0.5, -TAN_22_5_DEGREE_2},
					{      -0.5,  TAN_22_5_DEGREE_2},
					{-TAN_22_5_DEGREE_2,        0.5},
					{ TAN_22_5_DEGREE_2,        0.5},
					{       0.5,  TAN_22_5_DEGREE_2}};


		/* loop over maximum number of different sizes */
	for (i = MAX(MAX_PINORVIASIZE, MAX_LINESIZE); i != -1; i--)
		for (j = 0; j < 8; j++)
		{
			Outline[i][j].x = (p[j].X * TO_SCREEN(i));
			Outline[i][j].y = (p[j].Y * TO_SCREEN(i));
		}
}

/* ---------------------------------------------------------------------------
 * draws one polygon
 * x and y are already in display coordinates
 * the points are numbered:
 *
 *          5 --- 6
 *         /       \
 *        4         7
 *        |         |
 *        3         0
 *         \       /
 *          2 --- 1
 *
 */
static void DrawSpecialPolygon(Drawable d, GC DrawGC,
	Position X, Position Y, XPoint *PolyPtr)
{
	int		i;
	XPoint	polygon[8];

		/* add line offset */
	for (i = 0; i < 8; i++)
	{
		polygon[i].x = X+ PolyPtr[i].x;
		polygon[i].y = Y+ PolyPtr[i].y;
	}
	XFillPolygon(Dpy, d, DrawGC,
		polygon, ENTRIES(polygon), Convex, CoordModeOrigin);
}

/* ---------------------------------------------------------------------------
 * lowlevel drawing routine for pins and vias
 */
static void DrawPinOrViaLowLevel(PinTypePtr Ptr, Boolean drawHole)
{
	Dimension	half = Ptr->Thickness/2;

	UpdateRect.x = TO_DRAW_X(Ptr->X - half);
        UpdateRect.y = TO_DRAW_Y(Ptr->Y) -TO_SCREEN(half);
        UpdateRect.width = UpdateRect.height = TO_SCREEN(Ptr->Thickness);
	if (Gathering)
	{
		AddPart();
		return;
	}

	if (drawHole && TEST_FLAG(HOLEFLAG, Ptr))
	{
		XSetLineAttributes(Dpy, Output.bgGC, TO_SCREEN(Ptr->Thickness),
			LineSolid, CapRound, JoinRound);
		XDrawLine(Dpy, DrawingWindow, Output.bgGC, TO_DRAW_X(Ptr->X),
			TO_DRAW_Y(Ptr->Y), TO_DRAW_X(Ptr->X), TO_DRAW_Y(Ptr->Y));
		XSetLineAttributes(Dpy, Output.fgGC, 1, LineSolid, CapRound, JoinRound);
		XDrawArc(Dpy, DrawingWindow, Output.fgGC,
			TO_DRAW_X(Ptr->X - half),
			TO_DRAW_Y(Ptr->Y - TO_SCREEN_SIGN_Y(half)),
			TO_SCREEN(Ptr->Thickness), TO_SCREEN(Ptr->Thickness),
			0, 23040);
		return;
	}
	if (TEST_FLAG(SQUAREFLAG, Ptr))
	{
		XFillRectangle(Dpy, DrawingWindow, Output.fgGC,
			UpdateRect.x, UpdateRect.y,
			UpdateRect.width, UpdateRect.height);
	}
	else if (TEST_FLAG(OCTAGONFLAG, Ptr))
	{
		XSetLineAttributes(Dpy, Output.fgGC,
			TO_SCREEN((Ptr->Thickness -Ptr->DrillingHole) /2),
			LineSolid, CapRound, JoinRound);

			/* transform X11 specific coord system */
		DrawSpecialPolygon(DrawingWindow, Output.fgGC,
			TO_DRAW_X(Ptr->X), TO_DRAW_Y(Ptr->Y),
			&Outline[Ptr->Thickness][0]);
	}
	else
	{		/* draw a round pin or via */
		XSetLineAttributes(Dpy, Output.fgGC, TO_SCREEN(Ptr->Thickness),
			LineSolid, CapRound, JoinRound);
		XDrawLine(Dpy, DrawingWindow, Output.fgGC, TO_DRAW_X(Ptr->X),
			TO_DRAW_Y(Ptr->Y), TO_DRAW_X(Ptr->X), TO_DRAW_Y(Ptr->Y));
	}

		/* and the drilling hole  (which is always round */
	if (drawHole)
	{
		XSetLineAttributes(Dpy, Output.bgGC, TO_SCREEN(Ptr->DrillingHole),
			LineSolid, CapRound, JoinRound);
		XDrawLine(Dpy, DrawingWindow, Output.bgGC, TO_DRAW_X(Ptr->X),
			TO_DRAW_Y(Ptr->Y), TO_DRAW_X(Ptr->X), TO_DRAW_Y(Ptr->Y));
	}
}

/**************************************************************
 * draw pin/via hole
 */
static void DrawHole(PinTypePtr Ptr)
{
	XSetLineAttributes(Dpy, Output.bgGC, TO_SCREEN(Ptr->DrillingHole),
		LineSolid, CapRound, JoinRound);
	XDrawLine(Dpy, DrawingWindow, Output.bgGC, TO_DRAW_X(Ptr->X),
		TO_DRAW_Y(Ptr->Y), TO_DRAW_X(Ptr->X), TO_DRAW_Y(Ptr->Y));
	if (TEST_FLAG(HOLEFLAG, Ptr))
	{
		Dimension half = Ptr->Thickness/2;

		if (TEST_FLAG(SELECTEDFLAG, Ptr))
			XSetForeground(Dpy, Output.fgGC, PCB->ViaSelectedColor);
		else
			XSetForeground(Dpy, Output.fgGC,
				BlackPixelOfScreen(XtScreen(Output.Output)));
		XSetLineAttributes(Dpy, Output.fgGC, 1, LineSolid, CapRound, JoinRound);
		XDrawArc(Dpy, DrawingWindow, Output.fgGC,
			TO_DRAW_X(Ptr->X - half),
			TO_DRAW_Y(Ptr->Y - TO_SCREEN_SIGN_Y(half)),
			TO_SCREEN(Ptr->DrillingHole), TO_SCREEN(Ptr->DrillingHole),
			0, 23040);
	}
}

/*******************************************************************
 * draw clearance in pixmask arround pins and vias that pierce polygons
 */
static void ClearOnlyPin(PinTypePtr Pin)
{
	Dimension	half = (Pin->Thickness + Pin->Clearance)/2;

	if (TEST_FLAG(HOLEFLAG, Pin))
		return;

	   /* Clear the area around the pin */
	if (TEST_FLAG(SQUAREFLAG, Pin))
	{
		XFillRectangle(Dpy, Offmask, Output.pmGC,
			TO_MASK_X(Pin->X - TO_SCREEN_SIGN_X(half)),
			TO_MASK_Y(Pin->Y - TO_SCREEN_SIGN_Y(half)),
			TO_SCREEN(Pin->Thickness + Pin->Clearance),
			TO_SCREEN(Pin->Thickness + Pin->Clearance));
	}
	else if (TEST_FLAG(OCTAGONFLAG, Pin))
	{
		XSetLineAttributes(Dpy, Output.pmGC,
			TO_SCREEN((Pin->Thickness + Pin->Clearance -Pin->DrillingHole) /2),
			LineSolid, CapRound, JoinRound);

			/* transform X11 specific coord system */
		DrawSpecialPolygon(Offmask, Output.pmGC,
			TO_MASK_X(Pin->X), TO_MASK_Y(Pin->Y),
			&Outline[Pin->Thickness + Pin->Clearance][0]);
	}
	else
	{
		XSetLineAttributes(Dpy, Output.pmGC,
			TO_SCREEN(Pin->Thickness + Pin->Clearance),
			LineSolid, CapRound, JoinRound);
		XDrawLine(Dpy, Offmask, Output.pmGC, TO_MASK_X(Pin->X),
			TO_MASK_Y(Pin->Y), TO_MASK_X(Pin->X), TO_MASK_Y(Pin->Y));
	}
}

/**************************************************************************
 * draw thermals on pin
 */
static void ThermPin(LayerTypePtr layer, PinTypePtr Pin)
{
	Dimension	half = (Pin->Thickness + Pin->Clearance)/2;
	Dimension	halfs = (TO_SCREEN(half) * SQRT2OVER2 + 1);

	if (TEST_FLAG(SELECTEDFLAG | FOUNDFLAG, Pin))
	{
		if (TEST_FLAG(SELECTEDFLAG, Pin))
			XSetForeground(Dpy, Output.fgGC, layer->SelectedColor);
		else
			XSetForeground(Dpy, Output.fgGC, PCB->ConnectedColor);
	}
	else
		XSetForeground(Dpy, Output.fgGC, layer->Color);

	XSetLineAttributes(Dpy, Output.fgGC, TO_SCREEN(Pin->Clearance/2),
		LineSolid, CapRound, JoinRound);
	if (TEST_FLAG(SQUAREFLAG, Pin))
	{
		
		XDrawLine(Dpy, DrawingWindow, Output.fgGC,
			TO_DRAW_X(Pin->X) - TO_SCREEN(half),
			TO_DRAW_Y(Pin->Y) - TO_SCREEN(half),
			TO_DRAW_X(Pin->X) + TO_SCREEN(half),
			TO_DRAW_Y(Pin->Y) + TO_SCREEN(half));
		XDrawLine(Dpy, DrawingWindow, Output.fgGC,
			TO_DRAW_X(Pin->X) - TO_SCREEN(half),
			TO_DRAW_Y(Pin->Y) + TO_SCREEN(half),
			TO_DRAW_X(Pin->X) + TO_SCREEN(half),
			TO_DRAW_Y(Pin->Y) - TO_SCREEN(half));
	}
	else
	{
		XDrawLine(Dpy, DrawingWindow, Output.fgGC,
			TO_DRAW_X(Pin->X) - halfs,
			TO_DRAW_Y(Pin->Y) - halfs,
			TO_DRAW_X(Pin->X) + halfs,
			TO_DRAW_Y(Pin->Y) + halfs);
		XDrawLine(Dpy, DrawingWindow, Output.fgGC,
			TO_DRAW_X(Pin->X) - halfs,
			TO_DRAW_Y(Pin->Y) + halfs,
			TO_DRAW_X(Pin->X) + halfs,
			TO_DRAW_Y(Pin->Y) - halfs);
	}
}

/* ---------------------------------------------------------------------------
 * lowlevel drawing routine for pins and vias that pierce polygons
 */
void ClearPin(PinTypePtr Pin, int Type, int unused)
{
	int		ThermLayerFlag;
	LayerTypePtr	layer;
	Cardinal	i;
	Dimension	half = (Pin->Thickness + Pin->Clearance)/2;
	Dimension	halfs = (TO_SCREEN(half) * SQRT2OVER2 + 1);

	if (Gathering)
	{
		UpdateRect.x = TO_DRAW_X(Pin->X) - TO_SCREEN(half);
		UpdateRect.y = TO_DRAW_Y(Pin->Y) - TO_SCREEN(half);
		UpdateRect.width = UpdateRect.height = TO_SCREEN(2*half);
		AddPart();
		return;
	}
	   /* Clear the area around the pin */
	if (TEST_FLAG(SQUAREFLAG, Pin))
	{
		XFillRectangle(Dpy, DrawingWindow, Output.bgGC,
			TO_DRAW_X(Pin->X - TO_SCREEN_SIGN_X(half)),
			TO_DRAW_Y(Pin->Y - TO_SCREEN_SIGN_Y(half)),
			TO_SCREEN(Pin->Thickness + Pin->Clearance),
			TO_SCREEN(Pin->Thickness + Pin->Clearance));
	}
	else if (TEST_FLAG(OCTAGONFLAG, Pin))
	{
		XSetLineAttributes(Dpy, Output.bgGC,
			TO_SCREEN((Pin->Thickness + Pin->Clearance -Pin->DrillingHole) /2),
			LineSolid, CapRound, JoinRound);

			/* transform X11 specific coord system */
		DrawSpecialPolygon(DrawingWindow, Output.bgGC,
			TO_DRAW_X(Pin->X), TO_DRAW_Y(Pin->Y),
			&Outline[Pin->Thickness + Pin->Clearance][0]);
	}
	else
	{
		XSetLineAttributes(Dpy, Output.bgGC,
			TO_SCREEN(Pin->Thickness + Pin->Clearance),
			LineSolid, CapRound, JoinRound);
		XDrawLine(Dpy, DrawingWindow, Output.bgGC, TO_DRAW_X(Pin->X),
			TO_DRAW_Y(Pin->Y), TO_DRAW_X(Pin->X), TO_DRAW_Y(Pin->Y));
	}
	   /* draw all the thermal(s) */
	for (i = MAX_LAYER; i; i--)
	{
		layer = LAYER_ON_STACK(i-1);
		if (!layer->On)
			continue;
		ThermLayerFlag = L0THERMFLAG << GetLayerNumber(PCB->Data, layer);
		if (TEST_FLAG(ThermLayerFlag, Pin))
		{
			if (!Erasing)
			{
				if (TEST_FLAG(SELECTEDFLAG | FOUNDFLAG, Pin))
				{
					if (TEST_FLAG(SELECTEDFLAG, Pin))
						XSetForeground(Dpy, Output.fgGC, layer->SelectedColor);
					else
						XSetForeground(Dpy, Output.fgGC, PCB->ConnectedColor);
				}
				else
					XSetForeground(Dpy, Output.fgGC, layer->Color);
			}
			if (TEST_FLAG(SQUAREFLAG, Pin))
			{
				XSetLineAttributes(Dpy, Output.fgGC, TO_SCREEN(Pin->Clearance/2),
					LineSolid, CapRound, JoinRound);
				
				XDrawLine(Dpy, DrawingWindow, Output.fgGC,
					TO_DRAW_X(Pin->X) - TO_SCREEN(half),
					TO_DRAW_Y(Pin->Y) - TO_SCREEN(half),
					TO_DRAW_X(Pin->X) + TO_SCREEN(half),
					TO_DRAW_Y(Pin->Y) + TO_SCREEN(half));
				XDrawLine(Dpy, DrawingWindow, Output.fgGC,
					TO_DRAW_X(Pin->X) - TO_SCREEN(half),
					TO_DRAW_Y(Pin->Y) + TO_SCREEN(half),
					TO_DRAW_X(Pin->X) + TO_SCREEN(half),
					TO_DRAW_Y(Pin->Y) - TO_SCREEN(half));
			}
			else
			{
				XSetLineAttributes(Dpy, Output.fgGC, TO_SCREEN(Pin->Clearance/2),
					LineSolid, CapRound, JoinRound);
				XDrawLine(Dpy, DrawingWindow, Output.fgGC,
					TO_DRAW_X(Pin->X) - halfs,
					TO_DRAW_Y(Pin->Y) - halfs,
					TO_DRAW_X(Pin->X) + halfs,
					TO_DRAW_Y(Pin->Y) + halfs);
				XDrawLine(Dpy, DrawingWindow, Output.fgGC,
					TO_DRAW_X(Pin->X) - halfs,
					TO_DRAW_Y(Pin->Y) + halfs,
					TO_DRAW_X(Pin->X) + halfs,
					TO_DRAW_Y(Pin->Y) - halfs);
			}
		}
	}
	if ((!TEST_FLAG(PINFLAG, Pin) && !PCB->ViaOn) || (TEST_FLAG(PINFLAG, Pin) && !PCB->PinOn))
		return;
		/* now draw the pin or via as appropriate */
	switch (Type)
	{
		case VIA_TYPE:
		case PIN_TYPE:
			SetPVColor(Pin, Type);
			DrawPinOrViaLowLevel(Pin, True);
			break;
		case NO_TYPE:
		default:
			break;
	}
}

/* ---------------------------------------------------------------------------
 * lowlevel drawing routine for pin and via names
 */
static void DrawPinOrViaNameLowLevel(PinTypePtr Ptr)
{
	int direction, ascent, descent;
	XCharStruct overall;

	if (Gathering)
	{
		UpdateRect.x = TO_DRAW_X(Ptr->X -Ptr->Thickness/2 +Settings.PinoutTextOffsetX);
		UpdateRect.y = TO_DRAW_Y(Ptr->Y -Ptr->Thickness/2 +Settings.PinoutTextOffsetY);
		XTextExtents(Settings.PinoutFont[MAX(0,PCB->Zoom)], EMPTY(Ptr->Name),
			MIN(Settings.PinoutNameLength, strlen(EMPTY(Ptr->Name))),
			&direction, &ascent, &descent, &overall);
		UpdateRect.x += overall.lbearing;
		UpdateRect.width = overall.width;
		UpdateRect.y -= overall.ascent;
		UpdateRect.height = ascent + descent;
		AddPart();
		return;
	}
	UpdateRect.x = TO_DRAW_X(Ptr->X -Ptr->Thickness/2 +Settings.PinoutTextOffsetX);
	UpdateRect.y = TO_DRAW_Y(Ptr->Y -Ptr->Thickness/2 +Settings.PinoutTextOffsetY);
	XDrawString(Dpy, DrawingWindow, Output.fgGC,
		UpdateRect.x, UpdateRect.y, EMPTY(Ptr->Name),
		MIN(Settings.PinoutNameLength, strlen(EMPTY(Ptr->Name))));
}

/* ---------------------------------------------------------------------------
 * lowlevel drawing routine for pads
 */
static void DrawPadLowLevel(PadTypePtr Pad)
{
	if (Gathering)
	{
		Dimension size = Pad->Thickness + Pad->Clearance;

		UpdateRect.x = MIN(TO_DRAW_X(Pad->Point1.X), TO_DRAW_X(Pad->Point2.X));
		UpdateRect.x -= TO_SCREEN(size/2);
		UpdateRect.y = MIN(TO_DRAW_Y(Pad->Point1.Y), TO_DRAW_Y(Pad->Point2.Y));
		UpdateRect.y -= TO_SCREEN(size/2);
		UpdateRect.width = TO_SCREEN(abs(Pad->Point1.X - Pad->Point2.X) + size);
		UpdateRect.height = TO_SCREEN(abs(Pad->Point1.Y - Pad->Point2.Y) + size);
		AddPart();
		return;
	}

	XSetLineAttributes(Dpy, Output.fgGC,
		TO_SCREEN(Pad->Thickness), LineSolid, 
			  TEST_FLAG(SQUAREFLAG, Pad) ? CapProjecting : CapRound,
			  JoinRound);
	XDrawLine(Dpy, DrawingWindow, Output.fgGC,
		TO_DRAW_X(Pad->Point1.X), TO_DRAW_Y(Pad->Point1.Y),
		TO_DRAW_X(Pad->Point2.X), TO_DRAW_Y(Pad->Point2.Y));
}

/* ---------------------------------------------------------------------------
 * lowlevel drawing routine for pad names
 */
static void DrawPadNameLowLevel(PadTypePtr Pad)
{
	int direction, ascent, descent;
	XCharStruct overall;


	if (Gathering)
	{
		UpdateRect.x = TO_DRAW_X((Pad->Point1.X + Pad->Point2.X)/2 + Settings.PinoutTextOffsetX);
		UpdateRect.y = TO_DRAW_Y((Pad->Point1.Y + Pad->Point2.Y)/2 + Settings.PinoutTextOffsetY);
		XTextExtents(Settings.PinoutFont[MAX(0,PCB->Zoom)], EMPTY(Pad->Name),
			MIN(Settings.PinoutNameLength, strlen(EMPTY(Pad->Name))),
			&direction, &ascent, &descent, &overall);
		UpdateRect.x += overall.lbearing;
		UpdateRect.width = overall.width;
		UpdateRect.y -= overall.ascent;
		UpdateRect.height = ascent + descent;
		AddPart();
		return;
	}

	UpdateRect.x = TO_DRAW_X((Pad->Point1.X + Pad->Point2.X)/2 + Settings.PinoutTextOffsetX);
	UpdateRect.y = TO_DRAW_Y((Pad->Point1.Y + Pad->Point2.Y)/2 + Settings.PinoutTextOffsetY);
	XDrawString(Dpy, DrawingWindow, Output.fgGC,
		UpdateRect.x, UpdateRect.y,
		EMPTY(Pad->Name),
		MIN(Settings.PinoutNameLength, strlen(EMPTY(Pad->Name))));
}

/* ---------------------------------------------------------------------------
 * clearance for pads
 */
static void ClearPad(PadTypePtr Pad)
{
	XSetLineAttributes(Dpy, Output.pmGC,
		TO_SCREEN(Pad->Thickness + Pad->Clearance), LineSolid, 
			  TEST_FLAG(SQUAREFLAG, Pad) ? CapProjecting : CapRound,
			  JoinRound);
	XDrawLine(Dpy, Offmask, Output.pmGC,
		TO_MASK_X(Pad->Point1.X), TO_MASK_Y(Pad->Point1.Y),
		TO_MASK_X(Pad->Point2.X), TO_MASK_Y(Pad->Point2.Y));
}

/* ---------------------------------------------------------------------------
 * clearance for lines
 */
static void ClearLine(LineTypePtr Line)
{
	XSetLineAttributes(Dpy, Output.pmGC,
		TO_SCREEN(Line->Clearance + Line->Thickness), LineSolid, CapRound, JoinRound);
	XDrawLine(Dpy, Offmask, Output.pmGC,
		TO_MASK_X(Line->Point1.X), TO_MASK_Y(Line->Point1.Y),
		TO_MASK_X(Line->Point2.X), TO_MASK_Y(Line->Point2.Y));
}

/* ---------------------------------------------------------------------------
 * clearance for arcs 
 */
static void ClearArc(ArcTypePtr Arc)
{
	XSetLineAttributes(Dpy, Output.pmGC,
		TO_SCREEN(Arc->Thickness + Arc->Clearance),
			LineSolid, CapRound, JoinRound);
	XDrawArc(Dpy, Offmask, Output.pmGC,
		TO_MASK_X(Arc->X) -TO_SCREEN(Arc->Width),
		TO_MASK_Y(Arc->Y) -TO_SCREEN(Arc->Height),
		TO_SCREEN(2*Arc->Width),
		TO_SCREEN(2*Arc->Height),
		(TO_SCREEN_ANGLE(Arc->StartAngle) -180) *64,
		TO_SCREEN_DELTA(Arc->Delta) *64);
}

/* ---------------------------------------------------------------------------
 * lowlevel drawing routine for lines
 */
static void DrawLineLowLevel(LineTypePtr Line, Boolean HaveGathered)
{
	if (Gathering && !HaveGathered)
	{
		Dimension wide = Line->Thickness;

		if (TEST_FLAG(CLEARLINEFLAG, Line))
			wide += Line->Clearance;
		UpdateRect.x = MIN(TO_DRAW_X(Line->Point1.X), TO_DRAW_X(Line->Point2.X));
		UpdateRect.x -= TO_SCREEN(wide/2);
		UpdateRect.y = MIN(TO_DRAW_Y(Line->Point1.Y), TO_DRAW_Y(Line->Point2.Y));
		UpdateRect.y -= TO_SCREEN(wide/2);
		UpdateRect.width = TO_SCREEN(abs(Line->Point1.X - Line->Point2.X) + wide);
		UpdateRect.height = TO_SCREEN(abs(Line->Point1.Y - Line->Point2.Y) + wide);
		if (UpdateRect.width == 0)
			UpdateRect.width = 1;
		if (UpdateRect.height == 0)
			UpdateRect.height = 1;
		AddPart();
		return;
	}

	XSetLineAttributes(Dpy, Output.fgGC,
		TO_SCREEN(Line->Thickness), LineSolid, CapRound, JoinRound);
	if (TEST_FLAG(RATFLAG, Line))
	{
		XSetStipple(Dpy, Output.fgGC, Stipples[0]);
		XSetFillStyle(Dpy, Output.fgGC, FillStippled);
		XDrawLine(Dpy, DrawingWindow, Output.fgGC,
			TO_DRAW_X(Line->Point1.X), TO_DRAW_Y(Line->Point1.Y),
			TO_DRAW_X(Line->Point2.X), TO_DRAW_Y(Line->Point2.Y));
		XSetFillStyle(Dpy, Output.fgGC, FillSolid);
	}
	else
	XDrawLine(Dpy, DrawingWindow, Output.fgGC,
		TO_DRAW_X(Line->Point1.X), TO_DRAW_Y(Line->Point1.Y),
		TO_DRAW_X(Line->Point2.X), TO_DRAW_Y(Line->Point2.Y));
}

/* ---------------------------------------------------------------------------
 * lowlevel drawing routine for text objects
 */
static void DrawTextLowLevel(TextTypePtr Text)
{
	Position	x = 0;
	int		maxthick = 0;
	unsigned char	*string = (unsigned char *) Text->TextString;
	Cardinal	n;
	FontTypePtr	font = &PCB->Font;

	if (Gathering)
	{
		while (string && *string)
		{
				/* draw lines if symbol is valid and data is present */
			if (*string <= MAX_FONTPOSITION && font->Symbol[*string].Valid)
			{
				LineTypePtr	line = font->Symbol[*string].Line;

				for (n = font->Symbol[*string].LineN; n; n--, line++)
					if (line->Thickness > maxthick)
						maxthick = line->Thickness;
			}
			string++;
		}
		maxthick *= Text->Scale/100;
		UpdateRect.x = MIN(TO_DRAW_X(Text->BoundingBox.X1), TO_DRAW_X(Text->BoundingBox.X2));
		UpdateRect.y = MIN(TO_DRAW_Y(Text->BoundingBox.Y1), TO_DRAW_Y(Text->BoundingBox.Y2));
		UpdateRect.x -= TO_SCREEN(maxthick);
		UpdateRect.y -= TO_SCREEN(maxthick);
		UpdateRect.width = TO_SCREEN(abs(Text->BoundingBox.X2 - Text->BoundingBox.X1) + 2 * maxthick);
		UpdateRect.height = TO_SCREEN(abs(Text->BoundingBox.Y2 - Text->BoundingBox.Y1) + 2 * maxthick);
		AddPart();
		return;
	}

	while (string && *string)
	{
			/* draw lines if symbol is valid and data is present */
		if (*string <= MAX_FONTPOSITION && font->Symbol[*string].Valid)
		{
			LineTypePtr	line = font->Symbol[*string].Line;
			LineType	newline;

			for (n = font->Symbol[*string].LineN; n; n--, line++)
			{
					/* create one line, scale, move, rotate and swap it */
				newline = *line;
				newline.Point1.X = (newline.Point1.X +x) *Text->Scale /100;
				newline.Point1.Y = newline.Point1.Y      *Text->Scale /100;
				newline.Point2.X = (newline.Point2.X +x) *Text->Scale /100;
				newline.Point2.Y = newline.Point2.Y      *Text->Scale /100;
				newline.Thickness = newline.Thickness *Text->Scale /100;
				if (newline.Thickness > maxthick)
					maxthick = newline.Thickness;

				RotateLineLowLevel(&newline, 0, 0, Text->Direction);

					/* the labels of SMD objects on the bottom
					 * side haven't been swapped yet, only their offset
					 */
				if (TEST_FLAG(ONSOLDERFLAG, Text))
				{
					newline.Point1.X = SWAP_SIGN_X(newline.Point1.X);
					newline.Point1.Y = SWAP_SIGN_Y(newline.Point1.Y);
					newline.Point2.X = SWAP_SIGN_X(newline.Point2.X);
					newline.Point2.Y = SWAP_SIGN_Y(newline.Point2.Y);
				}
					/* add offset and draw line */
				newline.Point1.X += Text->X;
				newline.Point1.Y += Text->Y;
				newline.Point2.X += Text->X;
				newline.Point2.Y += Text->Y;
				DrawLineLowLevel(&newline, True);
			}

				/* move on to next cursor position */
			x += (font->Symbol[*string].Width +font->Symbol[*string].Delta);
		}
		else
		{
				/* the default symbol is a filled box */
			BoxType		defaultsymbol = PCB->Font.DefaultSymbol;
			Position	size = (defaultsymbol.X2 -defaultsymbol.X1) *6/5;

			defaultsymbol.X1 = (defaultsymbol.X1 +x) *Text->Scale /100;
			defaultsymbol.Y1 = defaultsymbol.Y1      *Text->Scale /100;
			defaultsymbol.X2 = (defaultsymbol.X2 +x) *Text->Scale /100;
			defaultsymbol.Y2 = defaultsymbol.Y2      *Text->Scale /100;

			if (TEST_FLAG(ONSOLDERFLAG, Text))
			{
				defaultsymbol.X1 = TO_SCREEN_SIGN_X(defaultsymbol.X1);
				defaultsymbol.Y1 = TO_SCREEN_SIGN_Y(defaultsymbol.Y1);
				defaultsymbol.X2 = TO_SCREEN_SIGN_X(defaultsymbol.X2);
				defaultsymbol.Y2 = TO_SCREEN_SIGN_Y(defaultsymbol.Y2);
			}
			RotateBoxLowLevel(&defaultsymbol, 0, 0, Text->Direction);

				/* add offset and draw box */
			defaultsymbol.X1 += Text->X;
			defaultsymbol.Y1 += Text->Y;
			defaultsymbol.X2 += Text->X;
			defaultsymbol.Y2 += Text->Y;
			XFillRectangle(Dpy, DrawingWindow, Output.fgGC,
				TO_DRAW_X(defaultsymbol.X1),
				TO_DRAW_Y(SWAP_IDENT ? defaultsymbol.Y2 : defaultsymbol.Y1),
				TO_SCREEN(abs(defaultsymbol.X2 -defaultsymbol.X1)),
				TO_SCREEN(abs(defaultsymbol.Y2 -defaultsymbol.Y1)));

				/* move on to next cursor position */
			x += size;
		}
		string++;
	}
}

/* ---------------------------------------------------------------------------
 * lowlevel drawing routine for polygons
 */
static void DrawPolygonLowLevel(PolygonTypePtr Polygon, Boolean OnMask)
{
	static	XPoint		*data = NULL;	/* tmp pointer */
	static	Cardinal	max = 0;

	if (Gathering)
	{
		UpdateRect.x = MIN(TO_DRAW_X(Polygon->BoundingBox.X1), TO_DRAW_X(Polygon->BoundingBox.X2));
		UpdateRect.y = MIN(TO_DRAW_Y(Polygon->BoundingBox.Y1), TO_DRAW_Y(Polygon->BoundingBox.Y2));
		UpdateRect.width = TO_SCREEN(Polygon->BoundingBox.X2 - Polygon->BoundingBox.X1);
		UpdateRect.height = TO_SCREEN(Polygon->BoundingBox.Y2 - Polygon->BoundingBox.Y1);
		AddPart();
		return;
	}
		/* allocate memory for data with screen coordinates */
	if (Polygon->PointN > max)
	{
		max = Polygon->PointN;
		data = (XPoint *) MyRealloc(data, max *sizeof(XPoint),
			"DrawPolygonLowLevel()");
	}

		/* copy data to tmp array and convert it to screen coordinates */
	POLYGONPOINT_LOOP(Polygon,
		if (OnMask)
		{
			data[n].x = TO_MASK_X(point->X);
			data[n].y = TO_MASK_Y(point->Y);
		}
		else
		{
			data[n].x = TO_DRAW_X(point->X);
			data[n].y = TO_DRAW_Y(point->Y);
		}
	);
	if (OnMask)
		XFillPolygon(Dpy, Offmask, Output.pmGC,
			data, Polygon->PointN, Complex, CoordModeOrigin);
	else
		XFillPolygon(Dpy, DrawingWindow, Output.fgGC,
			data, Polygon->PointN, Complex, CoordModeOrigin);
}

/* ---------------------------------------------------------------------------
 * lowlevel routine to element arcs
 */
static void DrawArcLowLevel(ArcTypePtr Arc)
{
	Dimension	width = Arc->Thickness;

	if (Gathering)
	{
		if (TEST_FLAG(CLEARLINEFLAG, Arc))
			width += Arc->Clearance;
		UpdateRect.x = TO_DRAW_X(Arc->X) -TO_SCREEN(Arc->Width + width/2);
		UpdateRect.y = TO_DRAW_Y(Arc->Y) -TO_SCREEN(Arc->Height + width/2);
		UpdateRect.width = TO_SCREEN(2*Arc->Width + width);
		UpdateRect.height = TO_SCREEN(2*Arc->Height + width);
		AddPart();
		return;
	}
		/* angles have to be converted to X11 notation */
	XSetLineAttributes(Dpy, Output.fgGC,
		TO_SCREEN(Arc->Thickness), LineSolid, CapRound, JoinRound);
	XDrawArc(Dpy, DrawingWindow, Output.fgGC,
		TO_DRAW_X(Arc->X) -TO_SCREEN(Arc->Width),
		TO_DRAW_Y(Arc->Y) -TO_SCREEN(Arc->Height),
		TO_SCREEN(2*Arc->Width),
		TO_SCREEN(2*Arc->Height),
		(TO_SCREEN_ANGLE(Arc->StartAngle) -180) *64,
		TO_SCREEN_DELTA(Arc->Delta) *64);
}

/* ---------------------------------------------------------------------------
 * draws the package of an element
 */
static void DrawElementPackageLowLevel(ElementTypePtr Element, int unused)
{
		/* draw lines, arcs, text and pins */
	ELEMENTLINE_LOOP(Element, DrawLineLowLevel(line, False););
	ARC_LOOP(Element, DrawArcLowLevel(arc););
}

/* ---------------------------------------------------------------------------
 * draw a via object
 */
void DrawVia(PinTypePtr Via, int unused)
{
	if (!Gathering)
		SetPVColor(Via, VIA_TYPE);
	if (!TEST_FLAG(HOLEFLAG, Via) && TEST_FLAG(ALLPIPFLAGS, Via))
		ClearPin(Via, VIA_TYPE, 0);
	else
		DrawPinOrViaLowLevel(Via, True);
	if (!TEST_FLAG(HOLEFLAG, Via) && TEST_FLAG(DISPLAYNAMEFLAG, Via))
		DrawPinOrViaNameLowLevel(Via);
}

/* ---------------------------------------------------------------------------
 * draw a via without dealing with polygon clearance 
 */
static void DrawPlainVia(PinTypePtr Via, int unused)
{
	if (!Gathering)
		SetPVColor(Via, VIA_TYPE);
	DrawPinOrViaLowLevel(Via, False);
	if (!TEST_FLAG(HOLEFLAG, Via) && TEST_FLAG(DISPLAYNAMEFLAG, Via))
		DrawPinOrViaNameLowLevel(Via);
}

/* ---------------------------------------------------------------------------
 * draws the name of a via
 */
void DrawViaName(PinTypePtr Via, int unused)
{
	if (!Gathering)
	{
		if (TEST_FLAG(SELECTEDFLAG, Via))
			XSetForeground(Dpy, Output.fgGC, PCB->ViaSelectedColor);
		else
			XSetForeground(Dpy, Output.fgGC, PCB->ViaColor);
	}
	DrawPinOrViaNameLowLevel(Via);
}

/* ---------------------------------------------------------------------------
 * draw a pin object
 */
void DrawPin(PinTypePtr Pin, int unused) 
{
	if (!TEST_FLAG(HOLEFLAG, Pin) && TEST_FLAG(ALLPIPFLAGS, Pin))
		ClearPin(Pin, PIN_TYPE, 0);
	else
	{
		if (!Gathering)
			SetPVColor(Pin, PIN_TYPE);
		DrawPinOrViaLowLevel(Pin, True);
	}
	if (!TEST_FLAG(HOLEFLAG, Pin) && TEST_FLAG(DISPLAYNAMEFLAG, Pin))
		DrawPinOrViaNameLowLevel(Pin);
}

/* ---------------------------------------------------------------------------
 * draw a pin without clearing around polygons 
 */
static void DrawPlainPin(PinTypePtr Pin, int unused)
{
	if (!Gathering)
		SetPVColor(Pin, PIN_TYPE);
	DrawPinOrViaLowLevel(Pin, False);
	if (!TEST_FLAG(HOLEFLAG, Pin) && TEST_FLAG(DISPLAYNAMEFLAG, Pin))
		DrawPinOrViaNameLowLevel(Pin);
}

/* ---------------------------------------------------------------------------
 * draws the name of a pin
 */
void DrawPinName(PinTypePtr Pin, int unused)
{
	if (!Gathering)
	{
		if (TEST_FLAG(SELECTEDFLAG, Pin))
			XSetForeground(Dpy, Output.fgGC, PCB->PinSelectedColor);
		else
			XSetForeground(Dpy, Output.fgGC, PCB->PinColor);
	}
	DrawPinOrViaNameLowLevel(Pin);
}

/* ---------------------------------------------------------------------------
 * draw a pad object
 */
void DrawPad(PadTypePtr Pad, int unused)
{
	if (!Gathering)
	{
		if (TEST_FLAG(WARNFLAG | SELECTEDFLAG | FOUNDFLAG, Pad))
		{
			if (TEST_FLAG(WARNFLAG, Pad))
				XSetForeground(Dpy, Output.fgGC, PCB->WarnColor);
			else if (TEST_FLAG(SELECTEDFLAG, Pad))
				XSetForeground(Dpy, Output.fgGC, PCB->PinSelectedColor);
			else
				XSetForeground(Dpy, Output.fgGC, PCB->ConnectedColor);
		}
		else if (FRONT(Pad))
			XSetForeground(Dpy, Output.fgGC, PCB->PinColor);
		else
			XSetForeground(Dpy, Output.fgGC, PCB->InvisibleObjectsColor);
	}
	DrawPadLowLevel(Pad);
	if (TEST_FLAG(DISPLAYNAMEFLAG, Pad))
		DrawPadNameLowLevel(Pad);
}

/* ---------------------------------------------------------------------------
 * draws the name of a pad
 */
void DrawPadName(PadTypePtr Pad, int unused)
{
	if (!Gathering)
	{
		if (TEST_FLAG(SELECTEDFLAG, Pad))
			XSetForeground(Dpy, Output.fgGC, PCB->PinSelectedColor);
		else if (FRONT(Pad))
			XSetForeground(Dpy, Output.fgGC, PCB->PinColor);
		else
			XSetForeground(Dpy,Output.fgGC, PCB->InvisibleObjectsColor);
	}
	DrawPadNameLowLevel(Pad);
}

/* ---------------------------------------------------------------------------
 * draws a line on a layer
 */
void DrawLine(LayerTypePtr Layer, LineTypePtr Line, int unused)
{
	if (!Gathering)
	{
		if (TEST_FLAG(SELECTEDFLAG | FOUNDFLAG, Line))
		{
			if (TEST_FLAG(SELECTEDFLAG, Line))
				XSetForeground(Dpy, Output.fgGC, Layer->SelectedColor);
			else
				XSetForeground(Dpy, Output.fgGC, PCB->ConnectedColor);
		}
		else
			XSetForeground(Dpy, Output.fgGC, Layer->Color);
	}
	DrawLineLowLevel(Line, False);
}

/* ---------------------------------------------------------------------------
 * draws a ratline
 */
void DrawRat(RatTypePtr Line, int unused)
{
	if (!Gathering)
	{
		if (TEST_FLAG(SELECTEDFLAG | FOUNDFLAG, Line))
		{
			if (TEST_FLAG(SELECTEDFLAG, Line))
				XSetForeground(Dpy, Output.fgGC, PCB->RatSelectedColor);
			else
				XSetForeground(Dpy, Output.fgGC, PCB->ConnectedColor);
		}
		else
			XSetForeground(Dpy, Output.fgGC, PCB->RatColor);
	}
	DrawLineLowLevel((LineTypePtr) Line, False);
}

/* ---------------------------------------------------------------------------
 * draws an arc on a layer
 */
void DrawArc(LayerTypePtr Layer, ArcTypePtr Arc, int unused)
{
	if (!Gathering)
	{
		if (TEST_FLAG(SELECTEDFLAG | FOUNDFLAG, Arc))
		{
			if (TEST_FLAG(SELECTEDFLAG, Arc))
				XSetForeground(Dpy, Output.fgGC, Layer->SelectedColor);
			else
				XSetForeground(Dpy, Output.fgGC, PCB->ConnectedColor);
		}
		else
			XSetForeground(Dpy, Output.fgGC, Layer->Color);
	}
	DrawArcLowLevel(Arc);
}

/* ---------------------------------------------------------------------------
 * draws a text on a layer
 */
void DrawText(LayerTypePtr Layer, TextTypePtr Text, int unused)
{
        if (!Layer->On)
                return;
        if (TEST_FLAG(SELECTEDFLAG, Text))
                XSetForeground(Dpy, Output.fgGC, Layer->SelectedColor);
        else    
                        XSetForeground(Dpy, Output.fgGC, Layer->Color);
        DrawTextLowLevel(Text);
}     

/* ---------------------------------------------------------------------------
 * draws text on a layer - assumes it's not on silkscreen
 */
static void DrawRegularText(LayerTypePtr Layer, TextTypePtr Text, int unused)
{
        if (TEST_FLAG(SELECTEDFLAG, Text))
                XSetForeground(Dpy, Output.fgGC, Layer->SelectedColor);
        else
                XSetForeground(Dpy, Output.fgGC, Layer->Color);
        DrawTextLowLevel(Text);
}     

/* ---------------------------------------------------------------------------
 * draws a polygon on a layer
 */
void DrawPolygon(LayerTypePtr Layer, PolygonTypePtr Polygon, int unused)
{
      int     Myflag, layernum;
   
      if (TEST_FLAG(SELECTEDFLAG | FOUNDFLAG, Polygon))
      {
              if (TEST_FLAG(SELECTEDFLAG, Polygon))
                      XSetForeground(Dpy, Output.fgGC, Layer->SelectedColor);
              else
                      XSetForeground(Dpy, Output.fgGC, PCB->ConnectedColor);
      }
      else
              XSetForeground(Dpy, Output.fgGC, Layer->Color);
      layernum = GetLayerNumber(PCB->Data, Layer);
      if (Settings.StipplePolygons)
      {
              XSetStipple(Dpy, Output.fgGC, Stipples[layernum]);
              XSetFillStyle(Dpy, Output.fgGC, FillStippled);
      }
      DrawPolygonLowLevel(Polygon, False);
      if (Settings.StipplePolygons)
              XSetFillStyle(Dpy, Output.fgGC, FillSolid);
      Myflag = L0THERMFLAG << GetLayerNumber(PCB->Data, Layer);
      if (TEST_FLAG(CLEARPOLYFLAG, Polygon))
      {
              ALLPIN_LOOP(PCB->Data,
                      if (IsPointInPolygon(pin->X, pin->Y,
                          0, Polygon))
                              {
                                      ClearPin(pin, PIN_TYPE, 0);
                              }
              );  
              VIA_LOOP(PCB->Data,
                      if (IsPointInPolygon(via->X, via->Y,
                          0, Polygon))
                              {
                                      ClearPin(via, VIA_TYPE, 0);
                              }
              );
      }       
}

/* ---------------------------------------------------------------------------
 * draws a polygon without cutting away the pin/via clearances
 */
static void DrawPlainPolygon(LayerTypePtr Layer, PolygonTypePtr Polygon)
{     
	if (TEST_FLAG(SELECTEDFLAG | FOUNDFLAG, Polygon))
	{
		if (TEST_FLAG(SELECTEDFLAG, Polygon))
			XSetForeground(Dpy, Output.fgGC, Layer->SelectedColor);
		else
			XSetForeground(Dpy, Output.fgGC, PCB->ConnectedColor);
	}
	else
		XSetForeground(Dpy, Output.fgGC, Layer->Color);
	DrawPolygonLowLevel(Polygon, False);
}

/* ---------------------------------------------------------------------------
 * draws an element
 */
void DrawElement(ElementTypePtr Element, int unused)
{
	DrawElementPackage(Element, unused);
	DrawElementName(Element, unused);
	DrawElementPinsAndPads(Element, unused);
} 

/* ---------------------------------------------------------------------------
 * draws the name of an element
 */
void DrawElementName(ElementTypePtr Element, int unused)
{
	if (TEST_FLAG(HIDENAMEFLAG, Element))
		return;
	if (TEST_FLAG(SELECTEDFLAG, &ELEMENT_TEXT(PCB, Element)))
		XSetForeground(Dpy, Output.fgGC, PCB->ElementSelectedColor);
	else
		if (FRONT(Element))
			XSetForeground(Dpy, Output.fgGC, PCB->ElementColor);
		else
			XSetForeground(Dpy, Output.fgGC, PCB->InvisibleObjectsColor);
	DrawTextLowLevel(&ELEMENT_TEXT(PCB, Element));
}

/* ---------------------------------------------------------------------------
 * draws the package of an element
 */
void DrawElementPackage(ElementTypePtr Element, int unused)
{
		/* set color and draw lines, arcs, text and pins */
	if (TEST_FLAG(SELECTEDFLAG, Element))
		XSetForeground(Dpy, Output.fgGC, PCB->ElementSelectedColor);
	else
		if (FRONT(Element))
			XSetForeground(Dpy, Output.fgGC, PCB->ElementColor);
		else
			XSetForeground(Dpy, Output.fgGC, PCB->InvisibleObjectsColor);
	DrawElementPackageLowLevel(Element, unused);
}

/* ---------------------------------------------------------------------------
 * draw pins of an element
 */
void DrawElementPinsAndPads(ElementTypePtr Element, int unused)
{
	PAD_LOOP(Element,
		if (FRONT(pad) || PCB->InvisibleObjectsOn)
			DrawPad(pad, unused);
	);
	PIN_LOOP(Element,
		DrawPin(pin, unused);
	);
}

/* ---------------------------------------------------------------------------
 * draw pins of an element without clearing around polygons
 */
static void DrawPlainElementPinsAndPads(ElementTypePtr Element, int unused)
{
		/* don't draw invisible pads, they're already handled */
	PAD_LOOP(Element,
		if (FRONT(pad)) 
			DrawPad(pad, unused);
	);
	PIN_LOOP(Element,
		DrawPlainPin(pin, unused);
	);
}

/* ---------------------------------------------------------------------------
 * erase a via
 */
void EraseVia(PinTypePtr Via)
{
	Erasing++;
        if (TEST_FLAG(ALLPIPFLAGS, Via))
                ClearPin(Via, NO_TYPE, 0);
	XSetForeground(Dpy, Output.fgGC, Settings.bgColor);
	DrawPinOrViaLowLevel(Via, False);
	if (TEST_FLAG(DISPLAYNAMEFLAG, Via))
		DrawPinOrViaNameLowLevel(Via);
	Erasing--;
}

/* ---------------------------------------------------------------------------
 * erase a ratline
 */
void EraseRat(RatTypePtr Rat)
{
	Erasing++;
	XSetForeground(Dpy, Output.fgGC, Settings.bgColor);
	DrawLineLowLevel((LineTypePtr) Rat, False);
	Erasing--;
}


/* ---------------------------------------------------------------------------
 * erase a via name
 */
void EraseViaName(PinTypePtr Via)
{
	Erasing++;
	XSetForeground(Dpy, Output.fgGC, Settings.bgColor);
	DrawPinOrViaNameLowLevel(Via);
	Erasing--;
}

/* ---------------------------------------------------------------------------
 * erase a pad object
 */
void ErasePad(PadTypePtr Pad)
{
	Erasing++;
	XSetForeground(Dpy, Output.fgGC, Settings.bgColor);
	DrawPadLowLevel(Pad);
	if (TEST_FLAG(DISPLAYNAMEFLAG, Pad))
		DrawPadNameLowLevel(Pad);
	Erasing--;
}

/* ---------------------------------------------------------------------------
 * erase a pad name
 */
void ErasePadName(PadTypePtr Pad)
{
	Erasing++;
	XSetForeground(Dpy, Output.fgGC, Settings.bgColor);
	DrawPadNameLowLevel(Pad);
	Erasing--;
}

/* ---------------------------------------------------------------------------
 * erase a pin object
 */
void ErasePin(PinTypePtr Pin)
{
	Erasing++;
        if (TEST_FLAG(ALLPIPFLAGS, Pin))
                ClearPin(Pin, NO_TYPE, 0);
	XSetForeground(Dpy, Output.fgGC, Settings.bgColor);
	DrawPinOrViaLowLevel(Pin, False);
	if (TEST_FLAG(DISPLAYNAMEFLAG, Pin))
		DrawPinOrViaNameLowLevel(Pin);
	Erasing--;
}

/* ---------------------------------------------------------------------------
 * erase a pin name
 */
void ErasePinName(PinTypePtr Pin)
{
	Erasing++;
	XSetForeground(Dpy, Output.fgGC, Settings.bgColor);
	DrawPinOrViaNameLowLevel(Pin);
	Erasing--;
}

/* ---------------------------------------------------------------------------
 * erases a line on a layer
 */
void EraseLine(LineTypePtr Line)
{
	Erasing++;
	XSetForeground(Dpy, Output.fgGC, Settings.bgColor);
	DrawLineLowLevel(Line, False);
	Erasing--;
}

/* ---------------------------------------------------------------------------
 * erases an arc on a layer
 */
void EraseArc(ArcTypePtr Arc)
{
	Erasing++;
	XSetForeground(Dpy, Output.fgGC, Settings.bgColor);
	DrawArcLowLevel(Arc);
	Erasing--;
}

/* ---------------------------------------------------------------------------
 * erases a text on a layer
 */
void EraseText(TextTypePtr Text)
{
	Erasing++;
	XSetForeground(Dpy, Output.fgGC, Settings.bgColor);
	DrawTextLowLevel(Text);
	Erasing--;
}

/* ---------------------------------------------------------------------------
 * erases a polygon on a layer
 */
void ErasePolygon(PolygonTypePtr Polygon)
{
	Erasing++;
	XSetForeground(Dpy, Output.fgGC, Settings.bgColor);
	DrawPolygonLowLevel(Polygon, False);
	Erasing--;
}

/* ---------------------------------------------------------------------------
 * erases an element
 */
void EraseElement(ElementTypePtr Element)
{
	Erasing++;
		/* set color and draw lines, arcs, text and pins */
	XSetForeground(Dpy, Output.fgGC, Settings.bgColor);
	ELEMENTLINE_LOOP(Element, DrawLineLowLevel(line, False););
	ARC_LOOP(Element, DrawArcLowLevel(arc););
	if (!TEST_FLAG(HIDENAMEFLAG, Element))
		DrawTextLowLevel(&ELEMENT_TEXT(PCB, Element));
	EraseElementPinsAndPads(Element);
	Erasing--;
} 

/* ---------------------------------------------------------------------------
 * erases all pins and pads of an element
 */
void EraseElementPinsAndPads(ElementTypePtr Element)
{
	Erasing++;
	XSetForeground(Dpy, Output.fgGC, Settings.bgColor);
	PIN_LOOP(Element,
        	if (TEST_FLAG(ALLPIPFLAGS, pin))
		{
			ClearPin(pin, NO_TYPE, 0);
			XSetForeground(Dpy, Output.fgGC, Settings.bgColor);
		}
		DrawPinOrViaLowLevel(pin, False);
		if (TEST_FLAG(DISPLAYNAMEFLAG, pin))
			DrawPinOrViaNameLowLevel(pin);
	);
	PAD_LOOP(Element,
		DrawPadLowLevel(pad);
		if (TEST_FLAG(DISPLAYNAMEFLAG, pad))
			DrawPadNameLowLevel(pad);
	);
	Erasing--;
}

/* ---------------------------------------------------------------------------
 * erases the name of an element
 */
void EraseElementName(ElementTypePtr Element)
{
	if (TEST_FLAG(HIDENAMEFLAG, Element))
		return;
	Erasing++;
	XSetForeground(Dpy, Output.fgGC, Settings.bgColor);
	DrawTextLowLevel(&ELEMENT_TEXT(PCB, Element));
	Erasing--;
}

/* ---------------------------------------------------------------------------
 * draws grid points if the distance is >= MIN_GRID_DISTANCE
 */
static void DrawGrid()
{
	Position	minx, miny,
			maxx, maxy,
			x, y, temp,
			delta;
	
	delta = GetGridFactor() *PCB->Grid;
	if (TO_SCREEN(delta) >= MIN_GRID_DISTANCE)
	{
		minx = TO_PCB_X(0);
		miny = TO_PCB_Y(0);
		maxx = TO_PCB_X(Output.Width);
		maxy = TO_PCB_Y(Output.Height);
		if (miny > maxy)
		{
			temp = maxy;
			maxy = miny;
			miny = temp;
		}
		minx -= delta;
		miny -= delta;
		maxx += delta;
		maxy += delta;
		maxx = MIN((Dimension) maxx, PCB->MaxWidth);
		maxy = MIN((Dimension) maxy, PCB->MaxHeight);
		miny = MAX(0, miny);
		for (y = miny; y <= maxy; y += delta)
			for (x = minx; x <= maxx; x += delta)
				XDrawPoint(Dpy, DrawingWindow,
					Output.GridGC, TO_DRAW_X(GRIDFIT_X(x, delta)),
					TO_DRAW_Y(GRIDFIT_Y(y, delta)));
	}
}

void EraseObject(int type, void *ptr)
{
	switch(type)
	{
		case VIA_TYPE:
		case PIN_TYPE:
			ErasePin((PinTypePtr) ptr);
			break;
		case TEXT_TYPE:
		case ELEMENTNAME_TYPE:
			EraseText((TextTypePtr) ptr);
			break;
		case POLYGON_TYPE:
			ErasePolygon((PolygonTypePtr) ptr);
			break;
		case ELEMENT_TYPE:
			EraseElement((ElementTypePtr) ptr);
			break;
		case LINE_TYPE:
		case RATLINE_TYPE:
			EraseLine((LineTypePtr) ptr);
			break;
		case PAD_TYPE:
			ErasePad((PadTypePtr) ptr);
			break;
		case ARC_TYPE:
			EraseArc((ArcTypePtr) ptr);
			break;
		default:
			Message("hace: Internal ERROR, trying to erase an unknown type\n");
	}
}



void DrawObject(int type, void *ptr1, void *ptr2, int unused)
{
	switch(type)
	{
		case VIA_TYPE:
			if (PCB->ViaOn)
				DrawVia((PinTypePtr) ptr2, 0);
			break;
		case LINE_TYPE:
			if (((LayerTypePtr) ptr1)->On)
				DrawLine((LayerTypePtr) ptr1, (LineTypePtr) ptr2, 0);
			break;
		case ARC_TYPE:
			if (((LayerTypePtr) ptr1)->On)
				DrawArc((LayerTypePtr) ptr1, (ArcTypePtr) ptr2, 0);
			break;
		case TEXT_TYPE:
			if (((LayerTypePtr) ptr1)->On)
				DrawText((LayerTypePtr) ptr1, (TextTypePtr) ptr2, 0);
			break;
		case POLYGON_TYPE:
			if (((LayerTypePtr) ptr1)->On)
				DrawPolygon((LayerTypePtr) ptr1, (PolygonTypePtr) ptr2, 0);
			break;
		case ELEMENT_TYPE:
			if (PCB->ElementOn &&
			    (FRONT((ElementTypePtr) ptr2) || PCB->InvisibleObjectsOn))
				DrawElement((ElementTypePtr) ptr2, 0);
			break;
		case RATLINE_TYPE:
			if (PCB->RatOn)
				DrawRat((RatTypePtr) ptr2, 0);
			break;
		case PIN_TYPE:
			if (PCB->PinOn)
				DrawPin((PinTypePtr) ptr2, 0);
			break;
		case PAD_TYPE:
			if (PCB->PinOn)
				DrawPad((PadTypePtr) ptr2, 0);
			break;
		case ELEMENTNAME_TYPE:
			if (PCB->ElementOn &&
			   (FRONT((ElementTypePtr) ptr2) || PCB->InvisibleObjectsOn))
				DrawElementName((ElementTypePtr) ptr1, 0);
			break;
	}
}
