// Copyright (c) 1995 The University of Cincinnati.
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Author: Dale E. Martin          dmartin@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: symbol_table.cc,v 1.5 1999/10/19 01:12:19 dmartin Exp $
// 
//---------------------------------------------------------------------------

#include "symbol_table.hh"
#include "IIR_GenericList.hh"
#include "IIR_PortList.hh"
#include "IIR_SignalInterfaceDeclaration.hh"
#include "IIR_ConstantInterfaceDeclaration.hh"
#include "IIR_TypeDeclaration.hh"
#include "IIR_EnumerationTypeDefinition.hh"
#include "IIR_AccessTypeDefinition.hh"
#include "IIR_PhysicalTypeDefinition.hh"
#include "IIR_ArrayTypeDefinition.hh"
#include "IIR_IntegerTypeDefinition.hh"
#include "IIR_FloatingSubtypeDefinition.hh"
#include "IIR_EnumerationLiteral.hh"
#include "IIR_PhysicalUnit.hh"
#include "error_func.hh"
#include "IIR_Identifier.hh"
#include "IIRScram_IncompleteTypeDefinition.hh"
#include "IIR_LibraryDeclaration.hh"
#include "IIR_LibraryUnit.hh"
#include "StandardPackage.hh"

#include <strstream.h>
#include <string.h>

symbol_table::symbol_table( bool load_std_library ){
  global_scope = new scope_entry( NULL, NULL );  
  current_scope = global_scope;

  if( load_std_library == true ){
    load_standard_library();
  }
}

symbol_table::symbol_table( int table_size, bool load_std_library ) : 
  visible_symbols(table_size), hidden_symbols(table_size){

  global_scope = new scope_entry( NULL, NULL ); 
  current_scope = global_scope;
  
  if( load_std_library == true ){
    load_standard_library();
  }
}

void 
symbol_table::load_standard_library(){
  //  ((IIR_TypeDefinition *)StandardPackage::savant_universal_integer)->_come_into_scope( this, 0 );
  //  ((IIR_TypeDefinition *)StandardPackage::savant_universal_real)->_come_into_scope( this, 0 );
  bool old_val = debug_symbol_table;
  debug_symbol_table = false;
  
  add_declaration( StandardPackage::std_decl );
  open_scope( StandardPackage::std_decl );
  add_declaration( StandardPackage::std_standard_decl );
  open_scope( StandardPackage::std_standard_decl );  
  add_declaration( &StandardPackage::std_standard_decl->package_declarative_part );
  close_scope( StandardPackage::std_standard_decl );  
  close_scope( StandardPackage::std_decl );  
  make_visible( StandardPackage::std_standard_decl );
  StandardPackage::std_standard_decl->_make_interface_visible( this );
  add_declaration( IIR::_get_library_manager()->get_work_library() );
  
  debug_symbol_table = old_val;
}

void 
symbol_table::add_declaration( IIR_Declaration *decl_ptr ){
  ASSERT( decl_ptr != NULL );
  ASSERT( decl_ptr->_get_type() != IIRScram_Declaration::ERROR );
  //  ASSERT( decl_ptr->_is_visible() == TRUE );
  
  if (debug_symbol_table) {
    cerr << this << " - adding declaration : |" << *decl_ptr << "|" << decl_ptr << "|";
  }

  IIR *current_declarative_region =  declarative_region_stack.get_top_of_stack();

  if( current_declarative_region != NULL && decl_ptr->_get_declarative_region() == NULL ){
    decl_ptr->_set_declarative_region( current_declarative_region );
  
    if( debug_symbol_table ){
      cerr << " - setting declarative region |";
      if( decl_ptr->_get_declarative_region() != NULL ){
	cerr << *decl_ptr->_get_declarative_region();
      }
      else{
	cerr << "NULL";
      }
      cerr << "|.";
    }
  }

  if( debug_symbol_table ){
    cerr << endl;
  }

  // Add the declaration to the list that are currently in scope...
  in_scope_list.append(decl_ptr);

  // Add the declaration to the list of its type that are in scope...
  in_scope_by_type[ decl_ptr->_get_type() ].append( decl_ptr );

  // Now, we need to add the declaration to the currently open scope.
  get_current_scope()->add_declaration( decl_ptr );

  visible_symbols.lookup_add( decl_ptr );
  decl_ptr->_come_into_scope( this );
  
  if( decl_ptr->_is_iir_library_unit() == TRUE ){
    IIR::_get_library_manager()->add_declaration( (IIR_LibraryUnit *)decl_ptr );
  }
}

void 
symbol_table::add_declaration(IIR_DeclarationList *list_ptr) {
  ASSERT( list_ptr != NULL );
  IIR_Declaration *current_decl = list_ptr->first();
  while( current_decl != NULL ){
    add_declaration( current_decl );
    current_decl = list_ptr->successor( current_decl );
  }
}

void 
symbol_table::add_declaration( set<IIR_Declaration> *set_ptr ){
  ASSERT( set_ptr != NULL );

  IIR_Declaration *current_decl = set_ptr->get_element();
  while( current_decl != NULL ){
    add_declaration( current_decl );
    current_decl = set_ptr->get_next_element();
  }
}

void 
symbol_table::remove_from_visibility( IIR_Declaration *decl_ptr ){
  ASSERT( decl_ptr != NULL );
  ASSERT( decl_ptr->_get_type() != IIRScram_Declaration::ERROR );
  
  if( debug_symbol_table == true ){
    cerr << this << " - Removing from visibility - |" << *decl_ptr << "| - " << decl_ptr << endl;
  }

  visible_symbols.lookup_remove( decl_ptr );
}

void 
symbol_table::remove_from_scope(IIR_Declaration *decl_ptr){
  ASSERT( decl_ptr != NULL );
  ASSERT( decl_ptr->_get_type() != IIRScram_Declaration::ERROR );

  if( debug_symbol_table == true ){
    cerr << this << " - Removing from scope - |" << *decl_ptr << "| - " << decl_ptr << endl;
  }
  
  remove_from_visibility( decl_ptr );
  decl_ptr->_come_out_of_scope( this );
  
  in_scope_list.remove(decl_ptr);
  in_scope_by_type[ decl_ptr->_get_type() ].remove( decl_ptr );
  
  if( use_clause_entries.in_set( decl_ptr ) ){
    use_clause_entries.remove( decl_ptr );
  }
}

 
void 
symbol_table::hide_declaration( IIR_Declaration *to_hide ){
  remove_from_visibility( to_hide );
}



bool
symbol_table::in_scope(IIR_Declaration *decl) {
  ASSERT( decl != NULL);
  ASSERT( decl->get_declarator() != NULL);

  return in_scope_by_type[ decl->_get_type() ].in_list( decl );
}

bool
symbol_table::is_visible( IIR_Declaration *decl ){
  ASSERT( decl != NULL);
  ASSERT( decl->get_declarator() != NULL);

  // First look in the visibility list.
  set<IIR_Declaration> *same_name_list = find_set( decl->get_declarator() );
  ASSERT( same_name_list != NULL );

  return same_name_list->in_set( decl );
}

void 
symbol_table::open_scope( IIR *declarative_region ){
  scope_entry *new_scope =  current_scope->open_scope( declarative_region );
  current_scope = new_scope;

  if( declarative_region != NULL ){
    declarative_region_stack.push( declarative_region );
  }

  if (debug_symbol_table) {
    cerr << this << "- opening new scope ";
    if( declarative_region != NULL ){
      cerr << " - declarative region |" << *declarative_region << "|" << endl;
    }
    else{
      cerr << endl;
    }
  }  
}

void 
symbol_table::close_scope( IIR *declarative_region ){
  if( debug_symbol_table ) {
    cerr << this << " - closing scope";
  }
  
  IIR *decl_region = declarative_region_stack.pop();  
  ASSERT( decl_region == declarative_region );

  if( debug_symbol_table == TRUE ){
    if( decl_region != NULL ){
      cerr << " - popping declarative region |" << *decl_region << "|." << endl;
    }
    else{
      cerr << endl;
    }
  }

  // Because we're assured that _this_ is the deepest scope open, we know
  // we only need to pull out the declarations in this scope.  We can make some
  // assertions to be sure, however.
  ASSERT( current_scope != NULL );
  ASSERT( current_scope->get_owner() == decl_region );
  ASSERT( current_scope->get_scopes()->get_element() == NULL || 
	  current_scope->get_scopes()->get_element()->is_closed() == TRUE );
  
  IIR_Declaration *current = current_scope->get_declarations()->pop();
  while( current != NULL ){
    if( current->_is_incomplete_type_declaration() == TRUE ){
      ASSERT( current->_is_iir_type_declaration() == TRUE );
      if( ((IIR_TypeDeclaration *)current)->_get_fully_defined_type() == NULL ){
	ostrstream err;
	err << "Type |" << *current << "| was defined as an incomplete type, but "
	    << "no full definition of it was found in this scope." << ends;
	report_error( current, err );
      }
    }
    remove_from_scope( current );
    current = current_scope->get_declarations()->pop();
  }
  
  current_scope->close_scope();
  current_scope = current_scope->get_previous_scope();
}

void 
symbol_table::reopen_scope( IIR *declarative_region ){
//   bool found_scope_marker = false;
//   IIR_Declaration *last = NULL;
  
//   // Walk the out of scope list backwards until we hit the first (in the
//   // list) declaration that came before this scope.  (I.e.  walk over all
//   // of the declarations in our scope.  When we've found _that_, then we
//   // need to walk forward again until we find the first declaration with
//   // that same scope as it's declarative region.  We have to do it this way
//   // since multiple scopes might have been opened/closed while the scope
//   // we're reopening was open.
//   IIR_Declaration *current = out_of_scope_list.last();
//   while( current != NULL ){
//     if( found_scope_marker == true ){
//       if( current->_get_declarative_region() != declarative_region ){
// 	// Move back up one.
// 	current = outsuccessor( current );
// 	break;
//       }
//     }
//     else{
//       if( current->_get_declarative_region() == declarative_region ){
// 	found_scope_marker = true;
//       }
//     }
//     current = out_of_scope_list.predecessor( current );
//   }

//   // So, we've walked backwards, and found the first declaration entered in
//   // this declarative region.  Now, we need to walk forwards until we found

  
//   // If current is NULL, something bad happened.  Either the symbol table
//   // mangled declarative regions, or someone asked to to reopen a
//   // non-existant scope.
//   ASSERT( current != NULL );

//   // So, "current" holds the _first_ declaration that we need to readd, and
//   // "last" holds the last.
//   while( current != last ){
//     out_of_scope_list.remove( current );
//     add_declaration( current );
//     current = out_of_scope_list.successor( current );
//     ASSERT( current != NULL );
//   }
//   // Add the last one.
//   out_of_scope_list.remove( current );
//   add_declaration( current );

}

IIR *
symbol_table::get_current_declarative_region(){
  return declarative_region_stack.get_top_of_stack();
}

void 
symbol_table::add_undefined( IIR_Char *name ){
  //	IIR_Declaration *undef_decl = new IIR_UndefinedDeclaration;
  //	undef_decl->name = new IIR_Identifier(); // : text_string(name);
  //	((IIR_Identifier *)undef_decl->name)->text_string = name;
  //	add_declaration( undef_decl );
}

void 
symbol_table::make_visible( IIR_Declaration *decl_ptr ){
  if( decl_ptr->_get_attribute_name() != NULL ){
    return;
  }

  if( debug_symbol_table ){
    cerr << this << " - making visible - " << *decl_ptr << "|" << decl_ptr << "|" << endl;
  }

  if( decl_ptr->get_kind() == IIR_USE_CLAUSE ){
    decl_ptr->_make_interface_visible( this );
  }
  
  // Since we're maintaining an out_of_scope list (temporarily), we 
  // need to take this declaration out of that list and put it back
  // in scope.
  if( decl_ptr != NULL && decl_ptr->_is_specification() == FALSE ){

    // First we'll make sure it's not already visible...
    if( is_visible( decl_ptr ) == true ){
      if( debug_symbol_table == true ){
	cerr << this << "- it was already visible - returning!" << endl;
      }
      return;
    }

//     if( in_scope( decl_ptr ) == false ){
//       // Let's grab the out of scope symbol out of the list.
//       out_of_scope_list.remove( decl_ptr );
//       out_of_scope_by_type[ decl_ptr->_get_type() ].remove( decl_ptr );
//     }
    
    add_declaration( decl_ptr );


    if (decl_ptr->get_kind() == IIR_TYPE_DECLARATION && decl_ptr->_is_enumeration_type() == TRUE ){
      IIR_EnumerationTypeDefinition *enum_type;
      IIR_EnumerationLiteral *lit;
      enum_type = (IIR_EnumerationTypeDefinition *)((IIR_TypeDeclaration*)decl_ptr)->get_type();
      for (lit = enum_type->enumeration_literals.first();
	   lit != NULL;
	   lit = enum_type->enumeration_literals.successor(lit)) {
	make_visible(lit);
      }
    }


    if( decl_ptr->get_kind() == IIR_TYPE_DECLARATION && decl_ptr->_is_physical_type() ){
      IIR_PhysicalTypeDefinition *physical_type =
	(IIR_PhysicalTypeDefinition *)decl_ptr->_get_subtype();
      if( physical_type->get_primary_unit() != NULL ){
	make_visible( physical_type->get_primary_unit() );
      }
      make_visible( (IIR_DeclarationList *)&physical_type->units );
    }
  }
}

void 
symbol_table::make_visible( IIR_DeclarationList *decl_list ){
  if( decl_list != NULL ){
    IIR_Declaration *current = decl_list->first();
    while( current != NULL ){
      make_visible( current );
      current = decl_list->successor( current );
    }
  }
}

void 
symbol_table::make_visible( set<IIR_Declaration> *decl_set ){
  if( decl_set != NULL ){
    IIR_Declaration *current = decl_set->get_element();
    while( current != NULL ){
      make_visible( current );
      current = decl_set->get_next_element();
    }
  }
}


void 
symbol_table::add_to_use_list( IIR_Declaration *declaration ){
  ASSERT( use_clause_entries.in_set( declaration ) == FALSE );
  use_clause_entries.add( declaration );
}

bool 
symbol_table::in_use_list( IIR_Declaration *to_find ){
  return use_clause_entries.in_set( to_find );
}

void 
symbol_table::incomplete_type_fixup( IIR_TypeDeclaration *satisfies_incomplete_type ){
  ASSERT( satisfies_incomplete_type != NULL );
  ASSERT( satisfies_incomplete_type->get_type() != NULL );
  
  bool got_one = false;
  IIR_TypeDeclaration *current_incomplete_type_declaration = incomplete_types.get_element();
  while( current_incomplete_type_declaration != NULL ){
    ASSERT( current_incomplete_type_declaration->get_type() == NULL );
    if( IIR_TextLiteral::_cmp( satisfies_incomplete_type->get_declarator(),
			       current_incomplete_type_declaration->get_declarator() ) == 0 ){
      current_incomplete_type_declaration->_set_fully_defined_type( satisfies_incomplete_type->get_type() );
      incomplete_types.remove( current_incomplete_type_declaration );
      got_one = true;
      break;
    }
    current_incomplete_type_declaration = incomplete_types.get_next_element();
  }
  
  if( got_one == true ){
    // This new type declaration satisifies an incomplete type!  Now we
    // need to go through the list of access types that were designating
    // this incomplete type, and point them at this new one.  When we're
    // done, we can remove this incomplete type from our list.
    IIR_TypeDeclaration *current_access_type_declaration;
    current_access_type_declaration = designates_incomplete_type.get_element();
    while( current_access_type_declaration != NULL ){
      ASSERT(current_access_type_declaration->_designates_incomplete_type() == TRUE );
      ASSERT(current_access_type_declaration->get_type()->_is_iir_access_type_definition()==TRUE);
      
      IIR_AccessTypeDefinition *access_type;
      access_type = (IIR_AccessTypeDefinition *)current_access_type_declaration->get_type();
      ASSERT( access_type->get_designated_type()->_is_incomplete_type_definition() == TRUE );
      
      IIRScram_IncompleteTypeDefinition *current_type_definition;
      current_type_definition =
	(IIRScram_IncompleteTypeDefinition *)access_type->get_designated_type();      
      if( IIR_TextLiteral::_cmp( satisfies_incomplete_type->get_declarator(), current_type_definition->_get_designated_type_name() ) == 0 ){
	access_type->set_designated_type( satisfies_incomplete_type->get_type() );	
      }
      current_access_type_declaration = designates_incomplete_type.get_next_element();
    }
  }
}

