/*
 *  Linux snipes, a text-based maze-oriented game for linux.
 *  Copyright (C) 1997 Jeremy Boulton.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Jeremy Boulton is reachable via electronic mail at
 *  boultonj@ugcs.caltech.edu.
 */


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/time.h>

#if 0
#include <pthread.h>
#endif

#ifdef USE_CURSES
#include <linux/kd.h>		/* linux specific keyboard ioctls */
#include <termios.h>
#include <curses.h>
#endif

#ifdef USE_SVGALIB
#include <vga.h>
#endif

#ifdef USE_XWIN
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include "screen.h"
#endif


#include "snipes.h"
#include "keys.h"

/* Interface */

enum KbdAction {
  MOVE_ACTION,
  QUIT_ACTION,
  SUSPEND_ACTION
};

enum KbdMode {
  KMODE_NORMAL,
  KMODE_ACTION
};

struct kbd_action {
  signed int fx, fy;		/* -1, 0, 1 for left/up, nothing, right/down */
  signed int mx, my;
  char accel;
  enum KbdAction action;
};

int init_kbd( enum DisplayType disptype, int use_raw_mode );
int close_kbd( void );
void emergency_close_kbd( void );
void kbd_setmode( enum KbdMode km );
int kbd_getch( void );
int get_kbd_action( struct kbd_action *ka );
void wait_kbd_action( void );


/* Private */

#define	KEY_CTRL_C	0x03

int terminal_open( void );
void terminal_close( void );
int terminal_set_noncanonical_mode( void );
void terminal_set_nonblocking_input( void );
void terminal_set_blocking_input( void );
void terminal_restore_canonical_mode( void );
int terminal_set_raw_kbd_mode( void );
void terminal_restore_kbd_mode( void );


#define FIRE_UP		0
#define FIRE_DOWN	1
#define	FIRE_LEFT	2
#define	FIRE_RIGHT	3
#define MOVE_UP		4
#define MOVE_DOWN	5
#define	MOVE_LEFT	6
#define	MOVE_RIGHT	7
#define	ACCEL_KEY	8
#define	CTRL_KEY	9
#define	C_KEY		10
#define	Z_KEY		11

#define KEY_WATCH_COUNT	12


#define END_OF_TABLE	-1

struct keytable {
  int code;
  int index;
  char pressed;
} keymap[]={
  {KEYBOARD_KEY_DOWN_A,		FIRE_LEFT,	1},	/* a */
  {KEYBOARD_KEY_UP_A,		FIRE_LEFT,	0},
  {KEYBOARD_KEY_DOWN_S,		FIRE_DOWN,	1},	/* s */
  {KEYBOARD_KEY_UP_S,		FIRE_DOWN,	0},
  {KEYBOARD_KEY_DOWN_D,		FIRE_RIGHT,	1},	/* d */
  {KEYBOARD_KEY_UP_D,		FIRE_RIGHT,	0},
  {KEYBOARD_KEY_DOWN_W,		FIRE_UP,	1},	/* w */
  {KEYBOARD_KEY_UP_W,		FIRE_UP,	0},
  {KEYBOARD_KEY_DOWN_ARR_U,	MOVE_UP,	1},	/* up */
  {KEYBOARD_KEY_UP_ARR_U,	MOVE_UP,	0},
  {KEYBOARD_KEY_DOWN_ARR_L,	MOVE_LEFT,	1},	/* left */
  {KEYBOARD_KEY_UP_ARR_L,	MOVE_LEFT,	0},
  {KEYBOARD_KEY_DOWN_ARR_D,	MOVE_DOWN,	1},	/* down */
  {KEYBOARD_KEY_UP_ARR_D,	MOVE_DOWN,	0},
  {KEYBOARD_KEY_DOWN_ARR_R,	MOVE_RIGHT,	1},	/* right */
  {KEYBOARD_KEY_UP_ARR_R,	MOVE_RIGHT,	0},
  {KEYBOARD_KEY_DOWN_SPACE,	ACCEL_KEY,	1},	/* space */
  {KEYBOARD_KEY_UP_SPACE,	ACCEL_KEY,	0},
  {KEYBOARD_KEY_DOWN_CTRL,	CTRL_KEY,	1},	/* ctrl */
  {KEYBOARD_KEY_UP_CTRL,	CTRL_KEY,	0},
  {KEYBOARD_KEY_DOWN_C,		C_KEY,		1},	/* c */
  {KEYBOARD_KEY_UP_C,		C_KEY,		0},
  {KEYBOARD_KEY_DOWN_Z,		Z_KEY,		1},	/* z */
  {KEYBOARD_KEY_UP_Z,		Z_KEY,		0},
  {END_OF_TABLE, 0, 0}					/* end of table */
};


char keylist[KEY_WATCH_COUNT];	/* array of key states (up/down) */

int kbd_thread_stop;		/* signal from main program to abort thread */
int quit;			/* user requested to quit the program */
int suspend;			/* user requested to suspend/resume program */
int handled_suspend;		/* main program has been informed of suspend */
int suspended;

#if 0
#define KBD_BUFSIZE	40

int kbd_in_play_mode;
int kbd_buffer[KBD_BUFSIZE];	/* Store keys in here when not playing the
				   game so we can use the keyboard normally. */
int kbd_buf_head, kbd_buf_tail;

pthread_t *kbd_thread;
#endif


enum KbdMode curr_km;


int (*kbd_ptr_close)( void ) = NULL;
void (*kbd_ptr_emergency_close)( void ) = NULL;
int (*kbd_ptr_getch)( void ) = NULL;
void (*kbd_ptr_playmode)( enum KbdMode ) = NULL;
int (*kbd_ptr_get_action)( struct kbd_action *ka ) = NULL;
void (*kbd_ptr_get_action_from_keylist_helper)( void ) = NULL;



#if 0

/* Why is this thread here?  Because getchar() is normally blocking
 * and I used to think I couldn't change that.  Therefore, I put
 * it in a separate thread.  As it turns out, you can make getchar()
 * nonblocking, so I'm trying to get by without this thread.  We
 * shall see.
 */

void *keyboard_thread( void *p )
{
  int i, ch;
  
  while(1) {
    if( kbd_thread_stop )
      return NULL;

    ch=getchar();
    
    if( kbd_in_play_mode ) {
      i=0;
      while( keymap[i].code != END_OF_TABLE ) {
	if( ch == keymap[i].code ) {
	  keylist[keymap[i].index]=keymap[i].pressed;
	  break;
	}
	i++;
      }
      
      /* CTRL-Z: Pause game and restore normal kbd functionality */
      if( keylist[CTRL_KEY] && ch == KEYBOARD_KEY_DOWN_Z ) {
	suspend = 1;

	terminal_restore_kbd_mode();
	while( getchar() != '\r' )
	  ;
	terminal_set_raw_kbd_mode();
	suspend = 2;

	continue;
      }
      
      /* CTRL-C: Quit */
      if( keylist[CTRL_KEY] && ch == KEYBOARD_KEY_DOWN_C ) {
	quit = 1;
	return NULL;
      }
    }
    else {
      if( ch < 128 ) {		/* key pressed */
	/* FIXME: use mutex. */
	if( (kbd_buf_head+1)%KBD_BUFSIZE != kbd_buf_tail ) {
	  kbd_buf_head = (kbd_buf_head+1)%KBD_BUFSIZE;
	  kbd_buffer[kbd_buf_head] = ch;
	}
      }
    }
  }
}
#endif


void kbd_setmode_no_op( enum KbdMode km )
{
}

/* get_kbd_action_from_keylist( struct kbd_action *ka )
 * 
 * Fills in ka if any interesting keyboard action has occurred.
 * Returns 1 if an action has occurred, else 0.
 * 
 * First calls the function pointed to by
 * kbd_ptr_get_action_from_keylist_helper() to update the keylist[] 
 * array with the state of the keys.
 */

int get_kbd_action_from_keylist( struct kbd_action *ka )
{
  ka->action=MOVE_ACTION;
  ka->accel = 0;
  ka->fx = ka->fy = ka->mx = ka->my = 0;

  (*kbd_ptr_get_action_from_keylist_helper)();

  if( keylist[CTRL_KEY] && keylist[C_KEY] ) {
    ka->action=QUIT_ACTION;
    return 1;
  }
  if( keylist[CTRL_KEY] && keylist[Z_KEY] ) {
    ka->action=SUSPEND_ACTION;
    return 1;
  }
  if( keylist[MOVE_UP]   || keylist[MOVE_DOWN] ||
      keylist[MOVE_LEFT] || keylist[MOVE_RIGHT] ||
      keylist[FIRE_UP]   || keylist[FIRE_DOWN] ||
      keylist[FIRE_LEFT] || keylist[FIRE_RIGHT] ) {
    if( keylist[ACCEL_KEY] )  ka->accel=1;
    if( keylist[MOVE_UP] )	ka->my--;
    if( keylist[MOVE_DOWN] )	ka->my++;
    if( keylist[MOVE_LEFT] )	ka->mx--;
    if( keylist[MOVE_RIGHT] )	ka->mx++;
    if( keylist[FIRE_UP] )	ka->fy--;
    if( keylist[FIRE_DOWN] )	ka->fy++;
    if( keylist[FIRE_LEFT] )	ka->fx--;
    if( keylist[FIRE_RIGHT] )	ka->fx++;
    return 1;
  }
  return 0;
}



#if defined(USE_CURSES) || defined(DOUBLEWIDE_FONT_HACK)
void kbd_console_process_all_actions( void )
{
  int i, ch;
  while( (ch = getchar()) != EOF && ch != 0 ) {
    i=0;
    while( keymap[i].code != END_OF_TABLE ) {
      if( ch == keymap[i].code ) {
	keylist[keymap[i].index]=keymap[i].pressed;
	break;
      }
      i++;
    }
  }
}


int non_raw_mode_get_kbd_action( struct kbd_action *ka )
{
  int key;
  int retval=0;
  
  ka->accel = 0;

  while( (key = getchar()) != EOF && key != 0 ) {
#if 0
    if( suspended ) {
      suspended = 0;
      ka->action=RESUME_ACTION;
      return 1;
    }
#endif
    switch(key) {
      case 'A':
      case 'a':
	ka->fx=-1;
	retval=1;
	break;
      case 'S':
      case 's':
	ka->fy=1;
	retval=1;
	break;
      case 'D':
      case 'd':
	ka->fx=1;
	retval=1;
	break;
      case 'W':
      case 'w':
	ka->fy=-1;
	retval=1;
	break;
      case KEY_LEFT:
	ka->mx=-1;
	retval=1;
	break;
      case KEY_DOWN:
	ka->my=1;
	retval=1;
	break;
      case KEY_RIGHT:
	ka->mx=1;
	retval=1;
	break;
      case KEY_UP:
	ka->my=-1;
	retval=1;
	break;
      case 'q':
      case KEY_CTRL_C:
	ka->action=QUIT_ACTION;
	retval=1;
	break;
      case KEY_SUSPEND:
	suspended = 1;
	ka->action=SUSPEND_ACTION;
	retval=1;
	break;
    }
  }
  return retval;
}


void kbd_raw_curses_setmode( enum KbdMode km )
{
  int i;
  if( km == KMODE_NORMAL ) {
    terminal_restore_kbd_mode();
    terminal_set_blocking_input();
  }
  else {
    for( i=0; i<KEY_WATCH_COUNT; i++ )
      keylist[i]=0;
    terminal_set_nonblocking_input();
    terminal_set_raw_kbd_mode();
  }
}


int kbd_threaded_raw_curses_getch( void )
{
  return getch();
}


int kbd_threadless_cooked_curses_getch( void )
{
  return getch();
}


int kbd_threaded_raw_curses_close( void )
{
#if 0
  void *status;

  kbd_thread_stop = 1;
  pthread_join( *kbd_thread, &status );
  free( kbd_thread );
#endif
  terminal_restore_kbd_mode();
  terminal_restore_canonical_mode();
  terminal_close();
  
  return 1;
}


int kbd_threadless_cooked_curses_close( void )
{
  terminal_restore_canonical_mode();
  
  return 1;
}


/* kbd_threaded_raw_curses_emergency_close( void )
 * 
 * This is called from a signal handler so that we can try to set
 * the keyboard back to cooked mode but ioctl() is probably not
 * re-entrant (it's not listed in Figure 10.3 of the 1993 edition
 * of Stevens' APUE) so it may not always work.  It's worth a shot,
 * though.
 */

void kbd_threaded_raw_curses_emergency_close( void )
{
  terminal_restore_kbd_mode();
}


void kbd_threadless_cooked_curses_emergency_close( void )
{
}


int kbd_threadless_cooked_curses_init( void )
{
  if( !terminal_set_noncanonical_mode() )
    return 0;

  nodelay( stdscr, TRUE );
#if 0
  /* curses: Pass through things like Ctrl-Z. */
  /* Note: not the same as keyboard raw mode. */
  raw();
#endif
  suspended = 0;

  kbd_ptr_close = kbd_threadless_cooked_curses_close;
  kbd_ptr_emergency_close = kbd_threadless_cooked_curses_emergency_close;
  kbd_ptr_playmode = kbd_setmode_no_op;
  kbd_ptr_getch = kbd_threadless_cooked_curses_getch;
  kbd_ptr_get_action = non_raw_mode_get_kbd_action;
  kbd_ptr_get_action_from_keylist_helper = NULL;
  
  return 1;
}


int kbd_threaded_raw_curses_init( void )
{
  int i;
  
  if( terminal_open() )
    if( terminal_set_noncanonical_mode() ) {
      if( terminal_set_raw_kbd_mode() ) {

	/* Initialize variables used to communicate with keyboard
	 * thread.
	 */
	for( i=0; i<KEY_WATCH_COUNT; i++ )
	  keylist[i]=0;
	kbd_thread_stop = 0;
	quit=0;
	suspend=0;
	handled_suspend=0;

#if 0
	/* Launch the keyboard thread. */
	kbd_thread = (pthread_t *) malloc( sizeof(pthread_t) );
	pthread_create( kbd_thread, NULL, keyboard_thread, NULL );
#endif

	kbd_ptr_close = kbd_threaded_raw_curses_close;
	kbd_ptr_emergency_close = kbd_threaded_raw_curses_emergency_close;
	kbd_ptr_playmode = kbd_raw_curses_setmode;
	kbd_ptr_getch = kbd_threaded_raw_curses_getch;
	kbd_ptr_get_action = get_kbd_action_from_keylist;
	kbd_ptr_get_action_from_keylist_helper = kbd_console_process_all_actions;
	
	return 1;
      }
      else {
	terminal_restore_canonical_mode();
	terminal_close();
      }
    }
    else {
      terminal_close();
    }

  return 0;
}
#endif


#ifdef USE_SVGALIB
int kbd_svgalib_getch( void )
{
  return vga_getkey();
}

void kbd_svgalib_close( void )
{
}

void kbd_svgalib_emergency_close( void )
{
}

int kbd_svgalib_init( void )
{
  suspended = 0;
  kbd_ptr_close = kbd_svgalib_close;
  kbd_ptr_emergency_close = kbd_svgalib_emergency_close;
  kbd_ptr_playmode = kbd_setmode_no_op;
  kbd_ptr_getch = kbd_svgalib_getch;
  return 1;
}
#endif


#ifdef USE_XWIN
extern Display *disp;

void kbd_X_process_all_actions( void )
{
  XEvent report;
  char buffer[40];
  int bufsize = 40;
  KeySym keysym;
  int stat;

  while( XPending(disp) != 0 ) {
    stat = 2;
    XNextEvent( disp, &report );
    switch( report.type )
    {
      case Expose:
	set_okay_to_draw_X();
	break;
      case KeyPress:
	stat = 1;
	XLookupString( &report.xkey, buffer, bufsize, &keysym, NULL );
	break;
      case KeyRelease:
	stat = 0;
	XLookupString( &report.xkey, buffer, bufsize, &keysym, NULL );
	break;
    }
    if( stat != 2 )
    {
      switch( keysym )
      {
	case XK_w:
	  keylist[FIRE_UP] = stat;
	  break;
	case XK_s:
	  keylist[FIRE_DOWN] = stat;
	  break;
	case XK_a:
	  keylist[FIRE_LEFT] = stat;
	  break;
	case XK_d:
	  keylist[FIRE_RIGHT] = stat;
	  break;
	case XK_Up:
	  keylist[MOVE_UP] = stat;
	  break;
	case XK_Down:
	  keylist[MOVE_DOWN] = stat;
	  break;
	case XK_Left:
	  keylist[MOVE_LEFT] = stat;
	  break;
	case XK_Right:
	  keylist[MOVE_RIGHT] = stat;
	  break;
	case XK_space:
	  keylist[ACCEL_KEY] = stat;
	  break;
	case XK_Control_L:
	case XK_Control_R:
	  keylist[CTRL_KEY] = stat;
	  break;
	case XK_c:
	  keylist[C_KEY] = stat;
	  break;
	case XK_z:
	  keylist[Z_KEY] = stat;
	  break;
      }
    }
  }
}


int kbd_X_close( void )
{
  return 1;
}


void kbd_X_emergency_close( void )
{
}


void kbd_X_setmode( enum KbdMode km )
{
  int i;
  if( km == KMODE_ACTION )
    for( i=0; i<KEY_WATCH_COUNT; i++ )
      keylist[i]=0;
}


int kbd_X_getch( void )
{
  XEvent report;
  char buffer[40];
  int bufsize = 40;
  KeySym keysym;

  while(1)
  {
    XNextEvent( disp, &report );
    if( report.type == KeyPress ) {
      XLookupString( &report.xkey, buffer, bufsize, &keysym, NULL );
      if( keysym == XK_Return )
	return (int) '\r';
      if( strlen(buffer) == 1 )
	return (int) buffer[0];
    }
  }
}


int kbd_X_init( void )
{
  kbd_ptr_close = kbd_X_close;
  kbd_ptr_emergency_close = kbd_X_emergency_close;
  kbd_ptr_playmode = kbd_X_setmode;
  kbd_ptr_getch = kbd_X_getch;
  kbd_ptr_get_action = get_kbd_action_from_keylist;
  kbd_ptr_get_action_from_keylist_helper = kbd_X_process_all_actions;
  return 1;
}
#endif


int init_kbd( enum DisplayType disptype, int use_raw_mode )
{
#if 0
  kbd_in_play_mode = 0;
  kbd_buf_head = kbd_buf_tail = 0;
#endif
  
  switch( disptype ) {
#if defined(USE_CURSES) || defined(DOUBLEWIDE_FONT_HACK)
    case DT_CURSES:
    case DT_CURSES_DOUBLEWIDE:
      if( use_raw_mode && kbd_threaded_raw_curses_init() )
	return 1;
      return kbd_threadless_cooked_curses_init();
      break;
#endif
#ifdef USE_SVGALIB
    case DT_SVGALIB:
      return kbd_svgalib_init();
      break;
#endif
#ifdef USE_XWIN
    case DT_XWIN:
      return kbd_X_init();
      break;
#endif
    default:
      return 0;		/* No support for that display type */
  }

  return 0;
}


/* void emergency_close_kbd( void )
 * 
 * This is called from a signal handler and does its best to set
 * things right before the program gives up and exits.
 */

void emergency_close_kbd( void )
{
  (*kbd_ptr_emergency_close)();
}


int close_kbd( void )
{
  return (*kbd_ptr_close)();
}


int get_kbd_action( struct kbd_action *ka )
{
  return (*kbd_ptr_get_action)( ka );
}


void kbd_setmode( enum KbdMode km )
{
  (*kbd_ptr_playmode)( km );
  curr_km = km;
}


/* kbd_getch( void )
 * 
 * This is a blocking call.  It has the side-effect of setting the
 * play mode to KMODE_NORMAL if that is not already the case.
 * 
 * Assumed: getchar() must be blocking in KMODE_NORMAL.
 */

int kbd_getch( void )
{
  if( curr_km != KMODE_NORMAL )
    (*kbd_ptr_playmode)( KMODE_NORMAL );

  return (*kbd_ptr_getch)();
}


/* wait_kbd_action( struct kbd_action *ka )
 * 
 * This function does not return until a key is pressed.
 * 
 * Side effects: the keyboard mode will be the same when the
 * function terminates, but it may be changed in the mean time,
 * so depending on what happens when the keyboard mode changes,
 * information may be lost about the current state of keys.
 * 
 * Assumed: getchar() must be blocking in KMODE_NORMAL.
 */

void wait_kbd_action( void )
{
  enum KbdMode km_temp;

  if( (km_temp = curr_km) != KMODE_NORMAL ) {
    kbd_setmode( KMODE_NORMAL );

    kbd_getch();

    kbd_setmode( km_temp );
  }
  else
    kbd_getch();
}


#ifdef USE_CURSES

int have_saved_kbd_mode=0;		/* boolean */
long save_kbd_mode;			/* initial keyboard mode */

int terminal_is_open=0;			/* boolean */
int console;				/* console file handle */

int have_saved_termios_struct=0;	/* boolean */
struct termios savet, newt;		/* The terminal I/O structure */


int terminal_open( void )
{
  if( terminal_is_open )
    return 1;

  if( (console=open("/dev/tty", O_RDONLY)) != -1 ) {
    if( isatty(console) ) {
      terminal_is_open = 1;
      return 1;
    }
    else
      close( console );
  }

  return 0;
}


void terminal_close( void )
{
  if( terminal_is_open ) {
    close( console );
    terminal_is_open = 0;
  }
}


int terminal_set_noncanonical_mode( void )
{
  struct termios temp_t;

  /* POSIX.1 defines tc[gs]etattr() as the functions to use here.
   * For reference, the previous implementation was:
   *    ioctl(0, TCGETA, &savet);
   */

  if( !have_saved_termios_struct ) {
    tcgetattr( 0, &savet );
    have_saved_termios_struct = 1;
  }
  
  /* Except for the settings that will be changed in the next couple
   * lines of code, leave the terminal settings the same as they
   * were.
   */

  newt = savet;
  
  /* Disable the following input options:
   *   - strip off high bit
   *   - map newlines to carriage returns
   *   - ignore carriage returns
   *   - map carriage returns to newlines
   *   - map uppercase to lowercase (SVR4-ism)
   *   - start/stop output flow control (^S/^Q processing)
   */

  newt.c_iflag &= ~(ISTRIP | INLCR | IGNCR | ICRNL | IUCLC | IXON );

#if 0
  /* Disable the following output option:
   *   - All special output processing
   */

  newt.c_oflag &= ~OPOST;
#endif

  /* Disable the following control options:
   *   - character echo
   *   - "canonical input": (line buffering?)
   *   - extended input processing
   *   - terminal-generated signals (^C,^Z)
   *   - "canonical upper/lower presentation" ("uppercase \ hack")
   *     NOTE: the above is an SVR4-ism and may not be very important.
   */

  newt.c_lflag &= ~( ECHO | ICANON | IEXTEN | ISIG | XCASE );

  /* The following code makes read() on stdin nonblocking.
   * This is a change from when they keyboard had its own
   * thread.  At that time, the settings were:
   *    c_cc[VMIN]  = 1;
   *    c_cc[VTIME] = 0;
   */

  newt.c_cc[VMIN] = 0;		      /* non-blocking reads on stdin */
  newt.c_cc[VTIME] = 0;		      /* no input timeout */
  tcsetattr( 0, TCSANOW, &newt );

  tcgetattr( 0, &temp_t );
  
#if 0
  /* Make sure the call succeeded.  Can't depend on the return
   * value from tcsetattr(), so compare to see what actually
   * happened.
   */
  if( memcmp( &temp_t, &newt, sizeof(struct termios) ) != 0 )
    return 0;
#endif

  setbuf(stdin, (char *)NULL);        /* disable input buffering */

#if 0
  setbuf(stdout,(char *)NULL);	      /* disable output buffering */
#endif

  return 1;
}


void terminal_set_nonblocking_input( void )
{
  tcgetattr( 0, &newt );
  newt.c_cc[VMIN] = 0;		      /* non-blocking reads on stdin */
  newt.c_cc[VTIME] = 0;		      /* no input timeout */
  tcsetattr( 0, TCSANOW, &newt );
}


void terminal_set_blocking_input( void )
{
  tcgetattr( 0, &newt );
  newt.c_cc[VMIN] = 1;		      /* blocking reads on stdin */
  newt.c_cc[VTIME] = 0;		      /* no input timeout */
  tcsetattr( 0, TCSANOW, &newt );
}


void terminal_restore_canonical_mode( void )
{
  /* As mentioned earlier, by POSIX.1, tcsetattr() replaces the
   * following ioctl(), which is provided for reference:
   *   ioctl(0, TCSETA, &savet);
   */

  if( have_saved_termios_struct )
    tcsetattr( 0, TCSANOW, &savet );
}


/*
 * KDGKBMODE
 * 
 *   Gets current keyboard mode.  argp points to a long
 *   which is set to one of these:
 * 
 *   K_RAW         0x00
 *   K_XLATE       0x01
 *   K_MEDIUMRAW   0x02
 *   K_UNICODE     0x03
 * 
 * KDSKBMODE
 * 
 *   Sets current keyboard mode.  argp is a long equal
 *   to one of the above values.
 *
 */


int terminal_set_raw_kbd_mode( void )
{
  if( !have_saved_kbd_mode ) {
    if( ioctl( console, KDGKBMODE, &save_kbd_mode ) == -1 )
      return 0;
    else
      have_saved_kbd_mode = 1;
  }

  if( ioctl( console, KDSKBMODE, K_MEDIUMRAW ) == -1 )
    return 0;

  /* Success */
  return 1;
}


void terminal_restore_kbd_mode( void )
{
  ioctl( console, KDSKBMODE, save_kbd_mode );
}
#endif
