## vim:ts=4:et:nowrap
##
##---------------------------------------------------------------------------##
##
## PySol -- a Python Solitaire game
##
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
## Markus F.X.J. Oberhumer
## <markus.oberhumer@jk.uni-linz.ac.at>
## http://wildsau.idv.uni-linz.ac.at/mfx/pysol.html
##
##---------------------------------------------------------------------------##


# imports
import os, sys, string, types, Tkinter

# PySol imports
if __name__ == "__main__":                                          #bundle#
    import pysoltk                                                  #bundle#
from mfxutil import destruct                                        #bundle#
from actions import PysolToolbarActions                             #bundle#

# Toolkit imports
from tkconst import TK_DASH_PATCH                                   #bundle#
from tkwidget import MfxTooltip                                     #bundle#


# /***********************************************************************
# // A canvas that supports hiding.
# //
# // Note: Applications should call show/hide after constructor.
# ************************************************************************/

class _MfxToolbar:
    def __init__(self, top, relief=0):
        self.top = top
        self.button_bg = None
        self.__setRelief(relief)
        self.side = -1
        self._tooltips = []
        self._widgets = []
        self._icon_height = 0
        # We must create a Canvas and a Frame with these parameters
        # in order to be able to hide the toolbar.
        self.canvas = Tkinter.Canvas(self.top, bd=0, highlightthickness=0)
        self.frame = Tkinter.Frame(self.canvas, bd=0, highlightthickness=1)
        self._initFrame()
        self.canvas.pack(side=Tkinter.TOP, fill=Tkinter.X)

    def __setRelief(self, relief):
        if type(relief) == types.IntType:
            relief = (Tkinter.RAISED, Tkinter.FLAT)[relief]
        elif relief in (Tkinter.RAISED, Tkinter.FLAT):
            pass
        else:
            relief = Tkinter.FLAT
        self.button_relief = relief
        if relief == Tkinter.RAISED:
            self.separator_relief = Tkinter.FLAT
        else:
            self.separator_relief = Tkinter.RAISED
        return relief

    # subclass overrideable
    def _initFrame(self):
        self.frame.pack(side=Tkinter.TOP, fill=Tkinter.X)

    # util
    def _createSeparator(self, width=12, side=Tkinter.LEFT, relief=None):
        if relief is None: relief = self.separator_relief
        if relief == Tkinter.FLAT or width <= 6:
            sep = Tkinter.Frame(self.frame, highlightthickness=0,
                                width=width, height=0, takefocus=0,
                                relief=relief)
            padx = 0
        else:
            height = max((self._icon_height or 38) - 4, 20)
            sep = Tkinter.Frame(self.frame, bd=1, highlightthickness=1,
                                width=4, height=height, takefocus=0,
                                relief=relief)
            padx = (width - 4) / 2
        sep.pack(side=side, padx=padx)
        self._widgets.append(sep)


    #
    # public methods
    #

    def show(self, side=1, resize=1):
        if self.side == side:
            return 0
        if resize:
            self.top.wm_geometry("")    # cancel user-specified geometry
        if not side:
            # hide
            if 0 and TK_DASH_PATCH:
                ### hmm, this doesn't work
                self.canvas.config(state="hidden")
            else:
                self.canvas.pack_propagate(0)
                self.canvas.config(height=0)
        else:
            # show
            if 0 and TK_DASH_PATCH:
                self.canvas.config(state="normal")
            else:
                self.canvas.pack_propagate(1)
            s = (None, Tkinter.TOP, Tkinter.BOTTOM)[side]
            self.canvas.pack(side=s, fill=Tkinter.X)
        self.side = side
        return 1

    def hide(self, resize=1):
        self.show(0, resize)

    def getSide(self):
        return self.side

    def destroy(self):
        for w in self._tooltips:
            if w: w.destroy()
        self._tooltips = []
        for w in self._widgets:
            if w: w.destroy()
        self._widgets = []

    def setCursor(self, cursor):
        if self.side:
            self.frame.config(cursor=cursor)
            self.frame.update_idletasks()


# /***********************************************************************
# //
# ************************************************************************/

class PysolToolbar(_MfxToolbar, PysolToolbarActions):
    def __init__(self, top, dir, relief=0):
        _MfxToolbar.__init__(self, top, relief)
        PysolToolbarActions.__init__(self)
        self.dir = dir
        self._createButton("new",     self.mNewGame, tooltip="New game")
        self._createButton("open",    self.mOpen   , tooltip="Open a \nsaved game")
        self._createSeparator()
        self._createButton("restart", self.mRestart, tooltip="Restart the \ncurrent game")
        self._createButton("save",    self.mSave,    tooltip="Save game")
        self._createSeparator()
        self._createButton("undo",    self.mUndo,    tooltip="Undo last move")
        self._createButton("redo",    self.mRedo,    tooltip="Redo last move")
        self._createButton("autodrop", self.mDrop,   tooltip="Auto drop cards")
        self._createSeparator()
        self._createButton("stats",   self.mStatus,  tooltip="View statistics")
        self._createButton("rules",   self.mHelpRules, tooltip="Rules for this game")
        self._createSeparator()
        self._createButton("quit",    self.mQuit,     tooltip="Quit PySol")
        self._createSeparator(width=20, relief=Tkinter.FLAT)
        ##self._createLabel("moves", padx=8, tooltip="Number of moves\nin this game")
        self._createLabel("player",  padx=8, tooltip="Change name of\ncurrent player")
        self.player_label.bind("<1>",self.mOptPlayerName)
        self.popup = None
        self.frame.bind("<2>", self.middleclickHandler)             #bundle#
        self.frame.bind("<Control-1>", self.middleclickHandler)     #bundle#
        self.frame.bind("<3>", self.rightclickHandler)

    # Change the look of the frame to match the platform look
    def _initFrame(self):
        if 0 or os.name == "nt":
            self.frame.config(bd=1, relief=Tkinter.SOLID)
            self.frame.pack(side=Tkinter.TOP, fill=Tkinter.X, ipady=4)
            self._createSeparator(width=4, side=Tkinter.LEFT, relief=Tkinter.FLAT)
            self._createSeparator(width=4, side=Tkinter.RIGHT, relief=Tkinter.FLAT)
        else:
            self.frame.pack(side=Tkinter.TOP, fill=Tkinter.X, ipady=1)

    def _createButton(self, name, command, padx=0, tooltip=None):
        file = os.path.join(self.dir, name+".gif")
        image = Tkinter.PhotoImage(file=file)
        if self._icon_height == 0:
            self._icon_height = image.height()
        button = Tkinter.Button(self.frame)
        button.config(command=command, image=image, takefocus=0,
                         relief=self.button_relief)
        if self.button_relief == Tkinter.FLAT:
            if self.button_bg is None:
                self.button_bg = self._getButtonBg(button["activebackground"])
                ##print self.button_bg
            button["activebackground"] = self.button_bg
        button.pack(side=Tkinter.LEFT, padx=padx)
        setattr(self, name + "_image", image)
        setattr(self, name + "_button", button)
        self._widgets.append(button)
        if tooltip:
            b = MfxTooltip(button)
            self._tooltips.append(b)
            b.setText(tooltip)

    def _getButtonBg(self, col):
        if type(col) != types.StringType or col[0] != "#" or len(col) != 7:
            return "#f0f0f0"
        c = "#"
        for i in (1, 3, 5):
            v = string.atoi(col[i:i+2], 16)
            v = int(v + 24)
            if (v > 255): v = 255
            c = c + ("%02x" % v)
        return c

    def _createLabel(self, name, padx=0, side=Tkinter.RIGHT, tooltip=None):
        label = Tkinter.Message(self.frame, relief="ridge", justify="center", aspect=300)
        label.pack(side=side, padx=padx)
        setattr(self, name + "_label", label)
        self._widgets.append(label)
        if tooltip:
            b = MfxTooltip(label)
            self._tooltips.append(b)
            b.setText(tooltip)

    def _busy(self):
        return not (self.side and self.game and not self.game.busy and self.menubar)


    #
    # public methods
    #

    def connectGame(self, game, menubar):
        PysolToolbarActions.connectGame(self, game, menubar)
        if self.popup:
            self.popup.destroy()
            destruct(self.popup)
            self.popup = None
        if menubar:
            tkopt = menubar.tkopt
            self.popup = Tkinter.Menu(self.canvas, tearoff=0)
            self.popup.add_command(label="Toolbar", state=Tkinter.DISABLED)
            self.popup.add_radiobutton(label="Hide", variable=tkopt.toolbar, value=0, command=menubar.mOptToolbar, underline=0)
            self.popup.add_radiobutton(label="Top", variable=tkopt.toolbar, value=1, command=menubar.mOptToolbar, underline=0)
            self.popup.add_radiobutton(label="Bottom", variable=tkopt.toolbar, value=2, command=menubar.mOptToolbar, underline=0)
            if 0:
                self.popup.add_separator()
                self.popup.add_radiobutton(label="Raised", variable=tkopt.toolbar_relief, value=0, command=menubar.mOptToolbarRelief, underline=0)
                self.popup.add_radiobutton(label="Flat", variable=tkopt.toolbar_relief, value=1, command=menubar.mOptToolbarRelief, underline=0)


    def setRelief(self, relief):
        pass    # FIXME


    def updateText(self, **kw):
        for name in kw.keys():
            label = getattr(self, name + "_label")
            label["text"] = str(kw[name])


    #
    # Mouse event handlers
    #

    def rightclickHandler(self, event):
        if self._busy(): return
        if self.popup:
            ##print event.x, event.y, event.x_root, event.y_root, event.__dict__
            self.popup.tk_popup(event.x_root, event.y_root)

    def middleclickHandler(self, event):
        if self._busy(): return
        if 1 <= self.side <= 2:
            self.menubar.setToolbarSide(3 - self.side)


# /***********************************************************************
# //
# ************************************************************************/

#%ifndef BUNDLE

class TestToolbar(PysolToolbar):
    def __init__(self, top, args):
        dir = "kde-large"
        dir = "gnome-large"
        if len(args) > 1: dir = args[1]
        dir = os.path.join(os.pardir, os.pardir, "data", "toolbar", dir)
        relief = 1
        relief = 0
        PysolToolbar.__init__(self, top, dir=dir, relief=relief)
        # test some settings
        self.updateText(player="Player\nPySol")
        self.undo_button["state"] = Tkinter.DISABLED
    def mQuit(self, *args):
        self.top.after_idle(self.top.quit)

def toolbar_main(args):
    tk = Tkinter.Tk()
    toolbar = TestToolbar(tk, args)
    toolbar.show()
    tk.mainloop()
    return 0

if __name__ == "__main__":
    sys.exit(toolbar_main(sys.argv))

#%endif

