/*****************************************************************************
 *                                                                           *
 * Program:   paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Uses:      GTK+ 1.2                                                       *
 * Modul:     filesel.c                                                      *
 *            common functions from paul and paulscan for fileselection      *
 * Author:    Andreas Tille                                                  *
 * Date:      14.06.1998                                                     *
 * Copyright: Andreas Tille, 1999; GNU Public License                        *
 *                                                                           *
 *****************************************************************************/

#include <stdlib.h>
#include <unistd.h>
#include "paul.h"

char *ChdirToLoadDir(char *filepath)
/* Change this directory where the file to load resides if neccessary
 * --- Parameter: ---
 * char *filepath        : file to load with complete path
 * --- Return: ---
 * char *ChdirToLoadDir(): plain filename of file to load
 */
{
   char *cwd, *file;
   
   if ( !(file = strrchr(filepath, '/')) ) return filepath;  /* unchanged */
   *(file++) = 0;
   cwd       = g_get_current_dir();

   if ( strcmp(cwd, filepath) ) {
      if ( chdir(filepath) ) 
         g_warning(_("chdir(%s) wasn't successful"), filepath);
   } 
   FREE(cwd);
   return file;
}

static char *CropFilenameInCurrentDir(char *filepath)
/* If filepath contains the current working directory crop this path else leave it as is
 * --- Parameter: ---
 * char *filepath                  : file to load with complete path
 * --- Return: ---
 * char *CropFilenameInCurrentDir(): plain or complete filename of file to load
 */
{
  gchar *cwd ;

  cwd      = g_get_current_dir();
  if ( cwd ) {
    register char *dp = cwd - 1, *fp = filepath - 1;

    while ( *++fp && *++dp ) if ( *dp != *fp ) break;
    if ( !*dp ) filepath = fp + 1;
    FREE(cwd);
  }
  return g_strdup(filepath);
}


static void FileSelectionMove2(GtkFileSelection *fs, char *name)
/* Set the current file in filelist to name 
 */
{
  gchar *filename;
  int    i = -1;
   
  if ( !fs || !name || !*name ) return;

  while ( gtk_clist_get_text(GTK_CLIST(fs->file_list), ++i, 0, &filename) ) 
    if ( filename && !strcmp(filename, name) ) {
      gtk_clist_select_row(GTK_CLIST(fs->file_list), i, 0);
      gtk_clist_moveto(GTK_CLIST(fs->file_list), i, 0, 0.5, 0.0);
      break;
  }
}

char *FilenameExtractFromSelection(GtkFileSelection *fs, PICTURE *bild)
/* useful for both paul and paulscan, that's why this was extracted from FileSaveOK()
 * --- Parameter: ---
 * GtkFileSelection *fs                 : fileselection widget to get name from
 * PICTURE          *bild               : picture structure which filename has to be set
 * --- Return: ---
 * char *FilenameExtractFromSelection() : new filename if it was changed and paul's
 *                                        list has to be updated, else NULL 
 */
{
   char *filename, *ptr;
   int   flag = 0;  /* determine if paul's filename has to be set.  This can't be *
                     * done in paulscan, so we have to decide here                */

   filename = CropFilenameInCurrentDir(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)));
   if ( (ptr = strstr(filename, ".png")) ) *ptr = 0;
   if ( (ptr = strstr(filename, ".tif")) ) *ptr = 0;
   if ( strcmp(bild->file, filename) ) {
      FREE(bild->file);
      bild->file = g_strdup(filename);
      bild->flag |= QUIET;
      flag = 1;
   }
   gtk_widget_destroy(GTK_WIDGET(fs));
   FREE(filename) ;
   if ( flag ) return bild->file;
   return NULL;
}


void FileSelectionLoadSave(void *data, GtkSignalFunc func, char *name, const char *title, 
                           const char *pattern)
/* Create file selection dialog, set callback functions and user data
 * --- Parameters: ---
 * void         *data  : set this as user_data to file_selection object
 *                       used in paul: PAUL *p; used in paulscan: PICTURE *bild
 * GtkSignalFunc func  : function to call when OK-button was pressed
 * char         *name  : initial filename to set
 * char         *title : title of file selection window
 */
{
  static GtkWidget *fs = NULL;

  g_return_if_fail ( func ) ;
  
  if ( fs ) {
    gtk_widget_destroy(fs);
    fs = NULL;
    return;
  }
   
  g_return_if_fail ( (fs = gtk_file_selection_new(title)) );

  gtk_window_set_position(GTK_WINDOW(fs), GTK_WIN_POS_MOUSE);
  gtk_signal_connect(GTK_OBJECT(fs), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &fs);

  if ( name ) {
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs), name);
    gtk_editable_select_region(GTK_EDITABLE(GTK_FILE_SELECTION(fs)->selection_entry), 0, -1);
  }
  if ( pattern ) 
    gtk_file_selection_complete(GTK_FILE_SELECTION(fs), pattern);
     
  gtk_object_set_user_data(GTK_OBJECT(fs), data);
  /* This helps the HTML functions to detect whether HTML file or link is edited */
  gtk_object_set_user_data(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button), name);

  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button), "clicked", 
                     GTK_SIGNAL_FUNC(func), fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button), "clicked", 
                            GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(fs));
  
  gtk_widget_show(fs);
  FileSelectionMove2(GTK_FILE_SELECTION(fs), name);
}


GtkWidget *FileEntryOrBrowse(char *desc, char *init)
/* create textentry to edit filename and button to browse
 * --- Parameter: ---
 * char         *desc               : Description of the file
 * char         *init               : initial value for text entry widget
 * GtkSignalFunc EntryFunc          : callback function for text entry
 * GtkSignalFunc ButtonFunc         : callback function for button
 * void         *entry_data         : userdata for text entry widget
 * void         *button_data        : userdata for button widget
 * --- Return: ---
 * GtkWidget    *FileEntryOrBrowse(): hbox containing text entry and button
 */
{
  GtkWidget *hbox, *label, *text, *button;
   
  g_return_val_if_fail (init, NULL) ;
  
  hbox = gtk_hbox_new(FALSE, 2);
  gtk_container_set_border_width(GTK_CONTAINER(hbox), 2);

  if ( desc ) {
    label = gtk_label_new (desc);
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 2);
  }

  text = gtk_entry_new ();
  button = gtk_button_new_with_label(_("Browse"));
  gtk_object_set_user_data(GTK_OBJECT(text), button);
  gtk_object_set_user_data(GTK_OBJECT(button), init);

  gtk_entry_set_text(GTK_ENTRY(text), init);
  gtk_box_pack_start(GTK_BOX(hbox), text, TRUE, TRUE, 2);
  gtk_signal_connect(GTK_OBJECT(text), "changed", FileSelectEntryChanged, button);
                                 
  gtk_box_pack_start(GTK_BOX (hbox), button, FALSE, FALSE, 2);
  gtk_signal_connect(GTK_OBJECT(button), "clicked", FileSelectBrowse, text);
 
  gtk_widget_show_all(hbox);
  return hbox;
}

void FileSelectChanged(GtkButton *button, char **file)
/* called when parameters were changed in menu
 * --- Parameter: ---
 * GtkButton *button: OK-Button of menu window
 * char     **file  : target pointer
 */
{
  char *filename;
   
  g_return_if_fail ( button ) ;
  g_return_if_fail ( GTK_IS_BUTTON(button) );
  g_return_if_fail ( (filename = gtk_object_get_user_data(GTK_OBJECT(button))) );

  *file = NULL;
  if ( filename ) {
    if ( *filename ) *file = g_strdup(filename);
    FREE(filename);
  }
}

static void FileSelectFilenameOk(GtkWidget *widget, GtkFileSelection *fs)
/*
 * --- Parameter: ---
 * GtkWidget        *widget : OK-button inside GtkFileSelection widget, pointer to old file
 *                            name set as data
 * GtkFileSelection *fs     : GtkFileSelection widget, text entry widget set as user data
 *                            text entry has  browse button set as user data, 
 *                            browse button has temporary string as user data
 */
{
  char      *filename, *tmp;
  GtkWidget *text, *button;

  g_return_if_fail ( widget && fs ) ;
  g_return_if_fail ( GTK_IS_BUTTON(widget) ) ;
  g_return_if_fail ( (filename = gtk_object_get_user_data(GTK_OBJECT(widget)) ) ) ;
  g_return_if_fail ( GTK_IS_FILE_SELECTION(fs) ) ;
  g_return_if_fail ( (text = gtk_object_get_user_data(GTK_OBJECT(fs)) ) ) ;
  g_return_if_fail ( GTK_IS_ENTRY(text) ) ;
  g_return_if_fail ( (button = gtk_object_get_user_data(GTK_OBJECT(text)) ) ) ;
  g_return_if_fail ( GTK_IS_BUTTON(button) ) ;
  g_return_if_fail ( (tmp = gtk_object_get_user_data(GTK_OBJECT(button))) );

  filename = CropFilenameInCurrentDir(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)));
  gtk_entry_set_text(GTK_ENTRY(text), filename);
  gtk_widget_destroy(GTK_WIDGET(fs));
  FREE(filename);
}

void FileSelectBrowse(GtkWidget *button, GtkWidget *text)
/* callback for browsing filename
 * --- Parameter: ---
 * GtkWidget *text  : text entry widget, GtkBin *helper set as user_data
 *                  : button (will be) set as helper->child, title set as helper user data
 * GtkWidget *button: browse button, temporary string set as user data
 */
{
  char   *tmp;
   
  g_return_if_fail ( text && button );
  g_return_if_fail ( GTK_IS_ENTRY(text) );
  g_return_if_fail ( GTK_IS_BUTTON(button) );
  g_return_if_fail ( (tmp = gtk_object_get_user_data(GTK_OBJECT(button)) ) );

  FileSelectionLoadSave(text, GTK_SIGNAL_FUNC(FileSelectFilenameOk), tmp, NULL, NULL);
}

void FileSelectEntryChanged(GtkWidget *text, GtkWidget *button)
/* callback for editing text entry of filename
 * --- Parameter: ---
 * GtkWidget *text  : text entry widget, button set as user_data
 * GtkWidget *button: browse button, temporary string set as user data
 */
{
  char *filename, *tmp;

  g_return_if_fail ( text && button );
  g_return_if_fail ( GTK_IS_ENTRY(text) );
  g_return_if_fail ( GTK_IS_BUTTON(button) );
  g_return_if_fail ( (tmp = gtk_object_get_user_data(GTK_OBJECT(button)) ) );
  
  filename = gtk_entry_get_text(GTK_ENTRY(text));
  strcpy(tmp, filename);
}







