/***************************************************************************
                          rgraphic.h  -  description
                             -------------------
    begin                : Mon Sep 27 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/

/****************************************************************************
** rgraphic.h 1998/08/24 A. Mustun RibbonSoft 
**
** Copyright (C) 1998 RibbonSoft.  All rights reserved.
**
*****************************************************************************/

#ifndef RGRAPHIC_H
#define RGRAPHIC_H

#include "reventdef.h"
#include "rgraphdef.h"

#include <qframe.h>
#include <qlist.h>
#include <qstack.h>

#include "relement.h"
#include "rflags.h"
#include "rlayer.h"
#include "rlog.h"
#include "rtypes.h"
  
class RPainter;
class RActZoom;
class RActDraw;
class RActEdit;
class RActTag;
class RActCam;
class RSnap;

/*!
   Graphic widget with all functions needed to sore, view and modify elements.
*/
class RGraphic : public QFrame, public RFlags
{
  Q_OBJECT
public:
  /*! Constructor
      \param _parent The parent widget
      \param _name name
      \param _f widget flags
  */
  RGraphic(QWidget* _parent=0, 
           const char* _name=0, 
           WFlags _f=0);
           
  /*! Destructor
  */
  ~RGraphic();

  // Copy from...
  //
  
  /*! Copy all elements, layers and properties from _source
      \param _source the source document
      \return true: successful
              false: otherwise (_source was 0)
  */
  bool    copyAllFrom(RGraphic* _source);

  /*! Copy selected elements, layers and properties from _source
      \param _source the source document
      \return true: successful
              false: otherwise (_source was 0)
  */
  bool    copyTaggedFrom(RGraphic* _source, float _ox, float _oy);
  
  /*! Copy only properties from _source
      \param _source the source document
      \return true: successful
              false: otherwise (_source was 0)
  */
  bool    copyPropertiesFrom(RGraphic* _source);
  /*! Copy only visible objects from _source
      \param _source the source document
      \param _tagged only tasgged objects
      \param _ox Offset in X
      \param _oy Offset in Y
      \return true: successful
              false: otherwise (_source was 0)
  */
  bool    copyVisibleObjectsFrom(RGraphic* _source, bool _tagged=false, float _ox=0.0, float _oy=0.0);

  /*! Copy only visible objects from _list which must be a QList containing RElement's
      \param _list the source list
      \return true: successful
              false: otherwise (no graphic or no visible elements)
  */
  bool    copyVisibleObjectsFrom(QList<RElement> &_list);
  
  /*! Copy only layerlist from _source
      \param _source the source document
      \return true: successful
              false: otherwise (_source was 0)
  */
  bool    copyLayerListFrom(RGraphic* _source);

  // Reset:
  //
  
  /*! Reset the "converted"-flag (E_CONV) from all objects
  */
  void    resetConvertedFlags();
  
  /*! Reset the "contur"-flag (E_CONT) from all objects
  */
  void    resetContourFlags();

  // File handling:
  //
  
  /*! Load a file with it's path given to this graphic
      \param _fileName Path/name and extension of the file
      \return true: successful loaded
  */
  bool    loadFile(const char* _fileName, bool add=false);
  
  /*! Save a file with it's path given from this graphic (Format: DXF or CXD)
      \param _fileName Path/name and extension of the file
      \return true: successful saved
  */
  bool    saveFile(const char* _fileName);
  
  /*! Set an individual file name for this graphic
      \param _fileName Path/name and extension of the file
  */
  void    setFileName(const char* _fileName) { fileName=_fileName; }
  
  /*! Get the current file name for this graphic
      \return The file name as a QString
  */
  QString getFileName() const { return fileName; }
  
  
  /*! Set menu auto ID
  */
  void    setMenuId(int _id) { menuId=_id; RLOG("\nMenuId: "); RLOG(_id); }
  
  /*! Get menu auto ID
  */
  int     getMenuId() const { return menuId; }

  // Action handling:
  //
  
  
  bool    isBusy() { return busy; }
  void    setBusy(bool _busy) { busy=_busy; }

  bool    isPrinting() { return printing; }

  void    setAutoUpdateGraphic(bool _upd) { autoUpdateGraphic=_upd; }

  bool    isActive() { return active; }
  void    setActive(bool _active) { active=_active;
                                    if(active) emit activated(); }

  bool    getUpdateBroken() { return updateBroken; }
  void    setUpdateBroken(bool _upd) { updateBroken=_upd; }

  void setCurrentAction(int _curAct) { currentAction=_curAct; }
  void resetCurrentAction() { currentAction=-1; }
  int  getCurrentAction() const { return currentAction; }
  void setCurrentState(int _curState) { currentState=_curState; }
  void resetCurrentState() { currentState=-1; }
  int  getCurrentState() const { return currentState; }
  
  /*! Set Action which gets active after Tagging elements (enough)
  */
  void setNextAction(int _nextAct) { nextAction=_nextAct; }
  /*! Reset Action which gets active after Tagging elements (enough)
  */
  void resetNextAction() { nextAction=-1; }
  /*! Get Action which gets active after Tagging elements (enough)
  */
  int getNextAction() { return nextAction; }
  /*! Activate action which will come after clicking 'enough' button
  */
  void activateNextAction() { setCurrentAction(nextAction);
                              distributeEvent(REVENT_BEGIN); }

  void getMousePos(int* _x, int* _y) { *_x=mouseX; *_y=mouseY; }
  int  getMousePosX() { return mouseX; }
  int  getMousePosY() { return mouseY; }
  void getRealMousePos(float* _x, float* _y) { *_x=screenToRealX(mouseX); *_y=screenToRealY(mouseY); }
  float getRealMousePosX() { return screenToRealX(mouseX); }
  float getRealMousePosY() { return screenToRealY(mouseY); }

  void  getKeyCoords(float *_x, float *_y) { *_x=keyX; *_y=keyY; }
  void  setKeyCoords(float _x, float _y) { keyX=_x; keyY=_y; enableKeyCoords=true; }

  void  disableKeyCoords() { enableKeyCoords=false; }
  bool  keyCoordsEnabled() { return enableKeyCoords; }
  
  RSnap* snapper();
  RGraphic* clipboard();

  // Element handling:
  //
  void deleteElements();
  
  int setCurrentElement(const RElement* _curEl);
  RElement* elementCurrent() { return element.current(); }  // Get a pointer to the last added element
  RElement* elementFirst() { return element.first(); }
  RElement* elementLast() { return element.last(); }
  RElement* elementNext() { return element.next(); }
  RElement* elementPrev() { return element.prev(); }
  int       elementAt()   { return element.at(); }
  RElement* elementAt(int _ind) { return element.at(_ind); }
  void      appendElement(const RElement* _newEl) { if(_newEl) element.append(_newEl); }
  void      exchangeElements(RElement* _el1, RElement* _el2);
  //QList<RElement>* getElementList() { return &element; }

  // Layer handling:
  //
  RLayer* layerCurrent();
  RLayer* layerFirst() { return layer.first(); }
  RLayer* layerLast() { return layer.last(); }
  RLayer* layerNext() { return layer.next(); }
  RLayer* layerPrev() { return layer.prev(); }
  RLayer* layerAt(int _ind) { return layer.at(_ind); }

  int  countElementsOnLayer(const char* _layerName);
  int  countElementsOnLayer(int _layerNumber);
  int  count();
  int  countTagged();
  int  addLayer(const char* _name);
  int  checkLayer(int _num);
  int  countLayers();
  bool removeEmptyLayers();
  bool removeLayer(const char* _name, bool _alsoObjects=true);
  bool removeLayer(int  _num, bool _alsoObjects=true);
  bool renameLayer(const char* _oldName, const char* _newName);
  bool renameLayer(int _num, const char* _newName);
  void clearLayers();
  void sortLayers();
  int  getLayerNum(const char* _name);
  const char* getLayerName(int _num);
  int  getLayerIndex(int _num);
  RLayer* getLayer(byte _index);
  bool isLayerVisible(int _num);
  bool isLayerVisible(const char* _name) { return isLayerVisible(getLayerNum(_name)); }
  
public slots:
  void  switchLayer(const char* _name);

public:
  
  // Add Elements:
  //
  int  addElement(RElement* _el,
                  bool _setCreationNum=true);
  int  addPoint(float _x1, float _y1, 
                byte _layer=0,
                bool _setCreationNum=true);
  int  addLine(float _x1, float _y1, 
               float _x2, float _y2, 
               byte _layer=0,
               bool _setCreationNum=true);
  int  addArc(float _cx, float _cy, 
              float _cr, 
              float _a1, float _a2, 
              bool _reversed, 
              byte _layer=0,
              bool _setCreationNum=true);
  int  addArcPPP(float _x1, float _y1,
                 float _x2, float _y2,
                 float _x3, float _y3,
                 bool _reversed,
                 byte _layer=0,
                 bool _setCreationNum=true);
  int  addCircle(float _cx, float _cy, 
                 float _cr,
                 float _a1=0.0, float _a2=360.0, 
                 bool _reversed=false, 
                 byte _layer=0,
                 bool _setCreationNum=true);
  int  addPolylineElement(float _x1, float _y1,
                          float _x2, float _y2,
                          float _ab, 
                          byte _layer=0,
                          bool _setCreationNum=true);
  int  addRect(float _x1, float _y1, 
               float _x2, float _y2, 
               byte _layer=0,
               bool _setCreationNum=true);
  int  addText(float _x, float _y,
               int   _font,
               const unsigned char* _text,
               unsigned int _flags,
               float _textheight,
               float _angle=0.0,
               float _radius=0.0,
               float _letterspace=DEF_AREAMAX,
               float _wordspace=DEF_AREAMAX,
               float _linedistance=DEF_AREAMAX,
               byte _layer=0,
               bool _setCreationNum=true);
  int  addDimension(float _v1, float _v2, float _v3, float _v4,
                    float _v5, float _v6,
                    float _pos,
                    uint _flags,
                    byte _layer=0,
                    bool _setCreationNum=true);
  int  addHatching(float _factor, QString _pattern,
                    byte _layer=0,
                    bool _setCreationNum=true);
  int  addRapid(float _x1, float _y1, 
                byte _layer=0,
                bool _setCreationNum=true);

  void singleElements(RElement* _el, 
                      bool _add,
                      bool _draw=false,
                      RPainter* _paint=0,
                      bool _border=false,
                      float* _left=0,
                      float* _top=0,
                      float* _right=0,
                      float* _bottom=0,
                      bool _setCreationNum=true,
                      bool _terminateAction=true);
	
  // Bad Allocation:
  //
  void badAllocation();

  // Tag:
  //
  
  /*! Tag nothing
      \param _upd Update graphic view
  */
  void tagNothing(bool _upd=true);
  
  /*! Tag all
      \param _upd Update graphic view
  */
  void tagAll(bool _upd=true);
  
  /*! Tag contour
      \param _el An element of the contour
      \param _tag Tag or untag
      \param _upd Update graphic view
  */
  void tagContour(RElement* _el, bool _tag, bool _upd=true);
  
  /*! Tag range
      \param _minX minimum X coordinate
      \param _minY minimum Y coordinate
      \param _maxX maximum X coordinate
      \param _maxY maximum Y coordinate
      \param _tag Tag or untag
      \param _upd Update graphic view
      \param _tagEndpoints Tag also enpoints which are in the range (for stretch function)
  */
  void tagRange(float _x1, float _y1, 
                float _x2, float _y2, 
                bool _tag, bool _upd=true,
                bool _tagEndpoints=false);
                
  /*! Tag Intersected elements
      \param _x1 x of 1st point of the line
      \param _y1 y of 1st point of the line
      \param _x2 x of 2nd point of the line
      \param _y2 y of 2nd point of the line
      \param _tag Tag or untag
      \param _upd Update graphic view
  */
  void tagInters(float _x1, float _y1, 
                 float _x2, float _y2, 
                 bool _tag, bool _upd=true);
  
  /*! Invert selection
      \param _upd Update graphic view
  */
  void tagInvert(bool _upd=true);
  
  /*! Tag Layer
      \param _layerNum Number of layer to tag
      \param _tag Tag or untag
      \param _upd Update graphic view
  */
  void tagLayer(byte _layerNum, bool _tag, bool _upd=true);
  
  /*! Tag double elements
      \param _tolerance tolerance in mm
      \param _upd Update graphic view
  */
  void tagDoubles(float _tolerance, bool _upd=true);
  
  // Edit:
  //
  /*! Move selected elements
      \param _copies Number of copies
      \param _ox  Reference point X
      \param _oy  Reference point Y
      \param _x   X movement (rel)
      \param _y   Y movement (rel)
      \param _all Move all (not only selected)
      \param _upd Update graphic view
  */
  void editMove(int _copies, 
                float _ox, float _oy,
                float _x, float _y, 
                float _angle=0.0,
                bool _all=false,
                bool _upd=true);

  /*! Rotate selected elements
      \param _copies Number of copies
      \param _ang Angle
      \param _ox Reference point X
      \param _oy Reference point Y
      \param _upd Update graphic view
  */

  void editRotate(int _copies,
                  float _ang, 
                  float _ox, float _oy,
                  bool _upd);

  /*! Scale selected elements
      \param _copies Number of copies
      \param _factor Scale Factor
      \param _ox Reference point X
      \param _oy Reference point Y
      \param _upd Update graphic view
  */
  void editScale(int _copies,
                 float _factor, 
                 float _ox, float _oy,
                 bool _upd);
                 
  /*! Mirror selected elements
      \param _copies Number of copies
      \param _x1 1st point of axe x
      \param _y1 1st point of axe y
      \param _x2 2nd point of axe x
      \param _y2 2nd point of axe y
      \param _upd Update graphic view
  */
  void editMirror(int _copies,
                  float _x1, float _y1,
                  float _x2, float _y2,
                  bool _upd);
  
  /*! Move and rotate selected elements
      \param _copies Number of copies
      \param _angle Angle for rotation
      \param _x Movement in x
      \param _y Movement in y
      \param _ox1 Reference point for moving
      \param _oy1 Reference point for moving
      \param _ox2 Reference point for rotation
      \param _oy2 Reference point for rotation
      \param _upd Update graphic view  
  */
  void editMoveRotate(int _copies, 
                      float _angle,
                      float _x, float _y,
                      float _ox1, float _oy1,
                      float _ox2, float _oy2,
                      bool _upd);
                      
  /*! Rotate and Counter-rotate selected elements
      \param _copies Number of copies
      \param _angle1 Angle for rotation 1 (big)
      \param _angle2 Angle for rotation 2 (small)
      \param _ox1 Reference point for rotation 1 (big)
      \param _oy1 Reference point for rotation 1 (big)
      \param _ox2 Reference point for rotation 2 (small)
      \param _oy2 Reference point for rotation 2 (small)
      \param _upd Update graphic view  
  */
  void editRotateRotate(int _copies, 
                        float _angle1,
                        float _angle2,
                        float _ox1, float _oy1,
                        float _ox2, float _oy2,
                        bool _upd);
                        
  /*! Trim an element to another
      \param _tEl    trim element
      \param _ep     trim endpoint (=true) or trim startpoint (=false)
      \param _ix1/_iy1 first  intersection point (trim to this point)
      \param _ix2/_iy2 second intersection point (circles also trim to this point)
      \param _mang mouse angle (define part of a circle to trim)
      \param _upd update graphic view?
  */
  void editTrim(RElement* _tEl,
                bool _ep,
                float _ix1, float _iy1,
                float _ix2, float _iy2,
                float _mang, 
                bool _upd);
                
  /*! Cut an element at a certain point
      \param _cEl    Cut element
      \param _px/_py Cutting point
      \param _upd update graphic view?
  */
  void editCut(RElement* _cEl,
               float _px, float _py,
               bool _upd);
               
  /*! Stretch range
      \param _x/_y   Move offset
      \param _upd update graphic view?
  */
  void editStretch(float _x, float _y,
                   bool _upd);
                   
  /*! Round
      \param _el1 Element 1
      \param _el2 Element 2
      \param _r   Radius
      \param _trim Trim elements to rounding?
      \param _trimEp1 true:  Trim endpoint of first element 
                      false: Trim startpoint of first element 
      \param _trimEp2 true:  Trim endpoint of second element 
                      false: Trim startpoint of second element
      \param _ix  X-Pos of intersection point
      \param _iy  Y-Pos of intersection point
      \param _mang1 angle for 1st element clicked (center to clicked point / only circles)
      \param _mang2 angle for 2nd element clicked (center to clicked point / only circles)
      \param _upd update graphic view?
  */
  void editRound(RElement* _el1,
                 RElement* _el2,
                 float _r,
                 bool _trim,
                 bool _trimEp1,
                 bool _trimEp2,
                 float _ix, float _iy,
                 float _mang1, float _mang2,
                 bool _upd);
                 
                 
  /*! Beveling
      \param _el1 bevel element 1
      \param _el2 bevel element 2
      \param _width1    bevel width 1
      \param _width2    bevel width 2
      \param _trim      Trim elements to rounding?
      \param _trimEp1   trim endpoint (=true) or trim startpoint (=false) of 1st element
      \param _trimEp2   trim endpoint (=true) or trim startpoint (=false) of 2nd element
      \param _ipx/_ipy  intersection point (trim to this point)
      \param _upd update graphic view?
  */
  void editBevel(RElement* _el1,
                 RElement* _el2,
                 float _width1,
                 float _width2,
                 bool _trim,
                 bool _trimEp1,
                 bool _trimEp2,
                 float _ix, float _iy,
                 bool _upd);
                 
  /*! Convert all tagged texts into elements
      \param _upd update graphic view?
  */
  void editTextElements(bool _upd);
  
  /*! Edit given text (show text dialog)
      \param _el  Original text element
      \param _upd Update graphic view?
  */
  void editText(RElement* _el,
                bool _upd);
                
  /*! Edit delete tagged elements
      \param _upd Update graphic view?
  */
  void editDelete(bool _upd);
  
  /*! Edit move tagged elements to another layer
      \param _layerName New Layer Name
      \param _upd       Update graphic view?
  */
  void editLayer(const char* _layerName, bool _upd);

  /** Edit change attributes of elements
   *  \param _color  New color (-1 won't change the color)
   *  \param _width  New width (-1 won't change the width)
   *  \param _style  New style (-1 won't change the style)
   *  \param _upd    Update graphic view?
   */
  void editAttrib(int _color, int _width, int _style, bool _upd);


  // Zoom:
  //
  float getZoom() const { return zoom; }
	void  setZoom(float _zoom) { zoom=_zoom; }
  int   getOffsetX() const { return offsetX; }
	void  setOffsetX(int _offsetX) { offsetX=_offsetX; }
  int   getOffsetY() const { return offsetY; }
	void  setOffsetY(int _offsetY) { offsetY=_offsetY; }
  void  getBorders(float& _left, float& _top, 
                   float& _right, float& _bottom);

  void zoomAuto(QRect* _rect=0);
  void zoomTagged(QRect* _rect=0);
  void setZoom(int _left,  int _top, 
               int _right, int _bottom, 
               int _zoomBorder=DEF_ZOOMBORDER,
               QRect* _rect=0);
  void setZoom(float _left,  float _top, 
               float _right, float _bottom, 
               int   _zoomBorder=DEF_ZOOMBORDER,
               QRect* _rect=0);
  void zoomIncrease(float _fact=2.0);
  void zoomDecrease(float _fact=1.5);
  void setZoomFromGraphic(RGraphic* _gr);

  int   realToScreenX(float _realX);
  int   realToScreenY(float _realY);
  int   realToScreenDist(float _realDist);

  float realToExactScreenX(float _realX);
  float realToExactScreenY(float _realY);
  float realToExactScreenDist(float _realDist);
  
  float screenToRealX(int _screenX);
  float screenToRealY(int _screenY);
  float screenToRealDist(int _screenDist);

  bool  isPointOnScreen(int _x, int _y);

  void  moveRelZero(float _x, float _y);

  float getRelZeroX() { return relZeroX; }
  float getRelZeroY() { return relZeroY; }

  // Attributes:
  //
  void  setColor(QColor _col);
  void  setColor(int _col);
  void  setWidth(int _width);
  void  setStyle(Qt::PenStyle _style);
  void  setStyle(QString _name) { setStyle(nameToStyle(_name)); }

  static int   colorToNumber(QColor _col);
  static QColor numberToColor(int _num);

  static Qt::PenStyle numberToStyle(int _num);
  static Qt::PenStyle nameToStyle(QString _name);
  static QString styleToName(Qt::PenStyle _style);

	
	/*! Sets the printer factors. This is used for printing 1:1 and for printing the zero point.
	    <br><i>(Measure in MM * printerFactor = Measure in printer units)</i>
	    \param _pfx The printer factor in X
	    \param _pfy The printer factor in Y
	*/
	void  setPrinterFactor(float _pfx, float _pfy) { printerFactorX=_pfx; printerFactorY=_pfy; }

  // CAM related:
  //
  bool      isInitialized() const { return initialized; }
  void      setInitialized(bool _ini) { initialized=_ini; }

  QString   getMachineGenerator() const { return machineGenerator; }
  void      setMachineGenerator(const char* _gen) { machineGenerator=_gen; }

  QString   getExtension();
  void      setExtension(const char* _ext) { extension=_ext; }

  char      getMode() const { return mode; }
  void      setMode(char _md) { mode=_md; }

  bool      getOptimization() const { return optimization; }
  void      setOptimization(bool _opt) { optimization=_opt; }
  
  bool      getContourCenter() const { return contourCenter; }
  void      setContourCenter(bool _cont) { contourCenter=_cont; }

  char      getSorting() const { return sorting; }
  void      setSorting(char _sort) { sorting=_sort; }

  char      getDirection() const { return direction; }
  void      setDirection(char _dir) { direction=_dir; }

  bool      getPointing() const { return pointing; }
  void      setPointing(bool _point) { pointing=_point; }

  bool      getContourStartInHeader() const { return contourStartInHeader; }
  void      setContourStartInHeader(bool _cih) { contourStartInHeader=_cih; }
  
  bool      getLayerStartInHeader() const { return layerStartInHeader; }
  void      setLayerStartInHeader(bool _lih) { layerStartInHeader=_lih; }
  
  int       getNumberStart() const;
  void      setNumberStart(const char* _ns) { numberStart=_ns; }
  
  int       getNumberStep() const;
  void      setNumberStep(const char* _ns) { numberStep=_ns; }
  
  float     getMaxArcAngle() const;
  void      setMaxArcAngle(const char* _ma) { maxArcAngle=_ma; }
  
  float     getMaxArcLength() const;
  void      setMaxArcLength(const char* _mal) { maxArcLength=_mal; }
  
  float     getTolerance() const;
  void      setTolerance(const char* _tol) { tolerance=_tol; }
  
  int       getDigits() const;
  void      setDigits(const char* _dig) { digits=_dig; }
  
  float     getFactor() const;
  void      setFactor(const char* _fact) { factor=_fact; }
  
  float     getPointingAngle() const { return pointingAngle; }
  void      setPointingAngle(float _pa) { pointingAngle=_pa; }
  
  float     getPointingDepth() const { return pointingDepth; }
  void      setPointingDepth(float _pd) { pointingDepth=_pd; }
  
  float     getPointingLimit() const { return pointingLimit; }
  void      setPointingLimit(float _pl) { pointingLimit=_pl; }
  
  float     getPointingMaxLength() const { return pointingMaxLength; }
  void      setPointingMaxLength(float _pml) { pointingMaxLength=_pml; }

  int       getContNum() { return contNum; }
  void      setContNum(int _cn) { contNum=_cn; }
  
  void      clearParameters();
  void      addParameter(char _code, const char* _value);
  QString   getParameterString() const
              { return parameterString; }
  QString   getParameter(int _ind) const
              { if(_ind>=0 && _ind<DEF_MAXPARAMETERS) 
                  return parameter[_ind]; 
                else
                  return ""; }
                  
  void      clearLayerParameters();
  void      addLayerParameter(int _lay, char _code, const char* _value);
  QString   getLayerParameterString() const
              { return layerParameterString; }
  QString   getLayerParameter(int _lay, int _ind) const
              { if(_lay>=0 && _lay<DEF_MAXLAYERS &&
                   _ind>=0 && _ind<DEF_MAXPARAMETERS) 
                  return layerParameter[_lay][_ind]; 
                else
                  return ""; }

  int    getDelay() { return delay; }
  int    getDelayPosition() { return delayPosition; }

  //bool   getRapid() { return rapid; }
  //bool   getDetails() { return details; }
  //bool   getBroken() { return broken; }
  
  /*! Get the cutting way (excluding rapid move) in this graphic (for info-widget)
  */
  float getCuttingWay();
  
  /*! Get the total way (including rapid move) in this graphic (for info-widget)
  */
  float getTotalWay();
  
  /*! Draw the graphic in its widget
      \param _paint        Pointer to paint device (screen or printer)
      \param _sim          Simulation mode (=true)
      \param _print        Printing mode (=true). In printing mode we don't clip.
      \param _specialFlags Special flags for this single draw 
                           (same possibilities as for the graphic itself)
			   Used for printing
			   -1: Don't use special flags
  */
  void  draw(RPainter* _paint, bool _sim=false, bool _print=false, int _specialFlags=G_DEFAULT);
  
  void  drawElement(int _ind=-1, bool _relZero=true);
  void  drawElement(RElement* _el);
  void  drawGrid(RPainter* _paint);
  void  drawAbsoluteZero(RPainter* _paint);
  void  delRelativeZero(RPainter* _paint);
  void  drawRelativeZero(RPainter* _paint);
  void  drawDirection(RPainter* _paint,
                      float _dx, float _dy,
                      float _dang);
  void  drawEndPoint(RPainter* _paint,
                     float _ex, float _ey);
                     
  void  enableMark(bool _enable) { markEnabled=_enable; }
  bool  isMarkEnabled() { return markEnabled; }
  void  delMark() { drawMark(0, 0, false, false); }
  void  xorMark() { drawMark(0, 0, false, true); }
  void  drawMark(float _x, float _y, bool _drawNew=true);
  void  drawMark(int _x, int _y, bool _drawNew=true, bool _xor=false);
  
  void  setPreviewUpdate(bool _pu) { previewUpdate=_pu; }
  bool  getPreviewUpdate() { return previewUpdate; }


  // CAD:
  //
  //! Get Font number
  int    getFontNumber() { return fontNumber; }
  //! Set Font number
  void   setFontNumber(int _num) { fontNumber=_num; }
  //! Get if Font is proportional
  bool   getFontProportional() { return fontProportional; }
  //! Set if Font is proportional
  void   setFontProportional(bool _prop) { fontProportional=_prop; }
  //! Get font fixed width flag
  bool   getFontFixedWidth() { return fontFixedWidth; }
  //! Set font fixed width
  void   setFontFixedWidth(bool _fix) { fontFixedWidth=_fix; }
  //! Get Font height
  float  getFontHeight() { return fontHeight; }
  //! Set Font height
  void   setFontHeight(float _h) { fontHeight=_h; }
  //! Get Font letter spacing
  float  getFontLetterSpacing() { return fontLetterSpacing; }
  //! Set Font letter spacing
  void   setFontLetterSpacing(float _lsp) { fontLetterSpacing=_lsp; }
  //! Get Font word spacing
  float  getFontWordSpacing() { return fontWordSpacing; }
  //! Set Font word spacing
  void   setFontWordSpacing(float _wsp) { fontWordSpacing=_wsp; }
  //! Get Font line distance
  float  getFontLineDistance() { return fontLineDistance; }
  //! Set Font line distance
  void   setFontLineDistance(float _ld) { fontLineDistance=_ld; }
  //! Get Font angle
  float  getFontAngle() { return fontAngle; }
  //! Set Font angle
  void   setFontAngle(float _ang) { fontAngle=_ang; }
  //! Get Font radius
  float  getFontRadius() { return fontRadius; }
  //! Set Font radius
  void   setFontRadius(float _rad) { fontRadius=_rad; }
  //! Get Font flags <br> (E_STRAIGHT, E_ROUNDIN, E_ROUNDOUT, E_LEFT, E_RIGHT, E_CENTER)
  RFlags* getFontFlags() { return &fontFlags; }
  //! Set ALL Font flags at once<br> (E_STRAIGHT, E_ROUNDIN, E_ROUNDOUT, E_LEFT, E_RIGHT, E_CENTER)
  void   setFontFlags(uint _fl) { fontFlags.setFlags(_fl); }

  //! Get current grid width (used by snapper)
  float  getCurrentGridWidth() { return currentGridWidth; }

  // Preview:
  //
  void  resetPreviewElement();
  bool  setPreviewElement(int _typ,
                          float _v1, float _v2,
                          float _v3, float _v4=0.0,
                          float _v5=0.0, float _v6=0.0,
                          float _v7=0.0, uint _flags=0);
  bool  setPreviewElement(RElement* _el);
  void  delPreviewElement() { drawPreviewElement(false); }
  void  xorPreviewElement() { drawPreviewElement(false, true); }
  
  /*! Draw preview
      \param _drawNew true: dont delete last preview
      \param _xor true: draw in xor mode
  */
  void  drawPreviewElement(bool _drawNew=true, bool _xor=false);
  
  /*! Reset the preview elements
  */
  void  resetPreview();
  
  /*! Fill preview with tagged elements
      \param _refx reference point X (Preview is relative)
      \param _refy reference point Y
      \param _all  Fill preview with all not only with tagged
  */
  void  fillPreviewWithTagged(float _refx, float _refl, bool _all=false);

  /*! Fill preview with clipboard elements
  */
  void  fillPreviewWithClipboard();
 
  /*! Fill preview with all elements
      \param _refx reference point X (Preview is relative)
      \param _refy reference point Y
  */
  void  fillPreviewWithAll(float _refx, float _refy) 
    { fillPreviewWithTagged(_refx, _refy, true); }

  /*! Has the preview moved since the last call?
  */
  bool  previewMoved(float _x, float _y, float _angle=0.0);
  
  void  delPreview() { drawPreview(DEF_AREAMAX, DEF_AREAMAX, false); }
  void  xorPreview() { drawPreview(DEF_AREAMAX, DEF_AREAMAX, false, true); }
  
  /*! Draw preview
      \param _x x offset
      \param _y y offset
      \param _drawNew true: dont delete last preview
      \param _xor true: draw in xor mode
  */
  void  drawPreview(float _x, float _y, float _angle=0.0,
                    bool _drawNew=true, bool _xor=false);

  // Highlighting:
  //
  void  resetHighlightElement();
  RElement* getHighlightElement() { return highlightElement; }
  void  setHighlightElement(RElement* _el);
  void  delHighlightElement() { drawHighlightElement(false); }
  void  drawHighlightElement(bool _drawNew=true);

  void  setHighlightFilter(elementEnum _typ=T_ALL, RElement* _exception=0);

  void  drawActiveBorder(bool _draw=true);
  
  void  setUndoP(byte _up) { undoP=_up; }
  void  setRedoP(byte _rp) { redoP=_rp; }
  void  setNextP(byte _np) { nextP=_np; }
  void  terminateAction();
  void  increaseNextP();
  byte  getUndoP() { return undoP; }
  byte  getRedoP() { return redoP; }
  byte  getNextP() { return nextP; }
  void  updateUndoRegister();
  void  shiftUndoRegister();

public slots:
  void  show();
  void  setDelay(int _val);
  void  setRapid(bool _on) { if(_on) setFlag(G_RAPID);
                             else    delFlag(G_RAPID);
                             if(!drawing && autoUpdateGraphic) repaint(); }
  void  setDetails(bool _on) { if(_on) setFlag(G_DETAILS);
                               else    delFlag(G_DETAILS);
                               if(!drawing && autoUpdateGraphic) repaint(); }
  void  setBroken(bool _on) { if(_on) setFlag(G_BROKEN);
                              else    delFlag(G_BROKEN);
                              if(!drawing && autoUpdateGraphic) repaint(); }
  void  setGrid(bool _on) { if(_on) setFlag(G_GRID);
                            else    delFlag(G_GRID);
                            if(!drawing && autoUpdateGraphic) repaint(); }
  void  simulateSmooth();
  void  simulate();
  void  simulationDelay();
  
  void  highlightBrokenContours();
  void  resetHighlighting();
  void  setBehavior(int _bvr);
  void  graphicHasChanged();

  void  selectLayer(const char* _name);

  void  editClipCopy(float _x, float _y);
  void  editClipCut(float _x, float _y);
  void  editClipPaste(float _x, float _y);
  void  undo();
  void  redo();

signals:
  void  updateLayers();
  void  zoomChanged();
  void  graphicChanged();
  void  activated();

protected:
  void    paintEvent(QPaintEvent *_ev);
  void    resizeEvent(QResizeEvent *_ev);
  void    enterEvent(QEvent *);
  void    leaveEvent(QEvent *);
  
  int     getREvent(QMouseEvent* _evm, QKeyEvent* _evk);
  
  void    mousePressEvent(QMouseEvent* _ev);
  void    mouseReleaseEvent(QMouseEvent* _ev);
  void    mouseMoveEvent(QMouseEvent* _ev);

  void    keyPressEvent(QKeyEvent* _ev);
  void    keyReleaseEvent(QKeyEvent* _ev);

public:
  void    distributeEvent(QMouseEvent* _evm,
                          QKeyEvent*   _evk );
  void    distributeEvent(int _rEvent);
  
  void    enableEvents() { eventsEnabled=true; }
  void    disableEvents() { eventsEnabled=false; }
  
  void    setPrevVisible(bool _visible) { prevVisible=_visible; }
  
  void    setMarkVisible(bool _visible) { markVisible=_visible; }
  void    setPrevElementVisible(bool _visible) { prevElementVisible=_visible; }
  void    setPrevElementsVisible(bool _visible) { prevElementsVisible=_visible; }
  
  void    saveCurrentActionSettings();
  void    restoreLastActionSettings();
  
  void    doBehavior(int _rEvent);
  void    doBehaviorFreePoint(int _rEvent);
  void    doBehaviorFreePointPrevRect(int _rEvent);
  void    doBehaviorFreePointPrevRectCenter(int _rEvent);
  void    doBehaviorSnapNextElement(int _rEvent);

  void    debugLayers();
  void    debugElements();
  void    debugParameters();

private:
  
  // Graphics data:
  //
  QString   fileName;     // name of loaded file (with path)
  int       menuId;       // Menu with this id is linked to
                          //   this graphic document
  float     zoom;         // Zoom factor 
                          //   (from real coordinates 
                          //    to screen coordinates)
  float     relZeroX,     // Relative zero point position X
            relZeroY;     //                              Y
  int       offsetX,      // Offset for zero point in 
            offsetY;      //   pixel from top left
	
  QRect     currentZoomRect; // Zoom rectangle which was used at last
	
  float     printerFactorX,  // Factor for printing in X (from MM to printer-units)
            printerFactorY;  // Factor for printing in Y (from MM to printer-units)

  bool autoUpdateGraphic; // Update Graphic automatically
                          //   (on changings of view) ?

  QColor currentColor;      // Current Color for elements
  int    currentWidth;      // Current width for elements
  Qt::PenStyle currentStyle; // Current pen style for elements

  float  currentGridWidth;   // The current grid width in mm

  // Element:
  //
  QList<RElement> element;  // List of elements

  // Preview / events:
  //
  QList<RElement> preview;   // List of preview elements
  bool prevElementsVisible;  // Are the preview elements visible?
  RElement borderElement;    // Element for saving the previews border
  
  RElement  previewElement;  // One preview Element
  bool prevElementVisible;   // Is the preview element visible?

  RElement* highlightElement;   // One highlight Element
  bool highlightElementVisible; // Is the highlight element visible?

  elementEnum snapFilterTyp;     // Filter for snapping elements (snap only 
                                 //   elements in this group (T_ALL, T_LINE, ..))
  RElement* snapFilterException; // Don't snap to this element
  
  bool prevVisible;          // Is the preview visible (behavior -> zoom func.)?
  bool markVisible;          // Is the preview mark visible?
  bool markEnabled;          // Is the preview mark enabled?
  
  /*! Is the event handling enabled?
  */
  bool eventsEnabled;

  // Layer:
  //
  QList<RLayer> layer;        // List of layers
  bool autoUpdateLayerList;   // Update Layerlist automatically
                              //   (on add / remove) ?
  int  selectedLayer;         // Currnetly selected layer (in layerlist)

  // Info:
  //
  bool autoUpdateInfo;        // Update Info automatically
                              //   (on changings) ?

  // Behavior / action handlers:
  //
  QCursor*  lastCursor;       // Last used cursor (last action)
  int       lastAction;       // action id which was active before current action or -1
  int       lastState;        // state of last action or -1
  int       currentAction;    // action id which is currently active or -1
  int       currentState;     // state of current action or -1
  int       nextAction;       // action id which gets active by clicking enough button or -1

  int       mouseX, mouseY;   // mouse coordinates

  float     keyX, keyY;       // Key coordinates (set by keyboard for REVENT_COORDSET)
  bool      enableKeyCoords;  // So we currently use key coordinates?

  RActZoom* actZoom;          // action handler for zoom functions
  RActDraw* actDraw;          // action handler for draw functions
  RActEdit* actEdit;          // action handler for edit functions
  RActTag*  actTag;           // action handler for tag functions
  RActCam*  actCam;           // action handler for cam functions
  RSnap*    snap;             // snap handler
  int       lastBehavior;     // Last Behavior (see "rbehaviordef.h")
  int       behavior;         // Behavior (see "rbehaviordef.h")

  bool      busy;             // Graphic is busy (no actions please)
  bool      previewUpdate;    // Distribute mouse move event on painting?
  bool      active;           // Graphic receives simulation events
  bool      updateBroken;     // Graphic contents has changed /
                              //  must update highlighting of broken contours
  bool      noMoreMemory;     // Flag that indicates lack of memory
  
  // Undo / redo:
  //
  byte undoP,                 // Position of next undo step
       redoP,                 // Position of next redo step
       nextP;                 // Position of next action
  byte undoReg[256];          // Undo register
                              //               '0' = no function
                              //               'u' = we can undo this
                              //               'r' = we can redo this

  // Simulation:
  //
  bool   drawing;             // we're currently drawing (dont break)
  bool   printing;            // we're currently printing (drawing is also true) dont clip
  int    delay;               // Simulation delay
  int    delayPosition;       // Simulation delay position (0-100)
  bool   smooth;              // Smooth simulation?

  // Nc-specific data:
  //
  bool   initialized;          // Graphic is initialized with
                               //   user defined datas (dont read
                               //   them again from ini or mas)
  
  QString machineGenerator;    // Machine generator file with path ("c:\\...\\din.cxm")

  QString extension;           // Extension of File (NC, D, CNC, ...)
                               //   always CAPITAL Letters
  char   mode;                 // mode
                               //   'a': auto
                               //   'm': manual
  bool   optimization;         // Optimization
  bool   contourCenter;        // Contours from center
  char   sorting;              // Sorting    n=none
                               //            l=layer
                               //            c=contours
  char   direction;            // Direction  n=no certain
                               //            2=clockwise
                               //            3=counterclockwise
                               //            s=longer side first
  bool   pointing;             // Pointing
  
  bool    contourStartInHeader; // Contour Start is included in header
  bool    layerStartInHeader;   // Layer Start is included in header
  QString numberStart;          // Start number
  QString numberStep;           // number step
  QString maxArcAngle;          // max angle for arcs / Parameter
  QString maxArcLength;         // max length for arcs / Parameter
  QString tolerance;            // tolerance as value
  QString digits;               // Exactness (digits after point)
  QString factor;               // factor (writing)
  
  float  pointingAngle;        // Pointing Angle
  float  pointingDepth;        //          Depth
  float  pointingLimit;        //          Limit
  float  pointingMaxLength;    //          Max length

  int    contNum;              // Number of contours

  // Parameter:
  //
  QString parameterString;     // String with parameter letters
                               //   ("lft" for $l, $f, $t)
  QString parameter[DEF_MAXPARAMETERS];
                               // Parameters

  // Layer Parameter:
  //
  QString layerParameterString;  // String with layer parameter letters
                                 //   ("lft" for $l, $f, $t)
  QString layerParameter[DEF_MAXLAYERS][DEF_MAXPARAMETERS];

  // CAD specific data:
  //
  //! Font number
  int    fontNumber;
  //! Font proportional
  bool   fontProportional;
  //! Font with fixed width
  bool   fontFixedWidth;
  //! Font height
  float  fontHeight;
  //! Font letter spacing
  float  fontLetterSpacing;
  //! Font word spacing
  float  fontWordSpacing;
  //! Font line distance
  float  fontLineDistance;
  //! Font angle
  float  fontAngle;
  //! Font radius
  float  fontRadius;
  //! Font flags <br> (E_STRAIGHT, E_ROUNDIN, E_ROUNDOUT, E_LEFT, E_RIGHT, E_CENTER)
  RFlags fontFlags;

};


#endif

// EOF




















