/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "GuiDialogPreview.h"				// include gtk, gtk--
#include "GlobalSanityCheck.h"
#include "GlobalTrace.h"



/* 
 * constructor: init everything and do sanity checks
 */
GuiDialogPreview::GuiDialogPreview (char *windowTitle, char *frameTitle, 
				    int tableX, int tableY) : 
		  d_vbPreview (FALSE, 0), 
		  d_btnOK ("OK"),
		  d_btnApply ("Apply"),
		  d_btnCancel ("Cancel"),
		  d_cbUsePreview ("Use Preview")
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ GuiDialogPreview\n");

	SanityCheck::bailout ((!frameTitle), "frameTitle==NULL", "GuiDialogPreview::GuiDialogPreview");

	if (windowTitle)
		p_windowTitle = strdup (windowTitle);
	else
		p_windowTitle = strdup ("GuiDialogPreview Window");

	p_frameTitle = strdup (frameTitle);	// required

	p_frmBase = p_frmPreview = p_frmPreviewGroup = NULL;
	p_pvwMain = NULL;
	p_drawArea = NULL;
	d_size = 100;
	d_frameBorder = 5;

	SanityCheck::bailout ((tableX<6), "tableX<6", "GuiDialogPreview::GuiDialogPreview");
	SanityCheck::bailout ((tableY<4), "tableY<4", "GuiDialogPreview::GuiDialogPreview");
	SanityCheck::bailout ((tableX>25), "tableX<25", "GuiDialogPreview::GuiDialogPreview");
	SanityCheck::bailout ((tableY>25), "tableY>25", "GuiDialogPreview::GuiDialogPreview");

	p_tblMain = new Gtk_Table (tableX, tableY, FALSE);
	b_applyHit = FALSE;

	SanityCheck::bailout ((!p_tblMain), "p_tblMain==NULL", "GuiDialogPreview::GuiDialogPreview");

        this->set_position (GTK_WIN_POS_MOUSE);
}


/* 
 * destructor: release allocated memory
 */
GuiDialogPreview::~GuiDialogPreview ()
{
	if (p_windowTitle) free (p_windowTitle);	// c-type alloc from strdup
	if (p_frameTitle) free (p_frameTitle);	// c-type alloc from strdup
	if (p_frmBase) delete p_frmBase;
	if (p_frmPreview) delete p_frmPreview;
	if (p_pvwMain) delete p_pvwMain;
	if (p_tblMain) delete p_tblMain;

	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- GuiDialogPreview\n");
}


/*
 *  (virtual) buildDialogWindow: call the proper methods to assemble the window
 */
void GuiDialogPreview::buildDialogWindow ()
{
	this->set_usize (300, 250);
	this->set_title (p_windowTitle);

	this->get_vbox()->set_border_width (2);

	setupVBox ();
	insertPreview ();
	insertOptions ();
	fillActionArea ();

	if (p_frmBase)
		p_frmBase->show ();
	this->show ();
}


/*
 *  setupVBox: setup the basic frame of the vbox 
 */
void GuiDialogPreview::setupVBox ()
{
	p_frmBase = new Gtk_Frame (p_frameTitle);
	p_frmBase->set_shadow_type (GTK_SHADOW_ETCHED_IN);
	p_frmBase->set_border_width (0);
	this->get_vbox()->pack_start (*p_frmBase, TRUE, TRUE, 0);
	p_frmBase->show ();	
	this->get_vbox()->show ();
}


/*
 *  insertPreview: insert the preview into the main frame/vbox
 */
void GuiDialogPreview::insertPreview ()
{
	// add main table
	p_frmBase->add (*p_tblMain);

	// add bounding frame + main preview vbox 
	p_frmPreviewGroup = new Gtk_Frame ("Preview");
	p_frmPreviewGroup->set_border_width (d_frameBorder);
	p_frmPreviewGroup->add (d_vbPreview);
	p_frmPreviewGroup->show ();

	// add preview frame to table
	p_tblMain->attach (*p_frmPreviewGroup, 1, 3, 1, 4, 0, 0, 0, 0);
		//GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 2, 3);

	// build preview Frame and add it to preview vbox
	p_frmPreview = new Gtk_Frame ();
	p_frmPreview->set_shadow_type (GTK_SHADOW_ETCHED_IN);
	p_frmPreview->set_border_width ((int)(d_frameBorder/2));
	d_vbPreview.pack_start (*p_frmPreview, FALSE, FALSE, 0);
	p_frmPreview->show ();

	// setup checkbutton
	d_cbUsePreview.set_active (TRUE);
	connect_to_method (d_cbUsePreview.clicked, this, &GuiDialogPreview::checkboxCallbackPreview);
	d_vbPreview.pack_start (d_cbUsePreview, FALSE, FALSE, 0);
	d_cbUsePreview.show ();

	d_vbPreview.show ();
	p_tblMain->show ();

	p_drawArea = new GuiBufferedDrawingArea ();
	p_drawArea->setSync (0);
	p_frmPreview->add (*p_drawArea);
	p_drawArea->show ();
	p_drawArea->set_usize (d_size, d_size);
}


/*
 *  clearPreview: clear the preview drawing area
 */
void GuiDialogPreview::clearPreview ()
{
	if (p_drawArea)
		p_drawArea->clear ();
	else
		SanityCheck::warning (TRUE, "p_drawArea==NULL", "GuiDialogPreview::clearPreview");
}


/*
 *  (virtual) fillVBox: fill the already set up d_vBoxOptions which remains 
 * 	should be used to populate the individual dialog's terraform content
 * 	to the right side of the preview. 
 */
void GuiDialogPreview::insertOptions ()
{
	printf ("Default insertOptions ()\n"); fflush (stdout);
}


/*
 *  (virtual) fillActionArea: fill the dialog's action area 
 */
void GuiDialogPreview::fillActionArea ()
{
	connect_to_method (d_btnOK.clicked, this, &GuiDialogPreview::buttonCallbackOK);
	this->get_action_area()->pack_start (d_btnOK, TRUE, TRUE, 0);
	d_btnOK.show ();

	connect_to_method (d_btnApply.clicked, this, &GuiDialogPreview::buttonCallbackApply);
	this->get_action_area()->pack_start (d_btnApply, TRUE, TRUE, 0);
	d_btnApply.show ();

	connect_to_method (d_btnCancel.clicked, this, &GuiDialogPreview::buttonCallbackCancel);
	this->get_action_area()->pack_start (d_btnCancel, TRUE, TRUE, 0);
	d_btnCancel.show ();
}


/*
 *  iterateEvents: process the pending GTK events
 */
void GuiDialogPreview::iterateEvents ()
{
	Gtk_Main *gtkMain = Gtk_Main::instance();
	while (gtkMain->events_pending())
		gtkMain->iteration ();
}



/*
 *  (virtual) buttonCallbackOK: process the OK button 
 */
void GuiDialogPreview::buttonCallbackOK ()
{
	this->hide ();
	gdk_flush ();

	// if apply has been hit in the previous round, just close the window
	if (!this->b_applyHit)
		buttonCallbackApply ();
	b_applyHit = FALSE;
}


/*
 *  (virtual) buttonCallbackApply: process the Apply button 
 */
void GuiDialogPreview::buttonCallbackApply ()
{
	cout << "GuiDialogPreview: Default Callback Apply\n";
	b_applyHit = TRUE;
}


/*
 *  (virtual) buttonCallbackCancel: process the Cancelbutton 
 */
void GuiDialogPreview::buttonCallbackCancel ()
{
	this->hide ();
	b_applyHit = FALSE;
}


/*
 *  (virtual) checkboxCallbackPreview: process the 'use preview' checkbox 
 */
void GuiDialogPreview::checkboxCallbackPreview ()
{
}


/*
 *  (virtual) delete_event_impl: provide a function to hande window close 
 *				 request
 */
gint GuiDialogPreview::delete_event_impl (GdkEventAny *e)
{
	this->hide ();
	return (0);
}

