/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <time.h>

#include "TFDialogContourLines.h"
#include "HeightFieldExport.h" 
#include "GlobalDefs.h"
#include "GlobalSanityCheck.h"
#include "GlobalTrace.h"
#include "tf_glistFlexArr.h"


/*
 *  constructor: initialize all data members 
 */
TFDialogContourLines::TFDialogContourLines (HeightField *HF, HeightFieldDraw *HFD)
	        : TFBaseDialog (HF, HFD, "Terraform Contour Lines Export Dialog", 
				"Parameters"),
		  d_hbFilename (TRUE, 5),
		  d_hbNLevels (TRUE, 5),
		  d_hbEpsilon (TRUE, 5),
		  d_hbSquareSize (TRUE, 5),
		  d_hbExportToFile (TRUE, 5),
		  d_lblFilename (_("Output file")),
		  d_lblNLevels (_("Number of levels")),
		  d_lblEpsilon (_("Breadth Factor")),
		  d_lblSquareSize (_("Square Radius")),
		  d_lblExportToFile (_("Export to File")),
		  d_adjNLevels (10, 0, 50, 1), 
		  d_adjEpsilon (0.25, 0, 1, 0.01), 
		  d_adjSquareSize (2, 1, 5, 1), 
		  d_hsNLevels (d_adjNLevels),
		  d_hsEpsilon (d_adjEpsilon),
		  d_hsSquareSize (d_adjSquareSize)
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ TFDialogContourLines\n");

	this->setHFobjs (HF, HFD);
	sprintf (this->p_windowTitle, _("Isograms (Contour Lines): %s"), p_HF->getName());
	buildDialogWindow ();
	connect_to_method (d_cbExportToFile.clicked, this, 
			&TFDialogContourLines::checkboxCallback);
}


/*
 *  destructor: clean up 
 */
TFDialogContourLines::~TFDialogContourLines ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- TFDialogContourLines\n");
}


/*
 *  fillVBox: fill the dialog's VBox
 */
void TFDialogContourLines::fillVBox ()
{
	SanityCheck::bailout ((!this->p_frmBase), "TFDialogContourLines called without Frame label", 
		"TFDialogContourLines::setHFobjs");

	d_hbNLevels.pack_start (d_lblNLevels, TRUE, TRUE, s_HBOff);
	d_lblNLevels.show ();
	d_hsNLevels.set_digits (0);
	d_hsNLevels.set_draw_value (TRUE);
	d_hbNLevels.pack_end (d_hsNLevels, TRUE, TRUE, s_HBOff);
	d_hsNLevels.show ();
	this->d_vbOptions.pack_start (d_hbNLevels, TRUE, TRUE, s_VBOff);
	d_hbNLevels.show ();

	d_hbEpsilon.pack_start (d_lblEpsilon, TRUE, TRUE, s_HBOff);
	d_lblEpsilon.show ();
	d_hsEpsilon.set_digits (2);
	d_hsEpsilon.set_draw_value (TRUE);
	d_hbEpsilon.pack_end (d_hsEpsilon, TRUE, TRUE, s_HBOff);
	d_hsEpsilon.show ();
	this->d_vbOptions.pack_start (d_hbEpsilon, TRUE, TRUE, s_VBOff);
	d_hbEpsilon.show ();

	d_hbSquareSize.pack_start (d_lblSquareSize, TRUE, TRUE, s_HBOff);
	d_lblSquareSize.show ();
	d_hsSquareSize.set_digits (0);
	d_hsSquareSize.set_draw_value (TRUE);
	d_hbSquareSize.pack_end (d_hsSquareSize, TRUE, TRUE, s_HBOff);
	d_hsSquareSize.show ();
	this->d_vbOptions.pack_start (d_hbSquareSize, TRUE, TRUE, s_VBOff);
	d_hbSquareSize.show ();

	this->d_vbOptions.pack_start (d_sepMain, TRUE, TRUE, s_VBOff);
	d_sepMain.show ();

	d_hbExportToFile.pack_start (d_lblExportToFile, TRUE, TRUE, s_HBOff);
	d_lblExportToFile.show ();
	d_hbExportToFile.pack_start (d_cbExportToFile, TRUE, TRUE, s_HBOff);
	d_cbExportToFile.set_active (FALSE);
	d_cbExportToFile.show ();
	this->d_vbOptions.pack_start (d_hbExportToFile, TRUE, TRUE, s_VBOff);
	d_hbExportToFile.show ();

	d_hbFilename.pack_start (d_lblFilename, TRUE, TRUE, s_HBOff);
	d_lblFilename.set_sensitive (FALSE);
	d_lblFilename.show ();
	d_hbFilename.pack_start (d_enFilename, TRUE, TRUE, s_HBOff);
	d_enFilename.set_text ("contour.ps");
	d_enFilename.set_sensitive (FALSE);
	d_enFilename.show ();
	this->d_vbOptions.pack_start (d_hbFilename, TRUE, TRUE, s_VBOff);
	d_hbFilename.show ();

	this->p_frmBase->add (this->d_vbOptions);

	this->d_vbOptions.show ();
	this->p_frmBase->show ();
	this->get_vbox()->show ();
}


/*
 *  buildDialogWindow:
 */
void TFDialogContourLines::buildDialogWindow ()
{
	this->set_usize (300, 275);
	this->set_title (this->p_windowTitle);

	this->get_vbox()->set_border_width (2);

	this->setupVBox ();
	fillVBox ();
	fillActionArea ();

	this->show ();
}


/*
 *  checkboxCallback: process a press on the ExportToFile checkbox 
 */
void TFDialogContourLines::checkboxCallback ()
{
	bool            status=d_enFilename.is_sensitive ();

	d_lblFilename.set_sensitive (!status);
	d_enFilename.set_sensitive (!status);
}



/*
 *  buttonCallbackApply: rotate the Height Field
 */
void TFDialogContourLines::buttonCallbackApply ()
{
	SanityCheck::bailout ((!p_HFO), "p_HFO==NULL", "TFDialogContourLines::buttonCallbackApply");

	char 			*fname;
	int				levels, squareSize;
	float			epsilon;
	TFGListFlexArr		*contourList;

	if (d_cbExportToFile.get_active())
		fname = const_cast<char*>(d_enFilename.get_text().c_str());
	levels = (int)d_adjNLevels.get_value ();
	epsilon = d_adjEpsilon.get_value ();
	squareSize = (int)d_adjSquareSize.get_value ();

	contourList = p_HFO->contourLines (levels, epsilon, squareSize);
	if (d_cbExportToFile.get_active())
		{
		HeightFieldExport	*HFE = new HeightFieldExport (p_HF);

		HFE->exportContourLinesToPS (contourList, p_HFD, fname);
		delete HFE;
		}
		
	p_HF->setContourList (contourList, time(NULL));
	p_HFD->draw ();

	this->b_applyHit = TRUE;
}


