// ---------------------------------------------------------------------------
// - Closure.hpp                                                             -
// - aleph engine - closure class implementation                             -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "Interp.hpp"
#include "Return.hpp"
#include "Lexical.hpp"
#include "Closure.hpp"
#include "Reserved.hpp"
#include "Argument.hpp"
#include "Exception.hpp"

namespace aleph {

  // create a new closure 

  Closure::Closure (t_ctype type, Cons* argl, Object* form) {
    // save the arguments
    d_type = type;
    p_form = Object::iref (form);
    d_argc = 0;
    // create a local nameset
    p_lset = new Localset;
    Object::iref (p_lset);
    p_lset->symcst ("self", this);
    // add the arguments
    try {
      while (argl != nilp) {
	addarg (argl->getcar ());
	argl = argl->getcdr ();
      }
    } catch (...) {
      Object::dref (p_lset);
      throw;
    }
  }

  // destroy this closure

  Closure::~Closure (void) {
    Object::dref (p_form);
    Object::dref (p_lset);
  }

  // return the class name

  String Closure::repr (void) const {
    return "Closure";
  }

  // add an argument by name to this closure

  void Closure::addarg (const String& name, const bool flag) {
    // check if the argument already exists
    if (p_lset->exists (name) == true)
      throw Exception ("argument-error", "duplicate argument name", name);
    // create a new argument and bind it
    Argument* arg = new Argument (name, d_argc++);
    arg->setconst (flag);
    p_lset->bind (name,arg);
  }

  // add an argument by object - we can either add a lexical name, a
  // form with a const flag or a string

  void Closure::addarg (Object* object) {
    String* sobj = dynamic_cast <String*> (object);
    if (sobj != nilp) {
      addarg (*sobj, false);
      return;
    }
    Lexical* lex = dynamic_cast <Lexical*> (object);
    if (lex != nilp) {
      addarg (lex->toLiteral (), false);
      return;
    }
    Cons* form = dynamic_cast <Cons*> (object);
    if (form != nilp) {
      if (form->length () != 2)
	throw Exception ("argument-error", "invalid argument list form");
      Reserved* crsv = dynamic_cast <Reserved*> (form->getcar  ());
      Lexical*  alex = dynamic_cast <Lexical*>  (form->getcadr ());
      if ((crsv == nilp) || (alex == nilp))
	throw Exception ("argument-error", "invalid argument list form");
      if (crsv->toLiteral () != "const")
	throw Exception ("argument-error", "invalid argument list form");
      addarg (alex->toLiteral (), true);
      return;
    }
    throw Exception ("argument-error", "invalid argument list form");
  }

  // add a closed variable to this closure

  void Closure::addclv (const String& name, Object* object) {
    // check if the argument already exists
    if (p_lset->exists (name) == true)
      throw Exception ("argument-error", "duplicate argument name", name);
    p_lset->bind (name, object);
  }

  // apply this closure with a set of arguments

  Object* Closure::apply (Interp* interp, Nameset* nset, Cons* args) {
    // get the stack context
    Stack*   stk = interp->getstk ();
    Object** sp  = stk->getsp ();
    Object** fp  = stk->getfp ();
    // evaluate the arguments on the stack
    long argc = 0;
    while (args != nilp) {
      argc++;
      Object* car = args->getcar ();
      stk->push ((car == nilp) ? nilp : car->eval (interp, nset));
      args = args->getcdr ();
    }

    // check the number of arguments
    if (argc != d_argc)
      throw Exception ("argument-error", "invalid number of arguments");

    // set the parent nameset
    Nameset* lset = new Localset (p_lset);
    Object::iref (lset);
    if (d_type == LAMBDA) 
      lset->setparent (nset);
    else
      lset->setparent (interp->getnset ());

    // execute this closure - catch any exception so we unlink and destroy
    // the local set. The stack is unwind after the execution. Note how the
    // stack frame is created and restored. If you have a better idea - well
    // let me know.
    Object*  result = nilp;
    try {
      stk->setfp (sp);
      result = Object::iref (p_form->eval (interp,lset));
      stk->unwind (sp, fp);
      Object::dref (lset);
    } catch (const Return& r) {
      result = Object::iref (r.getobj ());
      stk->unwind (sp, fp);
      Object::dref (lset);
    } catch (...) {
      stk->unwind (sp, fp);
      delete lset;
      throw;
    }
    Object::tref (result);
    return result;
  }
}
