// ---------------------------------------------------------------------------
// - Globalset.cpp                                                           -
// - aleph engine - global set class implementation                          -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "Symbol.hpp"
#include "Globalset.hpp"
#include "Exception.hpp"

namespace aleph {
  // create a new global set

  Globalset::Globalset (void) {
    p_table = new HashTable;
    symcst (".", this);
  }

  // create a new global set with a parent nameset

  Globalset::Globalset (Nameset* nset) {
    p_table = new HashTable;
    setparent (nset);
    symcst (".", this);
  }

  // destroy this global set

  Globalset::~Globalset (void) {
    delete p_table;
  }

  // return the class name

  String Globalset::repr (void) const {
    return "Globalset";
  }

  // bind a new object by name
  void Globalset::bind (const String& name, Object* object) {
    p_table->add (name, object);
  }

  // return true if the name exists in the global set

  bool Globalset::exists (const String& name) const {
    return p_table->exists (name);
  }

  // find an object by name but do not evaluate

  Object* Globalset::find (const String& name) const {
    Object* obj = p_table->get (name);
    if (obj != nilp) return obj;
    if (p_parent != nilp) return p_parent->find (name);
    return nilp;
  }

  // remove an object by name in this global set

  void Globalset::remove (const String& name) {
    p_table->remove (name);
  }

  // set an object as a const object by name

  Object* Globalset::cdef (Interp* interp, Nameset* nset, const String& name, 
			   Object* object) {
    // try first to find the object
    Object* obj = p_table->get (name);
    if (obj != nilp) {
      obj->cdef (interp, nset, object);
      return object;
    }

    // the object is not found - create a symbol and bind it
    Symbol* sym = new Symbol (name, object);
    sym->setconst (true);
    p_table->add (name, sym);
    return object;
  }

  // set an object to this object by name

  Object* Globalset::vdef (Interp* interp, Nameset* nset, const String& name,
			   Object* object) {
    // try first to find the object
    Object* obj = p_table->get (name);
    if (obj != nilp) {
      obj->vdef (interp, nset, object);
      return object;
    }

    // the object is not found - create a symbol and bind it
    Symbol* sym = new Symbol (name, object);
    p_table->add (name, sym);
    return object;
  }

  // evaluate an object in the current nameset by name

  Object* Globalset::eval (Interp* interp, Nameset* nset, const String& name) {
    // try first to find the object
    Object* obj = p_table->get (name);
    if (obj != nilp) return obj->eval (interp, nset);
    // try in the parent
    if (p_parent != nilp) return p_parent->eval (interp, nset, name);
    // not found
    throw Exception ("eval-error", "unbound symbol", name);
  }
}
