///-*-C++-*-//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998-2000, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

/*
  processheap.h
  ------------------------------------------------------------------------
  We use one processHeap for the whole program.
  ------------------------------------------------------------------------
  @(#) $Id: processheap.h,v 1.51 2000/02/12 21:45:53 emery Exp $
  ------------------------------------------------------------------------
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/

#ifndef _PROCESSHEAP_H_
#define _PROCESSHEAP_H_

#include "config.h"

#include <stdio.h>
#include <stdlib.h>

#include "arch-specific.h"
#include "heap.h"
#if USE_PRIVATE_HEAPS
#include "privateheap.h"
#define HEAPTYPE privateHeap
#else
#define HEAPTYPE threadHeap
#include "threadheap.h"
#endif

#if HEAP_LOG
#include "memstat.h"
#include "log.h"
#endif

class processHeap : public hoardHeap {

public:

  // Always grab at least this many superblocks' worth of memory which
  // we parcel out.
  enum { REFILL_NUMBER_OF_SUPERBLOCKS = 16 };

  processHeap (void);

  ~processHeap (void) {
#if HEAP_STATS
    stats();
#endif
#if HEAP_LOG
    for (int i = 0; i < MAX_HEAPS + 1; i++) {
      _log[i].close();
    }
#endif
  }

  // Memory deallocation routines.
  void free (void * ptr);

  // Print out statistics information.
  void stats (void);

  // Get a thread heap index.
  inline int getHeapIndex (void);

  // Get the thread heap with index i.
  inline HEAPTYPE& getHeap (int i);

  // Extract a superblock.
  inline superblock * acquire (const int c, HEAPTYPE * to);

  // Insert a superblock.
  inline void release (superblock * sb);

  // Obtain a superblock buffer from our store.  We always ask the
  // system for at least the REFILL number of superblocks (above), and
  // then parcel them out to the threads.
  // Returns 1 if this is a recycled superblock.
  static int getSuperblockBuffer (char *&);

  // Add an empty superblock to the recycle list.
  inline void recycle (superblock * sb);

#if HEAP_LOG
  // Get the log for index i.
  inline Log<MemoryRequest>& getLog (int i);
#endif

#if HEAP_FRAG_STATS
  // Declare that we have allocated an object.
  void setAllocated (int requestedSize,
		     int actualSize);

  // Declare that we have deallocated an object.
  void setDeallocated (int requestedSize,
		       int actualSize);

  // Return the number of wasted bytes at the high-water mark
  // (maxAllocated - maxRequested)
  inline int getFragmentation (void);

  int getMaxAllocated (void) {
    return _maxAllocated;
  }

  int getInUseAtMaxAllocated (void) {
    return _inUseAtMaxAllocated;
  }

  int getMaxRequested (void) {
    return _maxRequested;
  }
  
#endif

private:

  // Get and lock the owner of a given superblock.
  inline hoardHeap * getAndLockOwner (superblock *);

  // Prevent copying and assignment.
  processHeap (const processHeap&);
  const processHeap& operator= (const processHeap&);

  // The per-thread heaps.
  HEAPTYPE theap[MAX_HEAPS];

#if HEAP_FRAG_STATS
  int _currentAllocated;
  int _currentRequested;
  int _maxAllocated;
  int _maxRequested;
  int _inUseAtMaxAllocated;
  int _fragmentation;

  hoardLockType _statsLock;		// A lock to protect these statistics.
#endif

#if HEAP_LOG
  Log<MemoryRequest> _log[MAX_HEAPS + 1];
#endif

  // A lock for the buffer of superblocks.
  static hoardLockType _bufferLock;

  // Completely empty superblocks.
  static superblock * _recycleBin;
};


HEAPTYPE& processHeap::getHeap (int i)
{
  assert (i >= 0);
  assert (i < MAX_HEAPS);
  return theap[i];
}


#if HEAP_LOG
Log<MemoryRequest>& processHeap::getLog (int i)
{
  assert (i >= 0);
  assert (i < MAX_HEAPS + 1);
  return _log[i];
}
#endif


// Return ceil(log_2(num)).
// num must be positive.
static int lg (int num)
{
  assert (num > 0);
  int power = 0;
  int n = 1;
  // Invariant: 2^power == n.
  while (n < num) {
    n <<= 1;
    power++;
  }
  return power;
}


// Hash out the thread id to a heap and return an index to that heap.
int processHeap::getHeapIndex (void) {
  // Here we use the number of processors as the maximum number of heaps.
  // In fact, for efficiency, we just round up to the highest power of two,
  // times two.
  static const int numProcessors = hoardGetNumProcessors();
  static const int numProcessorsMask = (1 << (lg(numProcessors) + 1)) - 1;

  int tid = hoardGetThreadID() & numProcessorsMask;
  assert (tid < MAX_HEAPS);

  return tid;
}


superblock * processHeap::acquire (const int sizeclass,
				   HEAPTYPE * to)
{

  lock ();

  // This removes the warning when assertions are turned off.
  to = to;

  // We can't acquire a superblock from ourselves.
  assert ((hoardHeap *) to != (hoardHeap *) this);

  // Remove the superblock with the most free space.
  superblock * maxSb = removeMaxSuperblock (sizeclass);

#if HEAP_FRAG_STATS
  // Pretend the released superblock was allocated.
  if (maxSb) {
    // setAllocated (0, maxSb->getNumBlocks() * sizeFromClass(maxSb->getBlockSizeClass()));
  }
#endif

  unlock ();

  return maxSb;
}


// Put a superblock back into our list of superblocks.
void processHeap::release (superblock * sb)
{
  assert (EMPTY_FRACTION * sb->getNumAvailable() > sb->getNumBlocks());

  // Insert the superblock.
  insertSuperblock (sb->getBlockSizeClass(), sb);

#if HEAP_FRAG_STATS
  // Pretend the released superblock was de-allocated.
  // setDeallocated (0, sb->getNumBlocks() * sizeFromClass(sb->getBlockSizeClass()));
#endif
}


hoardHeap * processHeap::getAndLockOwner (superblock * sb)
{
  // Lock only the owner of this superblock.
  // Since the owner might change between executing
  // getOwner() and locking the owner's heap,
  // we need this loop to make sure we lock the right heap.

  hoardHeap * owner;
  while (1) {
    owner = sb->getOwner();
    if (owner) {
      owner->lock ();
      if (sb->getOwner() != owner) {
	// The superblock moved;
	// we'll have to try again.
	owner->unlock ();
      } else {
	// Success!
	break;
      }
    }
  }
  return owner;
}


void processHeap::recycle (superblock * sb)
{
  assert (sb->getNumAvailable() == sb->getNumBlocks());
  sb->insertBefore (_recycleBin);
  _recycleBin = sb;
}


#endif // _PROCESSHEAP_H_

