/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    5635-34 Springhouse Dr.
    Pleasanton, CA 94588 (USA)
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_sysvideo.c,v 1.11 1999/07/17 07:19:11 slouken Exp $";
#endif

#include <stdio.h>
#include <stdlib.h>

#include <LowMem.h>
#include <Gestalt.h>
#include <Devices.h>
#include <DiskInit.h>
#include <QDOffscreen.h>
#include <Palettes.h>

#include "SDL_video.h"
#include "SDL_error.h"
#include "SDL_syswm.h"
#include "SDL_sysvideo.h"
#include "SDL_video_c.h"
#include "SDL_syswm_c.h"
#include "SDL_lowvideo.h"

/* Hardware surface functions */
static int SDL_LockHWSurface(SDL_Surface *surface);
static void SDL_UnlockHWSurface(SDL_Surface *surface);

/* Private display data */
static GDevice **SDL_Display = nil;
WindowRef SDL_Window = nil;
static SDL_Rect **SDL_modelist = NULL;
static CTabHandle SDL_CTab = nil;
static PaletteHandle SDL_CPal = nil;

static void SDL_UpdateVideoInfo(SDL_VideoInfo *info)
{
	return;
}

int SDL_SYS_VideoInit(SDL_PixelFormat *vformat)
{
	long info;
	
	/* Check out some things about the system */
	Gestalt(gestaltQuickdrawVersion, &info);
	if ( info == gestaltOriginalQD ) {
		SDL_SetError("Color Quickdraw not available");
		return(-1);
	}

	/* Start Macintosh events */
	SDL_InitMacEvents();

	/* Get a handle to the main monitor */
	SDL_Display = GetMainDevice();

	/* Determine pixel format */
	vformat->BitsPerPixel = (**(**SDL_Display).gdPMap).pixelSize;
	switch (vformat->BitsPerPixel) {
		case 16:	/* 5-5-5 RGB */
			vformat->Rmask = 0x00007c00;
			vformat->Gmask = 0x000003e0;
			vformat->Bmask = 0x0000001f;
			break;
		default:
			break;
	}

	/* Create our palette */
	SDL_CTab = (CTabHandle)NewHandle(sizeof(ColorSpec)*256 + 8);
	if ( SDL_CTab == nil ) {
		SDL_OutOfMemory();
		return(-1);
	}
	(**SDL_CTab).ctSeed = GetCTSeed();
	(**SDL_CTab).ctFlags = 0;
	(**SDL_CTab).ctSize = 255;
	CTabChanged(SDL_CTab);
	SDL_CPal = NewPalette(256, SDL_CTab, pmExplicit+pmTolerant, 0);

	/* Get a list of available fullscreen modes */
	SDL_modelist = (SDL_Rect **)malloc((1+1)*sizeof(SDL_Rect *));
	if ( SDL_modelist ) {
		SDL_modelist[0] = (SDL_Rect *)malloc(sizeof(SDL_Rect));
		if ( SDL_modelist[0] ) {
			SDL_modelist[0]->x = 0;
			SDL_modelist[0]->y = 0;
			SDL_modelist[0]->w = (**SDL_Display).gdRect.right;
			SDL_modelist[0]->h = (**SDL_Display).gdRect.bottom;
		}
		SDL_modelist[1] = NULL;
	}

	/* Fill in some window manager capabilities */
	SDL_HWCaps.info.wm_available = 1;
	SDL_WMCaps.SetCaption = SDL_SYS_SetWMCaption;

	/* Fill in our hardware acceleration capabilities */
	SDL_UpdateVideoInfo(&SDL_HWCaps.info);
	SDL_HWCaps.LockHWSurface = SDL_LockHWSurface;
	SDL_HWCaps.UnlockHWSurface = SDL_UnlockHWSurface;

	return(0);
}

SDL_Rect **SDL_SYS_ListModes(SDL_Surface *screen,
					SDL_PixelFormat *format, Uint32 flags)
{
	if ( screen->format->BitsPerPixel == format->BitsPerPixel ) {
		if ( (flags & SDL_FULLSCREEN) == SDL_FULLSCREEN ) {
			return(SDL_modelist);
		} else {
			return((SDL_Rect **)-1);
		}
	} else {
		return((SDL_Rect **)0);
	}
}

static short mBarHeight;

static void SDL_HideMenuBar(void)
{
	if ( LMGetMBarHeight() > 0 ) {
		mBarHeight = LMGetMBarHeight();
		LMSetMBarHeight(0);
	}
}
static void SDL_ShowMenuBar(void)
{
	if ( LMGetMBarHeight() == 0 ) {
		LMSetMBarHeight(mBarHeight);
	}
}

/* Various screen update functions available */
static void SDL_WindowUpdate(SDL_Surface *screen,int numrects,SDL_Rect *rects);
static void SDL_DirectUpdate(SDL_Surface *screen,int numrects,SDL_Rect *rects);

static void SDL_SYS_UnsetVideoMode(SDL_Surface *current)
{
	/* Free the current window, if any */
	if ( SDL_Window != nil ) {
		GWorldPtr memworld;
		
		memworld = (GWorldPtr)GetWRefCon(SDL_Window);
		if ( memworld != nil ) {
			UnlockPixels(GetGWorldPixMap(memworld));
			DisposeGWorld(memworld);
		}
		CloseWindow(SDL_Window);
		if ( (current->flags & SDL_FULLSCREEN) == SDL_FULLSCREEN ) {
			SDL_ShowMenuBar();
		}
		SDL_Window = nil;
	}
	current->pixels = NULL;
	current->flags &= ~(SDL_HWSURFACE|SDL_FULLSCREEN);
}

SDL_Surface *SDL_SYS_SetVideoMode(SDL_Surface *current,
				int width, int height, int bpp, Uint32 flags)
{
	Rect wrect;

	/* Free any previous video mode */
	SDL_SYS_UnsetVideoMode(current);

	/* Create the Mac window and SDL video surface */
	current->flags = 0;		/* Clear flags */
	current->w = width;
	current->h = height;
	SetRect(&wrect, 0, 0, width, height);
	OffsetRect(&wrect,
		(SDL_modelist[0]->w-width)/2, (SDL_modelist[0]->h-height)/2);

	if ( (flags & SDL_FULLSCREEN) == SDL_FULLSCREEN ) {
		current->flags |= SDL_HWSURFACE|SDL_FULLSCREEN;
		SDL_Window = NewCWindow(nil, &wrect, "\p", true, plainDBox,
						(WindowPtr)-1, false, 0);
		current->pitch = (**(**SDL_Display).gdPMap).rowBytes & 0x3FFF;
		current->pixels = (**(**SDL_Display).gdPMap).baseAddr;
		SDL_HideMenuBar();
		SDL_SYS_UpdateRects = SDL_DirectUpdate;
	} else {
		GWorldPtr memworld;
		PixMapHandle pixmap;
		SDL_Window = NewCWindow(nil, &wrect, "\p", true, noGrowDocProc,
						(WindowPtr)-1, true, 0);
		SetPalette(SDL_Window, SDL_CPal, false);
		ActivatePalette(SDL_Window);
		if ( NewGWorld(&memworld, 0, &SDL_Window->portRect, SDL_CTab,
							nil, 0) != noErr ) {
			SDL_SetError("NewGWorld() failed");
			return(NULL);
		}
		SetWRefCon(SDL_Window, (long)memworld);
		pixmap = GetGWorldPixMap(memworld);
		LockPixels(pixmap);
		current->pitch = (**pixmap).rowBytes & 0x3FFF;
		current->pixels = GetPixBaseAddr(pixmap);
		SDL_SYS_UpdateRects = SDL_WindowUpdate;
	}
	SetPort(SDL_Window);
	SelectWindow(SDL_Window);
	
	/* We're live! */
	return(current);
}

static int SDL_LockHWSurface(SDL_Surface *surface)
{
	return(0);
}
static void SDL_UnlockHWSurface(SDL_Surface *surface)
{
	return;
}

void SDL_DirectUpdate(SDL_Surface *screen, int numrects, SDL_Rect *rects)
{
	/* The application is already updating the visible video memory */
	return;
}

void SDL_WindowUpdate(SDL_Surface *screen, int numrects, SDL_Rect *rects)
{
	GWorldPtr memworld;
	WindowPtr saveport;
	int i;
	Rect update;
	
	/* Copy from the offscreen GWorld to the window port */
	GetPort(&saveport);
	SetPort(SDL_Window);
	memworld = (GWorldPtr)GetWRefCon(SDL_Window);
	for ( i=0; i<numrects; ++i ) {
		update.left = rects[i].x;
		update.right = rects[i].x+rects[i].w;
		update.top = rects[i].y;
		update.bottom = rects[i].y+rects[i].h;
		CopyBits(&((GrafPtr)memworld)->portBits, &SDL_Window->portBits,
						&update, &update, srcCopy, nil);
	}
	SetPort(saveport);
}

int SDL_SYS_SetColors(SDL_Surface *screen, int firstcolor, int ncolors)
{
	SDL_Palette *palette;
	CTabHandle cTab;
	int i;

	/* Get the colortable from the either the display or window */
	if ( (screen->flags & SDL_FULLSCREEN) == SDL_FULLSCREEN ) {
		cTab = (**(**SDL_Display).gdPMap).pmTable;
	} else {
		cTab = SDL_CTab;
	}

	/* Verify the range of colors */
	if ( (firstcolor+ncolors) > ((**cTab).ctSize+1) ) {
		return(0);
	}
	
	/* Set the screen palette and update the display */
	palette = screen->format->palette;
	for ( i=firstcolor; i<(firstcolor+ncolors); ++i ) {
		(**cTab).ctTable[i].value = i;
		(**cTab).ctTable[i].rgb.red =
			(palette->colors[i].r << 8) | palette->colors[i].r;
		(**cTab).ctTable[i].rgb.green =
			(palette->colors[i].g << 8) | palette->colors[i].g;
		(**cTab).ctTable[i].rgb.blue =
			(palette->colors[i].b << 8) | palette->colors[i].b;
	}
	if ( (screen->flags & SDL_FULLSCREEN) == SDL_FULLSCREEN ) {
		GDevice **odisplay;
		odisplay = GetGDevice();
		SetGDevice(SDL_Display);
		SetEntries(0, (**cTab).ctSize, (ColorSpec *)&(**cTab).ctTable);
		SetGDevice(odisplay);
	}
	return(1);
}

void SDL_SYS_VideoQuit(SDL_Surface *screen)
{
	int i;

	/* Free current video mode */
	SDL_SYS_UnsetVideoMode(screen);

	/* Free palette and restore original one */
	if ( SDL_CTab != nil ) {
		DisposeHandle((Handle)SDL_CTab);
		SDL_CTab = nil;
	}
	if ( SDL_CPal != nil ) {
		DisposePalette(SDL_CPal);
		SDL_CPal = nil;
	}
	RestoreDeviceClut(GetMainDevice());

	/* Free list of video modes */
	if ( SDL_modelist != NULL ) {
		for ( i=0; SDL_modelist[i]; ++i ) {
			free(SDL_modelist[i]);
		}
		free(SDL_modelist);
		SDL_modelist = NULL;
	}
}

void SDL_SYS_FinalQuit(void)
{ }
