/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    5635-34 Springhouse Dr.
    Pleasanton, CA 94588 (USA)
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_sysaudio.c,v 1.8 1999/06/02 20:46:56 slouken Exp $";
#endif

/* Allow access to a raw mixing buffer */

#include <stdio.h>
#include <stdlib.h>
#include <windows.h>
#include <mmsystem.h>

#include "SDL_audio.h"
#include "SDL_mutex.h"
#include "SDL_timer.h"
#include "SDL_audio_c.h"


#define NUM_BUFFERS 2			/* -- Don't lower this! */

static HWAVEOUT  sound = (HWAVEOUT)-1;
static Uint8 *mixbuf;		/* The raw allocated mixing buffer */
static Uint8  playing;
static int buffers_queued = 0;
static int last_queued;
static SDL_mutex *audio_wait = NULL;
static WAVEHDR wavebuf[NUM_BUFFERS];	/* Wave audio fragments */

/* The format of the audio data */
extern int SDL_AudioPaused;
extern int SDL_AudioEnabled;
extern SDL_AudioSpec audio_spec;

/* The Win32 callback for filling the WAVE device */
static void CALLBACK FillSound(HWAVEOUT hwo, UINT uMsg, DWORD dwInstance,
						DWORD dwParam1, DWORD dwParam2)
{
	/* Only service "buffer done playing" messages */
	if ( uMsg != WOM_DONE )
		return;

	/* Increment which buffer is currently playing */
	--buffers_queued;
#ifdef SOUND_DEBUG
	fprintf(stderr, "Finished playing buffer %d\n", playing);
#endif
	playing = (playing+1)%NUM_BUFFERS;

	/* Signal the audio thread that a buffer is done playing */
	SDL_mutexV(audio_wait);
}

static void SetMMerror(char *function, MMRESULT code)
{
	int    err_offset;
	static char errbuf[MAXERRORLENGTH];

	err_offset = strlen(function)+strlen(": ");
	errbuf[err_offset] = 0;
	waveOutGetErrorText(code, &errbuf[err_offset],
						MAXERRORLENGTH-err_offset);
	sprintf(errbuf, "%s: %s", function, &errbuf[err_offset]);
	SDL_SetError("%s", errbuf);
	return;
}

/* Dummy function -- we don't care about a window */
void SDL_SYS_SoundFocus(HWND hwnd)
{
	return;
}

void SDL_SYS_WaitAudio(void)
{
	/* Queue up the wavebufs that have been finished */
	do {
		/* Wait for the current buffer to finish playing */
		SDL_mutexP(audio_wait);

		/* Queue it up */
		last_queued = (last_queued+1)%NUM_BUFFERS;
#ifdef SOUND_DEBUG
		fprintf(stderr, "Queuing up buffer %d  (%d buffers queued)\n",
						last_queued, buffers_queued);
#endif
		waveOutWrite(sound, &wavebuf[last_queued], sizeof(wavebuf[0]));
		++buffers_queued;
	} while ( buffers_queued < NUM_BUFFERS );
}
void SDL_SYS_PlayAudio(void)
{
	return;		/* The sound plays via the waveOut API */
}
Uint8 *SDL_SYS_GetAudioBuf(void)
{
#ifdef SOUND_DEBUG
	fprintf(stderr, "Filling buffer %d\n",
				(playing+NUM_BUFFERS-1)%NUM_BUFFERS);
#endif
	return(wavebuf[(playing+NUM_BUFFERS-1)%NUM_BUFFERS].lpData);
}

void SDL_SYS_CloseAudio(void)
{
	int i;

	/* Wait for the audio to drain */
#ifdef SOUND_DEBUG
	fprintf(stderr, "Sound closing, %d buffers queued\n", buffers_queued);
#endif
	while ( buffers_queued > 0 ) {
		SDL_Delay(250);
	}
	waveOutClose(sound);

	/* Clean up mixing buffers */
	for ( i=0; i<NUM_BUFFERS; ++i ) {
		if ( wavebuf[i].dwUser != 0xFFFF ) {
			waveOutUnprepareHeader(sound, &wavebuf[i],
							sizeof(wavebuf[i]));
			wavebuf[i].dwUser = 0xFFFF;
		}
	}
	/* Free raw mixing buffer */
	if ( mixbuf != NULL ) {
		free(mixbuf);
		mixbuf = NULL;
	}

	/* Release audio trigger */
	if ( audio_wait ) {
		SDL_DestroyMutex(audio_wait);
		audio_wait = NULL;
	}
}

int SDL_SYS_OpenAudio(SDL_AudioSpec *spec)
{
	MMRESULT     result;
	int          i;
	WAVEFORMATEX waveformat;

	/* Initialize the wavebuf structures for closing */
	buffers_queued = 0;
	audio_wait = NULL;
	for ( i = 0; i < NUM_BUFFERS; ++i )
		wavebuf[i].dwUser = 0xFFFF;
	mixbuf = NULL;

	/* Set basic WAVE format parameters */
	memset(&waveformat, 0, sizeof(waveformat));
	waveformat.wFormatTag = WAVE_FORMAT_PCM;

	/* Determine the audio parameters from the AudioSpec */
	switch ( spec->format & 0xFF ) {
		case 8:
			/* Unsigned 8 bit audio data */
			spec->format = AUDIO_U8;
			waveformat.wBitsPerSample = 8;
			break;
		case 16:
			/* Signed 16 bit audio data */
			spec->format = AUDIO_S16;
			waveformat.wBitsPerSample = 16;
			break;
		default:
			SDL_SetError("Unsupported audio format");
			return(-1);
	}
	waveformat.nChannels = spec->channels;
	waveformat.nSamplesPerSec = spec->freq;
	waveformat.nBlockAlign =
		waveformat.nChannels * (waveformat.wBitsPerSample/8);
	waveformat.nAvgBytesPerSec = 
		waveformat.nSamplesPerSec * waveformat.nBlockAlign;

	/* Check the buffer size -- minimum of 1/4 second (word aligned) */
	if ( spec->samples < (spec->freq/4) )
		spec->samples = ((spec->freq/4)+3)&~3;

	/* Update the fragment size as size in bytes */
	SDL_CalculateAudioSpec(spec);

	/* Open the audio device */
	result = waveOutOpen(&sound, WAVE_MAPPER, &waveformat,
					(DWORD)FillSound, 0, CALLBACK_FUNCTION);
	if ( result != MMSYSERR_NOERROR ) {
		SetMMerror("waveOutOpen()", result);
		return(-1);
	}

#ifdef SOUND_DEBUG
	/* Check the sound device we retrieved */
	{
		WAVEOUTCAPS caps;

		result = waveOutGetDevCaps((UINT)sound, &caps, sizeof(caps));
		if ( result != MMSYSERR_NOERROR ) {
			SetMMerror("waveOutGetDevCaps()", result);
			return(-1);
		}
		printf("Audio device: %s\n", caps.szPname);
	}
#endif

	/* Create the audio trigger event */
	audio_wait = SDL_CreateMutex();
	if ( audio_wait == NULL ) {
		return(-1);
	}
	SDL_mutexP(audio_wait);

	/* Create the sound buffers */
	mixbuf = (Uint8 *)malloc(NUM_BUFFERS*spec->size);
	if ( mixbuf == NULL ) {
		SDL_SetError("Out of memory");
		return(-1);
	}
	for ( i = 0; i < NUM_BUFFERS; ++i ) {
		memset(&wavebuf[i], 0, sizeof(wavebuf[i]));
		wavebuf[i].lpData = &mixbuf[i*spec->size];
		wavebuf[i].dwBufferLength = spec->size;
		wavebuf[i].dwFlags = 0;
		result = waveOutPrepareHeader(sound, &wavebuf[i],
							sizeof(wavebuf[i]));
		if ( result != MMSYSERR_NOERROR ) {
			SetMMerror("waveOutPrepareHeader()", result);
			return(-1);
		}
	}
	memset(mixbuf, spec->silence, NUM_BUFFERS*spec->size);

	/* Start playing the audio buffers -- seed with all of them */
	playing = 0;
	for ( i=0; i<NUM_BUFFERS; ++i ) {
		result = waveOutWrite(sound, &wavebuf[i], sizeof(wavebuf[0]));
		if ( result != MMSYSERR_NOERROR ) {
			SetMMerror("waveOutWrite()", result);
			return(-1);
		}
		++buffers_queued;
	}
	last_queued = i-1;
	return(0);
}
