/*
 * Copyright (c) 1997 Stanford University
 *
 * The use of this software for revenue-generating purposes may require a
 * license from the owners of the underlying intellectual property.
 * Specifically, the SRP protocol may not be used for revenue-generating
 * purposes without license.
 *
 * Within that constraint, permission to use, copy, modify, and distribute
 * this software and its documentation for any purpose is hereby granted
 * without fee, provided that the above copyright notices and this permission
 * notice appear in all copies of the software and related documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF
 * THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT
 * OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#ifndef T_PWD_H
#define T_PWD_H

#ifndef P
#if defined (__STDC__) || defined (__cplusplus)
#define P(x) x
#else
#define P(x) ()
#endif
#endif

#if defined (__STDC__) || defined (__cplusplus)
#define t_const	const
#else
#define t_const
#endif

#define MAXPARAMBITS	2048
#define MAXPARAMLEN	((MAXPARAMBITS + 7) / 8)
#define MAXB64PARAMLEN	((MAXPARAMBITS + 5) / 6 + 1)
#define MAXHEXPARAMLEN	((MAXPARAMBITS + 3) / 4 + 1)
#define MAXOCTPARAMLEN	((MAXPARAMBITS + 2) / 3 + 1)

#define MAXUSERLEN	32
#define MAXSALTLEN	32
#define MAXB64SALTLEN	44	/* 256 bits in b64 + null */
#define SALTLEN		10	/* Normally 80 bits */

#define RESPONSE_LEN	20	/* 160-bit proof hashes */
#define SESSION_KEY_LEN	(2 * RESPONSE_LEN)	/* 320-bit session key */

#define DEFAULT_PASSWD	"/etc/tpasswd"
#define DEFAULT_CONF	"/etc/tpasswd.conf"

struct t_num {	/* Standard byte-oriented integer representation */
  int len;
  unsigned char * data;
};

struct t_confent {	/* One configuration file entry (index, N, g) */
  int index;
  struct t_num modulus;
  struct t_num generator;
};

struct t_conf {		/* An open configuration file */
  FILE * instream;
  char close_on_exit;
  unsigned char modbuf[MAXPARAMLEN];
  unsigned char genbuf[MAXPARAMLEN];
  struct t_confent tcbuf;
};

/*
 * The configuration file routines are designed along the lines of the
 * "getpw" functions in the standard C library.
 *
 * "t_openconf" accepts a stdio stream and interprets it as a config file.
 * "t_openconfbyname" accepts a filename and does the same thing.
 * "t_closeconf" closes the config file.
 * "t_getconfent" fetches the next sequential configuration entry.
 * "t_getconfbyindex" fetches the configuration entry whose index
 *   matches the one supplied, or NULL if one can't be found.
 * "t_getconflast" fetches the last configuration entry in the file.
 * "t_makeconfent" generates a set of configuration entry parameters
 *   randomly.
 * "t_newconfent" returns an empty configuration entry.
 * "t_cmpconfent" compares two configuration entries a la strcmp.
 * "t_checkconfent" verifies that a set of configuration parameters
 *   are suitable.  N must be prime and should be a safe prime.
 * "t_putconfent" writes a configuration entry to a stream.
 */
extern struct t_conf * t_openconf P((FILE *));
extern struct t_conf * t_openconfbyname P((t_const char *));
extern void t_closeconf P((struct t_conf *));
extern void t_rewindconf P((struct t_conf *));
extern struct t_confent * t_getconfent P((struct t_conf *));
extern struct t_confent * t_getconfbyindex P((struct t_conf *, int));
extern struct t_confent * t_getconflast P((struct t_conf *));
extern struct t_confent * t_makeconfent P((struct t_conf *, int));
extern struct t_confent * t_makeconfent_c P((struct t_conf *, int));
extern struct t_confent * t_newconfent P((struct t_conf *));
extern int t_cmpconfent P((t_const struct t_confent *, t_const struct t_confent *));
extern int t_checkconfent P((t_const struct t_confent *));
extern void t_putconfent P((t_const struct t_confent *, FILE *));

struct t_pwent {	/* A single password file entry */
  char * name;
  struct t_num password;
  struct t_num salt;
  int index;
};

struct t_pw {		/* An open password file */
  FILE * instream;
  char close_on_exit;
  char userbuf[MAXUSERLEN];
  unsigned char pwbuf[MAXPARAMLEN];
  unsigned char saltbuf[SALTLEN];
  struct t_pwent pebuf;
};

/*
 * The password manipulation routines are patterned after the getpw*
 * standard C library function calls.
 *
 * "t_openpw" reads a stream as if it were a password file.
 * "t_openpwbyname" opens the named file as a password file.
 * "t_closepw" closes an open password file.
 * "t_rewindpw" starts the internal file pointer from the beginning
 *   of the password file.
 * "t_getpwent" retrieves the next sequential password entry.
 * "t_getpwbyname" looks up the password entry corresponding to the
 *   specified user.
 * "t_makepwent" constructs a password entry from a username, password,
 *   numeric salt, and configuration entry.
 * "t_putpwent" writes a password entry to a stream.
 */
extern struct t_pw * t_openpw P((FILE *));
extern struct t_pw * t_openpwbyname P((t_const char *));
extern void t_closepw P((struct t_pw *));
extern void t_rewindpw P((struct t_pw *));
extern struct t_pwent * t_getpwent P((struct t_pw *));
extern struct t_pwent * t_getpwbyname P((struct t_pw *, t_const char *));
extern struct t_pwent * t_makepwent P((struct t_pw *, t_const char *,
				       t_const char *, t_const struct t_num *,
				       t_const struct t_confent *));
extern void t_putpwent P((t_const struct t_pwent *, FILE *));

/*
 * Utility functions
 *
 * "t_verifypw" accepts a username and password, and checks against the
 *   system password file to see if the password for that user is correct.
 *   Returns > 0 if it is correct, 0 if not, and -1 if some error occurred
 *   (i.e. the user doesn't exist on the system).  This is intended ONLY
 *   for local authentication; for remote authentication, look at the
 *   t_client and t_server source.  (That's the whole point of SRP!)
 * "t_changepw" modifies the specified file, substituting the given password
 *   entry for the one already in the file.  If no matching entry is found,
 *   the new entry is simply appended to the file.
 */
extern int t_verifypw P((t_const char *, t_const char *));
extern int t_changepw P((t_const char *, t_const struct t_pwent *));

/* Conversion utilities */

/*
 * All these calls accept output as the first parameter.  In the case of
 * t_tohex and t_tob64, the last argument is the length of the byte-string
 * input.
 */
extern char * t_tohex P((char *, char *, unsigned));
extern int t_fromhex P((char *, char *));
extern char * t_tob64 P((char *, char *, unsigned));
extern int t_fromb64 P((char *, char *));

/* Miscellaneous utilities */

/*
 * "t_rand" is a conventional random number generator, which uses srandom()
 *   internally on most systems.
 * "t_random" is a cryptographic random number generator, which is seeded
 *   from various high-entropy sources and uses a one-way hash function
 *   in a feedback configuration.
 * "t_sessionkey" is the interleaved hash used to generate session keys
 *   from a large integer.
 * "t_getpass" reads a password from the terminal without echoing.
 */
extern void t_rand P((unsigned char *, unsigned));
extern void t_random P((unsigned char *, unsigned));
extern void t_stronginitrand();
extern unsigned char *
  t_sessionkey P((unsigned char *, unsigned char *, unsigned));
extern int t_getpass P((char *, unsigned, t_const char *));

/*
 * Return value of t_checkprime:
 *   < 0 : not prime
 *   = 0 : prime, but not safe
 *   > 0 : safe
 */
#define NUM_NOTPRIME	-1
#define NUM_NOTSAFE	0
#define NUM_SAFE	1

extern int t_checkprime P((t_const struct t_num *));

#endif
