## Copyright (C) 1995, 1996, 1997  Kurt Hornik
## 
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2, or (at your option)
## any later version.
## 
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details. 
## 
## You should have received a copy of the GNU General Public License
## along with this file.  If not, write to the Free Software Foundation,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

## Usage:  datediff (x, y [, u])
## Given two date strings x and y in the pseudo ISO form
##	"year month day hour min sec tz",
## where only the first three are necessary (i.e., to specify June 12,
## 1996 use "1996 6 12"), return the time elapsed between the first and
## the second date.
##
## With an optional argument, one can specify how the units in which
## the return value is given.  No argument or "s" gives seconds, "m"
## minutes, "h" hours and "d" days.

## Author:  KH <Kurt.Hornik@ci.tuwien.ac.at>
## Description:  Time elapsed between two ISO dates
  
function d = datediff (x, y, u)

  if ((nargin < 2) || (nargin > 4))
    usage ("datediff (x, y [, u])");
  elseif (nargin == 3)
    if (isstr (u))
      u = u (1);
    else
      error ("datediff:  u must be a string");
    endif
  else
    u = "s";
  endif

  if (!isstr (x))
    error ("datediff:  x must be a string");
  else
    ts_x = ts_from_ISO_date (x);
  endif

  if (!isstr (y))
    error ("datediff:  y must be a string");
  else
    ts_y = ts_from_ISO_date (y);
  endif

  d = mktime (ts_y) - mktime (ts_x);

  if (u == "d")
    d = d / 86400;
  elseif (u == "h")
    d = d / 3600;
  elseif (u == "m")
    d = d / 60;
  elseif !(u == "s")
    error ("datediff:  u must start with `d', `h', `m' or `s',");
  endif

endfunction
