/*
   Siag, Scheme In A Grid
   Copyright (C) 1996, 1997  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
 */

/*
 * fileio_html.c
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "../common/fonts.h"
#include "../common/common.h"

#include "calc.h"

#define START 0
#define INSIDE 1
#define OUTSIDE 2

static int col_last_changed(buffer *b, int s, int row)
{
	int i;

	if (row > b->sht[s].alloc_lines) return 0;
	for (i = b->sht[s].alloc_cols[row]; i > 1; i--)
		if (ret_type(b, s, row, i) != EMPTY)
			break;
	return i;
}

/* ---
Returns: 0 if successful, otherwise 1
*/

static int save(char *fn, buffer *buf)
{
	int i, j, lastcell, lr;
	int s = 0;	/* only use first sheet */
	FILE *fp;

	if ((fp = fopen(fn, "w")) == (FILE *) 0)
		return 1;

	lr = line_last_used(buf, s);

	fprintf(fp,	"<HTML>\n"
			"<HEAD>\n"
			"  <TITLE>%s</TITLE>\n"
			"</HEAD>\n\n"
			"<BODY>\n"
			"<H1>%s</H1>\n"
			"<TABLE BORDER=1>", fn, fn);

	for (i = 1; i <= lr; i++) {
		fprintf(fp, "\n<TR>\n");
		lastcell = col_last_changed(buf, s, i);
		for (j = 1; j <= lastcell; j++) {
			fprintf(fp, "  <TD>");


			if (ret_type(buf, s, i, j) != ERROR) {
				fprintf(fp, "%s",
					ret_pvalue(NULL, buf, s, i, j, -1));
			}

			fprintf(fp, "</TD>\n");
		}
		fprintf(fp, "</TR>\n");
	}

	fprintf(fp,	"</TABLE>\n"
			"</BODY>\n"
			"</HTML>\n");

	fclose(fp);
	return 0;
}				/* save */

#define STARTSTATE 0
#define BEFORETABLE 0
#define BEFOREROW 1
#define BEFOREDATA 2
#define DATA 3
#define ENDSTATE 4

static int state;

static double value;
static char instring[256];
static int pi;
static int sf = 0;

static int only_space(char *p)
{
	while (*p && isspace(*p)) p++;
	return (*p == '\0');
}

static int guess_type(char *p)
{
	char *endp;

	if (!strcmp(p, "")) return EMPTY;

	value = strtod(p, &endp);
	if (only_space(endp)) return EXPRESSION;

	return LABEL;
}

static void print_cell(buffer *b, int s, int r, int c, char *p)
{
	char *texti;
	cval value;
	value.number = 0;

	switch (guess_type(p)) {
	case EMPTY:
		break;
	case EXPRESSION:
		texti = p;
		ins_data(b, siod_interpreter, texti,
			value, EXPRESSION, s, r, c);
		ins_format(b, s, r, c, sf);
		break;
	default:	/* anything else is a label */
		texti = p;
		ins_data(b, siod_interpreter, texti,
			value, LABEL, s, r, c);
		ins_format(b, s, r, c, sf);
	}
}

static int row, col;

static void breakline(buffer *b, int s)
{
	if (pi > 0) print_cell(b, s, row, col, instring);
	pi = 0;
	row++;
	col = 1;
}

static void breakcell(buffer *b, int s)
{
	if (pi > 0) print_cell(b, s, row, col, instring);
	pi = 0;
	col++;
}

static char *tags[] =
	{"TABLE", "/TABLE", "TR", "/TR", "TH", "/TH", "TD", "/TD", NULL};

static void tagout(buffer *b, char *tag, int s)
{
	int i;
	char *p = strtok(tag, "\n\r\t ");

	for (i = 0; tags[i]; i++)
		if (!cstrcasecmp(tags[i], p)) break;

	switch (state) {
	case STARTSTATE:
		switch (i) {
		case 0: state = BEFOREROW; break;
		}
		break;
	case BEFOREROW:
		switch (i) {
		case 2: state = BEFOREDATA; break;
		}
		break;
	case BEFOREDATA:
		switch (i) {
		case 1: state = ENDSTATE; break;
		case 2: breakline(b, s); break;
		case 3: breakline(b, s); state = BEFOREROW; break;
		case 4:
		case 6: state = DATA; break;
		}
		break;
	case DATA:
		switch (i) {
		case 1: state = ENDSTATE; break;
		case 2: breakline(b, s); state = BEFOREDATA; break;
		case 3: breakline(b, s); state = BEFOREROW; break;
		case 4:
		case 6: breakcell(b, s); break;
		case 5:
		case 7: breakcell(b, s); state = BEFOREDATA; break;
		}
		break;
	default:
		break;
	}
}

static void emitchar(buffer *b, int s, int c)
{
	if (state == DATA) {
		if (isspace(c)) instring[pi++] = ' ';
		else instring[pi++] = c;
	}
	instring[pi] = '\0';
}

/* ---
Returns: 0 if successful, otherwise 1
*/

static int load(char *fn, buffer * buf)
{
	int i, j;
	int s = 0;
	FILE *fp;
	int c;
	int inatag = 0;
	char tag[256];
	cval value;
	value.number = 0;

	state = STARTSTATE;

	if ((fp = fopen(fn, "r")) == NULL)
		return 1;

	for (i = line_last_used(buf, s); i > 0; i--)
		for (j = col_last_changed(buf, s, i); j > 0; j--) {
			ins_data(buf, siod_interpreter, NULL,
				value, EMPTY, s, i, j);
			ins_format(buf, s, i, j, sf);
		}

	pi = 0;
	row = 1;
	col = 1;

	while ((c = getc(fp)) != EOF) {
		if (inatag) {
			if (c == '>') {
				inatag = 0;
				tag[i] = '\0';
				tagout(buf, tag, s);
			} else {
				if (i < 1020) tag[i++] = c;
			}
		} else {
			if (c == '<') {
				inatag = 1;
				i = 0;
			} else {
				emitchar(buf, s, c);
			}
		}
	}

	fclose(fp);
	return 0;
} /* load */

/* ---
conservative file format guessing:
   1. extension .html or .htm
   2. contains the string "table"
*/

static int myformat(char *fn)
{
	char *ext;
	FILE *fp;
	char b[256];

	ext = strrchr(fn, '.');
	if (!ext) return 0;	/* no extension */
	if (cstrcasecmp(ext, ".html") && cstrcasecmp(ext, ".htm"))
		return 0;	/* wrong extension */
	if ((fp = fopen(fn, "r")) == NULL) return 0;	/* can't open */
	while (fgets(b, sizeof b, fp)) {
		if (strstr(b, "table") || strstr(b, "TABLE")) {
			fclose(fp);
			return 1;
		}
	}
	fclose(fp);
	return 0;	/* yep, sure is ;-) */
}

/* ---
*/
void fileio_html_init(void)
{
	register_format(load, save, myformat, "Hypertext Markup Language (*.html)");
}

