/*
   Copyright (C) 1996, 1997  Ulric Eriksson <ulric@edu.stockholm.se>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the Licence, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>

#include "../common/cmalloc.h"
#include "../common/fonts.h"
#include "xfonts.h"
#include "embed.h"
#include <X11/xpm.h>

#include "TableP.h"

static void plugin_coords(Widget, XtPointer, int *, int *);

#define offset(field) XtOffsetOf(TableRec, table.field)
static XtResource resources[] = {
	{
		XtNtableMaxRow,		/* name */
		XtCTableMaxRow,		/* class */
		XtRInt,			/* type */
		sizeof(int),		/* size */
		offset(max_row),	/* offset */
		XtRImmediate,		/* default_type */
		(XtPointer)1000		/* default_addr */
	}, {
		XtNtableMaxCol,
		XtCTableMaxCol,
		XtRInt,
		sizeof(int),
		offset(max_col),
		XtRImmediate,
		(XtPointer)1000
	}, {
		XtNtableProtRow,
		XtCTableProtRow,
		XtRInt,
		sizeof(int),
		offset(prot_row),
		XtRImmediate,
		(XtPointer)1,
	}, {
		XtNtableProtCol,
		XtCTableProtCol,
		XtRInt,
		sizeof(int),
		offset(prot_col),
		XtRImmediate,
		(XtPointer)1,
	}, {
		XtNtableTopRow,
		XtCTableTopRow,
		XtRInt,
		sizeof(int),
		offset(top_row),
		XtRImmediate,
		(XtPointer)1
	}, {
		XtNtableTopCol,
		XtCTableTopCol,
		XtRInt,
		sizeof(int),
		offset(top_col),
		XtRImmediate,
		(XtPointer)1
	}, {
		XtNtableSelectTopRow,
		XtCTableSelectTopRow,
		XtRInt,
		sizeof(int),
		offset(sel_top_row),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNtableSelectBottomRow,
		XtCTableSelectBottomRow,
		XtRInt,
		sizeof(int),
		offset(sel_bottom_row),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNtableSelectLeftCol,
		XtCTableSelectLeftCol,
		XtRInt,
		sizeof(int),
		offset(sel_left_col),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNtableSelectRightCol,
		XtCTableSelectRightCol,
		XtRInt,
		sizeof(int),
		offset(sel_right_col),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNtablePointRow,
		XtCTablePointRow,
		XtRInt,
		sizeof(int),
		offset(point_row),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNtablePointCol,
		XtCTablePointCol,
		XtRInt,
		sizeof(int),
		offset(point_col),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNtableDefaultWidth,
		XtCTableDefaultWidth,
		XtRInt,
		sizeof(int),
		offset(default_width),
		XtRImmediate,
		(XtPointer)80
	}, {
		XtNtableDefaultHeight,
		XtCTableDefaultHeight,
		XtRInt,
		sizeof(int),
		offset(default_height),
		XtRImmediate,
		(XtPointer)20
	}, {
		XtNtableColWidth,
		XtCTableColWidth,
		XtRPointer,
		sizeof(XtPointer),
		offset(col_width),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNtableRowHeight,
		XtCTableRowHeight,
		XtRPointer,
		sizeof(XtPointer),
		offset(row_height),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNtableType,
		XtCTableType,
		XtRPointer,
		sizeof(XtPointer),
		offset(type),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNtableText,
		XtCTableText,
		XtRPointer,
		sizeof(XtPointer),
		offset(text),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNtableFormat,
		XtCTableFormat,
		XtRPointer,
		sizeof(XtPointer),
		offset(format),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNtableGC,
		XtCTableGC,
		XtRPointer,
		sizeof(XtPointer),
		offset(gc),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNtableAdjHoriz,
		XtCTableAdjHoriz,
		XtRPointer,
		sizeof(XtPointer),
		offset(adj_horiz),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNtableAdjVert,
		XtCTableAdjVert,
		XtRPointer,
		sizeof(XtPointer),
		offset(adj_vert),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNtableBackground,
		XtCTableBackground,
		XtRPixel,
		sizeof(Pixel),
		offset(background),
		XtRString,
		XtDefaultBackground
	}, {
		XtNtableForeground,
		XtCTableForeground,
		XtRPixel,
		sizeof(Pixel),
		offset(foreground),
		XtRString,
		XtDefaultForeground
	}, {
		XtNtableGrid,
		XtCTableGrid,
		XtRPixel,
		sizeof(Pixel),
		offset(grid),
		XtRString,
		XtDefaultForeground
	}, {
		XtNtableCursor,
		XtCTableCursor,
		XtRPixel,
		sizeof(Pixel),
		offset(cursor),
		XtRString,
		XtDefaultForeground
	}, {
		XtNtableHighlight,
		XtCTableHighlight,
		XtRPixel,
		sizeof(Pixel),
		offset(highlight),
		XtRString,
		XtDefaultForeground
	}, {
		XtNtableData,
		XtCTableData,
		XtRPointer,
		sizeof(XtPointer),
		offset(data),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNtableRedisplay,
		XtCTableRedisplay,
		XtRBoolean,
		sizeof(Boolean),
		offset(redisplay),
		XtRImmediate,
		(XtPointer)False
	}, {
		XtNtableGridLines,
		XtCTableGridLines,
		XtRBoolean,
		sizeof(Boolean),
		offset(grid_lines),
		XtRImmediate,
		(XtPointer)True
	}, {
		XtNtableVisibleCursor,
		XtCTableVisibleCursor,
		XtRBoolean,
		sizeof(Boolean),
		offset(visible_cursor),
		XtRImmediate,
		(XtPointer)False
	}, {
		XtNtablePluginCoords,
		XtCTablePluginCoords,
		XtRPointer,
		sizeof(XtPointer),
		offset(plugin_coords),
		XtRImmediate,
		(XtPointer)plugin_coords
	}, {
		XtNtable3D,
		XtCTable3D,
		XtRBoolean,
		sizeof(Boolean),
		offset(threeD),
		XtRImmediate,
		(XtPointer)False
	}
};
#undef offset

/* methods */
static void DoLayout();
static void Resize();
static XtGeometryResult GeometryManager();
static void ChangeManaged();
static void Redisplay(Widget, XEvent *, Region);
static void Realize(Widget, XtValueMask *, XSetWindowAttributes *);
static void Destroy(Widget);
static Boolean SetValues(Widget, Widget, Widget, ArgList, Cardinal *);

static void plugin_coords(Widget w, XtPointer p, int *x, int *y)
{
	*x = *y = 0;
}

/* actions */
static void TableAction(Widget, XEvent *, String *, Cardinal *);

static XtActionsRec actions[] =
{
	{"table", TableAction},
};

/* translations */
static char translations[] =
"<Key>:		table()		\n";

TableClassRec tableClassRec = {
  { /* core fields */
    /* superclass		*/	(WidgetClass) &compositeClassRec,
    /* class_name		*/	"Table",
    /* widget_size		*/	sizeof(TableRec),
    /* class_initialize		*/	NULL,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	NULL,
    /* initialize_hook		*/	NULL,
    /* realize			*/	Realize,
    /* actions			*/	actions,
    /* num_actions		*/	XtNumber(actions),
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	Resize,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	translations,
    /* query_geometry		*/	XtInheritQueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
    /* extension		*/	NULL
  },{
/* composite_class fields */
    /* geometry_manager   */    GeometryManager,
    /* change_managed     */    ChangeManaged,
    /* insert_child       */    XtInheritInsertChild,
    /* delete_child       */    XtInheritDeleteChild,
    /* extension          */    NULL
  },{ /* table fields */
    /* empty			*/	0
  }
};

WidgetClass tableWidgetClass = (WidgetClass)&tableClassRec;


/* supporting code copied from window.c */

static void TableAction(Widget w, XEvent *event, String *params, Cardinal *n)
{
	;	/* application should create its own actions */
}

static int return_type(Widget w, int row, int col)
{
	TableWidget tw = (TableWidget) w;
	if (tw->table.type)
		return (*tw->table.type)(tw->table.data, row, col);
	return TABLE_TEXT;
}

static char *return_pvalue(Widget w, int row, int col)
{
	TableWidget tw = (TableWidget) w;
	if (tw->table.text)
		return (*tw->table.text)(tw->table.data, row, col);
	else return NULL;
}

static int return_format(Widget w, int row, int col)
{
	TableWidget tw = (TableWidget) w;

	if (tw->table.format)
		return (*tw->table.format)(tw->table.data, row, col);
	return 0;
}

static int return_font(Widget w, int row, int col)
{
	return format_table[return_format(w, row, col)].font;
}

static int return_color(Widget w, int row, int col)
{
	return format_table[return_format(w, row, col)].fg;
}

static int return_bgcolor(Widget w, int row, int col)
{
	return format_table[return_format(w, row, col)].bg;
}

static Dimension cell_width(TableWidget tw, int col)
{
	Dimension wi;

	if (tw->table.col_width) {
		wi = (*tw->table.col_width)(tw->table.data, col);
		return wi;
	}
	return tw->table.default_width;
}

static Dimension cell_height(TableWidget tw, int row)
{
	Dimension h;
	if (tw->table.row_height) {
		h = (*tw->table.row_height)(tw->table.data, row);
		return h;
	}
	return tw->table.default_height;
}

/* ---
*/
void table_global_coords(TableWidget tw,
		int row, int col, int *x, int *y)
{
	int i;

	*x = *y = 0;

	for (i = 1; i < col; i++)
		*x += cell_width(tw, i);

	for (i = 1; i < row; i++)
		*y += cell_height(tw, i);
}

static int inblock(TableWidget tw, int r, int c)
{
	return r >= tw->table.sel_top_row && r <= tw->table.sel_bottom_row
		&& c >= tw->table.sel_left_col && c <= tw->table.sel_right_col;
}


static void draw_cell(TableWidget tw, Drawable scribble,
			int row, int col, int x_base, int y_base)
{
	Widget w = (Widget) tw;
	char *p;
	int font_index, color_index;
	unsigned long color;
	int x_pos, y_pos;
	unsigned int text_height, text_width;
	int cw, ch;
	int ct;
	int fmt;
	int siagfmt;
	rich_char *rcp;

	if (row > tw->table.max_row || col > tw->table.max_col) return;

	cw = cell_width(tw, col);
	ch = cell_height(tw, row);

	fmt = return_format(w, row, col);

	ct = return_type(w, row, col);
	if (ct == TABLE_EMBED) {
		char *p = return_pvalue(w, row, col);
		embed_draw(scribble, x_base, y_base, p);
		return;
	}

	if (ct == TABLE_PIXMAP) {
		XtWarning("The Table widget does not yet display pixmaps");
		return;
	}

	font_index = return_font(w, row, col);

	color_index = return_color(w, row, col);
	color = get_color(color_index);
	XSetForeground(XtDisplay(w), tw->table.cell_gc, color);

	p = return_pvalue(w, row, col);
	if (p == NULL) p = "";
	rcp = rc_makerich(p, fmt);

	if (strlen(p) > 1000) p[1000] = '\0';

	rc_strgeom(rcp, -1, &text_width, &text_height);

	siagfmt = format_table[fmt].siagfmt;
	switch (siagfmt & HADJ_MASK) {
	case HADJ_CENTER:
		x_pos = (cw-text_width) / 2;
		break;
	case HADJ_RIGHT:
		x_pos = cw-text_width-5;
		break;
	default:	/* HADJ_LEFT */
		x_pos = 5;
	}

	switch (siagfmt & VADJ_MASK) {
	case VADJ_BOTTOM:
		y_pos = ch-5;
		break;
	case VADJ_TOP:
		y_pos = text_height+font_descent(font_index);
		break;
	default:	/* VADJ_CENTER */
		y_pos = (ch+text_height)/2
			-font_descent(font_index);
	}

	rc_strdraw(scribble, tw->table.cell_gc,
		x_base+x_pos, y_base+y_pos,
		rcp, -1);
	cfree(rcp);
	XSetForeground(XtDisplay(w), tw->table.grid_gc, tw->table.black);
	if (tw->table.grid_lines) {
		XDrawLine(XtDisplay(w), scribble, tw->table.grid_gc,
			x_base, y_base+ch-1,
			x_base+cw-1, y_base+ch-1);
		XDrawLine(XtDisplay(w), scribble, tw->table.grid_gc,
			x_base+cw-1, y_base+ch-1,
			x_base+cw-1, y_base);
	}
	/* these lines are from the format */
	if (siagfmt & BORDER_LEFT)
		XDrawLine(XtDisplay(w), scribble, tw->table.grid_gc,
			x_base, y_base, x_base, y_base+ch-2);
	if (siagfmt & BORDER_RIGHT)
		XDrawLine(XtDisplay(w), scribble, tw->table.grid_gc,
			x_base+cw-2, y_base, x_base+cw-2, y_base+ch-2);
	if (siagfmt & BORDER_TOP)
		XDrawLine(XtDisplay(w), scribble, tw->table.grid_gc,
			x_base, y_base, x_base+cw-2, y_base);
	if (siagfmt & BORDER_BOTTOM)
		XDrawLine(XtDisplay(w), scribble, tw->table.grid_gc,
			x_base, y_base+ch-2, x_base+cw-2, y_base+ch-2);
	XSetForeground(XtDisplay(w), tw->table.grid_gc,
			WhitePixelOfScreen(XtScreen(w)));
	if (tw->table.threeD) {
		XDrawLine(XtDisplay(w), scribble, tw->table.grid_gc,
			x_base, y_base, x_base, y_base+ch-2);
		XDrawLine(XtDisplay(w), scribble, tw->table.grid_gc,
			x_base, y_base, x_base+cw-2, y_base);
	}
}

static void erase_cell(TableWidget tw, Drawable scribble,
			int row, int col, int x_base, int y_base)
{
	Widget w = (Widget) tw;
	int color_index;
	unsigned long color;
	int cw, ch;

	if (row > tw->table.max_row || col > tw->table.max_col) return;

	cw = cell_width(tw, col);
	ch = cell_height(tw, row);

	if (inblock(tw, row, col)) {
		XFillRectangle(XtDisplay(w), scribble, tw->table.block_gc,
			x_base, y_base, cw, ch);
	} else {
		color_index = return_bgcolor(w, row, col);
		color = get_color(color_index);
		XSetForeground(XtDisplay(w), tw->table.cell_gc, color);
		XFillRectangle(XtDisplay(w), scribble, tw->table.cell_gc,
			x_base, y_base, cw, ch);
	}
}

static void cell_row(TableWidget tw, Drawable scribble,
			int width, int y_base, int i)
{
	int j, x_base;
	x_base = 0;

	for (j = 1; (j < tw->table.prot_col) && (x_base < width); j++) {
		erase_cell(tw, scribble, i, j, x_base, y_base);
		x_base += cell_width(tw, j);
	}
	for (j = tw->table.top_col; x_base < width; j++) {
		erase_cell(tw, scribble, i, j, x_base, y_base);
		x_base += cell_width(tw, j);
	}

	x_base = 0;
	for (j = 1; (j < tw->table.prot_col) && (x_base < width); j++) {
		draw_cell(tw, scribble, i, j, x_base, y_base);
		x_base += cell_width(tw, j);
	}
	for (j = tw->table.top_col; x_base < width; j++) {
		draw_cell(tw, scribble, i, j, x_base, y_base);
		x_base += cell_width(tw, j);
	}
}

static void draw_table(TableWidget tw, Drawable d)
{
	int y_base = 0;
	int width = tw->core.width;
	int height = tw->core.height;
	int i;

	for (i = 1; (i < tw->table.prot_row) && (y_base < height); i++) {
		cell_row(tw, d, width, y_base, i);
		y_base += cell_height(tw, i);
	}
	for (i = tw->table.top_row; y_base < height; i++) {
		cell_row(tw, d, width, y_base, i);
		y_base += cell_height(tw, i);
	}
}

static int cell_next_row(TableWidget w, int row)
{
        if (row+1 == w->table.prot_row) return w->table.top_row;
        return row+1;
}

static int cell_next_col(TableWidget w, int col)
{
        if (col+1 == w->table.prot_col) return w->table.top_col;
        return col+1;
}

static int cell_prev_row(TableWidget w, int row)
{
        if (row == w->table.top_row) return w->table.prot_row-1;
        return row-1;
}

static int cell_prev_col(TableWidget w, int col)
{
        if (col == w->table.top_col) return w->table.prot_col-1;
        return col-1;
}

/* ---
*/
void table_cell2coords(TableWidget tw,
			int cell_row, int cell_col,
			int *cell_x, int *cell_y)
{
        int i;
        *cell_y = 0;
        for (i = 1; i < tw->table.prot_row; i++)
                *cell_y += cell_height(tw, i);

        while (cell_row < tw->table.top_row) {
                *cell_y -= cell_height(tw, cell_row);
                cell_row = cell_next_row(tw, cell_row);
        }
        while (cell_row > tw->table.top_row) {
                cell_row = cell_prev_row(tw, cell_row);
                *cell_y += cell_height(tw, cell_row);
        }
        *cell_x = 0;
        for (i = 1; i < tw->table.prot_col; i++)
                *cell_x += cell_width(tw, i);

        while (cell_col < tw->table.top_col) {
                *cell_x -= cell_width(tw, cell_col);
                cell_col = cell_next_col(tw, cell_col);
        }
        while (cell_col > tw->table.top_col) {
                cell_col = cell_prev_col(tw, cell_col);
                *cell_x += cell_width(tw, cell_col);
        }
}

/* ---
*/
void table_coords2cell(TableWidget tw,
			int *cur_row, int *cur_col,
			int cur_x, int cur_y)
{
        int prot_x = 0, prot_y = 0, i;

        for (i = 1; i < tw->table.prot_col; i++)
              cur_x -= cell_width(tw, i);
        for (i = 1; i < tw->table.prot_row; i++)
              cur_y -= cell_height(tw, i);

        *cur_row = tw->table.top_row;
        *cur_col = tw->table.top_col;
        while (cur_y < prot_y) {
                cur_y += cell_height(tw, *cur_row);
                (*cur_row) = cell_prev_row(tw, *cur_row);
        }
        while (cur_y > cell_height(tw, *cur_row)) {
                cur_y -= cell_height(tw, *cur_row);
                (*cur_row) = cell_next_row(tw, *cur_row);
        }
        while (cur_x < prot_x) {
                cur_x += cell_width(tw, *cur_col);
                (*cur_col) = cell_prev_col(tw, *cur_col);
        }
        while (cur_x > cell_width(tw, *cur_col)) {
                cur_x -= cell_width(tw, *cur_col);
                (*cur_col) = cell_next_col(tw, *cur_col);
        }
}

/* ---
create, draw and return a pixmap of the stage at time now.
   Caller must free
Strategy:
   1. Clear the window, or create a new empty Pixmap
   2. Paint a rectangle for the block, if there is one (coords != 0)
   3. Draw the grid
   4. Draw the data
   5. Draw the cursor using XOR, if one is desired (coords != 0)
   The purpose of using XOR is to make sure the cursor is visible
   on any background. Technicolor is a potential side effect.
*/

Pixmap table_pixmap(Widget w)
{
	Pixmap scribble;
	TableWidget tw = (TableWidget)w;
	int top_x, top_y, bot_x, bot_y, max_r, max_c, width, height;

	width = tw->core.width;
	height = tw->core.height;

	if (width > 2000 || height > 2000) return None;

	scribble = XCreatePixmap(XtDisplay(w), XtWindow(w),
		width, height, tw->core.depth);
	table_cell2coords(tw,
		tw->table.sel_top_row, tw->table.sel_left_col,
		&top_x, &top_y);
	table_coords2cell(tw, &max_r, &max_c, width, height);
	if (max_r > tw->table.sel_bottom_row)
		max_r = tw->table.sel_bottom_row;
	if (max_c > tw->table.sel_right_col)
		max_c = tw->table.sel_right_col;

	table_cell2coords(tw, max_r, max_c, &bot_x, &bot_y);
	if (bot_x >= top_x && bot_x >= 0 && top_x <= width
			&& bot_y >= top_y && bot_y >= 0 && top_y <= height) {
		int wi = bot_x-top_x+cell_width(tw, tw->table.sel_right_col);
		int he = bot_y-top_y+cell_height(tw, tw->table.sel_bottom_row);
		if (top_x < 0) {
			wi += top_x;
			top_x = 0;
		}
		if (top_y < 0) {
			he += top_y;
			top_y = 0;
		}
	}

	draw_table(tw, scribble);
	return scribble;
}


static GC get_gc(Widget w, unsigned long fg, unsigned long bg)
{
	unsigned long valuemask = 0;
	XGCValues values;
	GC gc = XCreateGC(XtDisplay(w), XtWindow(w),
				valuemask, &values);
	XSetForeground(XtDisplay(w), gc, fg);
	XSetBackground(XtDisplay(w), gc, bg);
	return gc;
}

#define superclass (&coreClassRec)
static void Realize(Widget w, XtValueMask *valueMask,
		XSetWindowAttributes *attributes)
{
	TableWidget tw = (TableWidget) w;
	XColor screen_color, exact_color;

	(*superclass->core_class.realize) (w, valueMask, attributes);
	tw->table.black = BlackPixelOfScreen(XtScreen(w));
	tw->table.white = tw->core.background_pixel;
	XAllocNamedColor(XtDisplay(w),
		DefaultColormap(XtDisplay(w), DefaultScreen(XtDisplay(w))),
		"gray", &screen_color, &exact_color);
	tw->table.grey = screen_color.pixel;
	tw->table.clear_gc = get_gc(w, tw->table.white, tw->table.black);
	tw->table.cell_gc = get_gc(w, tw->table.black, tw->table.white);
	tw->table.grid_gc = get_gc(w, tw->table.black, tw->table.white);
#if 1	/* this is too much */
	XSetLineAttributes(XtDisplay(w), tw->table.grid_gc,
			0, LineSolid, CapButt, JoinMiter);
#endif
	tw->table.cursor_gc = get_gc(w, tw->table.black^tw->table.white, 0);
	XSetFunction(XtDisplay(w), tw->table.cursor_gc, GXxor);
	XSetLineAttributes(XtDisplay(w), tw->table.cursor_gc,
			2, LineSolid, CapButt, JoinMiter);
#if 1	/* grey block */
	tw->table.block_gc = get_gc(w, tw->table.grey, tw->table.white);
#else	/* reverse block */
	tw->table.block_gc = get_gc(w, tw->table.black^tw->table.white, 0);
	XSetFunction(XtDisplay(w), tw->table.block_gc, GXxor);
#endif
}

static void Destroy(Widget w)
{
	TableWidget tw = (TableWidget) w;

	XFreeGC(XtDisplay(w), tw->table.clear_gc);
	XFreeGC(XtDisplay(w), tw->table.cell_gc);
	XFreeGC(XtDisplay(w), tw->table.grid_gc);
	XFreeGC(XtDisplay(w), tw->table.cursor_gc);
	XFreeGC(XtDisplay(w), tw->table.block_gc);
}


static Boolean move_top(TableWidget tw)
{
	Boolean pr_scr_flag = False;
	int cur_x, cur_y, prot_x = 0, prot_y = 0, i;
	unsigned int width, height;

        for (i = 1; i < tw->table.prot_col; i++)
                prot_x += cell_width(tw, i);
        for (i = 1; i < tw->table.prot_row; i++)
                prot_y += cell_height(tw, i);

        /* Figure out how big the window is */
	width = tw->core.width;
	height = tw->core.height;

        table_cell2coords(tw,
                        tw->table.point_row, tw->table.point_col,
                        &cur_x, &cur_y);
        /* this isn't efficient, but it will work */
	while (cur_y < prot_y) {
		tw->table.top_row--;
		cur_y += cell_height(tw, tw->table.top_row);
		pr_scr_flag = TRUE;
	}
	while (cur_y + cell_height(tw, tw->table.point_row) > height) {
		cur_y -= cell_height(tw, tw->table.top_row);
		tw->table.top_row++;
		pr_scr_flag = TRUE;
	}
	while (cur_x < prot_x) {
		tw->table.top_col--;
		cur_x += cell_width(tw, tw->table.top_col);
		pr_scr_flag = TRUE;
	}
	while (cur_x + cell_width(tw, tw->table.point_col) > width) {
		cur_x -= cell_width(tw, tw->table.top_col);
		tw->table.top_col++;
		pr_scr_flag = TRUE;
	}
	return pr_scr_flag;
}

static void toggle_cursor(TableWidget tw)
{
	int row = tw->table.point_row, col = tw->table.point_col;
	int cur_x, cur_y;
	GC gc = tw->table.cursor_gc;
	unsigned long bg;

	table_cell2coords(tw, row, col, &cur_x, &cur_y);
	if (row >= tw->table.sel_top_row && row <= tw->table.sel_bottom_row &&
		col >= tw->table.sel_left_col && col <= tw->table.sel_right_col) {
		bg = tw->table.grey;
	} else {
		bg = tw->table.white;
	}
	XSetForeground(XtDisplay((Widget)tw), gc, tw->table.black^bg);
	XDrawRectangle(XtDisplay((Widget)tw), XtWindow((Widget)tw),
		tw->table.cursor_gc, cur_x+2, cur_y+2,
		cell_width(tw, col)-5, cell_height(tw, row)-5);
}

static void Redisplay(Widget w, XEvent *event, Region r)
{
	Pixmap scribble;
	GC gc;
	unsigned long valuemask = 0;
	XGCValues values;
	TableWidget tw = (TableWidget) w;

	if (!XtIsRealized(w)) return;	/* but that doesn't work */
	if (!XtIsManaged(w)) return;	/* what about this */
	scribble = table_pixmap(w);
	if (scribble == None) return;
	gc = XCreateGC(XtDisplay(w), XtWindow(w),
			valuemask, &values);
	XCopyArea(XtDisplay(w), scribble, XtWindow(w),
		gc, 0, 0, tw->core.width, tw->core.height, 0, 0);
	XFreePixmap(XtDisplay(w), scribble);
	XFreeGC(XtDisplay(w), gc);

	/* draw the cursor, perhaps? */
	if (tw->table.visible_cursor) {
		toggle_cursor(tw);
	}
	/* update plugin positions */
	DoLayout(tw);
}


/* ---
A dilemma: if Xt handles the redisplaying, it also takes care of
   compressing expose events. On the other hand, it clears the window
   which makes it flicker. We'll redisplay ourselves and try to
   make it more efficient later.
*/

static Boolean SetValues(Widget current, Widget request, Widget new,
		ArgList args, Cardinal *n)
{
	TableWidget curtw = (TableWidget) current;
	TableWidget newtw = (TableWidget) new;
	Boolean do_redisplay = False;
#if 1	/* this is way too complex */
	do_redisplay |= (curtw->table.sel_top_row != newtw->table.sel_top_row
		|| curtw->table.sel_bottom_row != newtw->table.sel_bottom_row
		|| curtw->table.sel_left_col != newtw->table.sel_left_col
		|| curtw->table.sel_right_col != newtw->table.sel_right_col);

	if (newtw->table.visible_cursor)
		do_redisplay |= move_top(newtw);

	if (newtw->table.redisplay) {
		do_redisplay = True;
		newtw->table.redisplay = False;
	}
	/* can't let Xt handle this because it flickers */
	if (do_redisplay) {
		Redisplay(new, NULL, None);
		do_redisplay = False;
	} else 
	if (!do_redisplay) {
		if (curtw->table.visible_cursor) {
			toggle_cursor(curtw);
		}
		if (newtw->table.visible_cursor) {
			toggle_cursor(newtw);
		}
	}
#else	/* any change redraws the table */
	if (newtw->table.visible_cursor)
		move_top(newtw);
	do_redisplay = False;
	Redisplay(new, NULL, None);
#endif
	return do_redisplay;
}

/*
 * Do a layout, actually assigning positions.
 */

static void DoLayout(TableWidget sw)
{
	int i;
	int x, y, top_x, top_y, prot_x, prot_y;

	if (sw->composite.num_children) {
		table_global_coords(sw,
			sw->table.prot_row, sw->table.prot_col,
			&prot_x, &prot_y);
		table_global_coords(sw,
			sw->table.top_row, sw->table.top_col,
			&top_x, &top_y);
	}
	for (i = 0; i < sw->composite.num_children; i++) {
		(*sw->table.plugin_coords)(sw->composite.children[i],
			sw->table.data, &x, &y);
		XtMoveWidget(sw->composite.children[i],
			x-top_x+prot_x, y-top_y+prot_y);
	}
}

/*
 * Actually layout the table
 */

static void Resize(Widget w)
{
        DoLayout((TableWidget)w);
} /* Resize */

/*
 * Geometry Manager
 *
 * 'reply' is unused; we say only yeay or nay, never almost.
 */

static XtGeometryResult GeometryManager(Widget w,
                XtWidgetGeometry request, XtWidgetGeometry reply)
{
        return XtGeometryYes;
}

static void ChangeManaged(Widget w)
{
    DoLayout((TableWidget)w);
}

