/*
    Copyright (C) 1997-1999  Ulric Eriksson <ulric@edu.stockholm.se>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston,
    MA 02111-1307, USA.
*/

/*
 * fonts.c
 *
 * Support for figuring out richtext metrics and displaying rich text.
 * Includes support for T1lib.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Intrinsic.h>
#include <X11/cursorfont.h>
#include "../config.h"
#ifdef HAVE_LIB_T1X
#include <t1lib.h>
#include <t1libx.h>
#endif

#include "../common/common.h"
#include "../common/cmalloc.h"
#include "../common/oldfonts.h"
#include "../common/fonts.h"
#include "../common/richchar.h"
#include "../common/traceme.h"
#include "xfonts.h"

#define has_t1_font(i) (font_table[(i)].id != -1)

static Display *dpy;

siag_color color_table[1000];
static siag_fontname fontname_table[256];
siag_font font_table[256];
static int ncolor, nfontname, nfont;

static struct {
	char *alias;
	char *name;
} fontalias[256];

static int nfontalias = 0;

static Display *t1_dpy;
static Screen *t1_scr;
static int t1_scrn;
static int t1_nfont;
static char **t1_fontname;
static int t1_mode;

static int compar(const void *p, const void *q)
{
	return strcmp(*(const char **)p, *(const char **)q);
}

/* ---
returns a null-terminated, sorted list of all known fonts.
Caller must free
*/

char **font_list(int *n)
{
	int i, j;
	char **list = (char **)cmalloc((nfontalias+nfontname+1)*sizeof(char *));
	for (i = 0; i < nfontalias; i++)
		list[i] = cstrdup(fontalias[i].alias);
	for (j = 0; j < nfontname; j++)
		list[i+j] = cstrdup(fontname_table[j].name);
	list[i+j] = NULL;
	qsort(list, i+j, sizeof(char *), compar);
	*n = i+j;
	return list;
}

/* ---
returns a null-terminated, sorted list of all known colours.
Caller must free
*/

char **color_list(int *n)
{
	int i;
	char **list = (char **)cmalloc((ncolor+1)*sizeof(char *));
	for (i = 0; i < ncolor; i++)
		list[i] = cstrdup(color_table[i].name);
	list[i] = NULL;
	qsort(list, i, sizeof(char *), compar);
	*n = i;
	return list;
}

/* ---
The index is into the font_table. The index must have come from
lookup_font, so we know that the font is already loaded.
*/

#define font_struct(i) (font_table[(i)].fs)

int font_ascent(int index)
{
	if (has_t1_font(index)) {
#ifdef HAVE_LIB_T1X
		GLYPH *glyph = T1_SetChar(font_table[index].id, 'd',
				font_table[index].size/10.0, NULL);
		if (glyph) return glyph->metrics.ascent;
#endif
		return font_table[index].size/10;	/* BOGUS */
	} else {
		return font_struct(index)->max_bounds.ascent;
	}
}

int font_descent(int index)
{
	if (has_t1_font(index)) {
#ifdef HAVE_LIB_T1X
		GLYPH *glyph = T1_SetChar(font_table[index].id, 'p',
				font_table[index].size/10.0, NULL);
		if (glyph) return -glyph->metrics.descent;
#endif
		return font_table[index].size/10;	/* BOGUS */
	} else {
		return font_struct(index)->max_bounds.descent;
	}
}

int font_height(int index)
{
#ifdef HAVE_LIB_T1X
	if (has_t1_font(index)) {
		BBox bbox = T1_GetFontBBox(font_table[index].id);
		return (bbox.ury-bbox.lly)*font_table[index].size/10000;
	}
#endif
	return font_ascent(index)+font_descent(index);
}

int font_width(int index, char *s)
{
	if (has_t1_font(index)) {
#ifdef HAVE_LIB_T1X
		return T1_GetStringWidth(font_table[index].id,
			s, strlen(s), 0, FALSE)*font_table[index].size/10000;
#else
		return font_table[index].size/20;
#endif
	} else {
		return XTextWidth(font_struct(index), s, strlen(s));
	}
}

/* ---
index to color_table. The index must come from lookup_color,
so we know that the colour has already been allocated
*/

unsigned long get_color(int index)
{
	return color_table[index].color.pixel;
}

/* ---
Take an index into the format table (i.e. a new-style format code) and
decode it into the components specified by the mask.
*/

void decode_format(int n, long mask, sfmt *fmt)
{
	int font = format_table[n].font;

	if (mask & SFMT_FAMILY) {
		fmt->family = fontname_table[font_table[font].name].name;
		if (!fmt->family) fmt->family = "Helvetica";
	}
	if (mask & SFMT_SIZE)
		fmt->size = font_table[font].size;
	if (mask & SFMT_BOLD)
		fmt->bold = font_table[font].bold;
	if (mask & SFMT_ITALIC)
		fmt->italic = font_table[font].italic;
	if (mask & SFMT_ULINE)
		fmt->uline = format_table[n].uline;
	if (mask & SFMT_FG) {
		fmt->fg = color_table[format_table[n].fg].name;
		if (!fmt->fg) fmt->fg = "black";
	}
	if (mask & SFMT_BG) {
		fmt->bg = color_table[format_table[n].bg].name;
		if (!fmt->bg) fmt->bg = "white";
	}
	if (mask & SFMT_BORDERS)
		fmt->borders = format_table[n].siagfmt & BORDER_MASK;
	if (mask & SFMT_VADJ)
		fmt->vadj = format_table[n].siagfmt & VADJ_MASK;
	if (mask & SFMT_HADJ)
		fmt->hadj = format_table[n].siagfmt & HADJ_MASK;
	if (mask & SFMT_STYLE)
		fmt->style = format_table[n].style;
}

/* ---
Encode the components from fmt specified by the mask into a new-style
format code, allocating colours and fonts as necessary.
*/

int encode_format(long mask, sfmt *fmt)
{
	char *family = "Helvetica";
	int fg = 0, bg = 0;
	int size = 120;
	int bold = 0, italic = 0, uline = 0;
	int siagfmt = 0;
	int style = 0;

	if (mask & SFMT_FAMILY) {
		family = lookup_fontalias(fmt->family);
		if (!family) family = "Helvetica";
	}
	if (mask & SFMT_SIZE)
		size = fmt->size;
	if (mask & SFMT_BOLD)
		bold = fmt->bold;
	if (mask & SFMT_ITALIC)
		italic = fmt->italic;
	if (mask & SFMT_ULINE)
		uline = fmt->uline;
	if (mask & SFMT_FG) {
		fg = lookup_color(fmt->fg);
		if (fg == -1) fg = 0;	/* black */
	}
	if (mask & SFMT_BG) {
		bg = lookup_color(fmt->bg);
		if (bg == -1) bg = 7;	/* white */
	}
	if (mask & SFMT_BORDERS)
		siagfmt |= fmt->borders;
	if (mask & SFMT_VADJ)
		siagfmt |= fmt->vadj;
	if (mask & SFMT_HADJ)
		siagfmt |= fmt->hadj;
	if (mask & SFMT_STYLE)
		style = fmt->style;
	return lookup_format(lookup_font(family, size, bold, italic),
				uline, fg, bg, style, siagfmt);
}

/* ---
Here is the whole song and dance to get a format id for Siag versions >= 3.1.

Start by deciding on font family, point size, weight and slant.
Get an index into the font table by calling lookup_font with
the family name as argument along with the size, weight and slant.
The font is automatically loaded, if necessary.

Next pick foreground and background colours. The colours are loaded
by calling lookup_color with the colour name as argument. An index into
the colour table is returned.

Finally we are ready to call lookup_format. Decide if the character
should be underlined. If the application is Siag, also combine the
cell attributes. Pass the font index, the colour index, the underlining
flag and the Siag attributes to lookup_format. An index into the
format table is returned, and it is this index we can use to replace
the old style formats.

Yes, this is more complex than the old stuff. But it is also in fact
easier to extract individual pieces of the formats, since they are
normal C structures. To my eyes, font_table[format_table[fmt].font].size
is much clearer than (fmt & SIZE_MASK)>>SIZE_SHIFT. And it is less
prone to errors. And it leaves room for future expansion, for example
for overstrike and double underline, or bidirectional printing.
--- */

/* ---
Initialise the X format code.
*/

void init_format(Display *d)
{
	dpy = d;
}


/* ---
returns the index of a structure with the requested color, or -1
   for failure. Pixel value in color_table[i].color.pixel is allocated
*/

int lookup_color(char *name)
{
	int i;
	Colormap cmap;

	if (!name) {
		abort();
	}
	if (!dpy) return 0;

	cmap = DefaultColormap(dpy, DefaultScreen(dpy));

	for (i = 0; i < ncolor; i++) {
		if (!cstrcasecmp(name, color_table[i].name))
			break;
	}
	if (i == ncolor) return -1;	/* fail */
	if (color_table[i].need_init) {
		color_table[i].color.red = color_table[i].red;
		color_table[i].color.green = color_table[i].green;
		color_table[i].color.blue = color_table[i].blue;
		if (!XAllocColor(dpy, cmap, &color_table[i].color))
			return -1;
		color_table[i].need_init = 0;
	}
	return i;
}

/* ---
returns the index of a font with the specified name, or -1 for failure
*/

static int lookup_fontname(char *name)
{
	int i;

	for (i = 0; i < nfontname; i++) {
		if (!cstrcasecmp(name, fontname_table[i].name))
			return i;
	}
	return -1;
}


/* ---
for the given font with the given properties, encode X, T1 and PS strings.
example (result in x_name will vary depending on resolution):

	font = index of font with name "Helvetica"
	size = 12
	bold = 0
	italic = 1;

	x_name = "-adobe-helvetica-medium-o-*--*-120-75-75-*-*-iso8859-1"
	t1_name = "Helvetica-Oblique"
	ps_name = "Helvetica-Oblique"

t1_name may be NULL. The caller must then use the X font.
*/

static void encode_font(Display *dpy, int font, int size, int bold, int italic,
		char *x_name, char **t1_name, char **ps_name)
{
	char *x_fmt;
	int i = 0;
	int res_x, res_y;	/* screen resolution */
#if 0
	int screen = DefaultScreen(dpy);

	res_x = DisplayWidth(dpy, screen)/(DisplayWidthMM(dpy, screen)/25.4);
	res_y = DisplayHeight(dpy, screen)/(DisplayHeightMM(dpy, screen)/25.4);
#else	/* cheat to get ps right */
	res_x = res_y = 72;
#endif

	if (bold) i += 2;
	if (italic) i += 1;

	x_fmt = fontname_table[font].x_name[i];
	if (!x_fmt) x_fmt = fontname_table[0].x_name[0];
	sprintf(x_name, x_fmt, size, res_x, res_y);
	*t1_name = fontname_table[font].t1_name[i];
	*ps_name = fontname_table[font].ps_name[i];
}

/* ---
return index into font table or -1 for failure. Load font if necessary.
This function is wasteful in that it always loads an X font, even if
a T1 font is found and will be used.
*/

int lookup_font(char *fname, int size, int bold, int italic)
{
	int i, name;
	char x_name[1000], *t1_name, *ps_name;

	name = lookup_fontname(fname);
	if (name == -1) name = 0;	/* fallback value */

	for (i = 0; i < nfont; i++) {
		if (font_table[i].name == name &&
			font_table[i].size == size &&
			font_table[i].bold == bold &&
			font_table[i].italic == italic)
				return i;	/* already loaded */
	}
	nfont++;
	font_table[i].name = name;
	font_table[i].size = size;
	font_table[i].bold = bold;
	font_table[i].italic = italic;
	encode_font(dpy, name, size, bold, italic, x_name, &t1_name, &ps_name);
	font_table[i].x_name = cstrdup(x_name);
	if (ps_name) font_table[i].ps_name = cstrdup(ps_name);
	else font_table[i].ps_name = NULL;
	if (t1_name) font_table[i].t1_name = cstrdup(t1_name);
	else font_table[i].t1_name = NULL;
	font_table[i].fs = XLoadQueryFont(dpy, x_name);
	font_table[i].id = t1_getfontid(t1_name);
	if (font_table[i].fs == NULL) {
		font_table[i].fs =
			XLoadQueryFont(dpy, "*helvetica-medium-r*12*");
	}
	if (font_table[i].fs == NULL) {
		fprintf(stderr, "Panic: can't load any fonts!\n");
		exit(EXIT_FAILURE);
	}
	return i;
}

/* ---
X, T1 and PS fonts are registered separately because they are device
dependent on different devices. This is slightly lame because they will
probably be called from the same Scheme wrapper function, and they are
in the same C translation unit. Never mind.
*/

int register_x_font(char *name, char *x, char *xi, char *xb, char *xbi)
{
	int i;

	for (i = 0; i < nfontname; i++) {
		if (!strcasecmp(name, fontname_table[i].name)) break;
	}
	if (i == nfontname) {
		nfontname++;
		fontname_table[i].name = cstrdup(name);
		fontname_table[i].ps_name[0] = NULL;
		fontname_table[i].ps_name[1] = NULL;
		fontname_table[i].ps_name[2] = NULL;
		fontname_table[i].ps_name[3] = NULL;
		fontname_table[i].t1_name[0] = NULL;
		fontname_table[i].t1_name[1] = NULL;
		fontname_table[i].t1_name[2] = NULL;
		fontname_table[i].t1_name[3] = NULL;
	}
	fontname_table[i].x_name[0] = cstrdup(x);
	fontname_table[i].x_name[1] = cstrdup(xi);
	fontname_table[i].x_name[2] = cstrdup(xb);
	fontname_table[i].x_name[3] = cstrdup(xbi);
	return i;
}

/* ---
As above but for Postscript.
*/

int register_ps_font(char *name, char *ps, char *psi, char *psb, char *psbi)
{
	int i;

	for (i = 0; i < nfontname; i++) {
		if (!cstrcasecmp(name, fontname_table[i].name)) break;
	}
	if (i == nfontname) {
		nfontname++;
		fontname_table[i].name = cstrdup(name);
		fontname_table[i].x_name[0] = NULL;
		fontname_table[i].x_name[1] = NULL;
		fontname_table[i].x_name[2] = NULL;
		fontname_table[i].x_name[3] = NULL;
		fontname_table[i].t1_name[0] = NULL;
		fontname_table[i].t1_name[1] = NULL;
		fontname_table[i].t1_name[2] = NULL;
		fontname_table[i].t1_name[3] = NULL;
	}
	fontname_table[i].ps_name[0] = cstrdup(ps);
	fontname_table[i].ps_name[1] = cstrdup(psi);
	fontname_table[i].ps_name[2] = cstrdup(psb);
	fontname_table[i].ps_name[3] = cstrdup(psbi);
	/* and now, a piece written in Hackish */
	fontname_table[i].iso8859_1 = strcmp(ps, psi)?1:0;
	return i;
}

/* ---
As above but for Type 1 (T1lib).
*/

int register_t1_font(char *name, char *ps, char *psi, char *psb, char *psbi)
{
	int i;

	for (i = 0; i < nfontname; i++) {
		if (!cstrcasecmp(name, fontname_table[i].name)) break;
	}
	if (i == nfontname) {
		nfontname++;
		fontname_table[i].name = cstrdup(name);
		fontname_table[i].x_name[0] = NULL;
		fontname_table[i].x_name[1] = NULL;
		fontname_table[i].x_name[2] = NULL;
		fontname_table[i].x_name[3] = NULL;
		fontname_table[i].ps_name[0] = NULL;
		fontname_table[i].ps_name[1] = NULL;
		fontname_table[i].ps_name[2] = NULL;
		fontname_table[i].ps_name[3] = NULL;
	}
	fontname_table[i].t1_name[0] = cstrdup(ps);
	fontname_table[i].t1_name[1] = cstrdup(psi);
	fontname_table[i].t1_name[2] = cstrdup(psb);
	fontname_table[i].t1_name[3] = cstrdup(psbi);
	/* and now, a piece written in Hackish */
	fontname_table[i].iso8859_1 = strcmp(ps, psi)?1:0;
	return i;
}

/* ---
handle mapping between e.g. Arial and Helvetica

Store the name of a font we don't have and the name of a replacement
*/

int font_alias(char *alias, char *name)
{
	int i;

	for (i = 0; i < nfontalias; i++) {
		if (!cstrcasecmp(alias, fontalias[i].alias)) break;
	}
	if (i == nfontalias) {
		nfontalias++;
		fontalias[i].alias = cstrdup(alias);
	}
	fontalias[i].name = cstrdup(name);
	return i;
}

/* ---
If there is a font with exactly the alias name, return that. Otherwise
if there is an alias, return the name of the real font. Otherwise NULL.
*/

char *lookup_fontalias(char *alias)
{
	int i;

	if (alias == NULL) return NULL;
	if (lookup_fontname(alias) >= 0) return alias;
	for (i = 0; i < nfontalias; i++) {
		if (!cstrcasecmp(alias, fontalias[i].alias))
			return fontalias[i].name;
	}
	return NULL;
}

/* ---
stores a colour name along with the red, green and blue values,
   or overwrites an existing colour with the same name.
   The colour is not allocated until it is looked up in lookup_color.
*/

int register_color(char *name,
	unsigned short red, unsigned short green, unsigned short blue)
{
	int i;

	for (i = 0; i < ncolor; i++)
		if (!cstrcasecmp(name, color_table[i].name)) break;
	if (i == ncolor) {
		ncolor++;
		color_table[i].name = cstrdup(name);
	}
	color_table[i].red = red;
	color_table[i].green = green;
	color_table[i].blue = blue;
	color_table[i].need_init = 1;
	return i;
}

/* ---
fmt is an index into fontname_table
*/

char *family2name(int fmt)
{
	return fontname_table[fmt].name;
}

/* ---
fmt is index into font_table
*/

char *size2name(int fmt)
{
	static char b[80];

	sprintf(b, "%d", font_table[fmt].size/10);
	return b;
}

/* ---
fmt is index into color_table
*/

char *color2name(int fmt)
{
	return color_table[fmt].name;
}

/* ---
Postscript font name
*/

char *ps_fontname(int fmt)
{
	char *p = font_table[format_table[fmt].font].ps_name;
	if (p == NULL) p = font_table[format_table[fmt].font].t1_name;
	if (p == NULL) p = font_table[0].ps_name;
	return p;
}

/* ---
Postscript font size
*/

int ps_fontsize(int fmt)
{
	return font_table[format_table[fmt].font].size / 10;
}

static int lastfont;

/* ---
Define Latin-1 fonts

Improved: only makes the fonts that are actually used.

This is a mess, but here is a strategy I believe in. Go through the entire
format table, looking for font, italic and bold information. For each
format, call a function that figures out whether to use ps or t1 fonts,
encodes it, uploads the font if necessary and adds it to a list of "done"
fonts. Fonts that have already been done are not repeated even if they are
found again in the format table.

Because of the way the font system works (the same font may be used for
regular and bold, for example), it is necessary to encode the fonts
before checking if it is already done.
*/

static struct done {
	char *name;
	struct done *next;
} *done_fonts;

static void ps_makefont(FILE *fp, int id, char *ps_name, int pfab, char *enc)
{
	struct done *df;

	for (df = done_fonts; df; df = df->next)
		if (!strcmp(df->name, ps_name)) return;

	df = cmalloc(sizeof(struct done));
	df->name = cstrdup(ps_name);
	df->next = done_fonts;
	done_fonts = df;

#ifdef HAVE_LIB_T1X
	if (pfab) {	/* dump font into ps file */
		char b[1024], cmd[10000];
		FILE *ffp;
		if (id == -1) return;
		sprintf(cmd, "%s/common/readpfa %s %s", datadir,
			T1_GetFileSearchPath(T1_PFAB_PATH),
			T1_GetFontFileName(id));
		ffp = popen(cmd, "r");
		if (ffp == NULL) {
			fprintf(stderr, "Couldn't read from %s\n", cmd);
			return;
		}
		while (fgets(b, sizeof b, ffp))
			fputs(b, fp);
		pclose(ffp);
		/* and presto, we have a font */
	}
#endif
	if (!enc) return;	/* don't bother */

	fprintf(fp, "/%s findfont\n", ps_name);

	fprintf(fp, "dup length dict begin\n");
	fprintf(fp, "  {1 index /FID ne {def} {pop pop} ifelse} forall\n");
	fprintf(fp, "  /Encoding %sEncoding def\n", enc);
	fprintf(fp, "  currentdict\n");
	fprintf(fp, "end\n");
	fprintf(fp, "/%s-%s exch definefont pop\n", ps_name, enc);
}

void ps_makefonts(FILE *fp)
{
	int i, j;
	struct done *df;

	done_fonts = NULL;

	for (i = 0; i < nformat; i++) {
		char *ps_name, *enc;
		int fi = format_table[i].font;	/* index to font_table */
		int id = font_table[fi].id;
		int bold = font_table[fi].bold;
		int italic = font_table[fi].italic;
		int name = font_table[fi].name;
		j = 0;
		if (bold) j += 2;
		if (italic) j += 1;

		if (fontname_table[name].iso8859_1) enc = "ISOLatin1";
		else enc = NULL;

		ps_name = fontname_table[name].ps_name[j];
		if (ps_name) ps_makefont(fp, id, ps_name, 0, enc);
		else {
			ps_name = fontname_table[name].t1_name[j];
			if (!ps_name) continue;		/* give up */
			ps_makefont(fp, id, ps_name, 1, enc);
		}
	}
	while (done_fonts) {
		df = done_fonts->next;
		cfree(done_fonts->name);
		cfree(done_fonts);
		done_fonts = df;
	}
}

/* ---
Set the font
*/

void ps_set_font(FILE *fp, int newfont)
{
	int name;

	if (newfont == lastfont)
		return;
	lastfont = newfont;
	if (newfont == -1) return;	/* starter mark */
	name = font_table[newfont].name;
	fprintf(fp, "/%s%s findfont\n",
		ps_fontname(newfont),
		fontname_table[name].iso8859_1?"-ISOLatin1":"");
	fprintf(fp, "%d scalefont\n", ps_fontsize(newfont));
	fprintf(fp, "setfont\n");
}


/* ---
   Code to handle a combination of X and T1 fonts. Currently pretty
   suboptimal: makes several passes over strings, doesn't use kerning
   or ligatures.
--- */

/* X code borrowed from Richchar.c */

int rc_width(rich_char c)
{
	int w;
	int index = format_table[c.fmt].font;

	if (has_t1_font(index)) {
#ifdef HAVE_LIB_T1X
		return T1_GetCharWidth(font_table[index].id, c.c)
			*font_table[index].size/10000;
#else
		return font_table[index].size/10;	/* BOGUS */
#endif
	} else {
		w = XTextWidth(font_struct(index), &c.c, 1);
	}

	return w;
}

int rc_height(rich_char c)
{
	int index = format_table[c.fmt].font;

	return font_height(index);
}

void rc_geom(rich_char c, int *width, int *height)
{
	*width = rc_width(c);
	*height = rc_height(c);
}

int rc_strwidth(rich_char *p, int len)
{
	int i, w = 0;

	if (len == -1) len = rc_strlen(p);

	for (i = 0; i < len; i++) {
		w += rc_width(p[i]);
	}
	return w;
}

int rc_strheight(rich_char *p, int len)
{
	int i, h = 0;

	if (len == -1) len = rc_strlen(p);

	for (i = 0; i < len; i++) {
		int h1 = rc_height(p[i]);
		if (h1 > h) h = h1;
	}
	return h;
}

void rc_strgeom(rich_char *p, int len, int *width, int *height)
{
	*width = rc_strwidth(p, len);
	*height = rc_strheight(p, len);
}

/* TODO: handle tabs */
int rc_draw(Drawable d, GC gc, int x, int y, rich_char c, int opaque)
{
	int index, color_index;
	int vadj, width;
	unsigned long color;

	index = format_table[c.fmt].font;
	switch (format_table[c.fmt].siagfmt & VADJ_MASK) {
	case VADJ_TOP:
		vadj = -6;
		break;
	case VADJ_BOTTOM:
		vadj = 6;
		break;
	default:
		vadj = 0;
	}

	color_index = format_table[c.fmt].fg;
	color = get_color(color_index);
	XSetForeground(t1_dpy, gc, color);

	if (has_t1_font(index)) {
#ifdef HAVE_LIB_T1X
		T1_SetCharX(d, gc, t1_mode, x, y+vadj,
			font_table[index].id, c.c,
			font_table[index].size/10.0, NULL);
if (opaque) fprintf(stderr, "Warning: can't draw opaque\n");
#endif
	} else {
		Font font = font_table[index].fs->fid;
		XSetFont(t1_dpy, gc, font);
		if (opaque) XDrawImageString(t1_dpy, d, gc, x, y+vadj, &c.c, 1);
		else XDrawString(t1_dpy, d, gc, x, y+vadj, &c.c, 1);
	}
	if (format_table[c.fmt].uline) {
	/* this should rather use the built-in T1 underlining */
		width = rc_width(c);
		XDrawLine(t1_dpy, d, gc, x, y+1, x+width, y+1);
	}
	return 0;
}

#if 0
static void draw_segment(rich_char *p, size_t len)
{
	printf("draw_segment(%p, %d)\n", p, (int)len);
}
#endif

int rc_strdraw(Drawable d, GC gc, int x, int y, rich_char *p, int len)
{
	int i;

#if 0
rc_forparts(p, draw_segment);
#endif

	if (len == -1) len = rc_strlen(p);

	for (i = 0; i < len; i++) {
		rc_draw(d, gc, x, y, p[i], 0);
		x += rc_width(p[i]);
	}
	return 0;
}

int t1_getfontid(char *name)
{
	int i;

	if (!name) return -1;
	for (i = 0; i < t1_nfont; i++) {
		if (t1_fontname[i] && !strcmp(t1_fontname[i], name))
			return i;
	}
	return -1;
}

char *t1_getfontname(int id)
{
	if (id < 0 || id >= t1_nfont) return NULL;
	return t1_fontname[id];
}

int t1_init(Widget w)
{
#ifdef HAVE_LIB_T1X
	int i;
	char **enc;
#endif

	t1_dpy = XtDisplay(w);
	t1_scr = XtScreen(w);
	t1_scrn = DefaultScreen(t1_dpy);
	t1_mode = 0;

#ifdef HAVE_LIB_T1X
	if (!getenv("T1LIB_CONFIG")) {
		char dir[1024];
		sprintf(dir, "T1LIB_CONFIG=%s/common/t1lib.config", datadir);
		putenv(dir);
	}

	T1_SetBitmapPad(16);

        if ((T1_InitLib(NO_LOGFILE) == NULL)) {
                fprintf(stderr, "Initialization of t1lib failed\n");
                return -1;
        }

        t1_nfont = T1_Get_no_fonts();

        T1_SetX11Params(t1_dpy, DefaultVisual(t1_dpy, t1_scrn),
                DefaultDepth(t1_dpy, t1_scrn),
                DefaultColormap(t1_dpy, t1_scrn));

        enc = T1_LoadEncoding("IsoLatin1.enc");
        if (enc == NULL) {
                fprintf(stderr, "Couldn't load encoding\n");
                exit(EXIT_FAILURE);
        } else {
                T1_SetDefaultEncoding(enc);
        }

        t1_fontname = cmalloc(t1_nfont*sizeof *t1_fontname);
        for (i = 0; i < t1_nfont; i++) {
		char *p;
		T1_LoadFont(i);
		p = T1_GetFontName(i);
		T1_DeleteFont(i);
		if (p == NULL) t1_fontname[i] = NULL;
                else t1_fontname[i] = cstrdup(p);
        }
#endif	/* HAVE_LIB_T1X */
        return 0;
}

int t1_exit(void)
{
#ifdef HAVE_LIB_T1X
        T1_CloseLib();
#endif
	return 0;
}

