/*
   Siag, Scheme In A Grid
   Copyright (C) 1996-1998  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
 */

/* ---
   Module name:    selection.c

   This module handles selection: grabbing, releasing, cutting, pasting.

   The selection uses a custom target SIAG_BLOCK which has this format:

   <HEIGHT>\0<WIDTH>\0<TYPE><TEXT>\0<TYPE><TEXT>\0...

   HEIGHT = numbers of rows in selection
   WIDTH = number of columns in selection
   TYPE = type char where:
	'"' = label
	'#' = empty
	'=' = expression
	'$' = string
	'+' = expression with named interpreter
	'm' = image
	'n' = new line (don't waste effort copying empty cells)
	'z' = end of selection (don't copy empty rows)
   TEXT = verbatim text from the cell

   This is pretty similar to the file format but different enough to
   be confusing.
--- */

#include <stdio.h>
#include <string.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <X11/Xmu/Atoms.h>
#include <X11/Xmu/StdSel.h>

#include "../common/cmalloc.h"
#include "../common/common.h"
#include "../siag/calc.h"
#include "../siag/selection.h"
#include "xsiag.h"

Atom target_atom;	/* used for selection */

/* ---
*/
Boolean convert_proc(Widget w,
	Atom *selection, Atom *target, Atom *type_return,
	XtPointer *value_return, unsigned long *length_return,
	int *format_return)
{
	unsigned int lr;
	window *wl = find_window_by_widget(w);
	XSelectionRequestEvent *req = XtGetSelectionRequest(w,
		*selection, (XtRequestId) NULL);

	/* handle all required atoms, and the one that we use */
	/* Xt already handles MULTIPLE, no branch necessary */
	if (*target == XA_TARGETS(XtDisplay(w))) {
		Atom *targetP;
		Atom *std_targets;
		unsigned long std_length;
		XmuConvertStandardSelection(w, req->time, selection,
			target, type_return,
			(XPointer *)&std_targets,
			&std_length, format_return);
		*value_return = XtMalloc(sizeof(Atom)*(std_length+1));
		targetP = *(Atom **)value_return;
		*length_return = std_length+1;
		*targetP++ = target_atom;
		memcpy(std_targets, targetP, sizeof(Atom)*std_length);
		XtFree((char *)std_targets);
		*type_return = XA_ATOM;
		*format_return = sizeof(Atom)*8;
		return True;
	} else if (*target == target_atom) {
		/* handle normal selection */
		char *data = pack_area(wl->buf, wl->bsht,
				block_upper(wl).row, block_upper(wl).col,
				block_lower(wl).row, block_lower(wl).col,
				&lr);
		*length_return = lr;
		*value_return = XtMalloc(*length_return);
		memcpy(*value_return, data, *length_return);
		cfree(data);

		*type_return = target_atom;

		*format_return = 8;
		return True;
	} else if (*target == XA_STRING) {
		/* handle string selections (from outside Siag) */
		char *data;
		*length_return = 32000;
		data = XtMalloc(*length_return);
		pack_string_selection(wl->buf, data,
				wl->bsht,
				block_upper(wl).row, block_upper(wl).col,
				block_lower(wl).row, block_lower(wl).col);
		*value_return = data;
		*length_return = strlen(data);
		*type_return = XA_STRING;
		*format_return = 8;
		return True;
	} else {
		if (XmuConvertStandardSelection(w, CurrentTime, selection,
				target, type_return, (XPointer *)value_return,
				length_return, format_return))
			return True;
		else {
			XtWarning("Siag: unsupported selection type\n");
			return False;
		}
	}
	/*NOTREACHED*/
}

/* ---
*/
void lose_ownership_proc(Widget w, Atom *selection)
{
	window *wl = find_window_by_widget(w);

	wl->bsht = -1;
	set_top(wl, make_position(-1, -1));
	pr_scr_flag = TRUE;	/* make sure it's redrawn */
	show_cur(wl);
}

/* ---
*/
void string_requestor_callback(Widget w, XtPointer client_data,
	Atom *selection, Atom *type, XtPointer value,
	unsigned long *length, int *format)
{
	if ((value == NULL) && (*length == 0)) {
		XBell(XtDisplay(w), 100);
		XtWarning("Siag: no selection or selection timed out\n");
	} else {
		unpack_string_selection(w_list->buf, (char *)value,
			w_list->sht,
			get_point(w_list).row, get_point(w_list).col);
		w_list->buf->change = TRUE;
		calc_matrix(w_list->buf);
		XtFree((char *)value);
		pr_scr_flag = TRUE;
	}
}

/* ---
*/
void requestor_callback(Widget w, XtPointer client_data,
	Atom *selection, Atom *type, XtPointer value,
	unsigned long *length, int *format)
{
	if ((value == NULL) && (*length == 0)) {
	/* if we asked for a SIAG_BLOCK and got a null response,
	   we'll ask again, this time for an XA_STRING */
		XtGetSelectionValue(w, XA_PRIMARY, XA_STRING,
			string_requestor_callback,
			NULL, CurrentTime);	/* NULL is bogus event */
	} else {
		unpack_area(w_list->buf, (char *)value,
			w_list->sht,
			get_point(w_list).row, get_point(w_list).col);
		w_list->buf->change = TRUE;
		calc_matrix(w_list->buf);
		XtFree((char *)value);
		pr_scr_flag = TRUE;
	}
}

