/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* This file contains basic types, macros and functions used everywhere. */

/* constants ================================================================*/

#define MALAGA_VERSION "4.3"

#undef NULL
#define NULL 0 /* null pointer */

#define BITS_PER_BYTE 8

/* attribute for a function that never returns */
#ifdef __GNUC__
#define NO_RETURN __attribute__ ((noreturn))
#else
#define NO_RETURN
#endif

extern char lower_letter[256];
/* This table contains the lower case letter for each letter code,
 * and 0 for each non-letter. */

/* basic types ==============================================================*/

#define GLOBAL extern /* item that may be used in other modules */
#define LOCAL static /* item that belongs to this module */
#define FORWARD static /* item that will be defined later in this module */

/* numeric types */
typedef signed char byte_t; /* signed 8 bits */
typedef unsigned char u_byte_t; /* unsigned 8 bits */
typedef signed short int short_t; /* signed 16 bits */
typedef unsigned short int u_short_t; /* unsigned 16 bits */
typedef signed long int int_t; /* signed 32 bits */
typedef unsigned long int u_int_t; /* unsigned 32 bits */

/* character types */
typedef char *string_t; /* a EOS-terminated C string */
#define EOS '\0' /* End-Of-String control character */
#define ORD(char) ((u_byte_t) (char)) /* the ordinal number of a char */

/* boolean type */
#undef bool_t /* <rpc/rpc.h> conflicts with "bool_t" definition */
#undef TRUE
#undef FALSE
typedef enum {FALSE, TRUE} bool_t;

typedef struct TEXT_T *text_t;
/* a data structure that contains a string that may grow indefinitely */

/* macros ===================================================================*/

#define MIN(a,b) ((a) < (b) ? (a) : (b)) /* minimum of <a> and <b> */
#define MAX(a,b) ((a) > (b) ? (a) : (b)) /* maximum of <a> and <b> */
#define ABS(a) ((a) >= 0 ? (a) : (-a)) /* absolute value of <a> */

#define TO_LOWER(c) (lower_letter[ORD(c)] != 0 ? lower_letter[ORD(c)] : (c))
/* Return <c> as a lower letter, or <c> itself if it is not a letter. */

#define IS_ALPHA(c) (lower_letter[ORD(c)] != 0)
/* Return whether <c> is a letter */

#define IS_PRINT(c) ((ORD (c) >= 0x20 && ORD (c) < 0x80) || ORD (c) >= 0xa0)
/* Return whether <c> is a printable character. */

/* The following macro only expands when DEBUG is defined. */
#ifdef DEBUG
/* If <cond_expr> doesn't hold, print an error message and exit. */ 
#define DB_ASSERT(cond_expr) \
  do \
  { \
    if (!(cond_expr)) \
      { \
        fprintf (stderr, "DEBUG: %s:%d: assertion \"%s\" failed\n", \
	         __FILE__, __LINE__, #cond_expr); \
        exit (1); \
      } \
  } while (FALSE)
#else
#define DB_ASSERT(cond_expr) do {} while (FALSE)
#endif

/* variables ================================================================*/

extern void (*error) (string_t, ...) NO_RETURN;
/* print an error message, like "printf", and return to the user input loop.
 * (exit the program if there is no user input loop).
 * THIS FUNCTION SHOULD NEVER RETURN. */

/* memory functions =========================================================*/

extern void *new_mem (int_t item_size);
/* Allocate a memory block of size <item_size>, clear it and return it.
 * If memory is out, call the function "error". */

extern void *new_vector (int_t item_size, int_t items);
/* Allocate a vector of <items> items of size <item_size> each, clear it
 * and return it. If memory is out, call the function "error". */

extern int_t renew_vector (void *buffer_ptr, 
			   int_t item_size, 
			   int_t new_size);
/* Realloc the buffer at <*buffer_ptr> to contain <new_size> items of
 * <item_size> each and return <new_size>. If memory is out, call the
 * function "error". */

extern void free_mem (void *pointer);
/* Free memory <*pointer> points to, and set <*pointer> to NULL. */

/* string functions =========================================================*/

extern string_t new_string (string_t string, string_t end);
/* Allocate memory and copy <string> into it.
 * If <end> != NULL, it marks the end of the string.
 * The string can be deleted with "free". */

extern string_t new_string_readable (string_t from, string_t from_end);
/* Like "new_string", but copy a "\" in front of quotes
 * and copy any control chars in octal code: "\000". 
 * If <from_end> != NULL, it marks the end of the string. 
 * The result string must be freed with "free" after use. */

extern string_t concat_strings (string_t first_string, ...);
/* Concatenate a list of strings and return the result string.
 * Must have NULL-terminated list of strings as parameters.
 * The result string can be deleted with "free". */

extern int_t strcmp_no_case (string_t str1, string_t str2);
/* Return (case insensitive) lexical order of <str1> and <str2>:
 * Result is -1 if <str1> < <str2>,
 *            0 if <str1> = <str2>,
 *            1 if <str1> > <str2>. */

extern int_t strncmp_no_case (string_t str1, string_t str2, int_t n);
/* Return (case insensitive) lexical order of <str1> and <str2>,
 * but compare only the first <n> characters.
 * Result is -1 if <str1> < <str2>,
 *            0 if <str1> = <str2>,
 *            1 if <str1> > <str2>. */

extern string_t next_non_space (string_t string);
/* Return <string>, but without leading spaces. */

extern string_t double_to_string (double number);
/* Convert <number> to a string. It must be freed with "free" after use. */

extern string_t int_to_string (int_t number);
/* Convert <number> to a string. It must be freed with "free" after use. */

extern string_t replace_arguments (string_t format, string_t chars, ...);
/* Create a new string with a copy of <format>.
 * Replace each sequence "%<c>" in <format>, where <c> is the <n>-th
 * char in <chars>, by the <n>-th additional string argument. 
 * Return the result string. It must be freed with "free" after use. */

/* text functions ===========================================================*/

extern text_t new_text (void);
/* Return a new text structure. */

extern void clear_text (text_t *text_ptr);
/* Initialize <*text_ptr> to an empty string. 
 * If <*text_ptr> is NULL, a new text structure is allocated. */

extern void free_text (text_t *text_ptr);
/* Free the content of <*text_ptr>. */

extern void add_to_text (text_t text, string_t string);
/* Add <string> to <text>. */

extern void add_char_to_text (text_t text, char character);
/* Add <character> to <text>. */

extern void insert_in_text (text_t text, string_t string, int_t position);
/* Insert <string> at <position> in <text> (position counts from 0 onward). */

extern void insert_char_in_text (text_t text, char character, int_t position);
/* Insert <character> at <position> in <text>. */

extern int_t text_length (text_t text);
/* Return length of <text>. */

extern string_t text_string (text_t text);
/* Return content of <text> as a string. 
 * This string is valid until another text function on <text> is called. */

extern string_t text_to_string (text_t *text_ptr);
/* Return content of <*text_ptr> as a string and delete <*text_ptr>.
 * The string must be freed with "free" after use. */

/* end of file ==============================================================*/
