// Command-line Darxite FTP client

#include <signal.h>
#include "commands.h"

#define OPTION_STRING	"vhc:u:"
#define VERSION			"0.23"
#define AUTHORS			"Ashley Montanaro"
#define BASENAME(x)	(strrchr((x), '/') ? strrchr((x), '/') + 1 : (x))

char *ReadCmdLine(int argc, char **argv);
void ReadServersFile(void);
void PrintVersion(void);
void Usage(char *prog_name);
BOOL ParseCommand(const char *cmd);

void PrintVersion(void)
{
    printf("dxFTP utility v%s, release %s (%s) by %s\n", VERSION,
           RELEASE_VER, RELEASE_NAME, AUTHORS);
}

void Usage(char *prog_name)
{
    printf("Usage: %s [options] [URL]\n"
           "where options are:\n"
           "-v, --version\t\t\tShow version and exit\n"
           "-h, --help\t\t\tShow some usage information\n"
           "-c <host>:<port>\t\tConnect to daemon on <host> via <port>\n"
           "--connect <host>:<port>\t\tAs above\n"
           "-u <name>, --user <name>\tLog in as user <name>\n",
           prog_name);
    exit(0);
}

char *ReadCmdLine(int argc, char **argv)
{
    char opt;
    int option_index;
    
    static struct option long_options[] = {
        { "version", 	0, 0, 'v' },
        { "help",		0, 0, 'h' },
        { "connect",	1, 0, 'c' },
        { "user",		1, 0, 'u' },
        { 0, 			0, 0,  0  }
    };

    opt = getopt_long(argc, argv, OPTION_STRING, long_options,
                      &option_index);
    while (opt != -1)
    {
        switch (opt)
        {
        case 'c':
            if (strchr(optarg, ':'))
            {
                memset(DaemonHost, 0, sizeof(DaemonHost));
                strncpy(DaemonHost, optarg, strchr(optarg, ':') - optarg);
                DaemonPort = atoi(strchr(optarg, ':') + 1);
                if ((strlen(DaemonHost) == 0) || (DaemonPort <= 0))
                {
                    printf("Invalid host/port to connect to daemon\n");
                }
            }
            else
            {
                printf("Syntax: -c <host>:<port>\n");
            }
            break;
            
        case 'u':
            strcpy(UserName, optarg);
            break;
            
        case 'v':
            PrintVersion();
            exit(0);
            break;

        case 'h':
            Usage(argv[0]);
            break;
        }
        opt = getopt_long(argc, argv, OPTION_STRING, long_options,
                          &option_index);        
    }
    
    if (optind < argc)
        return argv[optind];
    else
       return NULL;
}

BOOL ParseCommand(const char *cmd)
{
    char buffer[1024], buffer2[256], flags[10];
    char argv[10][256], *arg[10], options[256];
    char *last_space = buffer;
    int i, rc, argc = 0, argsc = 0;
    BOOL quoted = FALSE;

    strcpy(buffer, cmd);
    strcpy(options, "");
    for (i = 0; i < 10; i++)
        memset(argv[i], 0, sizeof(argv[i]));
    for (i = 0; i < strlen(buffer); i++)
    {
        if ((buffer[i] == '"') && (!quoted))
        {
            last_space = buffer + i + 1;
            quoted = !quoted;
        }
        
        if ((buffer[i] == ' ') && (!quoted))
        {
            memset(argv[argc], 0, sizeof(argv[argc]));
            if ((i > 0) && (buffer[i - 1] == '"'))
                strncpy(argv[argc++], last_space, i - (last_space-buffer) - 1);
            else
                strncpy(argv[argc++], last_space, i - (last_space - buffer));
            last_space = buffer + i + 1;
        }
    }
    if ((i > 0) && (buffer[i - 1] == '"'))
        strncpy(argv[argc++], last_space, i - (last_space-buffer) - 1);
    else
        strncpy(argv[argc++], last_space, i - (last_space - buffer));
    for (i = 0; i < argc; i++)
    {
        if ((argv[i][0] == '-') && (strlen(argv[i]) > 1))
            strcat(options, argv[i] + 1);
        else if (strcmp(argv[i], ""))
            arg[argsc++] = argv[i];
    }
    if (argsc == 0)
        return FALSE;
    if (!strcasecmp(arg[0], "get"))
    {
        if (argsc < 2)
        {
            printf("Syntax: get [-r] <file>\n");
        }
        else
        {
            if (strchr(options, 'r'))
                strcpy(flags, "d");
            else
                strcpy(flags, "");
            if (*arg[1] == '/')
            {
                sprintf(buffer2, "ftp://%s%s", Server, arg[1]);
            }
            else
            {
                sprintf(buffer2, "ftp://%s%s%s", Server, Path, arg[1]);
            }
            nospace(buffer2);
            // FIXME: local path should be in current directory!
            if (strcasecmp(UserName, "anonymous"))
            {
                sprintf(buffer, "get \"%s\" | \"%s/\" | %s | %s | | %s",
                        buffer2, LocalDir, UserName, Password, flags);
            }
            else
            {
                sprintf(buffer, "get \"%s\" | \"%s/\" | | | %s", buffer2,
                        LocalDir, flags);
            }
            DaemonWrite(TRUE, buffer);
        }
    }
    else if (!strcasecmp(arg[0], "put"))
    {
        if (argsc < 2)
        {
            printf("Syntax: put [-r] <file>\n");
        }
        else
        {
            if (strchr(options, 'r'))
                strcpy(flags, "du");
            else
                strcpy(flags, "u");
            sprintf(buffer2, "ftp://%s%s%s", Server, Path, BASENAME(arg[1]));
            nospace(buffer2);
            if (strcasecmp(UserName, "anonymous"))
            {
                sprintf(buffer, "get \"%s\" | \"%s\" | %s | %s | %s",
                        buffer2, arg[1], UserName, Password, flags);
            }
            else
            {
                sprintf(buffer, "get \"%s\" | \"%s\" | | | %s", buffer2,
                        arg[1], flags);
            }
            DaemonWrite(TRUE, buffer);
        }
    }
    else if (!strcasecmp(arg[0], "open"))
    {
        if (argsc < 2)
        {
            printf("Syntax: open [-u] <host>\n");
        }
        else if (strchr(options, 'u'))
        {
            strcpy(Password, "");
            strcpy(UserName, "");
            OpenUrl(arg[1], FALSE);
        }
        else
        {
            OpenUrl(arg[1], TRUE);
        }
    }
    else if (!strcasecmp(arg[0], "site"))
    {
        if (argsc < 2)
            printf("Syntax: site <command>\n");
        else
            DaemonWrite(TRUE, "FTP %s", buffer);
    }
    else if (!strcasecmp(arg[0], "quote"))
    {
        if (argsc < 2)
            printf("Syntax: quote <string>\n");
        else
            DaemonWrite(TRUE, "FTP %s", buffer + 6);
    }
    else if (!strcasecmp(arg[0], "mkdir"))
    {
        if (argsc < 2)
        {
            printf("Syntax: mkdir <dir1> <dir2>...\n");
        }
        else
        {
            for (i = 1; i < argsc; i++)
                DaemonWrite(TRUE, "FTP MKD %s", arg[i]);
        }
    }
    else if (!strcasecmp(arg[0], "lmkdir"))
    {
        if (argsc < 2)
        {
            printf("Syntax: lmkdir <dir1> <dir2>...\n");
        }
        else
        {
            for (i = 1; i < argsc; i++)
            {
                rc = mkdir(arg[i], 0644);
                if (rc != 0)
                {
                    printf("Couldn't make directory \"%s\": %s\n", arg[i],
                           strerror(errno));
                }
            }
        }
    }
    else if (!strcasecmp(arg[0], "status"))
    {
        DaemonWrite(TRUE, "Status");
    }
    else if (!strcasecmp(arg[0], "close"))
    {
        LoggedIn = FALSE;
        DaemonWrite(TRUE, "close");
    }
    else if (!strcasecmp(arg[0], "pwd"))
    {
        DaemonWrite(TRUE, "FTP pwd");
    }
    else if (!strcasecmp(arg[0], "lpwd"))
    {
            printf("%s\n", LocalDir);
    }
    else if (!strcasecmp(arg[0], "mv"))
    {
        if (argsc != 3)
        {
            printf("Syntax: mv <oldfile> <newfile>\n");
        }
        else
        {
            DaemonWrite(FALSE, "FTP RNFR %s", arg[1]);
            if (*RespBuf <= '4')
            {
                sprintf(buffer, "FTP RNTO %s", arg[2]);
                // if we want to move into a directory
                if (arg[2][strlen(arg[2]) - 1] == '/')
                    strcat(buffer, BASENAME(arg[1]));
                DaemonWrite(TRUE, buffer);
            }
            else
            {
                printf(RespBuf);
            }
        }
    }
    else if (!strcasecmp(arg[0], "rename"))
    {
        if (argsc != 3)
        {
            printf("Syntax: rename <oldfile> <newfile>\n");
        }
        else
        {
            DaemonWrite(FALSE, "FTP RNFR %s", arg[1]);
            if (*RespBuf <= '4')
                DaemonWrite(TRUE, "FTP RNTO %s", arg[2]);
            else
                printf(RespBuf);
        }
    }
    else if (!strcasecmp(arg[0], "lrename") || !strcasecmp(arg[0], "lmv"))
    {
        if (argsc != 3)
        {
            printf("Syntax: %s <oldfile> <newfile>\n", arg[0]);
        }
        else
        {
            rc = rename(arg[1], arg[2]);
            if (rc != 0)
            {
                printf("Couldn't %s \"%s\" to \"%s\": %s\n", arg[0] + 1,
                       arg[1], arg[2], strerror(errno));
            }
        }
    }
    else if (!strcasecmp(arg[0], "rm"))
    {
        if (argsc < 2)
        {
            printf("Syntax: rm [-f] <file1> <file2>...\n");
        }
        else
        {
            for (i = 1; i < argsc; i++)
            {
                printf("Are you sure you want to delete \"%s\"? ", arg[i]);
                if (fgets(buffer2, sizeof(buffer2), stdin) &&
                    (*buffer2 == 'y'))
                {
                    DaemonWrite(TRUE, "FTP DELE %s", arg[i]);
                }
            }
        }
    }
    else if (!strcasecmp(arg[0], "lrm"))
    {
        if (argsc < 2)
        {
            printf("Syntax: lrm [-f] <file1> <file2>...\n");
        }
        else
        {
            for (i = 1; i < argsc; i++)
            {
                if (!strchr(options, 'f'))
                    printf("Are you sure you want to delete \"%s\"? ", arg[i]);
                if (strchr(options, 'f') ||
                    (fgets(buffer2, sizeof(buffer2), stdin) &&
                    (*buffer2 == 'y')))
                {
                    rc = remove(arg[i]);
                    if (rc != 0)
                    {
                        printf("Couldn't delete \"%s\": %s\n", arg[i],
                               strerror(errno));
                    }
                }
            }
        }
    }
    else if (!strcasecmp(arg[0], "rmdir"))
    {
        if (argsc < 2)
        {
            printf("Syntax: rmdir [-f] <dir1> <dir2>...\n");
        }
        else
        {
            for (i = 1; i < argsc; i++)
            {
                printf("Are you sure you want to delete \"%s\"? ", arg[i]);
                if (fgets(buffer2, sizeof(buffer2), stdin) &&
                    (*buffer2 == 'y'))
                {
                    DaemonWrite(TRUE, "FTP RMD %s", arg[i]);
                }
            }
        }
    }
    else if (!strcasecmp(arg[0], "lrmdir"))
    {
        if (argsc < 2)
        {
            printf("Syntax: lrmdir [-f] <dir1> <dir2>...\n");
        }
        else
        {
            for (i = 1; i < argsc; i++)
            {
                if (!strchr(options, 'f'))
                    printf("Are you sure you want to delete \"%s\"? ", arg[i]);
                if (strchr(options, 'f') ||
                    (fgets(buffer2, sizeof(buffer2), stdin) &&
                    (*buffer2 == 'y')))
                {
                    rc = rmdir(arg[i]);
                    if (rc != 0)
                    {
                        printf("Couldn't delete \"%s\": %s\n", arg[i],
                               strerror(errno));
                    }
                }
            }
        }
    }
    else if (!strcasecmp(arg[0], "ls"))
    {
        if (argsc == 1)
            RemoteLs(".", options);
        else
            RemoteLs(arg[1], options);
    }
    else if (!strcasecmp(arg[0], "lls"))
    {
        if (argsc < 2)
            strcpy(buffer, ".");
        else
            strcpy(buffer, arg[1]);
        LocalLs(buffer, strchr(options, 'l') ? TRUE : FALSE,
                strchr(options, 'a') ? TRUE : FALSE);
    }
    else if (!strcasecmp(arg[0], "cd"))
    {
        if (argsc != 2)
            printf("Syntax: cd <directory>\n");
        else
            Cwd(arg[1]);
    }
    else if (!strcasecmp(arg[0], "lcd"))
    {
        if (argsc != 2)
        {
            printf("Syntax: lcd <directory>\n");
        }
        else
        {
            if (chdir(arg[1]) == 0)
            {
                if (getcwd(LocalDir, sizeof(LocalDir)) == NULL)
                {
                    strcpy(LocalDir, DX_OutputDir);
                    printf("Couldn't get cwd: %s\n", strerror(errno));
                }
            }
            else
            {
                printf("Couldn't change directory: %s\n", strerror(errno));
            }
        }
    }
    else if (!strcasecmp(arg[0], "help"))
    {
        printf("dxFTP understands the following commands:\n"
               "open [-u] <site>: opens an FTP site\n"
               "close           : closes connection\n"
               "get <file>      : gets a remote file\n"
               "put <file>      : puts a local file\n"
               "[l]ls [dir]     : lists the directory\n"
               "[l]cd <dir>     : changes the directory\n"
               "[l]mkdir <dir>  : creates a directory\n"
               "[l]mv           : moves a file\n"
               "[l]pwd          : shows the current directory\n"
               "[l]rm <file>    : deletes a file\n"
               "[l]rmdir <dir>  : deletes a directory\n"
               "[l]rename       : renames a file\n"
               "site <cmd>      : executes a site-specific command\n"
               "status          : displays the daemon's status\n"
               "quote <cmd>     : sends a raw FTP string to the server\n"
               "bye,quit,exit   : exits dxFTP.\n"
               "Commands beginning with an 'l' act locally, not remotely.\n");
    }
    else if (!strcasecmp(arg[0], "bye") ||
             !strcasecmp(arg[0], "exit") ||
             !strcasecmp(arg[0], "quit"))
    {
        return TRUE;
    }
    else
    {
        printf("Don't know that command, sorry\n");
    }
    return FALSE;
}

void ReadServersFile(void)
{
    FILE *file;
    char buffer[256], *field_ptr, *buf_ptr;
    char user_name[256], password[256];
    char alias[256], url[256];

    ServerCount = 0;
    sprintf(buffer, "%s/servers", DX_ProgDir);
    file = fopen(buffer, "r");
    if (file)
    {
        memset(buffer, 0, sizeof(buffer));
        while (DX_ReadFileLine(buffer, sizeof(buffer), file))
        {
            if (buffer[strlen(buffer) - 1] == '\n')
                buffer[strlen(buffer) - 1] = '\0';
            if (buffer[strlen(buffer) - 1] == '\r')
                buffer[strlen(buffer) - 1] = '\0';
            
            if (!(buf_ptr = DX_StripComments(buffer)))
                continue;
            
// URL | UserName | Password | Alias | Comment | Flags | Mirrors
            
            strcpy(url, "");
            strcpy(user_name, "");
            strcpy(password, "");
            strcpy(alias, "");
            
            field_ptr = DX_GetNextField(buffer);
            if (*field_ptr)
            {
                // because the GetNextField call does not remove
                // quotes, we have to do it ourselves
                if (*field_ptr == '"')
                    field_ptr++;
                strcpy(url, field_ptr);
                if (url[strlen(url) - 1] == '"')
                    url[strlen(url) - 1] = '\0';
            }
            field_ptr += strlen(field_ptr) + 1;
            
            field_ptr = DX_GetNextField(field_ptr);
            if (*field_ptr)
            {
                if (*field_ptr == '"')
                    field_ptr++;
                strcpy(user_name, field_ptr);
                if (user_name[strlen(user_name) - 1] == '"')
                    user_name[strlen(user_name) - 1] = '\0';
            }
            field_ptr += strlen(field_ptr) + 1;
            
            field_ptr = DX_GetNextField(field_ptr);
            if (*field_ptr)
            {
                if (*field_ptr == '"')
                    field_ptr++;
                strcpy(password, field_ptr);
                if (password[strlen(password) - 1] == '"')
                    password[strlen(password) - 1] = '\0';
            }
            field_ptr += strlen(field_ptr) + 1;
            
            field_ptr = DX_GetNextField(field_ptr);
            if (*field_ptr)
            {
                if (*field_ptr == '"')
                    field_ptr++;
                strcpy(alias, field_ptr);
                if (alias[strlen(alias) - 1] == '"')
                    alias[strlen(alias) - 1] = '\0';
            }
            
            if (strcmp(url, ""))
            {
                ServHost[ServerCount] = malloc(256);
                ServPath[ServerCount] = malloc(256);
                DX_ParseUrl(url, buffer, ServHost[ServerCount], 256,
                            ServPath[ServerCount], 256);
                ServLogin[ServerCount] = strdup(user_name);
                ServPassword[ServerCount] = strdup(password);
                ServAlias[ServerCount] = strdup(alias);
                ServerCount++;
            }
        }
        fclose(file);
    }
}

void exit_func(void)
{
    if (LoggedIn)
        DaemonWrite(TRUE, "Close");
    DX_DisconnectClient(DaemonFd);
}

int main(int argc, char *argv[])
{
    BOOL quit = FALSE;
    char buffer[2048], *url, *cmd;

    url = ReadCmdLine(argc, argv);
    // if we don't ignore this, we die when connecting to a dodgy port
    signal(SIGPIPE, SIG_IGN);
    if ((strlen(DaemonHost) > 0) && (DaemonPort > 0))
    {
        printf("Connecting to daemon on host %s, port %d...\n", DaemonHost,
               DaemonPort);
        sprintf(buffer, "Enter password: ");
        DaemonFd = DX_ConnectRemoteClient(DaemonHost, DaemonPort,
                                          getpass(buffer), "dxftp");
        if (DaemonFd < 0)
        {
            fprintf(stderr, "Couldn't connect to daemon: %s\n",
                    strerror(DX_errno));
            return 1;
        }
        else
        {
            printf("Connected OK.\n");
        }
    }
    else
    {
        DaemonFd = DX_ConnectClient("dxftp");
        if (DaemonFd < 0)
        {
            fprintf(stderr, "Couldn't connect to daemon: %s\n",
                    strerror(DX_errno));
            exit(1);
        }
    }
    
    DX_ReadConfigFiles();
    ReadServersFile();
    if (!strcmp(DX_OutputDir, ".") || !strcmp(DX_OutputDir, "./"))
    {
        getcwd(LocalDir, sizeof(LocalDir));
    }
    else
    {
        if (chdir(DX_OutputDir) != 0)
        {
            printf("Couldn't change dir to \"%s\": %s", DX_OutputDir,
                   strerror(errno));
            getcwd(LocalDir, sizeof(LocalDir));
        }
        else
        {
            strcpy(LocalDir, DX_OutputDir);
        }
    }
    atexit(exit_func);
    
	rl_readline_name = strdup("dxftp");
	rl_terminal_name = getenv("TERM");
    rl_completion_entry_function = (Function *) Completion;
	using_history();
    
    // open the start url anonymously if no user name's been specified
    OpenUrl(url, !*UserName);
    
    while (!quit)
    {
        /*if (LoggedIn)
            printf("%s %s > ", Server, Path);
        else
        printf("dxftp> ");*/
        if (LoggedIn)
            sprintf(buffer, "%s %s > ", Server, Path);
        else
            sprintf(buffer, "dxftp> ");
        /*if (!fgets(buffer, sizeof(buffer), stdin))
        {
            printf("\n");
            quit = TRUE;
        }
        else
        {
            buffer[strlen(buffer) - 1] = '\0'; // strip the final \n
            quit = ParseCommand(buffer);
            }*/
        cmd = readline(buffer);
        if (!cmd)
        {
            printf("\n");
            quit = TRUE;
        }
        else
        {
            quit = ParseCommand(cmd);
            if (*cmd)
                add_history(cmd);
            free(cmd);
        }
    }
    return 0;
}
