/*
 sound.c : sound plugin for irssi

    Copyright (C) 1999 Trever Adams

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <string.h>

#include <irssi-plugin.h>
#include <ui-common/txt.h>
#include "sound.h"

static int start_bg_sound (gchar *sound_name);
static int check_child (int pid);

/* For custom format it is: $1 = sender, $2 = sendaddr, $3 = target, *
 * $4 = sound, $5 = action text */

static FORMAT_REC formats[] =
{
    /* ---- */
    { NULL, PLUGIN_LONG_NAME " plugin", 0 },

    { "start", PLUGIN_LONG_NAME " plugin loaded", 0 },
    { "stop", PLUGIN_LONG_NAME " plugin unloaded", 0 },
    { "sound", "%8[%7%_$1 %11SOUND%_%8] $4", 4, {0, 0, 0, 0} },
    { "soundaction", "%15 (*) $1%7 $5 %8[%6sound: %10$4%8]", 5, {0, 0, 0, 0, 0} },
    { "soundbusy", "%15 (*) $1%7 $5 %8[%7sound: %10$4%8] $7- %4%_Hardware/Software "
                   "Busy - Could NOT Play Sound", 6, {0, 0, 0, 0, 0, 0} },
    { "nosound", "%15 (*) $1%7 $5 %8[%7sound: %10$4%8] $7- %4%_No Such File",
                 6, {0, 0, 0, 0, 0, 0} },

    { NULL, NULL }
};

enum
{
    PLUGTXT_START,
    PLUGTXT_STOP,
    PLUGTXT_SOUND,
    PLUGTXT_SOUND_ACTION,
    PLUGTXT_SOUND_BUSY,
    PLUGTXT_NOSOUND
};

static PLUGIN_REC *plug;

static gint sound_pid=0;

gchar *get_scommand(char *sound_name)
{
PLUGIN_DATA *data=plug->data;
gint len;

    len = strlen(sound_name);
    if (len > 3 && g_strcasecmp(sound_name+len-3, "mp3") == 0)
        return(data->mp3_path);
    else if (len > 4 && g_strcasecmp(sound_name+len-4, "midi") == 0)
        return(data->midi_path);
    else return(data->default_path);
}

static gint send_sound_func(gchar *args)
{
gchar *str, *str2, *sound, *rest, *params;
PLUGIN_DATA *data = plug->data;
DCC_REC *dcc;

    if (cur_channel->type != CHANNEL_TYPE_CHANNEL &&
	cur_channel->type != CHANNEL_TYPE_QUERY &&
	cur_channel->type != CHANNEL_TYPE_DCC_CHAT)
    {
	/* /sound is useless in (status), (msgs) and (empty) windows .. */
	return TRUE;
    }

    params = event_get_params(args, 2 | PARAM_FLAG_GETREST, &sound, &rest);
    g_strdown(sound);

    if(sound_pid>0 && check_child(sound_pid)==0)
    {
        kill(sound_pid, SIGTERM);
    }

    str = g_strdup_printf("%s/%s", data->sound_dir, sound);
    str2 = convert_home(str);
    sound_pid=start_bg_sound(str2);
    g_free(str);
    g_free(str2);

    if(*rest!='\0') /* We have an embedded action, might as well show it */
    {
        str=NULL;
        if (cur_channel->type == CHANNEL_TYPE_DCC_CHAT)
        {
            dcc = dcc_find_item(DCC_TYPE_CHAT, cur_channel->name+1, NULL);
            if (dcc == NULL)
            {
                printformat(NULL, NULL, MSGLEVEL_CLIENTERROR,
                            IRCTXT_DCC_CHAT_NOT_FOUND, cur_channel->name+1);
            }
            else
            {
                str = g_strdup_printf("%s\001SOUND %s %s\001\n", dcc->mirc_ctcp ? "" :
                                      "CTCP_MESSAGE ", sound, rest);
                net_transmit(dcc->handle, str, strlen(str));
                printformat_plugin(cur_channel->server, cur_channel->name, MSGLEVEL_DCC,
                                   plug, PLUGTXT_SOUND_ACTION, dcc->mynick, source_host_ip,
                                   dcc->nick, sound, rest);
            }
        }
        else /* All channels/targets that are not a dcc chat */
        {
            str = g_strdup_printf("PRIVMSG %s :\001SOUND %s %s\001",
                                  cur_channel->name, sound, rest);
            irc_send_cmd(cur_channel->server, str);
            printformat_plugin(cur_channel->server, cur_channel->name, MSGLEVEL_CTCPS, plug,
                               PLUGTXT_SOUND_ACTION, cur_channel->server->nick,
                               source_host_ip, cur_channel->name, sound, rest);
        }
        if(str!=NULL) g_free(str);
    }

    else { /* No embedded action so lets just play the blasted sound!! */
	str = NULL;
        if (cur_channel->type == CHANNEL_TYPE_DCC_CHAT)
        {
            dcc = dcc_find_item(DCC_TYPE_CHAT, cur_channel->name+1, NULL);
            if (dcc == NULL)
            {
                printformat(NULL, NULL, MSGLEVEL_CLIENTERROR,
                            IRCTXT_DCC_CHAT_NOT_FOUND, cur_channel->name+1);
            }
            else
            {
                str = g_strdup_printf("%s\001SOUND %s\001\n", dcc->mirc_ctcp ? "" :
                                      "CTCP_MESSAGE ", sound);
                net_transmit(dcc->handle, str, strlen(str));
			    printformat_plugin(cur_channel->server, cur_channel->name,
                                   MSGLEVEL_DCC, plug, PLUGTXT_SOUND,
                                   dcc->mynick, source_host_ip, dcc->nick, sound);
            }
        }
        else /* All channels/targets that are not a dcc chat */
        {
            str = g_strdup_printf("PRIVMSG %s :\001SOUND %s\001",
                                  cur_channel->name, sound);
            irc_send_cmd(cur_channel->server, str);
	    printformat_plugin(cur_channel->server, cur_channel->name,
			       MSGLEVEL_CTCPS, plug, PLUGTXT_SOUND,
                               cur_channel->server->nick, source_host_ip,
                               cur_channel->name, sound);
        }
        if(str!=NULL) g_free(str);
    }
    g_free(params);
    return FALSE;
}

/* Handle incomming sound requests.  This can have actions embedded or not! */
static gboolean sound_ctcp_msg(gchar *data, SERVER_REC *server, gchar *sender, gchar *sendaddr, gchar *target)
{
gchar *params, *sound, *rest, *str, *str2, *channel;
PLUGIN_DATA *plug_data = plug->data;

    params = event_get_params(data, 2 | PARAM_FLAG_GETREST, &sound, &rest);

	if(target[0]=='=' || ischannel(target[0])) channel=target;
	else channel=sender;

    if(check_child(sound_pid)!=0)
    {
        if(*rest!='\0')
        {
            /* Show rest as action from sender */
	    printformat_plugin(server, target, ischannel(target[0]) ? MSGLEVEL_PUBLIC :
			       (target[0]=='=' ? MSGLEVEL_DCC : MSGLEVEL_MSGS), plug,
                               PLUGTXT_SOUND_ACTION, sender, sendaddr, target, sound, rest);
        }
        else
        {
	    printformat_plugin(server, channel, ischannel(target[0]) ? MSGLEVEL_PUBLIC :
			       (target[0]=='=' ? MSGLEVEL_DCC : MSGLEVEL_MSGS), plug,
			       PLUGTXT_SOUND, sender, sendaddr, target, sound);
        }
        str = g_strdup_printf("%s/%s", plug_data->sound_dir, sound);
        str2 = convert_home(str);
        sound_pid=start_bg_sound(str2);
        g_free(str);
        g_free(str2);

    }
    else printformat_plugin(server, channel,
                       MSGLEVEL_CTCPS, plug, PLUGTXT_SOUND_BUSY,
                       cur_channel->server->nick,
                       source_host_ip, cur_channel->name, sound, rest);
    g_free(params);
return(TRUE);
}

/* Wrapper for dcc chat sound requests -- simply clean it up and pass *
 * it on to sound_ctcp_msg. */
static gboolean dcc_sound_ctcp_msg(gchar *data, DCC_REC *dcc)
{
gchar *name;
gboolean ret=0;

    name=g_strdup_printf("=%s", dcc->nick);
    ret=sound_ctcp_msg(data, dcc->server, dcc->nick, dcc->addrstr, name);
    g_free(name);
return(ret);
}

gchar *plugin_description(void)
{
    return PLUGIN_LONG_NAME " plug-in";
}

gint plugin_version(void)
{
    return PLUGIN_LAYER_VERSION;
}

/* If we call plugin_deinit() in this code, it doesn't necessarily point to
   _THIS_ module's plugin_deinit() but instead some other module's.. So,
   we create static deinit() function which should be used..

   Actually this isn't needed in this module since it never calls
   plugin_deinit(), but this is a sample module after all... */
static void deinit(PLUGIN_REC *plugin)
{
PLUGIN_DATA *data = plugin->data;

    if (data->loaded)
	printformat_plugin(NULL, NULL, MSGLEVEL_CLIENTNOTICE, plugin, PLUGTXT_STOP);
    plugin_sound_setup_deinit(plugin);

    if (data->sound_dir != NULL) g_free(data->sound_dir);
    if (data->mp3_path != NULL) g_free(data->mp3_path);
    if (data->midi_path != NULL) g_free(data->midi_path);
    if (data->default_path != NULL) g_free(data->default_path);
    g_free(data);
}

void plugin_deinit(PLUGIN_REC *plugin)
{
    deinit(plugin);
}

gboolean plugin_init(gint gui, PLUGIN_REC *plugin, gchar *args)
{
PLUGIN_DATA *data;

    plug = plugin;
    plugin->data = data = g_new0(PLUGIN_DATA, 1);
    data->plugin = plugin;
    data->gui = gui;

    plugin_sound_setup_init(plugin);

    printformat_plugin_init(plugin, formats);
    plugin_bind(plugin, "command sound", (SIGNAL_FUNC) send_sound_func, FALSE);
    plugin_bind(plugin, "ctcp msg sound", (SIGNAL_FUNC) sound_ctcp_msg, FALSE);
    plugin_bind(plugin, "dcc ctcp sound", (SIGNAL_FUNC) dcc_sound_ctcp_msg, FALSE);

    printformat_plugin(NULL, NULL, MSGLEVEL_CLIENTNOTICE, plugin, PLUGTXT_START);

    if (gui == IRSSI_GUI_GTK || gui == IRSSI_GUI_GNOME)
	plugin_sound_gtk_init(plugin);
    data->loaded = TRUE;
    return TRUE;
}

/* Check to see if we have a sound process going.  Special thanks to gfoot of EFNet: #C */
static int check_child (int pid)
{
    int x = waitpid (pid, NULL, WNOHANG);
    return x == pid ? 1 : x;
}

/* Black magic of the backgrounding/nonblocking sound playing... again Special thanks *
 * to gfoot of EFNet: #C */
static int start_bg_sound (gchar *sound_name)
{
gint pid, old_stdout_fd, dev_null_fd, old_stderr_fd;
gchar *str;

    old_stdout_fd = dup(1);
    old_stderr_fd = dup(2);
    dev_null_fd = open("/dev/null", O_WRONLY);
    dup2(dev_null_fd, 1);
    dup2(dev_null_fd, 2);

    str = g_strdup_printf(get_scommand(sound_name), sound_name);
    pid = execute(str);
    g_free(str);

    dup2(old_stdout_fd, 1);
    dup2(old_stderr_fd, 2);
    close(dev_null_fd);

return pid;
}
