/*
 gui-windows.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static gboolean gui_window_created(WINDOW_REC *window)
{
    MAIN_WINDOW_REC *parent;
    GUI_WINDOW_REC *gui;

    g_return_val_if_fail(window != NULL, FALSE);

    parent = (/*!toggle_use_tabbed_windows || */cur_channel == NULL) ?
	gui_mainwindow_create() : WINDOW_GUI(CHANNEL_PARENT(cur_channel))->parent;

    gui = g_new0(GUI_WINDOW_REC, 1);
    window->gui_data = gui;
    gui->parent = parent;

    if (parent->children == NULL) parent->active = window;
    parent->children = g_list_append(parent->children, window);

    signal_emit("gui window created", 1, window);
    return TRUE;
}

static gboolean gui_window_destroyed(WINDOW_REC *window)
{
    MAIN_WINDOW_REC *parent;
    GUI_WINDOW_REC *gui;
    GSList *tmp;

    g_return_val_if_fail(window != NULL, FALSE);

    gui = WINDOW_GUI(window);
    parent = gui->parent;
    parent->children = g_list_remove(parent->children, window);

    signal_emit("gui window destroyed", 1, window);

    for (tmp = gui->lines; tmp != NULL; tmp = tmp->next)
    {
        LINE_REC *line = tmp->data;

        g_free(line->line);
        g_free(line);
    }
    g_slist_free(gui->lines);

    g_free(gui);
    window->gui_data = NULL;

    if (parent->children == NULL)
        gui_mainwindow_destroy(parent);

    if (windows != NULL && !quitting)
	signal_emit("command window goto", 1, "1");
    return TRUE;
}

static gboolean gui_window_goto(gpointer window)
{
    WINDOW_REC *winrec;
    GList *list;

    list = g_list_nth(windows, GPOINTER_TO_INT(window));
    if (list == NULL) return TRUE;

    winrec = list->data;
    signal_emit("gui channel open", 1, winrec->active);

    return TRUE;
}

void gui_window_redraw(WINDOW_REC *window)
{
    GSList *line;
    gushort *l;
    gint pos, ypos;
    gint n;

    g_return_if_fail(window != NULL);

    pos = WINDOW_GUI(window)->starty;
    line = g_slist_nth(WINDOW_GUI(window)->lines, pos < 0 ? 0 : pos);

    for (n = 0; n < LINES-2; n++)
    {
        move(n, 0);
        clrtoeol();
    }

    for (ypos = 0; line != NULL && ypos < LINES-2; line = line->next, ypos++)
    {
        LINE_REC *rec = line->data;

        move(ypos, 0);
        l = rec->line;
        for (n = 0; n < COLS && n < rec->linelen; n++)
        {
            set_color(*l); l++;
            addch(*l); l++;
        }
    }
    screen_refresh();
}

static gboolean signal_window_focused(WINDOW_REC *window)
{
    CHANNEL_REC *channel;
    gchar *str;

    g_return_val_if_fail(window != NULL, FALSE);

    channel = window->active;
    if (channel == NULL)
    {
        gui_entry_set_prompt("");
        return TRUE;
    }

    /* set prompt */
    screen_refresh_freeze();
    str = channel->type == CHANNEL_TYPE_EMPTY && g_strcasecmp(channel->name, "(empty)") == 0 ? "" :
        g_strdup_printf("[%1.17s] ", channel->name);
    gui_entry_set_prompt(str);
    if (*str != '\0') g_free(str);

    gui_window_redraw(window);
    screen_refresh_thaw();
    return TRUE;
}

GSList *gui_window_find_text(WINDOW_REC *window, gchar *text)
{
    GSList *list, *tmp, *start;
    gchar *str;
    gint n, size;
    gboolean add_line;

    g_return_val_if_fail(window != NULL, NULL);
    g_return_val_if_fail(text != NULL, NULL);

    text = g_strdup(text); g_strup(text);
    list = NULL; size = -1; str = NULL; add_line = FALSE;
    for (start = tmp = WINDOW_GUI(window)->lines; tmp != NULL; tmp = tmp->next)
    {
        LINE_REC *rec = tmp->data;

        if (rec->linelen+1 > size)
        {
            if (str != NULL) g_free(str);
            size = rec->linelen+1;
            str = g_malloc(size);
        }
        for (n = 0; n < rec->linelen; n++)
            str[n] = toupper(rec->line[n*2+1]);
        str[n] = '\0';

        if (!add_line && strstr(str, text) != NULL)
            add_line = TRUE;

        if (!rec->next_joined)
        {
            if (add_line)
            {
                /* add the whole line */
                for (;;)
                {
                    list = g_slist_append(list, start->data);
                    if (start == tmp) break;
                    start = start->next;
                }
                add_line = FALSE;
            }
            start = tmp->next;
        }
    }

    if (str != NULL) g_free(str);
    g_free(text);
    return list;
}

void gui_windows_init(void)
{
    signal_add("window created", (SIGNAL_FUNC) gui_window_created);
    signal_add("window destroyed", (SIGNAL_FUNC) gui_window_destroyed);
    signal_add("window focused", (SIGNAL_FUNC) signal_window_focused);
    signal_add("gui window goto", (SIGNAL_FUNC) gui_window_goto);
}

void gui_windows_deinit(void)
{
    signal_remove("window created", (SIGNAL_FUNC) gui_window_created);
    signal_remove("window destroyed", (SIGNAL_FUNC) gui_window_destroyed);
    signal_remove("window focused", (SIGNAL_FUNC) signal_window_focused);
    signal_remove("gui window goto", (SIGNAL_FUNC) gui_window_goto);
}
