/*
 ui-windows.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

GList *windows;

WINDOW_REC *ui_window_create(CHANNEL_REC *channel)
{
    WINDOW_REC *rec;

    rec = g_new0(WINDOW_REC, 1);
    windows = g_list_append(windows, rec);

    /* Add the channel to channels list in window */
    channel->ui_data = rec;
    rec->channels = g_list_append(rec->channels, channel);

    signal_emit("window created", 1, rec);
    return rec;
}

void ui_window_destroy(WINDOW_REC *window)
{
    g_return_if_fail(window != NULL);

    if (window->destroying) return;
    window->destroying = TRUE;

    while (window->channels != NULL)
        channel_destroy(window->channels->data);

    windows = g_list_remove(windows, window);
    signal_emit("window destroyed", 1, window);

    g_free(window);
}

static gboolean cmd_window(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    command_runsub("window", data, server, channel);
    return TRUE;
}

static gboolean cmd_window_new(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    gint type;

    g_return_val_if_fail(data != NULL, FALSE);

    type = (g_strcasecmp(data, "hide") == 0 || g_strcasecmp(data, "tab") == 0) ? 1 :
	(g_strcasecmp(data, "split") == 0 ? 2 : 0);
    signal_emit("gui window create override", 1, GINT_TO_POINTER(type));

    channel_create(channel->server, "(empty)", CHANNEL_TYPE_EMPTY);
    return TRUE;
}

static gboolean cmd_window_close(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    g_return_val_if_fail(channel != NULL, FALSE);

    if (channels->next != NULL)
    {
	/* destroy window unless it's the last one */
	ui_window_destroy(CHANNEL_PARENT(channel));
    }
    return TRUE;
}

static gboolean cmd_window_goto(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    GList *tmp;
    gint num;

    g_return_val_if_fail(data != NULL, FALSE);

    num = 0;
    if (g_strcasecmp(data, "active") == 0)
    {
	/* go to first window with activity */
	num = 1;
	for (tmp = windows; tmp != NULL; tmp = tmp->next, num++)
	{
	    WINDOW_REC *rec = tmp->data;

	    if (rec->new_data)
		break;
	}
	if (tmp == NULL)
	{
	    /* No activities in windows, keep the current one */
	    num = 0;
	}
    }
    else if (!isdigit(*data))
    {
	/* channel name - first try channel from same server */
	SERVER_REC *server;

	server = channel->server;
	channel = channel_find(server, data);
	if (channel == NULL)
	{
	    /* from any server? */
	    channel = channel_find_any(data);
	}
	if (channel == NULL)
	{
	    gchar *chan;

	    /* still nothing? maybe user just left the # in front of
               channel, try again with it.. */
	    chan = g_strdup_printf("#%s", data);
	    channel = channel_find(server, data);
	    if (channel == NULL) channel = channel_find_any(chan);
	    g_free(chan);
	}
	if (channel != NULL)
	{
	    /* get the window number */
	    num = g_list_index(windows, CHANNEL_PARENT(channel));
	    if (num >= 0) num++;
	}
    }
    else
    {
	/* window number */
	sscanf(data, "%d", &num);
    }

    if (num > 0)
	signal_emit("gui window goto", 1, GINT_TO_POINTER(num-1));
    return TRUE;
}

static gboolean cmd_window_next(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    gint num;

    g_return_val_if_fail(channel != NULL, FALSE);

    num = g_list_index(windows, CHANNEL_PARENT(channel))+1;
    if (num >= g_list_length(windows)) num = 0;
    signal_emit("gui window goto", 1, GINT_TO_POINTER(num));
    return TRUE;
}

static gboolean cmd_window_prev(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    gint num;

    g_return_val_if_fail(channel != NULL, FALSE);

    num = g_list_index(windows, CHANNEL_PARENT(channel))-1;
    if (num < 0) num = g_list_length(windows)-1;
    signal_emit("gui window goto", 1, GINT_TO_POINTER(num));
    return TRUE;
}

static gboolean cmd_window_level(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(channel != NULL, FALSE);

    channel->level = combine_level(channel->level, data);
    printtext(NULL, NULL, MSGLEVEL_CLIENTNOTICE, "Window level is now %s",
              bits2level(channel->level));
    return TRUE;
}

static gboolean cmd_window_server(gchar *data, SERVER_REC *oldserver, CHANNEL_REC *channel)
{
    SERVER_REC *server;

    g_return_val_if_fail(data != NULL, FALSE);

    server = server_find_tag(data);
    if (server == NULL)
	printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_UNKNOWN_SERVER_TAG, data);
    else
    {
        channel->server = server;
	printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_SERVER_CHANGED, server->tag, server->address,
		    server->ircnet == NULL ? "" : server->ircnet);
	signal_emit("channel server changed", 1, channel);
    }

    return TRUE;
}

static gboolean cmd_channel(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    command_runsub("channel", data, server, channel);
    return TRUE;
}

static gboolean cmd_channel_prev(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    CHANNEL_REC *last;
    WINDOW_REC *win;
    GList *tmp;

    g_return_val_if_fail(channel != NULL, FALSE);

    last = NULL;
    win = CHANNEL_PARENT(channel);
    for (tmp = channels; tmp != NULL; tmp = tmp->next)
    {
	CHANNEL_REC *rec = tmp->data;

	if (tmp->data == channel)
	{
	    /* current channel. did we find anything? if not,
	       go to the last channel */
	    if (last != NULL) break;
	}
	else if (CHANNEL_PARENT(rec) == win)
	    last = rec;
    }

    if (last != NULL)
	signal_emit("gui channel open", 1, last);

    return TRUE;
}

static gboolean cmd_channel_next(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    CHANNEL_REC *next;
    WINDOW_REC *win;
    GList *tmp;
    gboolean gone;

    g_return_val_if_fail(channel != NULL, FALSE);

    next = NULL; gone = FALSE;
    win = CHANNEL_PARENT(channel);
    for (tmp = channels; tmp != NULL; tmp = tmp->next)
    {
	CHANNEL_REC *rec = tmp->data;

	if (tmp->data == channel)
	    gone = TRUE;
	else if (CHANNEL_PARENT(rec) == win)
	{
	    if (gone)
	    {
		/* found the next channel */
		next = rec;
		break;
	    }

	    if (next == NULL)
		next = rec; /* fallback to first channel */
	}
    }

    if (next != NULL)
	signal_emit("gui channel open", 1, next);

    return TRUE;
}

void ui_windows_init(void)
{
    command_bind("window", NULL, (SIGNAL_FUNC) cmd_window);
    command_bind("window new", NULL, (SIGNAL_FUNC) cmd_window_new);
    command_bind("window close", NULL, (SIGNAL_FUNC) cmd_window_close);
    command_bind("window server", NULL, (SIGNAL_FUNC) cmd_window_server);
    command_bind("window goto", NULL, (SIGNAL_FUNC) cmd_window_goto);
    command_bind("window prev", NULL, (SIGNAL_FUNC) cmd_window_prev);
    command_bind("window next", NULL, (SIGNAL_FUNC) cmd_window_next);
    command_bind("window level", NULL, (SIGNAL_FUNC) cmd_window_level);
    command_bind("channel", NULL, (SIGNAL_FUNC) cmd_channel);
    command_bind("channel prev", NULL, (SIGNAL_FUNC) cmd_channel_prev);
    command_bind("channel next", NULL, (SIGNAL_FUNC) cmd_channel_next);
}

void ui_windows_deinit(void)
{
    command_unbind("window", (SIGNAL_FUNC) cmd_window);
    command_unbind("window new", (SIGNAL_FUNC) cmd_window_new);
    command_unbind("window close", (SIGNAL_FUNC) cmd_window_close);
    command_unbind("window server", (SIGNAL_FUNC) cmd_window_server);
    command_unbind("window goto", (SIGNAL_FUNC) cmd_window_goto);
    command_unbind("window prev", (SIGNAL_FUNC) cmd_window_prev);
    command_unbind("window next", (SIGNAL_FUNC) cmd_window_next);
    command_unbind("window level", (SIGNAL_FUNC) cmd_window_level);
    command_unbind("channel", (SIGNAL_FUNC) cmd_channel);
    command_unbind("channel prev", (SIGNAL_FUNC) cmd_channel_prev);
    command_unbind("channel next", (SIGNAL_FUNC) cmd_channel_next);
}
