/*
  $Id: cf.c,v 1.11 1997/01/16 21:00:48 luik Exp $

  cf.c - omirrd config file handling: build `cmd' struct.
  Copyright (C) 1996, Andreas Luik, <luik@pharao.s.bawue.de>.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "common.h"

#if defined(RCSID) && !defined(lint)
static char rcsid[] UNUSED__ = "$Id: cf.c,v 1.11 1997/01/16 21:00:48 luik Exp $";
#endif /* defined(RCSID) && !defined(lint) */

#include <stdio.h>
#include <stdlib.h>
#include <fnmatch.h>		/* GNU fnmatch.h in ../lib */
#include "libomirr.h"
#include "debug.h"
#include "cfnl.h"
#include "cfvar.h"
#include "cf.h"
#include "optflags.h"
#include "conn.h"


/* cfAllocSubCmdEnt - make a sub command for lists of variables,
   commands, etc.  */
CfSubCmd cfAllocSubCmdEnt(int type)
{
    CfSubCmd sc;

    sc = (CfSubCmd) xmalloc(sizeof(CfSubCmdRec));
    sc->sc_type = type;
    sc->sc_args = NULL;
    sc->sc_next = NULL;
    sc->sc_name = NULL;
    return(sc);
}


/* cfAddCmd - insert or append ARROW command to list of commands to be
   performed, saved in global linked list `cmds'. */
void cfAddCmd(CfCmdType type, char *label,
	      CfNameList files, CfNameList hosts, CfSubCmd subcmds)
{
    CfCmd c, prev, nc;
    CfNameList h;
    CfNameList next;

    files = cfExpandVars(files);
    hosts = cfExpandVars(hosts);

    for (h = hosts; h; next = h->n_next, free(h), h = next) {
	/* Search command list for an update to the same host.  */
	for (prev = NULL, c = cmds; c; prev = c, c = c->c_next) {
	    if (strcmp(c->c_name, h->n_name) == 0) {
		do {
		    prev = c;
		    c = c->c_next;
		} while (c != NULL && strcmp(c->c_name, h->n_name) == 0);
		break;
	    }
	}
	/*
	 * Insert new command to update host.
	 */
	nc = (CfCmd) xmalloc(sizeof(CfCmdRec));
	nc->c_type = type;
	nc->c_name = h->n_name;
	nc->c_conn = NULL;
	nc->c_label = label;
	nc->c_files = files;
	nc->c_subcmds = subcmds;
	nc->c_next = c;
	if (prev == NULL)
	    cmds = nc;
	else
	    prev->c_next = nc;
    }
}


/* cfFreeCmds - free all elements of global linked command list `cmds'
   and set `cmds' to NULL.  */
void cfFreeCmds(void)
{
    CfCmd c;
    CfCmd nextc;
    CfSubCmd sc;
    CfSubCmd nextsc;

    for (c = cmds; c; c = nextc) {
	nextc = c->c_next;
	if (nextc == NULL || c->c_label != nextc->c_label)
	    xfree(c->c_label);
	xfree(c->c_name);
	if (nextc == NULL || c->c_files != nextc->c_files)
	    cfFreeNameList(c->c_files);
	if (nextc == NULL || c->c_subcmds != nextc->c_subcmds) {
	    for (sc = c->c_subcmds; sc; sc = nextsc) {
		nextsc = sc->sc_next;
		xfree(sc->sc_name);
		cfFreeNameList(sc->sc_args);
		free(sc);
	    }
	}
	free(c);
    }
    cmds = NULL;
}



/* filematch - check whether file name `path' matches path
   specification `pattern'.  Returns TRUE for match, FALSE otherwise.  */
static int filematch(char *path, const char *pattern)
{
    if (streq(path, pattern))
	return (1);

    if (fnmatch(pattern, path, FNM_PATHNAME | FNM_LEADING_DIR) == 0) /* match */
	return (1);

    return (0);
}


/* cfFileMatch - checks whether the (path-)name `name' matches any cmd
   entry (from `e->cmds') which represents a connection to `e'.  Sets
   `targetpathp' to pathname on remote machine using destination
   directory from config file and taking OPT_WHOLE into account.
   Returns TRUE or FALSE.  */
int cfFileMatch(char *path, ConnEntry e, char **targetpathp)
{
    CfCmd c;
    CfSubCmd sc;
    CfSubCmd inst_sc;
    int result;
    char *targetname;

    for (c = e->cmds; c; c = c->c_next) {
	if (c->c_conn != e)
	    continue;		/* wrong host/connection */

	if (c->c_type != C_TYPE_ARROW)
	    continue;		/* not an ARROW command */

	/* XXX check for c_label for partial update not yet supported */

	if (!cfMatchNameInList(path, c->c_files, filematch, &targetname))
	    continue;		/* file is not in install name list */

	/* Check for "install" subcommand, which is optional.  */
	for (inst_sc = c->c_subcmds; inst_sc; inst_sc = inst_sc->sc_next)
	    if (inst_sc->sc_type == SC_TYPE_INSTALL) {
		/* XXX process options, save destination directory */
		break;
	    }

	/* Check for except subcommand and whether `path' is in except
           list.  */
	for (sc = c->c_subcmds; sc; sc = sc->sc_next) {
	    if (sc->sc_type == SC_TYPE_EXCEPT) {
		if (cfMatchNameInList(path, sc->sc_args, filematch, NULL))
		    break;
	    }
	}
	if (sc)
	    continue;		/* r->path.path found in except list */

	break;			/* found it */
    }

    result = (c != NULL);
    if (result && targetpathp) {
	if (inst_sc == NULL || inst_sc->sc_name == NULL)
	    *targetpathp = NULL;
	else if (option_flag & OPT_WHOLE
		 || inst_sc->sc_options & OPT_WHOLE) {
	    *targetpathp = xmalloc(strlen(inst_sc->sc_name) + strlen(path) + 2);
	    sprintf(*targetpathp, "%s/%s", inst_sc->sc_name, path);
	}
	else if (targetname) {
	    *targetpathp = xmalloc(strlen(inst_sc->sc_name) + strlen(targetname) + 2);
	    sprintf(*targetpathp, "%s/%s", inst_sc->sc_name, targetname);
	}
	else {
	    *targetpathp = xstrdup(inst_sc->sc_name);
	}
    }
    debuglog(DEBUG_CF, ("cfFileMatch: file %s host %s: %s, target %s\n",
			path, e->name, (result ? "true" : "false"),
			(result && targetpathp && *targetpathp
			 ? *targetpathp : "<default>")));
    return (result);
}





#ifdef DEBUG
void cfDebugCmd(char *prefix, CfCmd cmds)
{
    CfCmd c;
    CfSubCmd sc;
    char buf[40];
    char *nested_prefix;

    if (strlen(prefix) < sizeof(buf) - 2) {
	sprintf(buf, "%s\t", prefix);
	nested_prefix = buf;
    }
    else {
	nested_prefix = prefix;
    }

    for (c = cmds; c; c = c->c_next) {
	printf("%s%-16s", prefix, c->c_label ? c->c_label : "<default>");
	printf("-> %s\n", c->c_name);
	cfDebugNameList(nested_prefix, c->c_files);
	for (sc = c->c_subcmds; sc; sc = sc->sc_next) {
	    switch (sc->sc_type) {
	      case SC_TYPE_INSTALL:
		printf("%s    install", prefix);
		if (sc->sc_options)
		    printf(" -o%s", optStrOptions(sc->sc_options));
		if (sc->sc_name)
		    printf(" %s", sc->sc_name);
		putchar('\n');
		break;
	      case SC_TYPE_EXCEPT:
		printf("%s    except:\n", prefix);
		cfDebugNameList(nested_prefix, sc->sc_args);
		break;
	      default:
		printf("%s    unknown subcommand!\n", prefix);
		break;
	    }
	}
    }
}
#endif /* DEBUG */

