/*
  $Id: iname.c,v 1.13 1997/01/16 20:52:53 luik Exp $

  iname.c - inode/ompath to pathname conversion including name cache for omirrd.
  Copyright (C) 1996-1997, Andreas Luik, <luik@pharao.s.bawue.de>.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "common.h"

#if defined(RCSID) && !defined(lint)
static char rcsid[] UNUSED__ = "$Id: iname.c,v 1.13 1997/01/16 20:52:53 luik Exp $";
#endif /* defined(RCSID) && !defined(lint) */

#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "libomirr.h"
#include "getmount.h"
#include "debug.h"
#include "error.h"
#include "iname.h"
#include "omirrd.h"

#ifndef HAVE_LSTAT
#define lstat stat
#endif /* !defined(HAVE_LSTAT) */


#define MAX_FIND_PROCS 4	/* maximum number of parallel find processes */

#define MAX_SYS_ERRORS 10	/* maximum number of consecutive
				   resolve errors before aborting */

#define INAM_GC_INTERVALL 300	/* call garbage collector every 5 min */

#define INAM_CACHE_TIMEOUT 30	/* validation timeout for cache entries */
#define INAM_NEG_CACHE_TIMEOUT 3 /* timeout for entries in negative cache */

#define HASHSIZE 1021		/* size of hash table */


static InamEntry inam_hashtab[HASHSIZE]; /* hash table for fast lookup */
static InamEntry inam_queue;	/* queue to be processed with find(1) */


static InamCallbackRec inam_lookup_done_callback; /* callback for delayed lookup */




/**********************************************************************
 * Primitives.
 **********************************************************************/


/* inamSetTimestamp - set timestamp of `e' to current time.  */
#define inamSetTimestamp(e) \
    e->timestamp = time((time_t) NULL)


/* inamUpdateTimestamp - update timestamp of `e' to current time if
   already set.  */
#define inamUpdateTimestamp(e) \
    if (!e->timestamp) ; else e->timestamp = time((time_t) NULL)


/* inamTimeout - return effective timeout for entry `e'.  */
#define inamTimeout(e) \
    (e->pathname ? INAM_CACHE_TIMEOUT : INAM_NEG_CACHE_TIMEOUT)


/* inamIsTimedOut - determine whether entry `e' is timed out.  */
#define inamIsTimedOut(e) \
    (e->timestamp == 0 || time((time_t) NULL) - e->timestamp > inamTimeout(e))


/* inamHashVal - computes hash value based on inode index.  Returns
   index.  This is implemented as a macro for performance reasons.  */
#define inamHashVal(dip)    ((dip)->ino % HASHSIZE)


/* inamAllocEntry - allocates a new entry and links it into the hash
   list.  Sets dev/ino values and initializes other components.
   Returns new allocated entry or NULL on failure (out of memory).  */
static InamEntry inamAllocEntry(DevIno dip)
{
    int hashval;
    InamEntry e;

    hashval = inamHashVal(dip);

    if ((e = malloc(sizeof(InamEntryRec)))) {
	e->next = inam_hashtab[hashval];
	e->dev_ino.dev = dip->dev;
	e->dev_ino.ino = dip->ino;
	e->pathname = NULL;
	e->count = 0;
	e->timestamp = (time_t) 0;
	e->state = 0;
	e->syserrno = 0;
	inam_hashtab[hashval] = e;
    }
    return (e);
}


/* inamAllocEntryInQueue - allocates a new entry and appends it to the
   queue list.  Sets dev/ino values and initializes other components.
   Returns new allocated entry or NULL on failure (out of memory).  */
static InamEntry inamAllocEntryInQueue(DevIno dip)
{
    InamEntry e;
    InamEntry queue;

    if ((e = malloc(sizeof(InamEntryRec)))) {
	e->next = NULL;
	e->dev_ino.dev = dip->dev;
	e->dev_ino.ino = dip->ino;
	e->pathname = NULL;
	e->count = 0;
	e->timestamp = (time_t) 0;
	e->state = 0;
	e->syserrno = 0;
	if (inam_queue) {
	    for (queue = inam_queue; queue->next; queue = queue->next)
		/* nothing */;
	    queue->next = e;
	}
	else inam_queue = e;
    }
    return (e);
}


/* inamMoveEntryFromQueue - removes `inam_entry' from the queue and
   adds it to the regular hashed linked lists.  Returns `inam_entry'.  */
static InamEntry inamMoveEntryFromQueue(InamEntry inam_entry)
{
    int hashval;
    InamEntry e, prev;

    /* First remove from queue.  */
    if (inam_queue == inam_entry)
	inam_queue = inam_entry->next;
    else {
	for (prev = inam_queue, e = inam_queue->next; e; prev = e, e = e->next) {
	    if (e == inam_entry) {
		prev->next = e->next;
		break;
	    }
	    assert(e);
	}
    }

    /* Add to hashed list.  */
    hashval = inamHashVal(&inam_entry->dev_ino);
    
    inam_entry->next = inam_hashtab[hashval];
    inam_hashtab[hashval] = inam_entry;
    return (inam_entry);
}


/* inamFreeEntry - removes the specified entry from the hash list and
   frees the pathname component (unless NULL) and the entry itself.  */
static void inamFreeEntry(InamEntry inam_entry)
{
    int hashval;
    InamEntry e, prev;
    InamEntry root;

    hashval = inamHashVal(&inam_entry->dev_ino);
    root = inam_hashtab[hashval];

    if (root == inam_entry)
	inam_hashtab[hashval] = inam_entry->next;
    else {
	for (prev = root, e = root->next; e; prev = e, e = e->next) {
	    if (e == inam_entry) {
		prev->next = e->next;
		break;
	    }
	    assert(e);
	}
    }

    if (inam_entry->pathname)
	free(inam_entry->pathname);
    free(inam_entry);
}


/* inamFindDevIno - searches for the entry with the specified dev/ino
   pair.  Returns the entry on success, NULL otherwise.  */
static InamEntry inamFindDevIno(DevIno dip)
{
    InamEntry e;

    for (e = inam_hashtab[inamHashVal(dip)]; e; e = e->next)
	if (dip->dev == e->dev_ino.dev && dip->ino == e->dev_ino.ino)
	    return e;
    return NULL;
}


/* inamFindDevInoInQueue - searches for the entry with the specified
   dev/ino pair in the `inam_queue'.  Returns the entry on success,
   NULL otherwise.  */
static InamEntry inamFindDevInoInQueue(DevIno dip)
{
    InamEntry e;

    for (e = inam_queue; e; e = e->next)
	if (dip->dev == e->dev_ino.dev && dip->ino == e->dev_ino.ino)
	    return e;
    return NULL;
}


/* inamFindName - searches for the entry with the specified pathname.
   Returns the entry on success, NULL otherwise.  */
static InamEntry inamFindName(const char *name)
{
    int n;
    InamEntry e;

    for (n = 0; n < HASHSIZE; n++) {
	for (e = inam_hashtab[n]; e; e = e->next)
	    if (e->pathname && streq(e->pathname, name))
		return e;
    }
    return NULL;
}



/**********************************************************************
 * Queue processing and handling of find(1) subprocesses.
 **********************************************************************/


/* abort_on_max_syserrs - counts number of consecutive system errors.
   Counter is cleared if `syserrno' argument is 0, otherwise it is
   incremented by one.  If counter reaches MAX_SYS_ERRORS, daemon is
   terminated with a fatal error message.  This function is used for
   the find(1) subprocess handling and should catch serious problems
   with fork(2) or I/O problems.  */
static void abort_on_max_syserrs(int syserrno)
{
    static int syserr_count;

    if (syserrno) {
	syserr_count++;
	if (syserr_count == MAX_SYS_ERRORS) {
	    fatal("iname: too many system errors - terminating\n");
	    exit(1);
	}
    }
    else syserr_count = 0;	/* reset counter */
}


/* execute_find - determines the mount point of device `dip->dev' and
   calls "find" (PATH_GFIND) with appropriate arguments.  Note, that
   this function is called in a forked subprocess, messages written to
   stdout and stderr are read and interpreted by the parent.
   Therefore avoid debug output!  */
static void execute_find(DevIno dip)
{
    int argc;
    char *argv[10];
    char ino[20 + 1];		/* enough for 64 bit inode value */
    char buffer[300];		/* error message buffer */
    struct mountent *mount_entp;
    
    if ((mount_entp = getmountentbydev(dip->dev)) == NULL) {
	sprintf(buffer, "cannot determine mount point of dev %#lx\n", dip->dev);
	write(2, buffer, strlen(buffer));
	return;
    }

    sprintf(ino, "%lu", (long) dip->ino);

    argc = 0;
    argv[argc++] = "find";
    argv[argc++] = mount_entp->dir;
    argv[argc++] = "-mount";
    argv[argc++] = "-inum";
    argv[argc++] = ino;
    argv[argc++] = "-print0";
    argv[argc++] = NULL;

    execv(PATH_GFIND, argv);

    /* If we reach this point, execv() failed. Write error message to
       stderr fd and return (we are in the subprocess - do not try to
       do much else).  */
    sprintf(buffer, "%s: exec failed: %s\n", PATH_GFIND, xstrerror(errno));
    write(2, buffer, strlen(buffer));
}


/* process_find_results - lio callback to read output of find
   subprocess.  The current `line' may be either a pathname (which is
   added to the iname cache) or an error message (which is logged).  */
static void process_find_results(char *line, void *data)
{
    InamEntry e = (InamEntry) data;

    if (*line == '/') {		/* find returned pathname */
	if (e->pathname) {
	    /* XXX should handle hard links returned by find(1) */
	    error("iname: find [%#lx,%#lx]: multiple names - probably links?\n",
		  (long) e->dev_ino.dev, (long) e->dev_ino.ino);
	    error("       already have [%#lx,%#lx] = %s\n",
		  (long) e->dev_ino.dev, (long) e->dev_ino.ino, e->pathname);
	    error("       got          [%#lx,%#lx] = %s\n",
		  (long) e->dev_ino.dev, (long) e->dev_ino.ino, line);
	}
	else {
	    e->pathname = xstrdup(line);
	    inamSetTimestamp(e);
	}
    }
    else {			/* probably find error message */
	if (e->syserrno == 0)
	    e->syserrno = EINVAL; /* we don't know better */
	error("iname: find [%#lx,%#lx]: %s%s",
	      (long) e->dev_ino.dev, (long) e->dev_ino.ino, line,
	      (line[strlen(line) - 1] != '\n' ? "\n" : ""));
    }
}


/* close_find_pipe - mp callback to process close of pipe to find
   subprocess.  This function handles errors, moves the entry `e' from
   the iname queue to the iname cache (thereby setting the timestamp
   if not yet set) and removes the lio/mp callbacks.  In addition, it
   calls the inam_lookup_done_callback, if set.  */
static int close_find_pipe(int fd, void *data)
{
    InamEntry e = (InamEntry) data;
    static int syserr_count;

    if (errno != 0) {
	error("iname: find [%#lx,%#lx]: read error: %s\n",
	      (long) e->dev_ino.dev, (long) e->dev_ino.ino, xstrerror(errno));
	if (e->syserrno == 0)
	    e->syserrno = errno;
    }

    e->state = 0;

    debuglog(DEBUG_INAME, ("iname: find [%#lx,%#lx] returned %s, errno %d\n",
			   (long) e->dev_ino.dev, (long) e->dev_ino.ino,
			   e->pathname ? e->pathname : "(null)", e->syserrno));

    if (!e->syserrno) {
	abort_on_max_syserrs(0); /* reset error counter */
	e = inamMoveEntryFromQueue(e);
	if (e->timestamp == 0)	/* still 0 if no file found by find process */
	    inamSetTimestamp(e);
    }

    /* Remove callbacks added in `inamProcessQueueEntry' for this pipe fd.  */
    lioRemoveReadCallback(lio_context, fd);
    mpRemoveReadCallback(mp_context, fd);
    mpRemoveErrorCallback(mp_context, fd);

    /* Call lookup-done callback, if set.  */
    if (inam_lookup_done_callback.callback)
	(*inam_lookup_done_callback.callback)(&e->dev_ino,
					      inam_lookup_done_callback.data);

    close(fd);
    return 0;
}


/* inamProcessQueueEntry - creates find subprocess for entry `e' if
   not yet running.  Also adds lio/mp callback for pipe fd to find.  */
static void inamProcessQueueEntry(InamEntry e)
{
    pid_t pid;
    int pipefd[2];

    if (e->state || e->syserrno)
	return;

    if (pipe(pipefd) == -1) {
	e->syserrno = errno;
	return;
    }

    switch (pid = fork()) {
      case -1:
	e->syserrno = errno;
	close(pipefd[0]);
	close(pipefd[1]);
	return;

      case 0:
	close(pipefd[0]);
	dup2(pipefd[1], 1);
	dup2(pipefd[1], 2);
	execute_find(&e->dev_ino); /* does not return, unless error */
	close(pipefd[1]);
	_exit(1);
    }	

    /* parent process */
    close(pipefd[1]);

    debuglog(DEBUG_INAME,
	     ("iname: find [%#lx,%#lx] starting find subprocess [%ld]\n",
	      (long) e->dev_ino.dev, (long) e->dev_ino.ino, (long) pid));

    e->state = 1;		/* find is running */

    /* lio_context: add `process_find_results' as read callback.
       This is the function processing the messages from the find
       process and assigning them to e->pathname.  */
    lioAddReadCallback(lio_context, pipefd[0], process_find_results, (void *) e);

    /* mp_context: add `lioMpReadFunc' as read callback, which will
       split the find output into lines and call the callback
       function of the lio context.  */
    mpAddReadCallback(mp_context, pipefd[0], lioMpReadFunc, (void *) lio_context);

    /* mp_context: add `close_find_pipe' as error callback to close
       pipe filedescriptor after processing.  */
    mpAddErrorCallback(mp_context, pipefd[0], close_find_pipe, (void *) e);
}



/* inamProcessQueue - checks each entry in `inam_queue'.  Entries
   which terminated with an error are removed.  Entries which are
   currently handled by a find subprocess are counted. The processing
   loop terminates if MAX_FIND_PROCS subprocesses are currently
   running.  Otherwise, a new find subprocess is started for the
   current entry and the next one is checked.  Additionally, this
   function calls the garbage collector for the iname cache, provided
   more than INAM_GC_INTERVALL seconds expired since the last call.  */
static void inamProcessQueue(void)
{
    InamEntry e;
    InamEntry next;
    int nfinds;			/* number of find processes running */
    static time_t last_garbage_collection;

    nfinds = 0;
    for (e = inam_queue; e && nfinds < MAX_FIND_PROCS; e = next) {
	next = e->next;
	if (e->state) {
	    nfinds++;
	}
	else if (e->syserrno) {
	    error("iname: find [%#lx,%#lx]: failed: %s\n",
		  (long) e->dev_ino.dev, (long) e->dev_ino.ino,
		  xstrerror(e->syserrno));
	    e = inamMoveEntryFromQueue(e);
	    inamFreeEntry(e);	/* drop this entry */
	    abort_on_max_syserrs(e->syserrno);
	}
	else inamProcessQueueEntry(e);
    }

    /* Call garbage collector if more than INAM_GC_INTERVALL seconds
       expired since last garbage collection.  */
    if (last_garbage_collection == 0)
	last_garbage_collection = time((time_t) NULL);

    if (time((time_t) NULL) - last_garbage_collection > INAM_GC_INTERVALL) {
	inamGarbageCollect();
	last_garbage_collection = time((time_t) NULL);
    }
}



/**********************************************************************
 * File I/O primitives (to check whether cache is consistent with disk
 * contents).
 **********************************************************************/


/* inamEntryConsistent - checks whether the pathname of `e' still
   matches the dev/ino pair of the entry by comparing it with the
   real dev/ino pair determined with lstat(2).  */
static int inamEntryConsistent(InamEntry e)
{
    DevIno dip = &e->dev_ino;
    struct stat statbuf;
    int ret;

  again:
    if (lstat(e->pathname, &statbuf) == -1) {
	if (errno == EINTR)
	    goto again;
	if (errno != ENOENT)
	    error("iname: stat(%s) failed: %s\n", e->pathname, xstrerror(errno));
	ret = 0;
    }
    else ret = (statbuf.st_dev == dip->dev && statbuf.st_ino == dip->ino);

    if (!ret)
	debuglog(DEBUG_INAME,
		 ("iname: inamEntryConsistent: [%#lx,%#lx] = %s FALSE\n",
		  (long) dip->dev, (long) dip->ino, e->pathname));

    return ret;
}



/**********************************************************************
 * Public functions.
 **********************************************************************/


/* inamAssocNameDevIno - associates the specified pathname with the
   dev/ino pair.  If an asociation for this dev/ino pair already
   exists, it checks the pathnames.  If they are equal, only the
   reference count is increased and 0 is returned.  Otherwise, if the
   pathname is different, either the old file is already removed, or
   the two files are hard links to the same inode.  If the check using
   lstat(2) returns ENOENT or a different dev/ino pair for the
   pathname, the old iname entry is free'd and processing continues.
   Othwise hard links (or an internal error) is assumed.  This is
   currently not handled, therefore an error is logged and -1 is
   returned.  Otherwise, if the dev/ino pair is not associated with a
   pathname, a new entry is allocated and filled.  Returns
   INAM_SUCCESS if that was successful, INAM_OTHERERROR otherwise.  */
InamStatus inamAssocNameDevIno(char *pathname, DevIno dip)
{
    InamEntry e;

    debuglog(DEBUG_INAME, ("iname: inamNameDevIno: [%#lx,%#lx] = %s\n",
	      (long) dip->dev, (long) dip->ino, pathname));

    if ((e = inamFindDevIno(dip))) {
	if (e->pathname == NULL) {
	    inamFreeEntry(e);	/* entry is obsolete */
	    e = NULL;
	}
	else if (strcmp(e->pathname, pathname) == 0) {
	    e->count++;
	    inamUpdateTimestamp(e);
	    return INAM_SUCCESS;
	}
	else if (!inamEntryConsistent(e)) {
	    inamFreeEntry(e);	/* dev/ino pair no longer matches */
	    e = NULL;
	}
    }

    /* XXX does still exist??? This may be a hard link or an error.  */
    if (e) {
	error("iname: multiple names for one dev/ino pair - probably links?\n");
	error("       already have [%#lx,%#lx] = %s\n",
	      (long) dip->dev, (long) dip->ino, e->pathname);
	error("       got          [%#lx,%#lx] = %s\n",
	      (long) dip->dev, (long) dip->ino, pathname);
	return INAM_OTHERERROR;
    }
    
    if ((e = inamAllocEntry(dip))) {
	e->pathname = xstrdup(pathname);
	e->count = 1;
	return INAM_SUCCESS;
    }
    return INAM_OTHERERROR;
}


/* inamLookupDevIno - searches the hashtable for the specified dev/ino
   pair and sets *pathname to the associated pathname, if found and
   not expired.  Note that *pathname may be NULL.  If no entry was
   found in the cache, the query is added to iname_queue and a find
   subprocess will be started from `inamProcessQueue'.  Returns
   INAM_SUCCESS on success and INAM_INPROCESS if query was queued.  */
InamStatus inamLookupDevIno(DevIno dip, char **pathname)
{
    InamEntry e;

    if ((e = inamFindDevIno(dip))) {
	if (e->pathname == NULL) {
	    /* Find was not able to find pathname, return negative
               result and remove entry if timeout expired.  */
	    *pathname = e->pathname;
	    if (inamIsTimedOut(e))
		inamFreeEntry(e);
	    return INAM_SUCCESS;
	}
	else if (inamEntryConsistent(e)) {
	    *pathname = e->pathname;
	    inamUpdateTimestamp(e);
	    return INAM_SUCCESS;
	}
	else if (e->count) {
	    *pathname = e->pathname;
	    return INAM_SUCCESS;
	}
	else {
	    inamFreeEntry(e);	/* dev/ino pair no longer matches */
	    e = NULL;
	}
    }

    if ((e = inamFindDevInoInQueue(dip)) == NULL) {
	inamAllocEntryInQueue(dip);
	inamProcessQueue();
    }

    return INAM_INPROCESS;
}


/* inamGarbageCollect - free all unused (count == 0) iname entries
   where the timeout has expired.  */
void inamGarbageCollect(void)
{
    int n;
    InamEntry e;
    InamEntry next;

    for (n = 0; n < HASHSIZE; n++) {
	for (e = inam_hashtab[n]; e; e = next) {
	    next = e->next;
	    if (e->count == 0 && inamIsTimedOut(e)) {
		debuglog(DEBUG_INAME,
			 ("iname: inamGarbageCollect: [%#lx,%#lx] free'd\n",
			  (long) e->dev_ino.dev, (long) e->dev_ino.ino));
		inamFreeEntry(e);
	    }
	}
    }
}


/* inamUseDevIno - increments the reference count of the entry
   associated with the specified dev/ino pair.  Returns INAM_SUCCESS
   on success or INAM_NOTFOUND if no entry for dev/ino was found.  */
InamStatus inamUseDevIno(DevIno dip)
{
    InamEntry e;

    if ((e = inamFindDevIno(dip)) && e->pathname) {
	e->count++;
	inamUpdateTimestamp(e);
	return INAM_SUCCESS;
    }
    return INAM_NOTFOUND;
}


/* inamFreeDevIno - decrements the reference count of the entry
   associated with the specified dev/ino pair.  If the reference count
   reaches 0, the entry is free'd.  Returns INAM_SUCCESS on success or
   INAM_NOTFOUND if no entry for dev/ino was found.  */
InamStatus inamFreeDevIno(DevIno dip)
{
    InamEntry e;

    if ((e = inamFindDevIno(dip))) {
	if (e->count)
	    e->count--;
	if (e->count == 0 && inamIsTimedOut(e)) {
	    debuglog(DEBUG_INAME,
		     ("iname: inamFreeDevIno: [%#lx,%#lx] free'd\n",
		      (long) e->dev_ino.dev, (long) e->dev_ino.ino));
	    inamFreeEntry(e);
	}
	return INAM_SUCCESS;
    }
    return INAM_NOTFOUND;
}


/* inamRename - renames the pathname `name' in the cache to `newname'
   in the assoc cache, if found.  Returns INAM_SUCCESS on success,
   INAM_NOTFOUND on failure.  */
InamStatus inamRename(const char *name, const char *newname)
{
    InamEntry e;

    if ((e = inamFindName(name))) {
	free(e->pathname);
	e->pathname = xstrdup(newname);
	debuglog(DEBUG_INAME, ("iname: inameRename: [%#lx,%#lx] %s -> %s\n",
			       (long) e->dev_ino.dev, (long) e->dev_ino.ino,
			       name, newname));
	return INAM_SUCCESS;
    }
    return INAM_NOTFOUND;
}


/* inamRemove - removes the assoc entry with pathname `name' from the
   cache, if found.  Returns INAM_SUCCESS on success, INAM_NOTFOUND on
   failure.  */
InamStatus inamRemove(const char *name)
{
    InamEntry e;

    if ((e = inamFindName(name))) {
	inamFreeEntry(e);
	debuglog(DEBUG_INAME, ("iname: inameRemove: [%#lx,%#lx] %s\n",
			       (long) e->dev_ino.dev, (long) e->dev_ino.ino,
			       name));
	return INAM_SUCCESS;
    }
    return INAM_NOTFOUND;
}


/* inamAddLookupDoneCallback - adds `func' as the
   `inam_lookup_done_callback' which is called if a delayed lookup
   using a find subprocess completes.  Only one callback can be added.
   Returns INAM_SUCCESS if successful, INAM_OTHERERROR otherwise.  */
InamStatus inamAddLookupDoneCallback(InamCallbackFunc func, void *data)
{
    if (inam_lookup_done_callback.callback == NULL) {
	inam_lookup_done_callback.callback = func;
	inam_lookup_done_callback.data = data;
	return INAM_SUCCESS;
    }
    return INAM_OTHERERROR;
}


/* inamRemoveLookupDoneCallback - compares `func' and `data' with
   installed `inam_lookup_done_callback' and removes it, if they
   match.  Returns INAM_SUCCESS if successful, INAM_NOTFOUND
   otherwise.  */
InamStatus inamRemoveLookupDoneCallback(InamCallbackFunc func, void *data)
{
    if ((inam_lookup_done_callback.callback == func
	 && inam_lookup_done_callback.data == data)) {
	inam_lookup_done_callback.callback = NULL;
	inam_lookup_done_callback.data = NULL;
	return INAM_SUCCESS;
    }
    return INAM_NOTFOUND;
}

