/*
  $Id: omirrd.c,v 1.29 1997/01/17 01:47:00 luik Exp $

  omirrd.c - online mirror daemon main file.
  Copyright (C) 1996, Andreas Luik, <luik@pharao.s.bawue.de>.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#define NEED_TIME
#include "common.h"

#if defined(RCSID) && !defined(lint)
static char rcsid[] UNUSED__ = "$Id: omirrd.c,v 1.29 1997/01/17 01:47:00 luik Exp $";
#endif /* defined(RCSID) && !defined(lint) */

#include <errno.h>		/* POSIX.1 headers */
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#if HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif
#include <sys/socket.h>		/* Networking headers */
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#ifdef HAVE_PATHS_H		/* Other headers */
#include <paths.h>
#endif
#include <getopt.h>		/* GNU getopt */
#include "libomirr.h"
#include "sig.h"
#include "version.h"
#include "debug.h"
#include "error.h"
#include "cfvar.h"
#include "cf.h"
#include "optflags.h"
#include "mp.h"
#include "lio.h"
#ifdef ENABLE_OMIRRK
#include "request.h"
#include "omirrk.h"
#endif /* defined(ENABLE_OMIRRK) */
#include "sock.h"
#include "conn.h"
#include "omirrd.h"


/* The following macros are usually defined in <sys/wait.h>.  */
#ifndef WIFEXITED
#define	WIFEXITED(S)	((int)((S) & 0xFF) == 0)
#endif
#ifndef	WIFSIGNALED
#define	WIFSIGNALED(S)	((int)((S) & 0xFF) > 0 && (int)((S) & 0xFF00) == 0)
#endif
#ifndef	WIFSTOPPED
#define	WIFSTOPPED(S)	((int)((S) & 0xFF) == 0177 && (int)((S) & 0xFF00) != 0)
#endif
#ifndef	WEXITSTATUS
#define	WEXITSTATUS(S)	((int)(((S) >> 8) & 0xFF))
#endif
#ifndef	WTERMSIG
#define	WTERMSIG(S)	((int)((S) & 0x7F))
#endif
#ifndef	WSTOPSIG
#define	WSTOPSIG(S)	((int)(((S) >> 8) & 0xFF))
#endif


/* If set, swap parent son relationship so that the kernel interface
   runs in the parent process and the regular processing in the
   son. This is sometimes useful for debugging purposes with a
   debugger but will result in incorrect signal propagation and has
   probably other problems, too.  */
#define SWAP_PROCESS_RELATIONSSHIP 0


#ifndef PATH_VARRUN		/* Pathname of omirrd.pid run file.  */
#ifdef _PATH_VARRUN
#define PATH_VARRUN _PATH_VARRUN
#else /* !defined(_PATH_VARRUN) */
#define PATH_VARRUN PATH_VARRUN_DEFAULT
#endif /* !defined(_PATH_VARRUN) */
#endif /* !defined(PATH_VARRUN) */

#define PID_FILENAME "omirrd.pid"

#ifndef PATH_VARTMP		/* Pathname of omirrd.run debug output file.  */
#ifdef _PATH_VARTMP
#define PATH_VARTMP _PATH_VARTMP
#else /* !defined(_PATH_VARTMP) */
#define PATH_VARTMP "/usr/tmp/"
#endif /* !defined(_PATH_VARTMP) */
#endif /* !defined(PATH_VARTMP) */

#define RUN_FILENAME "omirrd.run"


/* Global input/output contexts.  */
BioContext bio_context;
LioContext lio_context;
MpContext mp_context;

/* Command line options */
unsigned char block_mode_flag;	/* -b - set driver in block mode */
unsigned char clearbuf_flag;	/* -c - clear ring buffer before reading */
unsigned long debug_flag;	/* -d - enable debugging, run in foreground */
char *path_omirrdconf = PATH_OMIRRDCONF; /* -f omirrd.conf - set config file */
char *store_dir = PATH_STOREDIR;
static int help_flag;		/* --help - display help message */

static pid_t omirrk_pid;	/* pid of omirrk subprocess */

/* Some globals for error messages and other purposes */
char *program_name;		/* name of current program (from argv[0]) */
char *program_version;		/* version of omirrd */
char *own_hostname;		/* official name of this host */
char *current_file_name;	/* name of file currently processed (or NULL) */
int current_line = -1;		/* current line number in `current_file' or -1 */

/* The `CfCmd' pointer containing the commands to process while running.  */
CfCmd cmds;

/* zero_umask() saves the old umask in `saved_umask' and
   restore_umask() restores it from this value.  The daemon sets its
   umask to 0 at the beginning. All regular file creation calls should
   then be surrounded with a "restore_umask"-"zero_umask" block.  */
static mode_t saved_umask;



/*
 * Utility functions.
 */

/* zero_umask - set umask to 0 saving old value to allow restoring it
   with restore_umask.  */
static void zero_umask(void)
{
    saved_umask = umask(0);
}

static void restore_umask(void)
{
    umask(saved_umask);
}



/* init_program_version - set global variable `program_version' to
   string containing the current version number computed from the
   values set in "version.h". */
static void init_program_version(void)
{
    static char program_version_buf[12]; /* up to XXX.YYY.ZZZ */

#ifdef TEENY_VERSION
    sprintf(program_version_buf, "%d.%d.%d",
	    MAJOR_VERSION, MINOR_VERSION, TEENY_VERSION);
#else
    sprintf(program_version_buf, "%d.%d",
	    MAJOR_VERSION, MINOR_VERSION);
#endif
    program_version = program_version_buf;
}


/* init_own_hostname - set global variable `own_hostname' to the name
   of this host. It tries to determine the canonic host name using
   `gethostbyname' and only if that fails, the system host name
   returned by `gethostname' is used.  */
static void init_own_hostname(void)
{
    struct hostent *hp;
    static char own_hostname_buf[MAXHOSTNAMELEN + 1];

    own_hostname = own_hostname_buf;
    if (gethostname(own_hostname, MAXHOSTNAMELEN) == -1) {
	error("failed to determine my hostname: %s\n", xstrerror(errno));
	strcpy(own_hostname_buf, "unknown");
    }
    else if (((hp = gethostbyname(own_hostname))) == NULL) {
	error("failed to determine my official hostname: %s\n", xstrerror(errno));
    }
    else if (strlen(hp->h_name) <= MAXHOSTNAMELEN) {
	strcpy(own_hostname, hp->h_name);
    }
    else own_hostname = xstrdup(hp->h_name);
}



/* write_pid_file - creates the omirrd.pid run file in PATH_VARRUN
   (usually /var/run or /etc) and writes the current pid and the
   command line used to start the daemon process into it. (This
   follows the 4.4BSD convention).  */
static void write_pid_file(int argc, char **argv)
{
    FILE *f;
#ifdef __STDC__
    static char pid_filename[] = PATH_VARRUN "/" PID_FILENAME;
#else
    static char pid_filename[sizeof(PATH_VARRUN) + sizeof(PID_FILENAME) + 2];
    sprintf (pid_filename, "%s/%s", PATH_VARRUN, PID_FILENAME);
#endif

    restore_umask();
    if ((f = fopen(pid_filename, "w"))) {
	fprintf(f, "%ld\n", (long) getpid());
	fprintf(f, "%s", *argv++);
	while (--argc)
	    fprintf(f, " %s", *argv++);
	putc('\n', f);
	fclose(f);
    }
    else warning("open pid file %s failed: %s\n",
		 pid_filename, xstrerror(errno));
    zero_umask();
}


/* cleanup - cleanup before calling exit.  */
static void cleanup(int sig)
{
#ifdef DEBUG
    /* If compiled with -DDEBUG, try to free all allocated memory
       blocks to simplify searching for memory leaks with a debug
       malloc library or purify.  */
    cfFreeCmds();		/* frees `cmds' and sets it to NULL */
    cfUnsetVars(1);		/* remove all previous variables */
#endif

    if (omirrk_pid)
	kill(omirrk_pid, sig);
}



/*
 * Signal handlers. These are called both for omirrd and omirrk until
 * omirrk installs its own handlers. Therefore they check getpid()
 * against `omirrk_pid'.
 */

static RETSIGTYPE handle_sighup(int sig)
{
    debuglog(DEBUG_DAEMON, ("handle_sighup: processing SIGHUP\n"));
    /* Nothing to do here until a mp-signal handler is installed. This
       is handled in `process_signal' then.  */
}

static RETSIGTYPE handle_sigpipe(int sig)
{
    debuglog(DEBUG_DAEMON, ("handle_sigpipe: processing SIGPIPE\n"));
    /* Nothing to do because write errors to pipes/sockets are handled
       by mp error callbacks, e.g. `close_daemon_connection'.  */
}

static RETSIGTYPE handle_sigterm(int sig)
{
    debuglog(DEBUG_DAEMON, ("handle_sigterm: processing SIGTERM\n"));
    if (getpid() != omirrk_pid) { /* omirrd */
	errlog(LOG_INFO, "going down on signal TERM\n");
	cleanup(sig);
	exit(0);
    }
    else {			/* omirrk */
	exit(0);
    }
}

static RETSIGTYPE handle_sigchld(int sig)
{
    pid_t pid;
    int wait_status;
    char *msg;
    int msg_arg;

    while (1) {			/* repeat wait to catch all exited children */
#ifdef HAVE_WAITPID
	pid = waitpid((pid_t) -1, &wait_status, WNOHANG);
#else
	pid = wait3(&wait_status, WNOHANG, NULL);
#endif
	if (pid == -1 && errno == EINTR)
	    continue;		/* try again */

	if (pid == 0 || (pid == -1 && errno == ECHILD))
	    break;		/* no exited child available */

	if (WIFEXITED(wait_status)) {
	    msg = "exited with status";
	    msg_arg = WEXITSTATUS(wait_status);
	}
	else if (WIFSIGNALED(wait_status)) {
	    msg = "terminated due to signal";
	    msg_arg = WTERMSIG(wait_status);
	}
	else if (WIFSTOPPED(wait_status)) {
	    msg = "stopped due to signal";
	    msg_arg = WSTOPSIG(wait_status);
	}
	else {
	    msg = "returned unknown status";
	    msg_arg = wait_status;
	}
	debuglog(DEBUG_DAEMON, ("handle_sigchld: processing SIGCHLD, pid %ld %s %d\n",
				(long) pid, msg, msg_arg));

	if ((getpid() != omirrk_pid	/* omirrd */
	     && pid == omirrk_pid))	/* omirrk died */
	    error("omirrk[%ld] %s %d\n", (long) pid, msg, msg_arg);
	else
	    error("unknown child %ld %s %d\n", (long) pid, msg, msg_arg);
    }
}

static RETSIGTYPE handle_sigusr2(int sig)
{
    FILE *f;
    time_t time0;
    char *ctimep;

#ifdef __STDC__
    static char run_filename[] = PATH_VARTMP "/" RUN_FILENAME;
#else
    static char run_filename[sizeof(PATH_VARTMP) + sizeof(RUN_FILENAME) + 2];
    sprintf (run_filename, "%s/%s", PATH_VARTMP, RUN_FILENAME);
#endif

    restore_umask();
    if ((f = fopen(run_filename, "a"))) {
	fprintf(f, "omirrd version %s\n", program_version);
	if ((time0 = time(NULL)) != -1 && ((ctimep = ctime(&time0))))
	    fputs(ctimep, f);
#ifdef DEBUG
	connDebugConnections(f);
#else
	fprintf(f, "not compiled with -DDEBUG\n");
#endif
	fprintf(f, "\n");
	fclose(f);
    }
    else error("failed to open debug output file %s: %s\n",
	       run_filename, xstrerror(errno));
    zero_umask();
}


/*ARGSUSED*/
static RETSIGTYPE process_signal(int sig, void *data)
{
    switch (sig) {
      case SIGHUP:
	handle_sighup(sig);

	/* close all connections (not required, but simplifies resync,
           which is then done automagically during reconnect) */
	connDoTerminateConnectionsToServers();

	init_own_hostname();
	cfFreeCmds();		/* frees `cmds' and sets it to NULL */
	cfUnsetVars(0);		/* remove all previous variables */
	cfParse(path_omirrdconf); /* Parse config file again.  */
#ifdef DEBUG
	if (debug(DEBUG_CF))
	    cfDebugCmd("", cmds);
#endif

	/* connect to servers again, also performs filesystem resync */
	connDoConnectionsToServers(cmds);
	break;

      case SIGPIPE:
	handle_sigpipe(sig);
	break;

      case SIGTERM:
	handle_sigterm(sig);
	break;

      case SIGCHLD:
	handle_sigchld(sig);
	break;

      default:
	error("internal: process_signal: unknown signal %d\n", sig);
	break;
    }
}



/*
 * Message processing.
 */

#ifdef ENABLE_OMIRRK
static void process_omirrk_message(char *line, void *data)
{
    /* XXX queue really required??? If not: removed as callback,
       too. We don't need to allocate it even at all.  */
    ReqQueue req_queue = (ReqQueue) data;
    ReqRequest r;
    ReqState state;

    debuglog(DEBUG_PROT, ("%s\n", line));

    /* Add message to request queue and send it to all slaves.  */
    if ((r = reqParseMessage(req_queue, line))) {
	switch (r->op) {
	  case 'Z':
	    connDoSyncsToServers();
	    break;
	  default:
	    state = reqFillRequest(r);
	    if (state == REQ_COMPLETE)
		connWriteMessage(r);
	    else reqError("unresolved pathname", line);
	    break;
	}
	/* XXX until ack/nack processing available, remove this
           request from queue immediately.  */
	reqDropFirstRequest(req_queue);
    }
}

/*ARGSUSED*/
static int close_omirrk_connection(int fd, void *data)
{
    error("lost connection to omirrk: %s\n", xstrerror(errno));

    /* Remove callbacks added in `main' for this pipe fd.  */
    lioRemoveReadCallback(lio_context, fd);
    mpRemoveReadCallback(mp_context, fd);
    mpRemoveErrorCallback(mp_context, fd);

    close(fd);
    return 0;
}
#endif /* defined(ENABLE_OMIRRK) */



/*
 * Socket connections.
 */

/* accept_client - callback to accept a new client on `sockfd'. Uses
   connAcceptClient to do the actual work.  Always returns 0.  */
/*ARGSUSED*/
static int accept_client(int sockfd, void *data)
{
    connAcceptClient(sockfd);
    return 0;
}




/*
 * Command line parsing.
 */

static const struct option long_options[] =
{
  {"block-mode", no_argument, NULL, 'b'},
  {"clear-buffer", no_argument, NULL, 'c'},
  {"debug", required_argument, NULL, 'd'},
  {"file", required_argument, NULL, 'f'},
  {"store-dir", required_argument, NULL, 's'},
  {"help", no_argument, &help_flag, 1},
  {"option", optional_argument, NULL, 'o'},
  {"version", no_argument, NULL, 'V'},
  { 0, 0, 0, 0 },
};


static void help(void)
{
    printf("Usage: %s [OPTION]...\n", program_name);
    puts("    -b, --block-mode          set kernel to blocking-mode");
    puts("    -c, --clear-buffer        clear kernel ring buffer at startup");
    puts("    -d, --debug[=FLAGS]       set debug level(s)");
    puts("    -f, --file=FILE           use FILE as config file [" PATH_OMIRRDCONF "]");
    puts("    -o, --option=OPTS         set/clear option flags");
    puts("          supported options:  numchkgroup,numchkowner,whole");
    puts("    -s, --store-dir=DIR       use DIR for redo-store [" PATH_STOREDIR "]");
    puts("        --help                display this help and exit");
    puts("    -V, --version             output version information and exit");
    exit(0);
}


static void usage(void)
{
    fprintf(stderr, "Usage: %s [-bc] [-d[levels] [-oopts] [-f omirrfile] [-V]\n",
	    program_name);
    fprintf(stderr, "Use `%s --help' for full list of command line options\n",
	    program_name);
    exit(2);
}



/*
 * Omirrd main program.
 */

int main(int argc, char **argv, char **environ)
{
    int c;
    extern char *optarg;
    extern int optind;
    char *unknown_args;
    unsigned char version_flag = 0;
    int sockfd;			/* server socket */
#ifdef ENABLE_OMIRRK
    int pipefd[2];		/* pipe connection to omirrk */
    ReqQueue req_queue;
#endif

    program_name = argv[0];
    initsetproctitle(argc, argv, environ);
    init_program_version();

    while ((c = getopt_long(argc, argv, "bcd::f:o:s:V", long_options, NULL)) != EOF) {
	switch (c) {
	  case 0:		/* long options without short option */
	    break;
	  case 'b':
	    block_mode_flag = 1;
	    break;
	  case 'c':
	    clearbuf_flag = 1;
	    break;
	  case 'd':
#ifdef DEBUG
	    if (debugParseLevels(optarg, &debug_flag, &unknown_args) == -1) {
		fprintf(stderr, "%s: unknown debug level%s `%s' ignored\n",
			program_name, (strchr(unknown_args, ',') ? "s" : ""),
			unknown_args);
	    }
#else /* !defined(DEBUG) */
	    if (optarg)
		fprintf(stderr, "%s: not compiled with -DDEBUG, ignoring `%s'\n",
			program_name, optarg);
#endif /* !defined(DEBUG) */
	    debug_flag |= DEBUG_GENERAL;
	    break;
	  case 'f':
	    path_omirrdconf = optarg;
	    break;
	  case 'o': {
	      unsigned long set;
	      unsigned long clear;

	      if (optParseOptions(optarg, &set, &clear, &unknown_args) == -1) {
		  fprintf(stderr, "%s: unknown option%s `%s' ignored\n",
			  program_name, (strchr(unknown_args, ',') ? "s" : ""),
			  unknown_args);
	      }
	      option_flag |= set;
	      option_flag &= ~clear;
	      break;
	  }
	  case 's':
	    store_dir = optarg;
	    break;
	  case 'V':
	    version_flag = 1;
	    break;
	  default:
	    usage();
	    /*NOTREACHED*/
	}
    }

    if (optind < argc)
	usage();		/* does not return */

    if (version_flag) {
	printf("omirrd version %s\n", program_version);
	exit(0);
    }

    if (help_flag)
	help();			/* does not return */

    if (getuid()) {
	error("must be root to run - terminating\n");
	exit(1);
    }

#ifdef LOG_DAEMON		/* new openlog(3) with three args */
    openlog("omirrd", 0, LOG_OMIRRD);
#else
    openlog("omirrd", 0);
#endif
    errlog_syslog(1);		/* enable error messages to syslog(3) */

    mkdir(store_dir, 0700);	/* ensure that store_dir exists */

    zero_umask();		/* umask(0) */
    sigHandler(SIGHUP, handle_sighup);
    sigHandler(SIGTERM, handle_sigterm);
    sigHandler(SIGCHLD, handle_sigchld);
    sigHandler(SIGUSR2, handle_sigusr2);

    if (!debug_flag) {		/* go into background */
	if (daemon(1, 0) == -1) { /* 4.4BSD function to move process into bg */
	    error("cannot go into background: %s - terminating\n",
		  xstrerror(errno));
	    exit(1);
	}
	errlog_stderr(0);	/* disable error messages to stderr */
	write_pid_file(argc, argv);
    }

    init_own_hostname();

    /* Open server socket, exit on failure. Error message is logged by
       function.  */
    if (((sockfd = sockServer(SERVICE_OMIRR, IPPORT_OMIRR))) == -1)
	exit(1);

#ifdef ENABLE_OMIRRK
    /* Fork omirrk process (kernel interface). Communication is
       handled via a pipe.  */

    if (pipe(pipefd) == -1) {
	error("pipe: %s - omirrk disabled\n", xstrerror(errno));
	pipefd[0] = -1;
    }
	
    if (pipefd[0] != -1) {
	fflush(stdout);
	fflush(stderr);
	switch ((omirrk_pid = fork())) {
	  case -1:
	    error("fork: %s - omirrk disabled\n", xstrerror(errno));
	    close(pipefd[0]);
	    close(pipefd[1]);
	    pipefd[0] = -1;
	    break;
#if !SWAP_PROCESS_RELATIONSSHIP
	  case 0:		/* son: handles kernel messages (omirrk) */
#else
	  default:
#endif
	    close(pipefd[0]);
	    kernel_interface_main(pipefd[1]); /* never returns */
	    /*NOTREACHED*/
#if !SWAP_PROCESS_RELATIONSSHIP
	  default:		/* parent: simply continues */
#else
	  case 0:
#endif
	    close(pipefd[1]);
	    break;
	}
    }
#endif

    /* Allocate buffered I/O, line I/O and multiplex contexts.  */
    bio_context = bioAlloc();
    lio_context = lioAlloc();
    mp_context = mpAlloc();

    /* mp_context: add signal handlers.  */
    mpSetSignalCallback(mp_context, SIGHUP, process_signal, NULL, NULL);
    mpSetSignalCallback(mp_context, SIGPIPE, process_signal, NULL, NULL);
    mpSetSignalCallback(mp_context, SIGTERM, process_signal, NULL, NULL);
    mpSetSignalCallback(mp_context, SIGCHLD, process_signal, NULL, NULL);

    cfParse(path_omirrdconf);	/* Parse config file.  */
#ifdef DEBUG
    if (debug(DEBUG_CF))
	cfDebugCmd("", cmds);
#endif
    connDoConnectionsToServers(cmds); /* connect to all other servers */

    /* mp_context: add `accept_client' as callback, it will accept
       new clients and initiate the communication.  */
    mpAddReadCallback(mp_context, sockfd, accept_client, NULL);

#ifdef ENABLE_OMIRRK
    if (pipefd[0] != -1) {
	/* Allocate request queue */
	req_queue = reqAllocQueue();

	/* lio_context: add `process_omirrk_message' as callback.  */
	lioAddReadCallback(lio_context, pipefd[0],
			   process_omirrk_message, req_queue);
	
	/* mp_context: add `lioMpReadFunc' as callback, which will split
	   the input stream into lines and call the callback function of
	   the lio context.  */
	mpAddReadCallback(mp_context, pipefd[0],
			  lioMpReadFunc, (void *) lio_context);

	/* mp_context: add `close_omirrk_connection' as error callback
           to close pipe filedescriptor in case of an error.  */
	mpAddErrorCallback(mp_context, pipefd[0], close_omirrk_connection, NULL);
    }
#endif

    mpMainLoop(mp_context);
    error("internal: mpMainLoop returned - terminating\n");

    cleanup(SIGTERM);
    exit(0);
}
