#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <signal.h>

#include <config.h>

#ifdef	HAVE_READLINE
#include <readline/readline.h>
#endif

#include <support.h>
#include <xcio.h>
#include <msgcat.h>

#define	CMD_FULLNAME		1
#define	CMD_SECURE		2
#define	NUM_BIC	(sizeof(bicList)/sizeof(bicList[0]))

extern int ppxpFd, ttyMode;
static int CmdHelp(), CmdPrint(), CmdShow(), CmdPasswd(), CmdChdir();
static int CmdListup();
extern int CmdFg(), CmdJobs(), CmdQdial(), CmdTpage();

static struct {
    const char *name;
    const char *help;
    int (*cmd)();
    u_int8_t flags;
} bicList[]={
    {
	"print", "print envs\n", CmdPrint, 0
    },
    {
	"jobs", "jobs\n", CmdJobs, CMD_FULLNAME
    },
    {
	"fg", "fg\n", CmdFg, CMD_FULLNAME
    },
    {
	"help", "help\n", CmdHelp, CMD_FULLNAME
    },
    {
	"show", "show\n", CmdShow, CMD_FULLNAME
    },
    {
	"cd", "cd\n", CmdChdir, CMD_FULLNAME
    },
    {
	"passwd", "passwd\n", CmdPasswd, CMD_FULLNAME
    },
    {
	"qdial", "Quick dialup mode\n", CmdQdial, CMD_FULLNAME
    },
    {
	"tpage", "Telephone book\n", CmdTpage, CMD_FULLNAME
    },
    {
	"listup", "list-up labels\n", CmdListup, CMD_FULLNAME
    },
};

void
SetPasswd(char *entry)
{
    char *name, *key;

    name = readline("Name:");
    if (name) {
	if (strlen(name)) {
	    TtyEcho(FALSE);
	    key = readline("Password:");
	    TtyEcho(TRUE);
	    PPxPwdSet(ppxpFd, entry, name, key);
	    printf("\n");
	    free(key);
	}
	free(name);
    }
}

static int
CmdPasswd(int argc, char *argv[])
{
    struct xcio_s xc;
    int n, xargc;
    u_int8_t xid;
    char *xargv[6];

    if (argc == 1) {
	xid = PPxPwdRequest(ppxpFd, NULL);
	printf("%-12s %-12s %-12s %-12s %-12s\n",
	       "", "NAME", "PASSWORD", "GROUP", "SCRIPT");
	while ((n = PPxPRead(ppxpFd, xid, &xc)) >= 0) {
	    if (n > 0 && (xc.type & XCIO_MASK) == XCIO_PWD_SET) {
		xargc = DecodeArgs(xargv, xc.buf, xc.len, 6);
		printf("%-12.12s %-12.12s %-12.12s %-12.12s %-12.12s\n",
		       xargv[2], xargv[0], xargv[1], xargv[3], xargv[4]);
		FreeArgs(xargc, xargv);
	    }
	    if (xc.type & XCIO_LAST) break;
	}
	return(0);
    }
    SetPasswd(argv[1]);
    return(0);
}

static int
CmdListup(int argc, char *argv[])
{
    struct xcio_s xc;
    int n;
    u_int8_t xid;

    xid = PPxPListupRequest(ppxpFd);
    while ((n = PPxPRead(ppxpFd, xid, &xc)) >= 0) {
	if (n > 0 && (xc.type & XCIO_MASK) == XCIO_LISTUP) {
	    if (xc.buf[0] == XLABEL_COMMAND) printf("CMD ");
	    else printf("%3d.", xc.buf[0]);
	    printf("%2d %s\n", xc.buf[1], &xc.buf[3]);
	}
	if (xc.type & XCIO_LAST) break;
    }
    return(0);
}

static int
CmdChdir(int argc, char *argv[])
{
    return(argc > 1 ? chdir(argv[1]): 0);
}

static int
CmdShow(int argc, char *argv[])
{
    int a, n;
    unsigned int b;
    struct xcio_s xc;
    u_int8_t xid, func=0xff;
    static const char *subcmd[]={
	"modem", "phase", "time", "line", "network", "mode",
	"console", "login", NULL
    };
    enum {
	SHOW_MODEM, SHOW_PHASE, SHOW_TIME, SHOW_LINE, SHOW_NETWORK,
	SHOW_MODE, SHOW_CONSOLE, SHOW_LOGIN
    };

    for (a = 1; a < argc; a ++) {
	n = 0;
	func = 0;
	while (subcmd[n]) {
	    if (!strcasecmp(subcmd[n], argv[a])) func |= 1<<n;
	    n ++;
	}
    }
    if (func == (1<<SHOW_CONSOLE)) {
	char *name, *from, flags, *p;
	int cfd;

	xid = PPxPRequest(ppxpFd, XCIO_CONSOLES);
	printf("A C  ID NAME     FROM\n");
	while ((n = PPxPRead(ppxpFd, xid, &xc)) >= 0) if (n > 0) {
	    name = xc.buf;

	    from = name + strlen(name) + 1;
	    p = from + strlen(from) + 1;
	    flags = *p ++;
	    memcpy(&cfd, p, sizeof(cfd));

	    printf("%c %c %3d %-8s %s\n",
		   (flags & CONSOLE_AUTOF) ?
		   (flags & CONSOLE_AUTOC) ? '+': '-': ' ',
		   (flags & CONSOLE_CURRENT) ? '*': ' ',
		   cfd, name, from);
	    if (n < 0 || (xc.type & XCIO_LAST)) break;
	}
	return(0);
    }
    if (func == (1<<SHOW_LOGIN)) {
	char *name, *key;

	name = key = NULL;
	PPxPwdGet(ppxpFd, &name, &key);
	printf("Name: %s\nPassword: %s\n", name, key);
	Free(key);
	Free(name);
	return(0);
    }

    PPxPUpdateRequest(ppxpFd);
    while ((n = PPxPRead(ppxpFd, XID_UPDATE, &xc)) >= 0)
	if (n > 0 && xc.type == XCIO_UP_INFO) break;
    memcpy(&pppInfo, &xc.buf, sizeof(pppInfo));
    if ((func & (1<<SHOW_MODEM))
	&& (pppInfo.minfo != (unsigned int)-1)) {
	static struct {
	    unsigned int b;
	    const char *name;
	} sioInfo[]={
	    {TIOCM_DTR, "DTR"},
	    {TIOCM_RTS, "RTS"},
	    {TIOCM_CTS, "CTS"},
	    {TIOCM_CD, "CD"},
	    {TIOCM_RNG, "RNG"},
	    {TIOCM_DSR, "DSR"},
	    {0, 0}
	};

	printf("Modem:");
	n = 0; 
	while (sioInfo[n].b) {
	    if (pppInfo.minfo & sioInfo[n].b)
		printf(" %s", sioInfo[n].name);
	    n ++;
	}
	printf("\n");
    }
    if (func & (1<<SHOW_PHASE)) {
	static const char *phase[]={
	    "Dead", "Establish", "Authenticate", "Callback",
	    "Network", "Terminate"};

	printf("Phase: %s\n", phase[pppInfo.phase]);
    }
    if (func & (1<<SHOW_TIME)) {
	printf("Connect Time: %02d:%02d:%02d\n", pppInfo.connect / 3600,
	       (pppInfo.connect % 3600) / 60,
	       pppInfo.connect % 60);
	printf("Idle Time: %02d:%02d:%02d\n", pppInfo.idle / 3600,
	       (pppInfo.idle % 3600) / 60,
	       pppInfo.idle % 60);
    }
    if (func & (1<<SHOW_LINE)) {
	static const char *lstat[]={
	    "Tty", "Chat", "PPP", "Network", "Dial", NULL
	};

	printf("Line status:");
	for (b = 0; lstat[b]; b ++)
	    if (pppInfo.l_stat & (1 << b)) printf(" %s", lstat[b]);
	printf("\n");
    }
    if (func & (1<<SHOW_NETWORK)) {
	static const char *nstat[]={
	    "IP", "IPX", NULL
	};

	printf("Network status:");
	for (b = 0; nstat[b]; b ++)
	    if (pppInfo.n_stat & (1 << b)) printf(" %s", nstat[b]);
	printf("\n");
    }
    if (func & (1<<SHOW_MODE)) {
	static const char *mflag[]={
	    "AUTO", NULL
	};

	printf("Mode:");
	for (b = 0; mflag[b]; b ++)
	    if (pppInfo.m_flag & (1 << b)) printf(" %s", mflag[b]);
	printf("\n");
    }
    return(0);
}

static int
CmdPrint(int argc, char *argv[])
{
    struct xcio_s xc;
    char *envv[4];
    int n, envc;
    u_int8_t xid;

    xid = PPxPEnvRequest(ppxpFd, argc - 1, &argv[1]);
    while ((n = PPxPRead(ppxpFd, xid, &xc)) >= 0) if (n > 0) {
	if ((xc.type & XCIO_MASK) == XCIO_ENV_SET) {
	    envc = DecodeArgs(envv, xc.buf, xc.len, 4);
	    if (envc > 1 && *envv[1]) {
		if (envc >= 3 && *envv[2]) printf("%s.", envv[2]);
		printf("%s=%s\n", envv[0], envv[1]);
	    }
	    FreeArgs(envc, envv);
	}
	if (xc.type & XCIO_LAST) break;
    }
    return(0);
}

static int
CmdHelp(int argc, char *argv[])
{
    unsigned n;
    int i=0;
    char *help;
    extern struct command_s xcmdList[];

    if (argc > 1) {
	for (n = 0; n < XCMD_MAX; n ++) {
	    if (!strcasecmp(argv[1], xcmdList[n].name))
		if ((help = SysMsgGet(xcmdList[n].msg)) != NULL) {
		    printf("%s\n", help);
		    return(0);
		}
	}
	for (n = 0; n < NUM_BIC; n ++) {
	    if (!strcasecmp(argv[1], bicList[n].name))
		if (bicList[n].help) printf(bicList[n].help);
	}
	return(0);
    }
    for (n = 0; n < XCMD_MAX; n ++) {
	if (i && !(i % 4)) printf("\n");
	i ++;
	printf("%-20s", xcmdList[n].name);
    }
    for (n = 0; n < NUM_BIC; n ++) {
	if (i && !(i % 4)) printf("\n");
	i ++;
	printf("%-20s", bicList[n].name);
    }
    printf("\n");
    return(0);
}

int
RunBuiltinCommand(int argc, char *argv[])
{
    unsigned int n;
    int len=strlen(argv[0]), ret=-1;

    for (n = 0; n < NUM_BIC; n ++) {
	if (bicList[n].flags & CMD_FULLNAME) {
	    if (!strcasecmp(argv[0], bicList[n].name)) break;
	} else {
	    if (!strncasecmp(argv[0], bicList[n].name, len)) break;
	}
    }
    if (n < NUM_BIC) ret = bicList[n].cmd(argc, argv);
    return(ret);
}
