/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>
#include <string.h>

#include <glib.h>

#include "acache.h"
#include "server.h"
#include "group.h"
#include "article.h"
#include "articlelist.h"

/***
****
***/

PanCallback *article_deleted_callback = NULL;

/***
****
***/

static void
article_update_impl (
	article_db db,
	const article_data* adata,
	gboolean sync,
	gboolean update_ui);

static void
article_add_flag_impl (
	article_db db,
	article_data *adata,
	int flag,
	gboolean sync,
	gboolean update_ui);

static void
article_remove_flag_impl (
	const article_db db,
	article_data* adata,
	int flag,
	gboolean sync,
	gboolean update_ui);

/***
****
***/


void
article_init (void)
{
	article_deleted_callback = pan_callback_new ();
}

void
article_isolate (article_data* adata)
{
	GSList *l = NULL;
	article_data *parent = adata->parent;

	if (parent)
		parent->threads = g_slist_remove (parent->threads, adata);

	for (l=adata->threads; l!=NULL; l=l->next) {
		article_data *child = ARTICLE_DATA(l->data);
		child->parent = parent;
		if (parent)
			parent->threads = g_slist_append (parent->threads, child);
	}

	g_slist_free (adata->threads);
	adata->threads = NULL;
	adata->parent = NULL;
}

article_data*
article_new (void)
{
	article_data* adata = g_new0 (article_data, 1);
	adata->threads = NULL;
	return adata;
}

/***
****
***/

/**
 * @pararm article_db
 * @param adata article to be marked as read
 * @param marked pointer to an int to be incremented for each article marked read
 *
 * If an article is a multipart article, then the multipart articles are also
 * marked as read if they're not read already.
 */
static void
article_mark_read_impl (
	article_db db,
	article_data* adata,
	int* marked)
{
	if (!adata)
		return;

	if (!article_flag_on (adata, STATE_READ)) {
		article_add_flag_impl (db, adata, STATE_READ, FALSE, FALSE);
		articlelist_adata_read_changed (adata);
		++*marked;
	}

	/* special case: mark ALL of a multipart read */
	if (adata->part>0 && adata->parts>1 && adata->threads!=NULL) {
		GSList *l;
		for (l=adata->threads; l!=NULL; l=l->next) {
			article_data *child = ARTICLE_DATA(l->data);
			article_mark_read_impl (db, child, marked);
		}
	}
}


/**
 * @param server
 * @param group
 * @param article
 * @param sync database
 *
 * This marks an article as read as described in article_mark_read_impl.
 *
 * After being marked, then if sync is true, the database is synchronized.
 *
 * If the read state of more than one article has changed, the group attributes
 * are updated.
 */
void
article_mark_read (
	const server_data* sdata,
	const group_data* gdata,
	article_data* adata,
	gboolean sync )
{
	int marked = 0;
	article_db db = NULL;
	
	db = ahdb_ref (sdata, gdata);
	article_mark_read_impl (db, adata, &marked);

	if (sync)
		ahdb_sync (db);
	if (marked)
		group_add_attrib_i (sdata, gdata, "Read", marked);

	ahdb_unref (db);

}

/**
 * @param server
 * @param group
 * @param article
 * @param sync database
 *
 * This marks a list of articles as read as described in article_mark_read_impl.
 *
 * After being marked, then if sync is true, the database is synchronized.
 *
 * If the read state of more than one article has changed, the group attributes
 * are updated when done.
 */
void
articles_mark_read (
	const server_data* sdata,
	const group_data* gdata,
	article_data** articles,
	int article_qty,
	gboolean sync )
{
	int marked = 0;
	article_db db = NULL;
	int i;

	g_return_if_fail (articles!=NULL);
	g_return_if_fail (article_qty>=1);

	db = ahdb_ref (sdata, gdata);
	for (i=0; i!=article_qty; ++i)
		article_mark_read_impl (db, articles[i], &marked);

	if (sync && marked)
		ahdb_sync (db);
	if (marked)
		group_add_attrib_i (sdata, gdata, "Read", marked);

	ahdb_unref (db);
}


/**
***
**/

/**
 * @pararm article_db
 * @param adata article to be marked as read
 * @param marked pointer to an int to be incremented for each article marked unread
 *
 * If an article is a multipart article, then the multipart articles are also
 * marked as unread if they're not unread already.
 */
static void
article_mark_unread_impl (
	article_db db,
	article_data* adata,
	int *marked )
{
	if (!adata)
		return;

	if (article_flag_on (adata, STATE_READ)) {
		article_remove_flag_impl (db, adata, STATE_READ, FALSE, FALSE);
		articlelist_adata_read_changed (adata);
		++*marked;
	}

	/* special case: mark ALL of a multipart unread */
	if (adata->part>0 && adata->parts>1 && adata->threads!=NULL) {
		GSList *l;
		for (l=adata->threads; l!=NULL; l=l->next) {
			article_data *child = ARTICLE_DATA(l->data);
			article_mark_unread_impl (db, child, marked);
		}
	}
}

/**
 * @param server
 * @param group
 * @param article
 * @param sync database
 *
 * This marks an article as unread as described in article_mark_unread_impl.
 *
 * After being marked, then if sync is true, the database is synchronized.
 *
 * If the read state of more than one article has changed, the group attributes
 * are updated when done.
 */
void
article_mark_unread (
	const server_data* sdata,
	const group_data* gdata,
	article_data* adata,
	gboolean sync)
{
	int marked = 0;
	article_db db = 0;

	db = ahdb_ref (sdata, gdata);
	article_mark_unread_impl (db, adata, &marked);

	if (sync && marked)
		ahdb_sync (db);
	if (marked)
		group_add_attrib_i (sdata, gdata, "Read", -marked);

	ahdb_unref (db);
}

/**
 * @param server
 * @param group
 * @param article
 * @param sync database
 *
 * Marks a list of articles as unread as described in article_mark_unread_impl.
 *
 * After being marked, then if sync is true, the database is synchronized.
 *
 * If the read state of more than one article has changed, the group attributes
 * are updated when done.
 */
void articles_mark_unread (
	const server_data* sdata,
	const group_data* gdata,
	article_data** articles,
	int article_qty,
	gboolean sync )
{
	int marked = 0;
	article_db db = 0;
	int i;

	g_return_if_fail (articles!=NULL);
	g_return_if_fail (article_qty>=1);

	db = ahdb_ref (sdata, gdata);
	for (i=0; i!=article_qty; ++i)
		article_mark_unread_impl (db, articles[i], &marked);

	if (sync) {
		ahdb_sync (db);
		if (marked)
			group_add_attrib_i (sdata, gdata, "Read", -marked);
	}

	ahdb_unref (db);
}

/***
****
***/

static void
article_delete_impl (
	article_db db,
	article_data* adata,
	int* read,
	int* total)
{
	if (!adata)
		return;

	/* special case: if a multipart, delete ALL of it */
	if (adata->part>0 && adata->parts>1 && adata->threads!=NULL) {
		GSList *l;
		GSList *tmp = g_slist_copy (adata->threads);
		for (l=tmp; l!=NULL; l=l->next) {
			article_data *child = ARTICLE_DATA(l->data);
			article_delete_impl (db, child, read, total);
		}
		g_slist_free (tmp);
	}

	/* notify listeners before the delete takes place */
	pan_callback_call (article_deleted_callback, adata, NULL);

	/* update counts */
	++*total;
	if (article_flag_on (adata, STATE_READ))
		++*read;

	/* isolate */
	article_isolate (adata);

	/* erase from cache */
	acache_delete (adata->message_id);

	/* erase from db */
	ahdb_erase (db, adata, NULL);

	/* erase from memory */
	article_free (adata);
}

void
article_delete (
	const server_data* sdata,
	const group_data* gdata,
	article_data* adata,
	gboolean sync)
{
	int total = 0;
	int read = 0;
	article_db db = 0;

	/* sanity checks */
	g_return_if_fail (sdata!=NULL);
	g_return_if_fail (gdata!=NULL);
	g_return_if_fail (adata!=NULL);

	db = ahdb_ref (sdata, gdata);
	article_delete_impl (db, adata, &read, &total);
	g_assert (total >= read);

	if (sync)
		ahdb_sync (db);

	group_add_attrib_i (sdata, gdata, "Total", -total);
	group_add_attrib_i (sdata, gdata, "Read", -read);

	ahdb_unref (db);
}

void
articles_delete (
	const server_data* sdata,
	const group_data* gdata,
	article_data** articles,
	int article_qty,
	gboolean sync)
{
	int total = 0;
	int read = 0;
	int i;
	article_db db = 0;

	/* sanity checks */
	g_return_if_fail (sdata!=NULL);
	g_return_if_fail (gdata!=NULL);
	g_return_if_fail (article_qty>=0);
	g_return_if_fail (article_qty ? articles!=NULL : 1);

	db = ahdb_ref (sdata, gdata);
	for (i=0; i!=article_qty; ++i)
		article_delete_impl (db, articles[i], &read, &total);

	if (sync)
		ahdb_sync (db);
	if (total)
		group_add_attrib_i (sdata, gdata, "Total", -total);
	if (read)
		group_add_attrib_i (sdata, gdata, "Read", -read);

	ahdb_unref (db);
}

/***
****
***/

#if 0
gboolean article_flag_on (
	const article_data* adata,
	int flag)
{
	g_return_val_if_fail (adata!=NULL, FALSE);

	return adata->state & flag;
}
#endif

/***
****
***/

static void
article_add_flag_impl (
	article_db db,
	article_data *adata,
	int flag,
	gboolean sync,
	gboolean update_ui)
{
	g_return_if_fail (adata!=NULL);

	adata->state |= flag;
	article_update_impl (db, adata, sync, update_ui);
}
void
article_add_flag (const server_data *sdata,
		  const group_data *gdata,
		  article_data* adata,
		  int flag,
		  gboolean sync)
{
	article_db db = ahdb_ref (sdata, gdata);
	article_add_flag_impl (db, adata, flag, sync, TRUE);
	ahdb_unref (db);
}

/***
****
***/

static void
article_remove_flag_impl (
	const article_db db,
	article_data* adata,
	int flag,
	gboolean sync,
	gboolean update_ui)
{
	g_return_if_fail (adata!=NULL);

	adata->state &= ~flag;
	article_update_impl (db, adata, sync, update_ui);
}

void
article_remove_flag (
	const server_data *sdata,
	const group_data *gdata,
	article_data* adata,
	int flag,
	gboolean sync)
{
	article_db db = ahdb_ref (sdata, gdata);
	article_remove_flag_impl (db, adata, flag, sync, TRUE);
	ahdb_unref (db);
}

/***
****
***/

static void
article_update_impl (
	article_db db,
	const article_data* adata,
	gboolean sync,
	gboolean update_ui)
{
	/* entry assertions */
	g_assert (adata && adata->message_id);

	/* update article header db... */
	ahdb_save (db, adata, sync);

	/* update the article ui... */
	if (update_ui)
		articlelist_update_node (adata);
}

void
article_update (
	const server_data *sdata,
	const group_data *gdata,
	const article_data* adata,
	gboolean sync)
{
	article_db db = ahdb_ref (sdata, gdata);
	article_update_impl (db, adata, sync, TRUE);
	ahdb_unref (db);
}

/**
***
**/

article_data*
article_copy (article_data* target,
	      const article_data* source)
{
	target->number = g_strdup (source->number);
	target->subject = g_strdup (source->subject);
	target->author = g_strdup (source->author);
	target->message_id = g_strdup (source->message_id);
	target->references = g_strdup (source->references);
	target->date = source->date;
	target->linecount = source->linecount;
	target->parts = source->parts;
	target->part = source->part;
	target->followup_to = g_strdup (source->followup_to);
	target->reply_to = g_strdup (source->reply_to);
	target->newsgroups = g_strdup (source->newsgroups);
	target->state = source->state;

	return target;
}

void
article_free_gfunc (gpointer data, gpointer user_data)
{
	article_free ( (article_data*)data );
}

void
article_free (article_data *adata)
{
        g_return_if_fail (adata!=NULL);

	g_assert (adata->subject != NULL);
	g_free (adata->number);
	g_free (adata->subject);
	adata->subject = NULL;
	g_free (adata->author);
	g_free (adata->message_id);
	g_free (adata->references);
	g_free (adata->newsgroups);
	g_free (adata->reply_to);
	g_free (adata->followup_to);
	g_free (adata);
}



/*
 * Extracts the requested header from adata->header;
 * hdr is the case sensitive name, without the colon
 * Doesn't handle wrapped lines
 */
#if 0
gchar *
extract_header (article_data *adata, gchar *hdr)
{
	gchar* header = g_strconcat (hdr, ": ", NULL);
//	gchar* start = strstr (adata->header, header);
        gchar* end;
	g_free (header);
	if (start == NULL)
		return NULL;
	else {
		start = strchr (start, ':');
		end = strchr (start, '\n');
		return g_strndup (start + 2, end - start - 3);
	}
}
#endif


gchar*
article_get_thread_message_id (const article_data* adata)
{
	gchar *pch = g_strdup (adata->references);
	gchar *pch2 = strchr (pch, '>');

	if (pch2)
		pch2[1] = '\0';
	else {
		g_free (pch);
		pch = g_strdup (adata->message_id);
	}

	return pch;
}

static void
article_get_entire_thread_count (article_data* top, int* qty)
{
	GSList *l;
	++*qty;
	for (l=top->threads; l!=NULL; l=l->next)
		article_get_entire_thread_count (ARTICLE_DATA(l->data), qty);
}
static void
article_get_entire_thread_impl (article_data* top, article_data** a, int *idx)
{
	GSList *l;
	a[*idx] = top;
	++*idx;
	for (l=top->threads; l!=NULL; l=l->next)
		article_get_entire_thread_impl (ARTICLE_DATA(l->data), a, idx);
}

void
article_get_entire_thread (
	article_data* adata,
	int* setme_qty,
	article_data*** setme_array)
{
	article_data* top = adata;

	*setme_array = NULL;
	*setme_qty = 0;

	if (top != NULL)
	{
		int qty = 0;
		int index = 0;
		article_data ** buf = NULL;

		while (top->parent != NULL)
			top = top->parent;

		article_get_entire_thread_count (top, &qty);
		buf = g_new (article_data*, qty);
		article_get_entire_thread_impl (top, buf, &index);
		g_assert (index == qty);

		*setme_array = buf;
		*setme_qty = qty;
	}
}
