/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <stdlib.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gnome-uidefs.h>

#include <stdio.h>

#include "message.h"
#include "message-window.h"
#include "globals.h"
#include "gui.h"
#include "prefs.h"
#include "print.h"
#include "text.h"
#include "util.h"


/* private structure for a print dialog */
typedef struct {
	GtkWidget *window; /* Gnomedialog */

	GtkWidget *print_command; /* entry box, should likely be 'lpr' */

	Message *msg;
} PrintDialog;


static void print_dialog_print_clicked (PrintDialog *dialog);
static void print_dialog_destroy (PrintDialog *dialog);
static gint print_execute (gchar *command, gchar *message);
static void print_dialog_cb (GtkWidget *widget, gint num, gpointer data);
static void print_dialog (Message *msg);


/* Build and execute the actual print command */
/* DESIGN:
 * 1. open a file pointer to a temp (save in ~/.pan/tmp) file
 * 2. dump the message to the file
 * 3. run the print command on the temporary file with system()
 * 4. unlink() the temp file
 *
 * Return value: 0 for success, -1 for any error encountered
 */
static gint
print_execute (gchar *command, gchar *message)
{
	gchar *tmp_filename = NULL;
	gchar *fullcmd=NULL; /* full command line for system() */
	FILE *fp;

	tmp_filename = g_strdup_printf ("%s/printme", temp_dir);

	/* open the temp file */
	fp = fopen (tmp_filename, "wra+");
	if (!fp) {
		g_warning ("print_execute(): error in fopen()\n");
		g_free (tmp_filename);
		return -1;
	}
	/* dump the message to the file */
	fputs (message, fp);
	fflush (fp);
	fclose (fp);

	/* run the actual print command */
	fullcmd = g_strdup_printf ("%s %s", command, tmp_filename);
	g_free (tmp_filename);
	g_free (fullcmd);
	if (system (fullcmd) == -1) {
		g_warning ("print_execute(): error in system()\n");
		return -1;
	}

	/* delete the temp file */
	if (remove(tmp_filename)) {
		g_print ("print_execute(): error in unlink()\n");
		return -1;
	}

	return 0;
}


/* Destroy the dialog and free used memory */
static void
print_dialog_destroy (PrintDialog *dialog)
{
	gtk_widget_destroy (dialog->window);
	message_free (dialog->msg);
	g_free (dialog);
}


/* The user pressed the Print button, build the message and call for it
 * to be printed with the provided print command
 */
static void
print_dialog_print_clicked (PrintDialog *dialog)
{
	gchar *command;
	gchar *message;
	Message *msg;

	g_return_if_fail (dialog != NULL);

	msg = dialog->msg;

	/* FIXME: I don't think msg->organization is ever set, because the
	   article-db doesn't ever save/load this field */
	if ( !msg->organization )
		msg->organization = g_strdup ("None");

	message = g_strconcat (_("From: "), msg->from, "\n",
			       _("Organization: "), msg->organization, "\n",
			       _("Newsgroups: "), msg->newsgroups, "\n",
			       _("Date: "), msg->date, "\n",
			       _("Subject: "), msg->subject, "\n\n",
			       msg->body, NULL);

	command = gtk_editable_get_chars (GTK_EDITABLE (dialog->print_command), 0, -1);

	if ( print_execute (command, message) )
		pan_error_dialog ("Printing of message '%s' failed!", msg->subject);
	else
		gnome_config_set_string ("/Pan/Print/print_command", command);

	/* clean up */
	print_dialog_destroy (dialog);
	g_free (message);
	g_free (command);
}


/* The user pressed a button. Lets like, do things and stuff. Dude. */
static void
print_dialog_cb (GtkWidget *widget, gint num, gpointer data)
{
	g_return_if_fail (data);

	switch (num)
	{
	case 0: /* Print */
		print_dialog_print_clicked (data);
		break;
	case 1: /* Cancel */
		print_dialog_destroy (data);
		break;
	default:
		break;
	}
}


/* Make the dialog with Print and Cancel buttons */
static void
print_dialog (Message *msg)
{
	GtkWidget *table;
	GtkWidget *label;
	gchar *p;
	PrintDialog *dialog;

	dialog = g_malloc0 (sizeof (PrintDialog));
	dialog->window = gnome_dialog_new (_("Print"),
					   _("Print"),
					   GNOME_STOCK_BUTTON_CANCEL, NULL);
	dialog->msg = message_copy (msg);
	
	table = gtk_table_new (1, 2, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);
        gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
        gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);

	label = gtk_label_new (_("Print Command:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 4, 0);

	dialog->print_command = gtk_entry_new ();
	if ((p = gnome_config_get_string ("/Pan/Print/print_command")) != NULL)
		gtk_entry_set_text (GTK_ENTRY (dialog->print_command), p);
	else
		gtk_entry_set_text (GTK_ENTRY (dialog->print_command), "lpr");
	g_free (p);
	gtk_table_attach (GTK_TABLE (table), dialog->print_command, 1, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 4, 0);

	gtk_container_add (GTK_CONTAINER (GNOME_DIALOG(dialog->window)->vbox), table);
	gtk_signal_connect (GTK_OBJECT (dialog->window), "clicked",
			    GTK_SIGNAL_FUNC (print_dialog_cb), dialog);

	gui_popup_draw (
		dialog->window,
	       	msg->window ? msg->window->window : Pan.window);
}


/**
 * print_cb:
 * @data: the Message* to be printed
 *
 * The only public routine, gets called from the Print button on the toolbar
 * of a message window or File->Print from any menu in Pan.
 **/
void
print_cb (GtkWidget *widget, gpointer data)
{
	if (data)
		print_dialog (data);

	/* if they aren't printing from a message window, use the current
	   article in the message pane/page */
	else if ( !data && current_article ) {
		Message *msg = g_malloc0 (sizeof (Message));
		message_import_adata (msg, current_article);
		print_dialog (msg);
	}
	else if ( !data && !current_article ) {
		pan_error_dialog (_("You haven't loaded an article to print."));
		return;
	}
}
