/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <glib.h>

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-config.h>

#include "article.h"
#include "article-db.h"
#include "articlelist.h"
#include "date.h"
#include "debug.h"
#include "log.h"
#include "grouplist.h"
#include "queue-item-grouplist.h"
#include "util.h"

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

static gint queue_item_grouplist_run (QueueItem* item);

static int nntp_grouplist_download_all (QueueItemGrouplist* item, GSList** setme);

static int nntp_grouplist_download_new (QueueItemGrouplist* item, GSList** setme);

static gchar* queue_item_grouplist_describe (const StatusItem* item);

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

extern gboolean must_exit;

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PUBLIC ROUTINES
************/

PanObject*
queue_item_grouplist_new (server_data *sdata,
	                  gboolean high_priority,
                          GrouplistDownloadType dl_type)
{
	QueueItemGrouplist *item = g_new0 (QueueItemGrouplist, 1);

	debug (DEBUG_PAN_OBJECT, "queue_item_grouplist_new: %p", item);
	
	/* construct superclass... */
	queue_item_constructor (QUEUE_ITEM(item),
		queue_item_destructor,
		queue_item_grouplist_describe,
		queue_item_grouplist_run, sdata, high_priority, TRUE);

	/* construct this class... */
	item->sdata = sdata;
	item->download_type = dl_type;

	return PAN_OBJECT(item);
}

/*****
******
*****/

static gchar*
queue_item_grouplist_describe (const StatusItem* item)
{
	const gchar* action = NULL;

	/* sanity checks */
	g_return_val_if_fail (item!=NULL, NULL);

	switch (QUEUE_ITEM_GROUPLIST(item)->download_type) {
		case GROUPLIST_ALL: action=_("all groups"); break;
		case GROUPLIST_NEW: action=_("new groups"); break;
		default: pan_warn_if_reached(); action=_("BUG IN CODE"); break;
	}

	return g_strdup_printf ( _("Getting %s from %s"),
		action,
		QUEUE_ITEM_GROUPLIST(item)->sdata->name );
}


static gint
queue_item_grouplist_run (QueueItem *item)
{
	QueueItemGrouplist *item_g = QUEUE_ITEM_GROUPLIST(item);
	server_data *sdata = item->sdata;

	status_item_emit_status (STATUS_ITEM(item),
		_("Retrieved 0 groups for '%s'"), sdata->name);

	if (item_g->download_type==GROUPLIST_ALL)
	{
		GSList *glist = NULL;
		int retval = nntp_grouplist_download_all(item_g, &glist);
		if (retval)
			return retval;

		grouplist_free (sdata->grouplist);
		sdata->grouplist = glist;
		check_new_group_list_for_newness (sdata->grouplist);
		grouplist_save (sdata);
		grouplist_update_ui (sdata);
	}
	else if (item_g->download_type==GROUPLIST_NEW)
	{
		GSList *glist = NULL;
		int retval = nntp_grouplist_download_new(item_g, &glist);
		if (retval)
			return retval;

		g_slist_concat (sdata->grouplist, glist);
		g_slist_sort (sdata->grouplist, (GCompareFunc)compare_group_data);
		grouplist_save (sdata);
		grouplist_update_ui (sdata);
	}

	{ /* no special reason for this being bracketed - JEL */
		gchar *buf;
		struct tm *gmt;
		gchar date[32]; /* "YYYYMMDD HHMMSS GMT" */
		time_t secs;

		time (&secs);
		gmt = gmtime (&secs);

		strftime (date, sizeof (date), "%Y%m%d %H%M%S GMT", gmt);

		buf = g_strdup_printf ("/Pan/%s/Last_Grouplist_Update", sdata->name);
		gnome_config_set_string (buf, date);
		gnome_config_sync ();

		g_free (buf);
	}

	/* success */
	return 0;
}


/*---[ nntp_grouplist_download ]--------------------------------------
 *
 *--------------------------------------------------------------------*/
static int
nntp_grouplist_download_all (QueueItemGrouplist *item, GSList** setme)
{
	PanSocket *sock = QUEUE_ITEM(item)->sock;
	const server_data *sdata = QUEUE_ITEM(item)->sdata;
	gboolean get_descriptions = TRUE;
	const char *buffer = NULL;
	GPtrArray *array = NULL;
	GSList *glist = NULL;
	int read_status = 0;
	int count = 0;
	int i = 0;
	
	status_item_emit_status (STATUS_ITEM(item),
		_("Retrieved 0 groups for '%s'"), sdata->name);

	if (pan_socket_putline (sock, "LIST\r\n")) {
		queue_item_emit_sockwrite_err (QUEUE_ITEM(item));
		return QUEUE_ITEM_STATUS_ESOCKET;
	}
	if (pan_socket_getline (sock, &buffer)) {
		queue_item_emit_sockread_err (QUEUE_ITEM(item));
		return QUEUE_ITEM_STATUS_ESOCKET;
	}
	if (atoi(buffer)!=215) { /* 215 = list follows */
		status_item_emit_error (STATUS_ITEM(item),
			_("Group list could not be retrieved\nServer said: %s"),
			buffer);
		return -1;
	}

	array = g_ptr_array_new ();
	while (((read_status = pan_socket_getline (sock, &buffer))) == 0)
	{
		const gchar *pch;
		group_data *group = NULL;
	
		if (!strcmp(buffer,".\r\n"))
			break;
		
		/* init the group */
		group = group_new ();

		/* get the name */
		pch = strchr (buffer, ' ');
		if (!pch) pch = buffer + strlen(buffer);
		group->name = g_memdup (buffer, (pch-buffer)+1);
		group->name[pch-buffer] = '\0';

		/* update the list */
		g_ptr_array_add (array, group);

		/* periodic status feedback */
		if (++count % 43 == 0)
		{
			status_item_emit_status (STATUS_ITEM(item),
				_("Retrieved %d groups for '%s'"),
				count, sdata->name);
		}
	}
	if (read_status) {
		/*FIXME: prompt whether or not to keep the list.  For now, leak. */
		queue_item_emit_sockread_err (QUEUE_ITEM(item));
		return QUEUE_ITEM_STATUS_ESOCKET;
	}

	/* sort so that we can bsearch them... */
	qsort (array->pdata, array->len,
	       sizeof(group_data*),
	       compare_group_data_std);

	/* now try to get the descriptions...
	   this is implemented as a second pass because the overlap between
	   LIST and LIST NEWSGROUPS isn't well defined; the latter may only
	   show the groups with a description, moreover, it may not be
	   available on older servers.  To make sure we get all the groups,
	   and descriptions for as many as we can, we use both commands.
	   -- csk */

	if ((read_status=pan_socket_putline (sock, "LIST NEWSGROUPS *\r\n")))
	{
		queue_item_emit_sockwrite_err (QUEUE_ITEM(item));
		get_descriptions = FALSE;
	}
	if (get_descriptions && (read_status=pan_socket_getline (sock, &buffer)))
	{
		queue_item_emit_sockread_err (QUEUE_ITEM(item));
		get_descriptions = FALSE;
	}
	if (get_descriptions && atoi(buffer)!=215) /* 215 = list follows */
	{
		status_item_emit_error (STATUS_ITEM(item),
			_("Group list could not be retrieved\nServer said: %s"),
			buffer);
			get_descriptions = FALSE;
	}
	while (get_descriptions && !read_status)
	{
		const gchar* pch = NULL;
		gchar *name = NULL;
		gchar *description = NULL;

		read_status = pan_socket_getline (sock, &buffer);
		pch = buffer;

		if (read_status)
			continue;

		if (!strcmp(buffer,".\r\n"))
			break;	
		
		while (*pch && !isspace((int)*pch)) ++pch;
		name = g_strndup (buffer, pch-buffer);
		buffer = pch;
		if (( pch = strstr (buffer, "\r\n"))) {
			description = g_strndup (buffer, pch-buffer);
			g_strchug(g_strchomp (description));
		}

		if (name && description) {
			group_data tmp;
			group_data* ptmp = &tmp;
			int index;
			gboolean exact_match = FALSE;
			tmp.name = name;
			index = lower_bound (
				&ptmp, array->pdata,
				array->len, sizeof(group_data*),
				compare_group_data_std,
				&exact_match);
			if (exact_match)
				((group_data*)array->pdata[index])->description = g_strdup (description);
		}

		g_free (name);
		g_free (description);
	}
	if (read_status)
	{
		/*FIXME: prompt whether or not to keep the list.  For now, leak. */
		queue_item_emit_sockread_err (QUEUE_ITEM(item));
		return QUEUE_ITEM_STATUS_ESOCKET;
	}

	/* build the gslist */
	for (i=0; i!=array->len; ++i)
		glist = g_slist_prepend (glist, array->pdata[i]);
	glist = g_slist_reverse (glist);
	g_ptr_array_free (array, TRUE);
	array = 0;

	log_add_va (_("Retrieved %d groups"), count);

	*setme = glist;
	return 0;
}


/*--------------------------------------------------------------------
 * get "new" groups
 * 
 * based on RFC977 - JEL
 *--------------------------------------------------------------------*/
static int
nntp_grouplist_download_new (QueueItemGrouplist *item, GSList** setme)
{
	PanSocket *sock = QUEUE_ITEM(item)->sock;
	const server_data *sdata = QUEUE_ITEM(item)->sdata;
	const char *buffer = NULL;
	group_data *gdata = NULL;
	int count = 0;
	GSList *glist = NULL;
	gchar *buf;
	gchar *datestr;

	status_item_emit_status (STATUS_ITEM(item),
		_("Retrieving 0 new groups for '%s'"), sdata->name);

	/* get the date of the last grouplist update */
	buf = g_strdup_printf (
		"/Pan/%s/Last_Grouplist_Update=19700101 000000 GMT",
		sdata->name);
	datestr = gnome_config_get_string_with_default (buf,  NULL);
	g_free (buf);

	/* ask the server which groups have been added since then */
	if (pan_socket_putline_va (sock, "NEWGROUPS %s\r\n", datestr))
	{
		queue_item_emit_sockwrite_err (QUEUE_ITEM(item));
		g_free (datestr);
		return QUEUE_ITEM_STATUS_ESOCKET;
	}
	if (pan_socket_getline (sock, &buffer))
	{
		queue_item_emit_sockread_err (QUEUE_ITEM(item));
		return QUEUE_ITEM_STATUS_ESOCKET;
	}
	if (atoi (buffer) != 231) /* 231 = list of new groups follows */
	{
		status_item_emit_error (STATUS_ITEM(item),
			_("Group list could not be retrieved\nServer said: %s"),
			buffer);
		return -1;
	}

	if (pan_socket_getline (sock, &buffer))
	{
		queue_item_emit_sockread_err (QUEUE_ITEM(item));
		return QUEUE_ITEM_STATUS_ESOCKET;
	}
	while (strncmp(buffer, ".\r\n", 3))
	{
		const gchar *pch;

		if (must_exit) {
			grouplist_free (glist);
			return -1;
		}

		/* create a group_data */
		gdata = group_new ();

		/* get the name */
		pch = strchr (buffer, ' ');
		if (!pch) pch = buffer + strlen(buffer);
		gdata->name = g_memdup (buffer, (pch-buffer)+1);
		gdata->name[pch-buffer] = '\0';

		/* add the group to the list */
		glist = g_slist_prepend (glist, gdata);

		if (++count % 43 == 0)
		{
			status_item_emit_progress (STATUS_ITEM(item), count);

			status_item_emit_status (STATUS_ITEM(item),
				_("Retrieving %d new groups for '%s'"), 
				count, sdata->name);
		}

		if (pan_socket_getline (sock, &buffer))
		{
			queue_item_emit_sockread_err (QUEUE_ITEM(item));
			grouplist_free (glist);
			return QUEUE_ITEM_STATUS_ESOCKET;
		}
	}

	log_add_va (_("Retrieved %d new groups"), count);

	status_item_emit_status (STATUS_ITEM(item),
		_("Sorting %d news groups for '%s'"), count, sdata->name);
	glist = g_slist_sort ( glist, (GCompareFunc)compare_group_data );

	*setme = glist;
	return 0;
}
