/* Main function for gpasm
   Copyright (C) 1998 James Bowman

This file is part of gpasm.

gpasm is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

gpasm is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with gpasm; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"

#include "gpasm.h"
#include "symbol.h"
#include "opcode.h"
#include "lst.h"
#include "hex.h"

struct gpasm_state state = {
    16,		/* radix */
    inhx8m,	/* hex_format */
    1,		/* case_insensitive */
    0		/* quiet */
};

int yyparse(void);

int main( int argc, char *argv[] )
{
  int c;
  int usage = 0;

  /* #defines are case-insensitive */
  state.stDefines = push_symbol_table(NULL, 1);

  while ((c = getopt(argc, argv, "?a:cd:e:hp:qr:w:v")) != EOF) {
    switch (c) {
    case '?':
    case 'h':
      usage = 1;
      break;
    case 'a':
      if (strcasecmp(optarg, "inhx8m") == 0)
	state.hex_format = inhx8m;
      else if (strcasecmp(optarg, "inhx8s") == 0)
	state.hex_format = inhx8s;
      else if (strcasecmp(optarg, "inhx32") == 0)
	state.hex_format = inhx32;
      else {
	fprintf(stderr,
		"Error: unrecognised hex file format \"%s\"\n",
		optarg);
      }
      break;
    case 'c':
      state.case_insensitive = 0;
      break;
    case 'd':
      if ((optarg != NULL) && (strlen(optarg) > 0)) {
	struct symbol *sym;
	char *lhs, *rhs;

	lhs = strdup(optarg);
	rhs = strchr(lhs, '=');
	if (rhs != NULL) {
	  *rhs = '\0';	/* Terminate the left-hand side */
	  rhs++;	/* right-hand side begins after the '=' */
	}

	sym = get_symbol(state.stDefines, lhs);
	if (sym == NULL)
	  sym = add_symbol(state.stDefines, lhs);
	if (rhs)
	  annotate_symbol(sym, rhs);
      }
      break;
    case 'p':
      select_processor(optarg);
      break;
    case 'q':
      state.quiet = 1;
      break;
    case 'r':
      select_radix(optarg);
      break;
    case 'v':
      fprintf(stderr, "%s\n", GPASM_VERSION_STRING);
      exit(0);
      
    }
    if (usage)
      break;
  }
  
  if (optind < argc)
    state.srcfilename = argv[optind];
  else
    usage = 1;

  if (usage) {
      printf("\n");
      printf("usage: gpasm <options> <filename>\n");
      printf("Where <options> are:\n");
      printf(" -a <fmt>       Select hex format inhx8m, inhx8s, inhx32\n");
      printf(" -c             case sensitive \n");
      printf(" -d <sym>=<val> Define symbol with value\n");
      printf(" -h             Show this usage message \n");
      printf(" -p <processor> Select processor\n");
      printf(" -q             Quiet \n");
      printf(" -r <radix>     Select radix \n");
      printf(" -v             Show version \n");
      printf("\n");
      exit(0);
  }

  /* Builtins and macros are always case insensitive */
  state.stBuiltin = push_symbol_table(NULL, 1);
  state.stMacros = push_symbol_table(NULL, 1);
  state.stTop = 
    state.stGlobal = push_symbol_table(NULL, state.case_insensitive);

  opcode_init(0);

  state.maxram = (MAX_RAM - 1);

  open_file(SPECIAL_PATH);

  state.pass = 1;

  yyparse();

  assert(state.pass == 2);
  
  pop_symbol_table(state.stBuiltin);
  
  dump_hex();

  /* Maybe produce a symbol table */
  if (state.lst.symboltable) {
    lst_throw(); /* Start symbol table on a fresh page */
    dump_symbol_table(state.stGlobal);
  }

  /* Finish off the listing */
  lst_close();

  if (state.num.errors > 0)
    return 1;
  else
    return 0;
}
