# Module: HandleCfgFile.tcl
# 26.1.96 T.Niederreiter
#
# External called funktions:
# - loadcfgfile{}: loads all global variables from config-file
#
# - savecfgfile{}: saves all global variables in config-file
#
# - loadcdparanoiaconf { filename } : load CDParanoia-Config-file
# 
# - createcdparanoiaconf { filename }: create initial CDParanoia-Config-file 
#

# Opens the configuration file $CFGFILE and reads it line by line.
# When we find a line starting with "XCDR" we treat this line as a line
# containing a variablename and a variablevalue seperated by a "=".
# All lines not starting with "XCDR" are ignored.
# We extract the variablename and the value (the value can be set in 
# double-quotes (")). Then we set this variable to that value in the
# global context. (So you can access it from the whole program.) 

proc loadcfgfile { filename } {
global CFGFILE
global str1 str2

	set opnflg [catch { set fileid [open $filename r] }]

	if { $opnflg != 0 } {
	#	puts "Error opening $filename!"
		return -1
	}

	set str1 ""
	set str2 ""

	while {	[gets $fileid cfgline] >= 0 } {
		if { [string range $cfgline 0 3] == "XCDR" } {
			set mrk1 [string first "=" $cfgline]
			incr mrk1 -1
			# Set str1 to variablename 
			set str1 [string trim [string range $cfgline 0 $mrk1]]
			incr mrk1 2
			# Set str2 to variablevalue
			set str2 [string trim [string range $cfgline $mrk1 end]]
		 	set str2 [string trim $str2 \"]	

			# set the variables read from file in global context
			uplevel #0 { eval set $str1 \"$str2\" }
		}
	}
	unset str1 str2
	close $fileid
	return 0
}	

# Creates (or overwrites) the configuration file.
# We write each global variable starting with XCDR together with its
# value in a file. Each line is written in a syntax readable by 
# "loadcfgfile". The variables are alphabetically sorted.
 
proc savecfgfile { filename } {
eval global [info globals XCDR*]
global CFGFILE

	set varlist [lsort [info globals XCDR*]]

	set fileid [open $filename w]
	chownfile $filename

	puts $fileid "# X-CD-Roast V$XCDR_VERSION Configuration-File"
	puts $fileid "# Automatically created by the X-CD-Roast-Setup"
	puts $fileid "# Don't edit! (Unless you really know what you are doing)"
	puts $fileid ""

	foreach varname $varlist {
		eval set varvalue \$$varname
		fixstring varvalue
		puts $fileid "$varname = \"$varvalue\""
	}

	close $fileid
}


# load the cdparanoia-config-file. 
# set the global vars CDPARANOIA and CDPARAOPTIONS

proc loadcdparanoiaconf { filename } {
global CDPARANOIA CDPARAOPTIONS

	set CDPARANOIA ""
	set CDPARAOPTIONS ""

	set opnflg [catch { set fileid [open $filename r] }]

	if { $opnflg != 0 } {
	#	puts "Error opening $filename!"
		return -1
	}

	while {	[gets $fileid cfgline] >= 0 } {

		set cfgline [string trim $cfgline]
		if { [string index $cfgline 0] == "#" || $cfgline == "" } {
			# skip comment and empty-lines
			continue
		}

		# parsing here
		set lparam [split $cfgline "="]
		if { [string trim [lindex $lparam 0]] == "CDPARANOIA" } {
			set found [string trim [lindex $lparam 1]]
			set CDPARANOIA "$found"
		}
		if { [string trim [lindex $lparam 0]] == "OPTIONS" } {
			set found [string trim [lindex $lparam 1]]
			set CDPARAOPTIONS "$found"
		}
	}

	close $fileid

	# now check if valid options	
	if { $CDPARANOIA == "" } {
		puts "Error parsing $filename: No CDPARANOIA-path set\n"
		exit
	}
	# binary valid and executable?
	if { [file executable $CDPARANOIA] == 0 } {
		puts "Error parsing $filename: $CDPARANOIA is not a valid executable\n"
		exit
	}
	if { $CDPARAOPTIONS == "" } {
		puts "Error parsing $filename: No CDParanoia-Options given\n"
		exit
	}
	return 0
}


# create the initial CDParanoia-Config-File. This file should then
# edited by the user to suit his needs.

proc createcdparanoiaconf { filename } {


	set fileid [open $filename w]
	chownfile $filename

	set hash "#"
	set CONFFILEDATA " 
$hash
$hash Configuration-file for the CDParanoia-Mode of X-CD-Roast 0.96e
$hash
$hash You have to edit this file manually to try different CDParanoia-Options.
$hash

$hash Set this to point to your installed version of CDParanoia
CDPARANOIA=/usr/local/bin/cdparanoia

$hash
$hash Here edit ONE OPTIONS-string (comment all others with an #) to
$hash control how CDParanoia is going to do its job.
$hash
$hash Try some of this options to enhance the quality of the generated
$hash audio-tracks:
$hash
$hash  -Z --disable-paranoia           : disable all paranoia checking
$hash  -Y --disable-extra-paranoia     : only do cdda2wav-style overlap checking
$hash  -X --disable-scratch-detection  : do not look for scratches
$hash  -W --disable-scratch-repair     : disable scratch repair (still detect)

$hash For ATAPI-CDROMs: 
$hash     (the -d \$blkname-option will transformed into the correct 
$hash     block-device by X-CD-Roast. (e.g. it will become -d /dev/sr1))
OPTIONS=\"-d \$blkname -Z\"

$hash For SCSI-CDROMs or ATAPI-SCSI-Emulated CDROMs:
$hash     (the -g \$genname-option will transformed into the correct
$hash     generic-scsi-device by X-CD-Roast. (e.g. it will become -g /dev/sg0))
$hash OPTIONS=\"-g \$genname -Z\"
"

	puts $fileid $CONFFILEDATA
	close $fileid
}
